/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

/*------------------------------------------------------------------------
            Include Files
------------------------------------------------------------------------*/
#include <string.h>

#include "MC_regheaders.h"
#include "../ddr_utils.h"
#include "../ddr_config.h"
#include "ddr_phy_config.h"
#include "ddr_phase_tuning.h"
#include "ddr_write_leveling.h"
#include "ddr4_training.h"
#include "ddr_read_leveling.h"
#include "ID_utils.h"

/*------------------------------------------------------------------------
        Defines
------------------------------------------------------------------------*/
#define DUMP_MC_REGS
//#define R4MEMTEST

//#define msg(x,y,...) minPrintf(__VA_ARGS__)
#define msg(x,y,...) 

#define DDR4_VREF_PER_DRAM 51
#define DDR4_VREF_PER_PHY 40

/*------------------------------------------------------------------------
        Globals
------------------------------------------------------------------------*/


/*------------------------------------------------------------------------
        ddr_config_and_init
------------------------------------------------------------------------*/
/**
 *
 * \brief Call calibration routines to program the DDR and memory controller
 *
 * \param const DDR_config_t *MCConfig - pointer to the config values
 *                          to be used to program the MC
 *        cs_sa_t *startAddrs - starting address structure
 *        uint32_t mem_width - width of memory
 *        MC_REGS_t *mem_ctrl_regs - memory control register structure
 *        uint32_t ddrClkPd - period of DDR clock
 * \return void (no return value)
 *
 *
 */
void ddr_config_and_init_gs2(const DDR_config_t *MCConfig, cs_sa_t *startAddrs, uint32_t mem_width, MC_REGS_t *mem_ctrl_regs, uint32_t ddrClkPd)
{
    uint32_t dllTuneFail;

    set_DDR_configuration(MCConfig, startAddrs);
    msg(MSG_INFO, RAW_DATA, "DDR I/O Width: x%d\r\n", mem_width);
    write_level(mem_ctrl_regs, 0, mem_width);  // CS0 always valid 
    msg(MSG_INFO, RAW_DATA, "write leveling\r\nmem_ctrl_regs->CH0_PHY_WL_DATA_Control_CS0_B0  0x%08x\r\nmem_ctrl_regs->CH0_PHY_WL_DATA_Control_CS0_B1  0x%08x\r\n", mem_ctrl_regs->CH0_PHY_WL_DATA_Control_CS0_B0, mem_ctrl_regs->CH0_PHY_WL_DATA_Control_CS0_B1);


    
    if (!is_ddr4()) {
        uint8_t readlevelretry = 1;
            while ((read_leveling(mem_ctrl_regs, mem_width, startAddrs) != 0) && (readlevelretry < 6 ))
            {
                if (readlevelretry <= 5) 
                {
                    msg(MSG_INFO, RAW_DATA, "Failed Read Leveling Attempt %d.  Retrying.\r\n", readlevelretry);
                }
                readlevelretry++;
            } 

            if (readlevelretry >= 6) 
            {
                msg(MSG_INFO, RAW_DATA, "Failed Read Leveling.  Giving up.\r\n");
            }
    }

    dllTuneFail = dll_read_phase_tuning(mem_ctrl_regs, mem_width, startAddrs);

    // if dll_read_phase_tuning failed, do not VREF train DDR4.
    if (dllTuneFail == 0 && is_ddr4()) {
      dram_vref_training(mem_ctrl_regs, mem_width, DDR4_VREF_PER_DRAM);
      phy_vref_training(mem_ctrl_regs, mem_width, DDR4_VREF_PER_PHY);
    }

    #ifdef DUMP_MC_REGS
        DumpMcRegs(mem_ctrl_regs, startAddrs);
    #endif

    #ifdef R4MEMTEST
        uint32_t rc = DMA_DDR_Memory_Test ( 1, 1, 0, 0, 0, 0);          // AllNoise pattern, 1 loop, address 0x0
        msg(MSG_STATUS, RAW_DATA, "DMA_DDR_Memory_Test rc = 0x%0x\r\n", rc);
    #endif
}
