/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/
/*------------------------------------------------------------------------
            Include Files
------------------------------------------------------------------------*/
#include <string.h>

#include "MC_regheaders.h"
#include "../ddr_utils.h"
#include "cpu_api.h"

/*------------------------------------------------------------------------
        Defines
------------------------------------------------------------------------*/

//#define msg(x,y,...) minPrintf(__VA_ARGS__)
#define msg(x,y,...) 


#define UTF_waitForMilliSecs(x) cpu_spin_delay(1000 * x)
#define UTF_waitForMicroSecs(x) cpu_spin_delay(x)

#define MIN_DLL_WINDOW 10

/*------------------------------------------------------------------------
        Globals
------------------------------------------------------------------------*/
extern uint32_t __start_of_free_DDR__;  //from ld file



/*------------------------------------------------------------------------
        dll_read_phase_tuning
------------------------------------------------------------------------*/
/**
 *
 * \brief Adjust phase delay of DQS to center of the DQs
 *
 * \param const MC_REGS_t *mc_reg - pointer to memory controller registers
 * \return void (no return value)
 *
 *
 */
uint32_t dll_read_phase_tuning_gr2(MC_REGS_t *mc_reg, int mem_width, cs_sa_t *startAddrs)
{
    uint32_t i = 0;
    uint8_t j = 0;
    uint8_t k = 0;
    uint8_t cs = 0;
    uint32_t DLL_Control0;
    uint32_t DLL_Control1;
    uint32_t DLL_Control2;
    uint32_t DLL_Control3;
    uint8_t firstPass[8] = {0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff};  // {B0_P, B1_P, B2_P, B3_P, B0_N, B1_N, B2_N, B3_N}
    uint8_t lastPass[8] = {0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff};   // {B0_P, B1_P, B2_P, B3_P, B0_N, B1_N, B2_N, B3_N}
    uint8_t passArray[3][4][2][0x40];  // CS[0:2], B[3:3], Pos-edge/Neg-edge, DLL Settings
    uint8_t phsel[8];
    uint32_t dataWord[3][16];    // 16 data words per CS
    uint32_t expData[3][16];
    uint8_t csVld[3];
    uint32_t dataMask = 0x000000FF;
    uint32_t dataMaskn = 0x000000FF;
    uint32_t pattern = 0xAAAAAAAA;
    uint32_t pattern1 = 0xFFFFFFFF;
    uint32_t mapRecord[3];
    uint32_t rc = 0;

    //record map register
    mapRecord[0] = mc_reg->CH0_MMAP0;
    mapRecord[1] = mc_reg->CH0_MMAP1;
    mapRecord[2] = mc_reg->CH0_MMAP2;

    //set new mapping
    mc_reg->CH0_MMAP0 = 0x000d0001;
    mc_reg->CH0_MMAP1 = 0x200d0001;
    mc_reg->CH0_MMAP2 = 0x400d0001;

    if(mem_width == 16)
    {
        pattern = 0xAAAA5555;
        pattern1 = 0xFFFF0000;
    }

    memset(passArray,0x0,sizeof(passArray));

    volatile uint32_t* freeDDR_CS0 = &__start_of_free_DDR__;
    volatile uint32_t* freeDDR_CS1 = (uint32_t*)0x20002000;
    volatile uint32_t* freeDDR_CS2 = (uint32_t*)0x40004000;

    //msg(MSG_INFO, RAW_DATA, "pre tuning\r\nmc_reg->CH0_PHY_DLL_control_B0  0x%08x\r\nmc_reg->CH0_PHY_DLL_control_B1  0x%08x\r\nmc_reg->CH0_PHY_DLL_control_B2  0x%08x\r\nmc_reg->CH0_PHY_DLL_control_B3  0x%08x\r\n", mc_reg->CH0_PHY_DLL_control_B0, mc_reg->CH0_PHY_DLL_control_B1, mc_reg->CH0_PHY_DLL_control_B2,mc_reg->CH0_PHY_DLL_control_B3 );

    // Determine which CS are valid
    for (i=0; i<3; i++)
    {
        if (startAddrs->sa[i] != 0xDEADBEEF)
            csVld[i] = 1;
        else
            csVld[i] = 0;
    }

    if (csVld[0])  // CS0 is valid
    {
        expData[0][0] = pattern;
        expData[0][1] = ~pattern;
        expData[0][2] = pattern;
        expData[0][3] = ~pattern;
        expData[0][4] = ~pattern;
        expData[0][5] = pattern;
        expData[0][6] = ~pattern;
        expData[0][7] = pattern;
        expData[0][8] = pattern1;
        expData[0][9] = ~pattern1;
        expData[0][10] = pattern1;
        expData[0][11] = ~pattern1;
        expData[0][12] = ~pattern1;
        expData[0][13] = pattern1;
        expData[0][14] = ~pattern1;
        expData[0][15] = pattern1;
    }
    if (csVld[1])  // CS1 is valid
    {
        expData[1][0] = pattern;
        expData[1][1] = ~pattern;
        expData[1][2] = pattern;
        expData[1][3] = ~pattern;
        expData[1][4] = ~pattern;
        expData[1][5] = pattern;
        expData[1][6] = ~pattern;
        expData[1][7] = pattern;
        expData[1][8] = pattern1;
        expData[1][9] = ~pattern1;
        expData[1][10] = pattern1;
        expData[1][11] = ~pattern1;
        expData[1][12] = ~pattern1;
        expData[1][13] = pattern1;
        expData[1][14] = ~pattern1;
        expData[1][15] = pattern1;
    }
    if (csVld[2])  // CS2 is valid
    {
        expData[2][0] = pattern;
        expData[2][1] = ~pattern;
        expData[2][2] = pattern;
        expData[2][3] = ~pattern;
        expData[2][4] = ~pattern;
        expData[2][5] = pattern;
        expData[2][6] = ~pattern;
        expData[2][7] = pattern;
        expData[2][8] = pattern1;
        expData[2][9] = ~pattern1;
        expData[2][10] = pattern1;
        expData[2][11] = ~pattern1;
        expData[2][12] = ~pattern1;
        expData[2][13] = pattern1;
        expData[2][14] = ~pattern1;
        expData[2][15] = pattern1;
    }

    // Write data
    if (csVld[0])  // CS0 is valid
        xfer_DMA_DATA(expData[0],freeDDR_CS0, 64);
    if (csVld[1])  // CS1 is valid
        xfer_DMA_DATA(expData[1],freeDDR_CS1, 64);
    if (csVld[2])  // CS2 is valid
        xfer_DMA_DATA(expData[2],freeDDR_CS2, 64);

    //shmoo dll_phsel values
    for( i = 0; i < 0x40; i++ )
    {
        DLL_Control0 = MC_CH0_PHY_DLL_CONTROL_B0_DLL_PHSEL_REPLACE_VAL(mc_reg->CH0_PHY_DLL_control_B0, i);
        mc_reg->CH0_PHY_DLL_control_B0 = MC_CH0_PHY_DLL_CONTROL_B0_DLL_PHSEL1_REPLACE_VAL(DLL_Control0, i);
        DLL_Control1 = MC_CH0_PHY_DLL_CONTROL_B1_DLL_PHSEL_REPLACE_VAL(mc_reg->CH0_PHY_DLL_control_B1, i);
        mc_reg->CH0_PHY_DLL_control_B1 = MC_CH0_PHY_DLL_CONTROL_B1_DLL_PHSEL1_REPLACE_VAL(DLL_Control1, i);
        if(mem_width == 32)
        {
            DLL_Control2 = MC_CH0_PHY_DLL_CONTROL_B2_DLL_PHSEL_REPLACE_VAL(mc_reg->CH0_PHY_DLL_control_B2, i);
            mc_reg->CH0_PHY_DLL_control_B2 = MC_CH0_PHY_DLL_CONTROL_B2_DLL_PHSEL1_REPLACE_VAL(DLL_Control2, i);
            DLL_Control3 = MC_CH0_PHY_DLL_CONTROL_B3_DLL_PHSEL_REPLACE_VAL(mc_reg->CH0_PHY_DLL_control_B3, i);
            mc_reg->CH0_PHY_DLL_control_B3 = MC_CH0_PHY_DLL_CONTROL_B3_DLL_PHSEL1_REPLACE_VAL(DLL_Control3, i);
        }

        //ensure write is complete
        while((MC_CH0_PHY_DLL_CONTROL_B0_DLL_PHSEL_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B0)) != i);
        while((MC_CH0_PHY_DLL_CONTROL_B0_DLL_PHSEL1_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B0)) != i);
        while((MC_CH0_PHY_DLL_CONTROL_B1_DLL_PHSEL_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B1)) != i);
        while((MC_CH0_PHY_DLL_CONTROL_B1_DLL_PHSEL1_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B1)) != i);
        if(mem_width == 32)
        {
            while((MC_CH0_PHY_DLL_CONTROL_B2_DLL_PHSEL_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B2)) != i);
            while((MC_CH0_PHY_DLL_CONTROL_B2_DLL_PHSEL1_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B2)) != i);
            while((MC_CH0_PHY_DLL_CONTROL_B3_DLL_PHSEL_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B3)) != i);
            while((MC_CH0_PHY_DLL_CONTROL_B3_DLL_PHSEL1_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B3)) != i);
        }

        mc_reg->CH0_PHY_Control_9 = MC_CH0_PHY_CONTROL_9_PHY_DLL_RST_MASK;
        UTF_waitForMicroSecs(5);
        mc_reg->CH0_PHY_Control_9 = MC_CH0_PHY_CONTROL_9_DLL_UPDATE_EN_PULSE_MASK;
        UTF_waitForMicroSecs(5);

        // Read Data
        if (csVld[0])  // CS0 is valid
            xfer_DMA_DATA(freeDDR_CS0, dataWord[0], 64);
        if (csVld[1])  // CS1 is valid
            xfer_DMA_DATA(freeDDR_CS1, dataWord[1], 64);
        if (csVld[2])  // CS2 is valid
            xfer_DMA_DATA(freeDDR_CS2, dataWord[2], 64);

        //check for read error
        if(mem_width == 32)
        {
            for(j=0;  j<4; j++)  // Per byte lane
            {
                dataMask = 0x000000FF << 8*j;
                //Check reads
                for (cs=0; cs<3; cs++)
                {
                    if (csVld[cs])
                    {
                        //msg(MSG_DEBUG, RAW_DATA, "DLL Tuning for CS %0d, i = %0d, dataMask = 0x%08x\r\n",cs,i,dataMask);
                        for (k=0; k<8; k++)
                        {
                            //if ((j==0) && (i==0 || i==0x1f || i==0x3f))
                            //  msg(MSG_DEBUG, RAW_DATA, "i = 0x%0x: dataWord[%0d][%0d] = 0x%08x, dataWord[%0d][%0d] = 0x%08x\r\n",i,cs,2*k,dataWord[cs][2*k],cs,((2*k)+1),dataWord[cs][(2*k)+1]);
                            if ((dataWord[cs][2*k] & dataMask) != (expData[cs][2*k] & dataMask))
                                passArray[cs][j][0][i] |= 0x1;
                            if ((dataWord[cs][(2*k)+1] & dataMask) != (expData[cs][(2*k)+1] & dataMask))
                                passArray[cs][j][1][i] |= 0x1;
                        }
                        //msg(MSG_DEBUG, RAW_DATA, "passArray[%0d][%0d][0][%0d] = %0d, passArray[%0d][%0d][1][%0d] = %0d\r\n",cs,j,i,passArray[cs][j][0][i],cs,j,i,passArray[cs][j][1][i]);
                    }
                }
            }
        }
        else
        {
            for(j=0; j<2; j++) {  // per byte lane
                dataMask = 0x000000FF << 8*j;
                dataMaskn = 0x00FF0000 << 8*j;
                //Check reads
                for (cs=0; cs<3; cs++) {
                    if (csVld[cs]) {
                        for (k=0; k<16; k++) {
                            //msg(MSG_DEBUG, RAW_DATA, "dataWord[%0d][%0d] = 0x%08x, dataMask = 0x%08x, dataMaskn = 0x%08x\r\n",cs,k,dataWord[cs][k],dataMask,dataMaskn);
                            if ((dataWord[cs][k] & dataMask) != (expData[cs][k] & dataMask))
                                passArray[cs][j][0][i] |= 0x1;
                            if ((dataWord[cs][k] & dataMaskn) != (expData[cs][k] & dataMaskn))
                                passArray[cs][j][1][i] |= 0x1;
                        }
                        //msg(MSG_DEBUG, RAW_DATA, "passArray[%0d][%0d][0][%0d] = %0d, passArray[%0d][%0d][1][%0d] = %0d\r\n",cs,j,i,passArray[cs][j][0][i],cs,j,i,passArray[cs][j][1][i]);
                    }
                }
            }
        }


        //msg(MSG_DEBUG, RAW_DATA, "0x%02x  0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x    0x%08x 0x%08x 0x%08x 0x%08x\r\n",  i, dataWord[0],dataWord[0][1],dataWord[0][2],dataWord[0][3],dataWord[0][4],dataWord[0][5],dataWord[0][6],dataWord[0][7], pattern, ~pattern, pattern1, ~pattern1);
        //msg(MSG_DEBUG, RAW_DATA, "      0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x\r\n",  dataWord[0][8],dataWord[0][9],dataWord[0][10],dataWord[0][11],dataWord[0][12],dataWord[0][13],dataWord[0][14],dataWord[0][15]);
        //msg(MSG_DEBUG, RAW_DATA, "0x%02x  %d %d %d %d %d %d %d %d\r\n",  i, passArray[0][0][i],passArray[1][0][i],passArray[2][0][i],passArray[3][0][i],passArray[0][1][i],passArray[1][1][i],passArray[2][1][i],passArray[3][1][i]);
    }
    //find passing windows per byte lane
    for(k=0; k<4; k++) {
        if((mem_width == 32) || (k<2)) { // If x16, only do B[1:0]
            for(i=0; i<0x40; i++) {  // Per DLL tap
                // Pos-edge capture DLL setting
                // Only consider a passing value one where all CS pass (since passArray now defaults to 0, any unused CS should pass)
                if((passArray[0][k][0][i] == 0) && (passArray[1][k][0][i] == 0) && (passArray[2][k][0][i] == 0)) {
                    if(firstPass[k] == 0xff)
                        firstPass[k] = i;
                    else if (i == 0x3f) {    // Last DLL setting considered a pass.
                        lastPass[k] = i;
                        if(lastPass[k] - firstPass[k] < MIN_DLL_WINDOW) {
                            firstPass[k] = 0xff;
                            lastPass[k] = 0xff;
                            msg(MSG_DEBUG, RAW_DATA, "BL%0d_P:  Invalid window size of %0d\r\n",k,(lastPass[k] - firstPass[k]));
                        }
                    }
                } else {
                    if((firstPass[k] != 0xff) && (lastPass[k] == 0xff)) {
                        lastPass[k] = i-1;
                        if(lastPass[k] - firstPass[k] < MIN_DLL_WINDOW) {
                            firstPass[k] = 0xff;
                            lastPass[k] = 0xff;
                            msg(MSG_DEBUG, RAW_DATA, "BL%0d_P:  Invalid window size of %0d\r\n",k,(lastPass[k] - firstPass[k]));
                        }
                    }
                }
                //msg(MSG_DEBUG, RAW_DATA, "BL0 0x%02x  0x%02x 0x%02x\r\n",  i, firstPass[0], lastPass[0]);

                // Neg-edge capture DLL setting
                // Only consider a passing value one where all CS pass (since passArray now defaults to 0, any unused CS should pass)
                if((passArray[0][k][1][i] == 0) && (passArray[1][k][1][i] == 0) && (passArray[2][k][1][i] == 0)) {
                    if(firstPass[k+4] == 0xff)
                        firstPass[k+4] = i;
                    else if (i == 0x3f) {    // Last DLL setting considered a pass.
                        lastPass[k+4] = i;
                        if(lastPass[k+4] - firstPass[k+4] < MIN_DLL_WINDOW) {
                            firstPass[k+4] = 0xff;
                            lastPass[k+4] = 0xff;
                            msg(MSG_DEBUG, RAW_DATA, "BL%0d_N:  Invalid window size of %0d\r\n",k,(lastPass[k+4] - firstPass[k+4]));
                        }
                    }
                } else {
                    if((firstPass[k+4] != 0xff) && (lastPass[k+4] == 0xff)) {
                        lastPass[k+4] = i-1;
                        if(lastPass[k+4] - firstPass[k+4] < MIN_DLL_WINDOW) {
                            firstPass[k+4] = 0xff;
                            lastPass[k+4] = 0xff;
                            msg(MSG_DEBUG, RAW_DATA, "BL%0d_N:  Invalid window size of %0d\r\n",k,(lastPass[k+4] - firstPass[k+4]));
                        }
                    }
                }
            }
        }
    }
    msg(MSG_INFO, RAW_DATA, "0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\r\n0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\r\n\r\n",  firstPass[0],firstPass[1],firstPass[2],firstPass[3],firstPass[4],firstPass[5],firstPass[6],firstPass[7],lastPass[0],lastPass[1],lastPass[2],lastPass[3],lastPass[4],lastPass[5],lastPass[6],lastPass[7]);
    //get center of passing windows
    for(i=0;i<8;i++)
    {
        if(firstPass[i] != 0xff)
            phsel[i] = firstPass[i] + (lastPass[i] - firstPass[i])/2;
        else
        {
            if(mem_width==32 || (i==0 || i==1 || i==4 || i==5)) {
                msg(MSG_INFO, RAW_DATA, "Failed DLL Tuning.  Setting to 0x12\r\n");
                rc |= 1;
            }
            phsel[i] = 0x12;
        }

    }

    //update results
    DLL_Control0 = MC_CH0_PHY_DLL_CONTROL_B0_DLL_PHSEL_REPLACE_VAL(mc_reg->CH0_PHY_DLL_control_B0, phsel[0]);
    mc_reg->CH0_PHY_DLL_control_B0 = MC_CH0_PHY_DLL_CONTROL_B0_DLL_PHSEL1_REPLACE_VAL(DLL_Control0, phsel[4]);
    DLL_Control1 = MC_CH0_PHY_DLL_CONTROL_B1_DLL_PHSEL_REPLACE_VAL(mc_reg->CH0_PHY_DLL_control_B1, phsel[1]);
    mc_reg->CH0_PHY_DLL_control_B1 = MC_CH0_PHY_DLL_CONTROL_B1_DLL_PHSEL1_REPLACE_VAL(DLL_Control1, phsel[5]);
    if(mem_width == 32)
    {
        DLL_Control2 = MC_CH0_PHY_DLL_CONTROL_B2_DLL_PHSEL_REPLACE_VAL(mc_reg->CH0_PHY_DLL_control_B2, phsel[2]);
        mc_reg->CH0_PHY_DLL_control_B2 = MC_CH0_PHY_DLL_CONTROL_B2_DLL_PHSEL1_REPLACE_VAL(DLL_Control2, phsel[6]);
        DLL_Control3 = MC_CH0_PHY_DLL_CONTROL_B3_DLL_PHSEL_REPLACE_VAL(mc_reg->CH0_PHY_DLL_control_B3, phsel[3]);
        mc_reg->CH0_PHY_DLL_control_B3 = MC_CH0_PHY_DLL_CONTROL_B3_DLL_PHSEL1_REPLACE_VAL(DLL_Control3, phsel[7]);
    }
    //ensure write is complete
    while((MC_CH0_PHY_DLL_CONTROL_B0_DLL_PHSEL_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B0)) != phsel[0]);
    while((MC_CH0_PHY_DLL_CONTROL_B0_DLL_PHSEL1_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B0)) != phsel[4]);
    while((MC_CH0_PHY_DLL_CONTROL_B1_DLL_PHSEL_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B1)) != phsel[1]);
    while((MC_CH0_PHY_DLL_CONTROL_B1_DLL_PHSEL1_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B1)) != phsel[5]);
    if(mem_width == 32)
    {
        while((MC_CH0_PHY_DLL_CONTROL_B2_DLL_PHSEL_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B2)) != phsel[2]);
        while((MC_CH0_PHY_DLL_CONTROL_B2_DLL_PHSEL1_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B2)) != phsel[6]);
        while((MC_CH0_PHY_DLL_CONTROL_B3_DLL_PHSEL_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B3)) != phsel[3]);
        while((MC_CH0_PHY_DLL_CONTROL_B3_DLL_PHSEL1_MASK_SHIFT(mc_reg->CH0_PHY_DLL_control_B3)) != phsel[7]);
    }

    //pulse reset and enable to make sure new values are used
    mc_reg->CH0_PHY_Control_9 = MC_CH0_PHY_CONTROL_9_PHY_DLL_RST_MASK;
    UTF_waitForMicroSecs(5);
    mc_reg->CH0_PHY_Control_9 = MC_CH0_PHY_CONTROL_9_DLL_UPDATE_EN_PULSE_MASK;
    UTF_waitForMicroSecs(5);

    //restore map register
    mc_reg->CH0_MMAP0 = mapRecord[0];
    mc_reg->CH0_MMAP1 = mapRecord[1];
    mc_reg->CH0_MMAP2 = mapRecord[2];

    msg(MSG_INFO, RAW_DATA, "post tuning\r\nmc_reg->CH0_PHY_DLL_control_B0  0x%08x\r\nmc_reg->CH0_PHY_DLL_control_B1  0x%08x\r\nmc_reg->CH0_PHY_DLL_control_B2  0x%08x\r\nmc_reg->CH0_PHY_DLL_control_B3  0x%08x\r\n", mc_reg->CH0_PHY_DLL_control_B0, mc_reg->CH0_PHY_DLL_control_B1, mc_reg->CH0_PHY_DLL_control_B2,mc_reg->CH0_PHY_DLL_control_B3 );

    return rc;
}


