/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#define DEBUG

#include <stdbool.h>
#include "rom_types.h"
#include "error_types.h"
#include "crypto_callback.h"
#include "validate_code.h"
#include "ID_utils.h"
#include "minPrintf.h"
#include "bootCode.h"

static error_type_t ValidateImage(uint32_t ImageAddr, pImageInfo_t pImageInfo);


void initialize_CRYPTO(rom_types_e rom_ver)
{

}


bool signed_code_mode(void)
{
    rom_types_e rom_type = check_rom_type();
    uint32_t *TIMH       = *((uint32_t **) (((eROM_RevA == rom_type)||(eROM_RevA_bSPI == rom_type)) ? TIMH_BASE_ADDR_FIELD-4 : TIMH_BASE_ADDR_FIELD));

	//return false;
	//return true;

	return (( 1 == TIMH[2] ) ? true : false);
}


bool validate_code_signatures(rom_types_e rom_ver, void *image_info_addr)
{
    pImageInfo_t pImageInfo = (pImageInfo_t) image_info_addr;
    uint32_t ret;

    ENTRY();
    ret = initialize_security (0, rom_ver);
    assert (STATUS_OK == ret);
	return (STATUS_OK == ValidateImage(pImageInfo->LoadAddr, pImageInfo));
}


static error_type_t ValidateImage(uint32_t ImageAddr, pImageInfo_t pImageInfo)
{
    error_type_t Retval = STATUS_OK;
    uint32_t HashSize = 0;
    uint32_t i, ImageSize, HashAlgorithmID;
    uint32_t CalculatedHash[WordLengthOf_SHA512];   // Contains 16 32-bit words
    uint32_t HashInTIM[WordLengthOf_SHA512];        // Contains 16 32-bit words

    // First get relavant image data.
    ImageSize = pImageInfo->ImageSize;  
    HashSize  = pImageInfo->ImageSizeToHash;
    HashAlgorithmID = pImageInfo->HashAlgorithmID; 

    // Make sure we got an expected HashAlgorithmID
    switch (HashAlgorithmID)
    {
        case SHA160:
        case SHA256:
        case SHA512:
            break;
        default:
            return InvalidSecureBootMethodError;
    } 

    if(ImageSize < HashSize)
        return (HashSizeMismatchError);
    if( HashSize == 0 ) 
        return STATUS_OK;

    // Read the image hash into a local variable.
    for (i = 0; i < (HashAlgorithmID / 4); i++)
    {
        HashInTIM[i] = pImageInfo->Hash[i];         // Save the actual hash value
    }

    // Check the return value coming from the SHAMessageDigest function.
    // If there is an error, return with this error code.
    if (STATUS_OK != (Retval = SHA_message_digest((uint8_t*)ImageAddr,
                    HashSize,
                    (uint8_t*) CalculatedHash,
                    (HASHALGORITHMID_T) HashAlgorithmID)))
    {
        dbg_printf("SHA_message_digest FAILED!! Retval: %d\n",Retval);
        return Retval;
    }

    // Compare the 2 hashes
    for (i = 0; i < (HashAlgorithmID / 4); i++)
    {
        dbg_printf("HashInTim[%d]: %08xh CalculatedHash[%d]: %08xh\n",i,HashInTIM[i],i,CalculatedHash[i]);
        if (HashInTIM[i] != CalculatedHash[i])
        {
            dbg_printf("SHA compare FAILED!!\n");
            return (InvalidImageHash);
        }
    }
    dbg_printf("SHA compare Passed %d\n",Retval);

    return Retval;
}

