/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

/**
 * \file sha256_api.h
 * \brief This implements the SHA-256 hashing algorithm.  \
 * This was generated from psudo code found in wikipedia.  
 * The steps for running a hash are:
 *  -# Run sha256_init to get the data structures initialized.
 *  -# Run sha256_input_data to add data to the hash.  This is only used for 
 *      intermediate data and not for final.  Also the amount of data must sent 
 *      to this routine must be divisable by 64.
 *  -# run sha256_finish_data as the last command to send to the hash algorithm.
 *      The result is returned in the Digest variable.
 *  .
 * This can be run any number of times if this proceedure is followed.
 */

#ifndef SHA256_API_H
#define SHA256_API_H

/**
    Structure to provide support for doing the hash
*/

#define SHA256_LENGTH 64    ///< Length that can be processed at a time
#define SHA256_STATE_LENGTH 8  ///< Number of 32 bit words in the intermediate values

typedef struct 
{
	uint64_t	bit_count;  ///< The number of bits that have been processed
	uint32_t	sha256_state[SHA256_STATE_LENGTH]; ///< Intermediate state holding
	uint8_t	    buffer[SHA256_LENGTH];  ///< location to build the last buffer
} sha256_context_t;

/**
 * \brief Initialize the sha-256 structures
 * Prepare to run the sha transform.
 */
void sha256_init(sha256_context_t *buffer);

/**
 * \brief Add blocks of data to the hash
 * This adds blocks of data to the hash.  The data must be in lengths of 64 bytes.
 * \param[in] data Pointer to the data to add to the hash
 * \param[in] length The amount of data.  Must be evenly divisable by 64
 */
void sha256_input_data(uint8_t *data, uint32_t length, sha256_context_t *context);

/**
 * \brief Finish the hash and return the digest.
 * This takes the final block and returns the 256 bit hash of all the data.
 * This must be the last thing run.
 * \param[in] data Pointer to the data to add to the hash.
 * \param[in] length The amount of data to add to the hash.
 * \param[out] digest a pointer to a block of 32 bytes that is to receive the 256 bit hash.
 */
void sha256_finish_data(uint8_t *data, uint32_t length, uint32_t *digest, sha256_context_t *context);



#endif
