/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#include <stdint.h>
#include <stdbool.h>
#include <stddef.h>
#include "sha256_api.h"
#include "bootCode.h"

#define SHA256_LEN_LENGTH 8 ///< Length of the bit length at the end of the block
#define SHA256_SHORT SHA256_LENGTH - SHA256_LEN_LENGTH
#define SHA256_DIGEST_LENGTH 8  // number of 32 bit words in digest.


#define SWAP32(x) ((((x) & 0xff)<<24) | (((x) & 0xff00) << 8) | (((x) & 0xff0000)>>8) | (((x)>>24) & 0xff))

// define a rotate right macro
#define RROT(x,b) (((x) >> (b)) | ((x) << (32 - (b))))
//


// sha256 constants
// (first 32 bits of the fractional parts of the cube roots of the first 
// 64 primes 2..311):

const uint32_t SHA256_constants[64] = 
{
	0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5,
	0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
	0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3,
	0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
	0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc,
	0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
	0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7,
	0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
	0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13,
	0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
	0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3,
	0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
	0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5,
	0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
	0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208,
	0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
};

// Initial hash value  SHA-256
// (first 32 bits of the fractional parts of the square roots of the first 
// 8 primes 2..19):
const static uint32_t sha256_init_value[8] = 
{
	0x6a09e667UL,
	0xbb67ae85UL,
	0x3c6ef372UL,
	0xa54ff53aUL,
	0x510e527fUL,
	0x9b05688cUL,
	0x1f83d9abUL,
	0x5be0cd19UL
};
void sha256_init(sha256_context_t *job_context)
{
  //  job_context = MEM_MALLOC_ALIGNED(sizeof(job_context), cpu_get_dcache_line_size());
  //  ASSERT(job_context != NULL);
    Memcpy(job_context->sha256_state, sha256_init_value, sizeof(sha256_init_value));
    job_context->bit_count = 0;
    Memset(job_context->buffer, 0, sizeof(job_context->buffer));

}
/**
 * \brief Do the actual sha-256 processing on a given block
 * \param[in] context The context created by the init.
 * \param[in] data a block of data to hash, 64 bytes long.
 */
static void sha256_transform(sha256_context_t *context, uint32_t *data) 
{
	uint32_t	a, b, c, d, e, f, g, h, s0, s1;
	uint32_t	t1, t2, maj, ch;
	uint32_t	j;
    uint32_t w[64];

    // 
    //
    // data is a 512 bit chunk
    // first convert 512 bits into 32 bit words
    for(j = 0; j < 16; j++)
    {
        w[j] = SWAP32(data[j]);  // must be run big-endian.
    }
    //
    // now extend the 16 words into 64
    for(j = 16; j < 64; j++)
    {
        s0 = RROT(w[j-15], 7) ^ RROT(w[j-15],18) ^ (w[j-15] >> 3);
        s1 = RROT(w[j-2],17) ^ RROT(w[j-2],19) ^ (w[j-2] >> 10);
        w[j] = w[j-16] + s0 + w[j-7] + s1;

    }


	/* Initialize registers with the prev. intermediate value */
	a = context->sha256_state[0];
	b = context->sha256_state[1];
	c = context->sha256_state[2];
	d = context->sha256_state[3];
	e = context->sha256_state[4];
	f = context->sha256_state[5];
	g = context->sha256_state[6];
	h = context->sha256_state[7];

    // calculate the hash.
    for( j = 0; j < 64; j++)
    {
        s0 = RROT(a, 2) ^ RROT(a, 13) ^ RROT(a, 22);
        maj = (a & b) ^ (a & c) ^ (b & c);
        t2 = s0 + maj;
        s1 = RROT(e, 6) ^ RROT(e, 11) ^ RROT(e, 25);
        ch = (e & f) ^ ((~e) & g);
        t1 = h + s1 + ch + SHA256_constants[j] + w[j];

        h = g;
        g = f;
        f = e;
        e = d + t1;
        d = c;
        c = b;
        b = a;
        a = t1 + t2;

    }

	/* Compute the current intermediate hash value */
	context->sha256_state[0] += a;
	context->sha256_state[1] += b;
	context->sha256_state[2] += c;
	context->sha256_state[3] += d;
	context->sha256_state[4] += e;
	context->sha256_state[5] += f;
	context->sha256_state[6] += g;
	context->sha256_state[7] += h;

}

void sha256_input_data(uint8_t *data, uint32_t length, sha256_context_t *job_context)
{
   // ASSERT(length % SHA256_LENGTH == 0);
    while(length >= SHA256_LENGTH)
    {
        sha256_transform(job_context, (uint32_t *)data);
        data += SHA256_LENGTH;
        length -= SHA256_LENGTH;
        job_context->bit_count += SHA256_LENGTH<<3; // update bit length
    }
}

void sha256_finish_data(uint8_t *data, uint32_t length, uint32_t *digest, sha256_context_t *job_context)
{
//    int32_t left; // amount of space in the buffer.
    uint32_t len;
    uint32_t i;
    uint64_t final_length;
    
    // first do the block length stuff.
    while(length >= SHA256_LENGTH)
    {
        sha256_transform(job_context, (uint32_t *)data);
        data += SHA256_LENGTH;
        length -= SHA256_LENGTH;
        job_context->bit_count += SHA256_LENGTH<<3; // update bit length
    }
    // now get the final data length.

    final_length = job_context->bit_count + (length << 3);    // final bit count.
    // move the rest of the data into our buffer.
    Memcpy(job_context->buffer,data, length);
    len = length;
//    left = SHA256_LENGTH - len;
    // add on the termination for the data.
    job_context->buffer[len++] = 0x80;
    // check to see if the length of the data left enough room for the bit count.
    if(len > SHA256_SHORT)
    {
        // Not enough room, fill the rest of the buffer with zeros and
        // add it to the hash
        //
        Memset(&job_context->buffer[len], 0, SHA256_LENGTH - len);
        sha256_transform(job_context, (uint32_t *)job_context->buffer);
        // reset the counters to init the rest of the buffer.
        len = 0;
//        left = SHA256_LENGTH;
    }
    // now we have enough room for the bit length.  Zero out the buffer 
    // till the bit length boundary.
    Memset(&job_context->buffer[len], 0, SHA256_SHORT - len);
    //
    // Add in the bit length, make it big endian.
    //
    for(i = 0; i < SHA256_LEN_LENGTH; i++)
    {
        job_context->buffer[SHA256_LENGTH - i -1] = 
            ((uint8_t *)&final_length)[i];
    }
    // do the final transform.
    //
    sha256_transform(job_context, (uint32_t *)job_context->buffer);
    //
    // Now write the digest out.
    //
    for(i = 0; i < SHA256_DIGEST_LENGTH; i++)
    {
        digest[i] = job_context->sha256_state[i];
    }
    // free the context.
}
