/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#ifndef __CRYPTO_CALLBACK_H__
#define __CRYPTO_CALLBACK_H__

#include <stdint.h>

#define MAXRSAKEYSIZEWORDS	 	64				// 2048 bits
#define MAXECCKEYSIZEWORDS 		17				// 521 bits+

typedef enum
{
	Marvell_DS =		0x00000000,
	PKCS1_v1_5_Caddo =	0x00000001,
	PKCS1_v2_1_Caddo =	0x00000002,
	PKCS1_v1_5_Ippcp =	0x00000003,
	PKCS1_v2_1_Ippcp =	0x00000004,
	ECDSA_256 =			0x00000005,
	ECDSA_521 =			0x00000006,
	DUMMY_ENALG = 0x7FFFFFFF   // See _Cryptographic_Scheme for additional id's
} ENCRYPTALGORITHMID_T;


typedef enum 
{
	SHA160 = 0x00000014,   //20
	SHA256 = 0x00000020,   //32
	SHA512 = 0x00000040,   //64
	DUMMY_HASH = 0x7FFFFFFF
}
HASHALGORITHMID_T;


typedef struct
{
 ENCRYPTALGORITHMID_T DSAlgorithmID;   // See ENCRYPTALGORITHMID_T
 HASHALGORITHMID_T HashAlgorithmID; // See HASHALGORITHMID_T
 uint32_t KeySize;	          // Specified in bits
 uint32_t Hash[8];			  // Reserve 256 bits for optional key hash
 union 	// Note that this union should not be included as part of the hash for TIM in the Digital Signature
 {
	struct
	{
		uint32_t RSAPublicExponent[MAXRSAKEYSIZEWORDS];
		uint32_t RSAModulus[MAXRSAKEYSIZEWORDS];           	// Up to 2K bits
		uint32_t RSADigS[MAXRSAKEYSIZEWORDS];				// Contains TIM Hash
	}Rsa;

	struct
	{
		uint32_t ECDSAPublicKeyCompX[MAXECCKEYSIZEWORDS]; // Allow for 544 bits (17 words, 68 bytes for use with EC-521)
		uint32_t ECDSAPublicKeyCompY[MAXECCKEYSIZEWORDS];
		uint32_t ECDSADigS_R[MAXECCKEYSIZEWORDS];
		uint32_t ECDSADigS_S[MAXECCKEYSIZEWORDS];
		// Pad this struct so it remains consistent with RSA struct
		uint32_t Reserved[(MAXRSAKEYSIZEWORDS*3)-(MAXECCKEYSIZEWORDS*4)];
	} Ecdsa;
 };
} PLAT_DS, *pPLAT_DS;

typedef uint32_t (*pInitializeSecurity) (uint32_t ver_adv);
typedef uint32_t (*pSHAMessageDigest) (const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, uint8_t* pMesgDigestOut, uint32_t DigestByteLen);
typedef uint32_t (*pPKCS_DSA_Verify)(const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, const pPLAT_DS pDSA, uint8_t* DataBuffer);
typedef uint32_t (*pECCP_DSA_Verify)(const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, const pPLAT_DS pDSA, uint8_t* DataBuffer);
typedef uint32_t (*pGet_NonceBitLen) (void);
typedef uint32_t (*pGet_Nonce) (uint32_t* pNonceOut, uint32_t NonceBitLen);

typedef struct crypto_ops {
    uint32_t (*pInitializeSecurity) (uint32_t ver_adv);
    uint32_t (*pSHAMessageDigest) (const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, uint8_t* pMesgDigestOut, uint32_t DigestByteLen);
    uint32_t (*pPKCS_DSA_Verify)(const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, const pPLAT_DS pDSA, uint8_t* DataBuffer);
    uint32_t (*pECCP_DSA_Verify)(const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, const pPLAT_DS pDSA, uint8_t* DataBuffer);
    uint32_t (*pGet_NonceBitLen) (void);
    uint32_t (*pGet_Nonce) (uint32_t* pNonceOut, uint32_t NonceBitLen);
} crypto_ops_t;

extern uint32_t initialize_security (uint32_t ver_adv, int rom_version);
extern uint32_t SHA_message_digest (const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, uint8_t* pMesgDigestOut, uint32_t DigestByteLen);

#endif //ifndef __CRYPTO_CALLBACK_H__
