/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#include <stddef.h>
#include <stdint.h>
#include "rom_types.h"
#include "crypto_callback.h"

static const struct crypto_ops crypto_ops_cbs[] = {
	[eROM_RevC] = {
		.pInitializeSecurity = (pInitializeSecurity)0xffe10868,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xffe1086c,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xffe108d8,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xffe109d0,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xffe10bc4,
		.pGet_Nonce          = (pGet_Nonce)         0xffe10bcc,
	},
	[eROM_Ge2_RevA] = {
		.pInitializeSecurity = (pInitializeSecurity)0xffe10774,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xffe10778,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xffe107e4,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xffe108dc,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xffe10ad0,
		.pGet_Nonce          = (pGet_Nonce)         0xffe10ad8,
	},
	[eROM_Ge2_RevB] = {
		.pInitializeSecurity = (pInitializeSecurity)0xffe10758,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xffe1075c,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xffe107c8,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xffe108c0,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xffe10ab4,
		.pGet_Nonce          = (pGet_Nonce)         0xffe10abc,
	},
	[eROM_RevC_bSPI_Aspl] = {
		.pInitializeSecurity = (pInitializeSecurity)0xfe010840,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xfe010844,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xfe0108b0,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xfe0109a8,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xfe010b9c,
		.pGet_Nonce          = (pGet_Nonce)         0xfe010ba4,
	},
	[eROM_RevB_bSPI] = {
		.pInitializeSecurity = (pInitializeSecurity)0xfe0103a8,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xfe0103ac,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xfe010418,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xfe010510,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xfe010704,
		.pGet_Nonce          = (pGet_Nonce)         0xfe01070c,
	},
	[eROM_RevB_bSPI_Aspl] = {
		.pInitializeSecurity = (pInitializeSecurity)0xfe010434,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xfe010438,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xfe0104a4,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xfe01059c,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xfe010790,
		.pGet_Nonce          = (pGet_Nonce)         0xfe010798,
	},
	[eROM_RevB1] = {
		.pInitializeSecurity = (pInitializeSecurity)0xffe10460,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xffe10464,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xffe104d0,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xffe105c8,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xffe107bc,
		.pGet_Nonce          = (pGet_Nonce)         0xffe107c4,
	},
	[eFPGA_RevB_19_16] = {
		.pInitializeSecurity = (pInitializeSecurity)0xffe102a8,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xffe102ac,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xffe10318,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xffe10410,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xffe10604,
		.pGet_Nonce          = (pGet_Nonce)         0xffe1060c,
	},
	[eROM_RevA] = {
		.pInitializeSecurity = (pInitializeSecurity)0xffe0d58c,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xffe0d590,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xffe0d604,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xffe0d694,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xffe0d8b0,
		.pGet_Nonce          = (pGet_Nonce)         0xffe0d8b8,
	},
	[eROM_RevA_bSPI_Aspl] = {
		.pInitializeSecurity = (pInitializeSecurity)0xfe00d72c,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xfe00d730,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xfe00d7a4,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xfe00d83c,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xfe00da50,
		.pGet_Nonce          = (pGet_Nonce)         0xfe00da58,
	},
	[eFPGA_RevA] = {
		.pInitializeSecurity = (pInitializeSecurity)0xffe0d3cc,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xffe0d3d0,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xffe0d444,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xffe0d4d4,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xffe0d6f0,
		.pGet_Nonce          = (pGet_Nonce)         0xffe0d6f8,
	},
	[eFPGA_Ge2_RevA] = {
		.pInitializeSecurity = (pInitializeSecurity)0xffe105fc,
		.pSHAMessageDigest   = (pSHAMessageDigest)  0xffe10600,
		.pPKCS_DSA_Verify    = (pPKCS_DSA_Verify)   0xffe1066c,
		.pECCP_DSA_Verify    = (pECCP_DSA_Verify)   0xffe10764,
		.pGet_NonceBitLen    = (pGet_NonceBitLen)   0xffe10958,
		.pGet_Nonce          = (pGet_Nonce)         0xffe10960,
	},
};


static const struct crypto_ops *crypto_ops_cb = NULL;

uint32_t initialize_security (uint32_t ver_adv, int rom_version)
{
	crypto_ops_cb = &crypto_ops_cbs[rom_version];

	return crypto_ops_cb->pInitializeSecurity (ver_adv);
}


uint32_t SHA_message_digest (const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, uint8_t* pMesgDigestOut, uint32_t DigestByteLen)
{
	return crypto_ops_cb->pSHAMessageDigest(pSrcMesgIn, SrcMesgByteLen, pMesgDigestOut, DigestByteLen);
}


uint32_t PKCS_DSA_verify(const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, const pPLAT_DS pDSA, uint8_t* DataBuffer)
{
	return crypto_ops_cb->pPKCS_DSA_Verify(pSrcMesgIn, SrcMesgByteLen, pDSA, DataBuffer);
}

uint32_t ECCP_DSA_verify(const uint8_t* pSrcMesgIn, uint32_t SrcMesgByteLen, const pPLAT_DS pDSA, uint8_t* DataBuffer)
{
	return crypto_ops_cb->pECCP_DSA_Verify(pSrcMesgIn, SrcMesgByteLen, pDSA, DataBuffer);
}


uint32_t get_nonce_bit_len (void)
{
	return crypto_ops_cb->pGet_NonceBitLen();
}


uint32_t get_nonce(uint32_t* pNonceOut, uint32_t NonceBitLen)
{
	return crypto_ops_cb->pGet_Nonce(pNonceOut, NonceBitLen);
}

/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
