/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

/**
 *
 * \file UH_I2CReg88PG870.c
 *
 * \brief Contians functions to help control I2C regulator
 *
 *  The purpose of this file is to provide functions that help
 *  the I2C regulator to be controled by other tests and
 *  utilities.
 *
 * \warning This will alter core voltages - be aware of it's power!!!
 *
 **/

// #define DEBUG

/*------------------------------------------------------------------------
						 Include Files
------------------------------------------------------------------------*/
#include <stdint.h>
#include <stdbool.h>

#include "error_types.h"
#include "i2c_api.h"
#include "minPrintf.h"

#include "UH_I2CReg88PG870.h"

/* ------------------------------------------------------------------------
	Defines
------------------------------------------------------------------------ */

#define I2C_REGUL_88PG870_PORT		(1)
#define I2C_REGUL_88PG870_ADDR		(0x18)

#define REG_88PG870_REV_VAL			(0x11)

#define I2C_RC_OK					(STATUS_OK)

//88PG870 Voltage Regulator Registers
#define STATUS_REG_ADDR				(0x01)
#define SYSTEM_CTRL_REG_ADDR        (0x02)
#define SLEEP_MODE_CTRL_REG_ADDR    (0x04)
#define ACTIVE_VOLTAGE_0_REG_ADDR   (0x05)
#define ACTIVE_VOLTAGE_1_REG_ADDR   (0x06)
#define ACTIVE_VOLTAGE_2_REG_ADDR   (0x07)
#define ACTIVE_VOLTAGE_3_REG_ADDR   (0x08)
#define SLEEP_VOLTAGE_REG_ADDR      (0x0D)
#define BUCK_CTRL_REG_ADDR          (0x0E)
#define AVS_PATCH_REG_ADDR			(0x30)
#define CHIP_REVISION_REG_ADDR      (0x33)

#define SYS_CTRL_EN_AVS_BIT			(0x40)
#define AVS_PTCH_EN_AVS_BIT			(0x01)

/*------------------------------------------------------------------------
						Globals
-------------------------------------------------------------------------*/

/*------------------------------------------------------------------------
						Local Function Prototypes
------------------------------------------------------------------------*/

/*------------------------------------------------------------------------
			 board_has_88PG870_regulator
-------------------------------------------------------------------------*/
/**
 *
 * \brief  This test will determine through I2C if there is an 88PG870
 *
 * \param NONE
 *
 * \return [out] Boolean - True if 88PG870 found, false otherwise.
 *
 **/

bool board_has_88PG870_regulator(void)
{
	bool retval = false;
	uint8_t rbuf;

	if (I2C_RC_OK == (retval = initialize_I2C(I2C_REGUL_88PG870_PORT)))
	{
		retval = I2C_read(I2C_REGUL_88PG870_ADDR, CHIP_REVISION_REG_ADDR, I2C_ONE_BYTE_MODE, sizeof(rbuf), &rbuf);
		dbg_printf("I2C_read return: %d %02x\n",retval,rbuf);

		if (REG_88PG870_REV_VAL == rbuf)
		{
			retval = true;
		}
	}
	return retval;
}



/*------------------------------------------------------------------------
			Enable_AVS_Patch_I2CReg88PG870
-------------------------------------------------------------------------*/
/**
 *
 * \brief  This test will adjust voltage through I2C to change
 *
 * \param voltVal [in]  voltage value
 *
 * \return [out] actual voltage if set else 0
 *
 **/
uint32_t Enable_AVS_Patch_I2CReg88PG870(void)
{
	uint32_t retval = FAIL;
	uint8_t regVal;
	uint8_t voltRegAddr = AVS_PATCH_REG_ADDR;

	// Init I2C
	dbg_printf("Setting I2C Power Regulator to enable AVS patch\n");
	if (STATUS_OK == (retval = initialize_I2C(I2C_REGUL_88PG870_PORT)))
	{
		dbg_printf("I2C port Init completed\n");

		// Read/Modify/Write Register 0x30:bit0 = 0 - clear the bit
		if(I2C_RC_OK != (retval = I2C_read(I2C_REGUL_88PG870_ADDR, voltRegAddr, I2C_ONE_BYTE_MODE, sizeof(regVal), &regVal)))
		{
			dbg_printf("Failed to read from I2C 88PG870 regulator\n");
		} else
		{
			dbg_printf("Power Regulator Control Register Addr:0x%02X  Val:0x%02X\n", voltRegAddr, regVal);

			// write Voltage Reg AVS_PATCH Register with bottom bit cleared
			regVal &= ~AVS_PTCH_EN_AVS_BIT;
			if(I2C_RC_OK != (retval = I2C_write(I2C_REGUL_88PG870_ADDR, voltRegAddr, I2C_ONE_BYTE_MODE, sizeof(regVal), &regVal)))
			{
				dbg_printf("Failed to write to I2C 88PG870 regulator\n");
			} else
			{
				if(I2C_RC_OK != (retval = I2C_read(I2C_REGUL_88PG870_ADDR, voltRegAddr, I2C_ONE_BYTE_MODE, sizeof(regVal), &regVal)))
				{
					dbg_printf("Failed to read from I2C 88PG870 regulator\n");
				} else
				{
					dbg_printf("%s AVS Patch: Addr:0x%02X  Val:0x%02X\n\n",
							(AVS_PTCH_EN_AVS_BIT == (regVal & AVS_PTCH_EN_AVS_BIT)) ? "Failed to Enable" : "Enabled" , voltRegAddr, regVal);
				}
			}
		}
	}
	return retval;
}


/*------------------------------------------------------------------------
			Enable_AVS_Mode_I2CReg88PG870
-------------------------------------------------------------------------*/
/**
 *
 * \brief  This test will adjust voltage through I2C to change
 *
 * \param voltVal [in]  voltage value
 *
 * \return [out] actual voltage if set else 0
 *
 **/
uint32_t Enable_AVS_Mode_I2CReg88PG870(void)
{
	uint32_t	retval = FAIL;
	uint8_t		regVal;
	uint8_t		voltRegAddr = SYSTEM_CTRL_REG_ADDR;// Write 0x02[6] = 1

	// Init I2C
	dbg_printf("Setting I2C Power Regulator to enable AVS mode\n");
	if (STATUS_OK == (retval = initialize_I2C(I2C_REGUL_88PG870_PORT)))
	{
		dbg_printf("I2C port Init completed\n");

		// Read/Modify/Write SysCtrl Register 0x02:bit6 = 1
		if(I2C_RC_OK != (retval = I2C_read(I2C_REGUL_88PG870_ADDR, voltRegAddr, I2C_ONE_BYTE_MODE, sizeof(regVal), &regVal)))
		{
			dbg_printf("Failed to read from I2C 88PG870 regulator\n");
		} else
		{
			dbg_printf("Power Regulator Control Register Addr:0x%02X  Val:0x%02X\n", voltRegAddr, regVal);

			// write System Control Register with bit 5 on now
			regVal |= SYS_CTRL_EN_AVS_BIT;
			if(I2C_RC_OK != (retval = I2C_write(I2C_REGUL_88PG870_ADDR, voltRegAddr, I2C_ONE_BYTE_MODE, sizeof(regVal), &regVal)))
			{
				dbg_printf("Failed to write to I2C 88PG870 regulator\n");
			} else
			{
				if(I2C_RC_OK != (retval = I2C_read(I2C_REGUL_88PG870_ADDR, voltRegAddr, I2C_ONE_BYTE_MODE, sizeof(regVal), &regVal)))
				{
					dbg_printf("Failed to read from I2C 88PG870 regulator\n");
				} else
				{
					dbg_printf("%s Regulator EN_AVS Mode: Addr:0x%02X  Val:0x%02X\n\n",
							(SYS_CTRL_EN_AVS_BIT == (regVal & SYS_CTRL_EN_AVS_BIT)) ? "Enabled" : "Failed to Enable", voltRegAddr, regVal);
				}
			}
		}
	}
	return retval;
}

/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
