/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

/**
 *
 * \file UH_AVS.c
 *
 * \brief Contians functions to help control AVS
 *
 *  The purpose of this file is to provide functions that help
 *  AVS be controled by other tests and utilities.
 *
 * \warning This will alter core voltages - be aware of it's power!!!
 *
 **/

// #define DEBUG

/*------------------------------------------------------------------------
							Include Files
------------------------------------------------------------------------*/

#include "UH_AVS.h"
#include "UH_I2CReg88PG870.h"
#include "pmu_api.h"

/* ------------------------------------------------------------------------
							Defines
-------------------------------------------------------------------------*/
#define UTF_SOURCE_NAME __FILE__

typedef enum
{
	AVS_VOLT_CODE_0_747_MV =  0,
	AVS_VOLT_CODE_0_758_MV =  1,
	AVS_VOLT_CODE_0_770_MV =  2,
	AVS_VOLT_CODE_0_782_MV =  3,
	AVS_VOLT_CODE_0_793_MV =  4,
	AVS_VOLT_CODE_0_805_MV =  5,
	AVS_VOLT_CODE_0_817_MV =  6,
	AVS_VOLT_CODE_0_828_MV =  7,
	AVS_VOLT_CODE_0_840_MV =  8,
	AVS_VOLT_CODE_0_852_MV =  9,
	AVS_VOLT_CODE_0_863_MV =  10,
	AVS_VOLT_CODE_0_875_MV =  11,
	AVS_VOLT_CODE_0_887_MV =  12,
	AVS_VOLT_CODE_0_898_MV =  13,
	AVS_VOLT_CODE_0_910_MV =  14,
	AVS_VOLT_CODE_0_922_MV =  15,
	AVS_VOLT_CODE_0_933_MV =  16,
	AVS_VOLT_CODE_0_945_MV =  17,
	AVS_VOLT_CODE_0_957_MV =  18,
	AVS_VOLT_CODE_0_968_MV =  19,
	AVS_VOLT_CODE_0_980_MV =  20,
	AVS_VOLT_CODE_0_992_MV =  21,
	AVS_VOLT_CODE_1_003_MV =  22,
	AVS_VOLT_CODE_1_015_MV =  23,
	AVS_VOLT_CODE_1_027_MV =  24,
	AVS_VOLT_CODE_1_038_MV =  25,
	AVS_VOLT_CODE_1_050_MV =  26,
	AVS_VOLT_CODE_1_062_MV =  27,
	AVS_VOLT_CODE_1_073_MV =  28,
	AVS_VOLT_CODE_1_085_MV =  29,
	AVS_VOLT_CODE_1_097_MV =  30,
	AVS_VOLT_CODE_1_108_MV =  31,
	AVS_VOLT_CODE_1_120_MV =  32,
	AVS_VOLT_CODE_1_132_MV =  33,
	AVS_VOLT_CODE_1_143_MV =  34,
	AVS_VOLT_CODE_1_155_MV =  35,
	AVS_VOLT_CODE_1_167_MV =  36,
	AVS_VOLT_CODE_1_178_MV =  37,
	AVS_VOLT_CODE_1_190_MV =  38,
	AVS_VOLT_CODE_1_202_MV =  39,
	AVS_VOLT_CODE_1_213_MV =  40,
	AVS_VOLT_CODE_1_225_MV =  41,
	AVS_VOLT_CODE_1_237_MV =  42,
	AVS_VOLT_CODE_1_248_MV =  43,
	AVS_VOLT_CODE_1_260_MV =  44,
	AVS_VOLT_CODE_1_272_MV =  45,
	AVS_VOLT_CODE_1_283_MV =  46,
	AVS_VOLT_CODE_1_295_MV =  47,
	AVS_VOLT_CODE_1_307_MV =  48,
	AVS_VOLT_CODE_1_318_MV =  49,
	AVS_VOLT_CODE_1_330_MV =  50,
	AVS_VOLT_CODE_1_342_MV =  51,

} AVS_VOLTAGE_CODES;

/*------------------------------------------------------------------------
							Globals
-------------------------------------------------------------------------*/

static AVS_REGS_t* const p_avs_regs = (AVS_REGS_t*) AP_MPMU_AVS_BASE;

/*------------------------------------------------------------------------
						External Globals
-------------------------------------------------------------------------*/

/*------------------------------------------------------------------------
						Local Function Prototypes
------------------------------------------------------------------------*/
AVS_VOLTAGE_CODES  lookup_AVS_Volt_Code(AVS_VOLTAGE_OPTIONS voltage);
AVS_VOLTAGE_OPTIONS lookup_AVS_Code_Volt(AVS_VOLTAGE_CODES   code);


/*------------------------------------------------------------------------
			set_AVS_Voltage
-------------------------------------------------------------------------*/
/**
 *
 * \brief  This test will adjust voltage through AVS to change
 *
 * \param voltVal [in]  voltage value
 *
 * \return [out] actual voltage if set else 0
 *
 **/
uint32_t change_AVS_Voltage(AVS_VOLTAGE_OPTIONS voltage)
{
	uint8_t VLimitCode;

	AVS_DEBUG_PRINTF("setting AVS Voltage to %d\r\n",voltage);

	//lookup voltage value
	VLimitCode = (uint8_t)lookup_AVS_Volt_Code(voltage);
	if(0 == VLimitCode)
	{
		msg(MSG_DEBUG, UTF_SOURCE_NAME,"Unavailable AVS Voltage level = %d \r\n", voltage);
		return UTF_FAIL;
	}
	// set limits
	p_avs_regs->VddLimit = AVS_VDDLIMIT_HIGH_VDD_LIMIT_REPLACE_VAL(p_avs_regs->VddLimit, VLimitCode );
	p_avs_regs->VddLimit = AVS_VDDLIMIT_LOW_VDD_LIMIT_REPLACE_VAL(p_avs_regs->VddLimit, VLimitCode  );

	return voltage;
}


/*------------------------------------------------------------------------
			set_AVS_Voltage
-------------------------------------------------------------------------*/
/**
 *
 * \brief  This test will adjust voltage through AVS to change
 *
 * \param voltVal [in]  voltage value
 *
 * \return [out] actual voltage if set else 0
 *
 **/
uint32_t set_AVS_Voltage(AVS_VOLTAGE_OPTIONS voltage)
{
	uint32_t result = change_AVS_Voltage(voltage);
   /// if(UTF_FAIL != result)
		enable_AVS();

		// Wait 5 miliseconds for new setup to stablize, then reset DRO_min_Max();
		UTF_waitForMicroSecs(5000);
		reset_min_max();

	return result;
}

uint32_t set_AVS_FixedDRO(uint32_t targetDRO, AVS_VOLTAGE_OPTIONS lowLimit, AVS_VOLTAGE_OPTIONS highLimit) // Set AVS in DRO Mode
{
	uint8_t lowLimitCode,highLimitCode;
	AVS_PRINTF("setting AVS DRO Mode to Tartget DRO = %d\r\n", targetDRO);

	disable_AVS(); // just to be sure we don't go off into weeds while configuring

	//If AVS is in reset make sure that it has been initialized Correctly
	if(0 == AVS_CONTROL_RST_B_MASK_SHIFT(p_avs_regs->Control))
	{
		initAVS();
	}

	//set targed DRO value
	p_avs_regs->DroMode1 = AVS_DROMODE1_SPEED_TARGET_REPLACE_VAL( p_avs_regs->DroMode1, targetDRO );

	//Now that target DRO is set increase voltage limits.
	lowLimitCode = (uint8_t)lookup_AVS_Volt_Code(lowLimit);
	highLimitCode = (uint8_t)lookup_AVS_Volt_Code(highLimit);
	if((0 == lowLimitCode)||(0 == highLimitCode))
	{
		msg(MSG_DEBUG, UTF_SOURCE_NAME,"Unavailable AVS Voltage Limits, Low = %dmV, High = %dmV \r\n", lowLimit,highLimit);
		return UTF_FAIL;
	}
	p_avs_regs->VddLimit = AVS_VDDLIMIT_HIGH_VDD_LIMIT_REPLACE_VAL(p_avs_regs->VddLimit, highLimitCode);
	p_avs_regs->VddLimit = AVS_VDDLIMIT_LOW_VDD_LIMIT_REPLACE_VAL(p_avs_regs->VddLimit, lowLimitCode);

	AVS_DEBUG_PRINTF("AVS DRO Mode settings are Tartget DRO = %d, Low Vlimit = %d, High Vlimit = %d\r\n", targetDRO,lowLimitCode,highLimitCode);
	//Everything is configured turn AVS in DRO mode
	enable_AVS();

	// Wait 5 miliseconds for new setup to stablize, then reset DRO_min_Max();
	UTF_waitForMicroSecs(5000);
	reset_min_max();

	return targetDRO;
}

bool set_AVS_To_Efuse_Values(void)
{
	volatile APB_CONFIG_REGS_t *APB_config_reg = (void*) AP_AP_APB_APB_CONFIG_BASE;

	uint32_t Efuse_AVS_low_vdd_limit, Efuse_AVS_high_vdd_limit, Efuse_AVS_low_speed_target, AVS_Efuse_Programmed, AVS_Efuse_sense_select;

	if (!is_RevA())
		APB_config_reg = (APB_CONFIG_REGS_t*) ((uintptr_t)APB_config_reg - 0xC);

	Efuse_AVS_low_vdd_limit = (APB_config_reg->Fuse_Bank5_R0) & 0x3F;       // Bank5 bit [5:0]
	Efuse_AVS_high_vdd_limit = (APB_config_reg->Fuse_Bank5_R0 >> 6) & 0x3F; // Bank5 bit [11:6]
	Efuse_AVS_low_speed_target = (APB_config_reg->Fuse_Bank5_R0 >> 12) & 0xFFFF; // Bank5 bit [27:12]

	AVS_Efuse_sense_select = (APB_config_reg->Fuse_Bank5_R0 >> 28) & 1;       // Bank5 bit 28
	AVS_Efuse_Programmed = (APB_config_reg->Fuse_Bank5_R0 >> 29) & 1;       // Bank5 bit 29

    if(Efuse_AVS_low_vdd_limit==26)
    {
        uint32_t  additionalvalue=((APB_config_reg->Fuse_Bank5_R2&0xffff)<15000)?2:1;
        minPrintf("[ricoh] Low VDD Limit got additional value, %d+%d\n", Efuse_AVS_low_vdd_limit, additionalvalue);
        Efuse_AVS_low_vdd_limit+=additionalvalue;
        Efuse_AVS_high_vdd_limit+=additionalvalue;
    }

	AVS_DEBUG_PRINTF("AVS efuse Values: \r\n"
									"\t\t\t\tEfuse Programed = %d\r\n"
									"\t\t\t\tLow VDD Limit = %d\r\n"
									"\t\t\t\tHigh VDD Limit = %d\r\n"
									"\t\t\t\tTarget DRO = %d\r\n"
									"\t\t\t\tSelect Vsense0 = %d\r\n",
									AVS_Efuse_Programmed, Efuse_AVS_low_vdd_limit,
									Efuse_AVS_high_vdd_limit,Efuse_AVS_low_speed_target,
									AVS_Efuse_sense_select);

	// if Efuse are not burnned then don't setup AVS
	if(0 == AVS_Efuse_Programmed)
		return false;


	AVS_PRINTF("Setting AVS to match Efues Values\r\n");
	disable_AVS(); // just to be sure we don't go off into weeds while configuring

	//If AVS is not in reset make sure that it has been initialized Correctly
	if(0 == AVS_CONTROL_RST_B_MASK_SHIFT(p_avs_regs->Control))
	{
		initAVS();
	}

	p_avs_regs->VddLimit = AVS_VDDLIMIT_LOW_VDD_LIMIT_REPLACE_VAL(p_avs_regs->VddLimit, Efuse_AVS_low_vdd_limit);
	p_avs_regs->VddLimit = AVS_VDDLIMIT_HIGH_VDD_LIMIT_REPLACE_VAL(p_avs_regs->VddLimit, Efuse_AVS_high_vdd_limit);

	p_avs_regs->DroMode1 = AVS_DROMODE1_SPEED_TARGET_REPLACE_VAL(p_avs_regs->DroMode1, Efuse_AVS_low_speed_target);
	p_avs_regs->Control  = AVS_CONTROL_SELECT_VSENSE0_REPLACE_VAL(p_avs_regs->Control, AVS_Efuse_sense_select);

	enable_AVS();

	// Wait 5 miliseconds for new setup to stablize, then reset DRO_min_Max();
	UTF_waitForMicroSecs(5000);
	reset_min_max();

	return true;
}



/*------------------------------------------------------------------------
			check_AVS_enabled
-------------------------------------------------------------------------*/
/**
 *
 * \brief  sets the AVS enable bit
 *
 * \param [in] void
 *
 * \return [out] void
 *
 **/
uint32_t get_AVS_enable_status(void)
{
	return AVS_CONTROL_ENABLE_MASK_SHIFT(p_avs_regs->Control);
}

/*------------------------------------------------------------------------
			enable_AVS
-------------------------------------------------------------------------*/
/**
 *
 * \brief  sets the AVS enable bit
 *
 * \param [in] void
 *
 * \return [out] void
 *
 **/
void enable_AVS(void)
{
	// Enable AVS. AVS should automaticly wait until calibration is complete before actual start.(see AVS doc 5.2.1)
	p_avs_regs->Control = AVS_CONTROL_ENABLE_REPLACE_VAL(p_avs_regs->Control, 1);
}

/*------------------------------------------------------------------------
			disable_AVS
-------------------------------------------------------------------------*/
/**
 *
 * \brief  clears the AVS enable bit
 *
 * \param [in] void
 *
 * \return [out] void
 *
 **/
void disable_AVS(void)
{
	// disable AVS.
	p_avs_regs->Control = AVS_CONTROL_ENABLE_REPLACE_VAL(p_avs_regs->Control, 0);
}


/*------------------------------------------------------------------------
			reset_AVS
-------------------------------------------------------------------------*/
/**
 *
 * \brief  resets AVS
 *
 * \param [in] void
 *
 * \return [out] void
 *
 **/
void reset_AVS(void)
{
	// make sure we are in reset
	p_avs_regs->Control = AVS_CONTROL_RST_B_REPLACE_VAL(p_avs_regs->Control, 0);

	UTF_waitForMicroSecs(333);  //ensure AVS is though initial reset

	// take out of reset to begin calibration // should I check if already on??
	p_avs_regs->Control = AVS_CONTROL_RST_B_REPLACE_VAL(p_avs_regs->Control, 1);
}

void reset_min_max(void)
{
	p_avs_regs->Control = AVS_CONTROL_MIN_MAX_VAL_RST_REPLACE_VAL(p_avs_regs->Control, 1);

	UTF_waitForMicroSecs(333);  //wait a little bit just incase

	p_avs_regs->Control = AVS_CONTROL_MIN_MAX_VAL_RST_REPLACE_VAL(p_avs_regs->Control, 0);

}

AVS_VOLTAGE_CODES lookup_AVS_Volt_Code(AVS_VOLTAGE_OPTIONS voltage)
{
	switch(voltage)
	{
		case (AVS_VOLTS_747MV ) : return AVS_VOLT_CODE_0_747_MV;
		case (AVS_VOLTS_758MV ) : return AVS_VOLT_CODE_0_758_MV;
		case (AVS_VOLTS_770MV ) : return AVS_VOLT_CODE_0_770_MV;
		case (AVS_VOLTS_782MV ) : return AVS_VOLT_CODE_0_782_MV;
		case (AVS_VOLTS_793MV ) : return AVS_VOLT_CODE_0_793_MV;
		case (AVS_VOLTS_805MV ) : return AVS_VOLT_CODE_0_805_MV;
		case (AVS_VOLTS_817MV ) : return AVS_VOLT_CODE_0_817_MV;
		case (AVS_VOLTS_828MV ) : return AVS_VOLT_CODE_0_828_MV;
		case (AVS_VOLTS_840MV ) : return AVS_VOLT_CODE_0_840_MV;
		case (AVS_VOLTS_852MV ) : return AVS_VOLT_CODE_0_852_MV;
		case (AVS_VOLTS_863MV ) : return AVS_VOLT_CODE_0_863_MV;
		case (AVS_VOLTS_875MV ) : return AVS_VOLT_CODE_0_875_MV;
		case (AVS_VOLTS_887MV ) : return AVS_VOLT_CODE_0_887_MV;
		case (AVS_VOLTS_898MV ) : return AVS_VOLT_CODE_0_898_MV;
		case (AVS_VOLTS_910MV ) : return AVS_VOLT_CODE_0_910_MV;
		case (AVS_VOLTS_922MV ) : return AVS_VOLT_CODE_0_922_MV;
		case (AVS_VOLTS_933MV ) : return AVS_VOLT_CODE_0_933_MV;
		case (AVS_VOLTS_945MV ) : return AVS_VOLT_CODE_0_945_MV;
		case (AVS_VOLTS_957MV ) : return AVS_VOLT_CODE_0_957_MV;
		case (AVS_VOLTS_968MV ) : return AVS_VOLT_CODE_0_968_MV;
		case (AVS_VOLTS_980MV ) : return AVS_VOLT_CODE_0_980_MV;
		case (AVS_VOLTS_992MV ) : return AVS_VOLT_CODE_0_992_MV;
		case (AVS_VOLTS_1003MV) : return AVS_VOLT_CODE_1_003_MV;
		case (AVS_VOLTS_1015MV) : return AVS_VOLT_CODE_1_015_MV;
		case (AVS_VOLTS_1027MV) : return AVS_VOLT_CODE_1_027_MV;
		case (AVS_VOLTS_1038MV) : return AVS_VOLT_CODE_1_038_MV;
		case (AVS_VOLTS_1050MV) : return AVS_VOLT_CODE_1_050_MV;
		case (AVS_VOLTS_1062MV) : return AVS_VOLT_CODE_1_062_MV;
		case (AVS_VOLTS_1073MV) : return AVS_VOLT_CODE_1_073_MV;
		case (AVS_VOLTS_1085MV) : return AVS_VOLT_CODE_1_085_MV;
		case (AVS_VOLTS_1097MV) : return AVS_VOLT_CODE_1_097_MV;
		case (AVS_VOLTS_1108MV) : return AVS_VOLT_CODE_1_108_MV;
		case (AVS_VOLTS_1120MV) : return AVS_VOLT_CODE_1_120_MV;
		case (AVS_VOLTS_1132MV) : return AVS_VOLT_CODE_1_132_MV;
		case (AVS_VOLTS_1143MV) : return AVS_VOLT_CODE_1_143_MV;
		case (AVS_VOLTS_1155MV) : return AVS_VOLT_CODE_1_155_MV;
		case (AVS_VOLTS_1167MV) : return AVS_VOLT_CODE_1_167_MV;
		case (AVS_VOLTS_1178MV) : return AVS_VOLT_CODE_1_178_MV;
		case (AVS_VOLTS_1190MV) : return AVS_VOLT_CODE_1_190_MV;
		case (AVS_VOLTS_1202MV) : return AVS_VOLT_CODE_1_202_MV;
		case (AVS_VOLTS_1213MV) : return AVS_VOLT_CODE_1_213_MV;
		case (AVS_VOLTS_1225MV) : return AVS_VOLT_CODE_1_225_MV;
		case (AVS_VOLTS_1237MV) : return AVS_VOLT_CODE_1_237_MV;
		case (AVS_VOLTS_1248MV) : return AVS_VOLT_CODE_1_248_MV;
		case (AVS_VOLTS_1260MV) : return AVS_VOLT_CODE_1_260_MV;
		case (AVS_VOLTS_1272MV) : return AVS_VOLT_CODE_1_272_MV;
		case (AVS_VOLTS_1283MV) : return AVS_VOLT_CODE_1_283_MV;
		case (AVS_VOLTS_1295MV) : return AVS_VOLT_CODE_1_295_MV;
		case (AVS_VOLTS_1307MV) : return AVS_VOLT_CODE_1_307_MV;
		case (AVS_VOLTS_1318MV) : return AVS_VOLT_CODE_1_318_MV;
		case (AVS_VOLTS_1330MV) : return AVS_VOLT_CODE_1_330_MV;
		case (AVS_VOLTS_1342MV) : return AVS_VOLT_CODE_1_342_MV;
		default:
			msg(MSG_ERROR, UTF_SOURCE_NAME, "ERROR: Invalide Voltage =%0d\r\n", voltage);
			return 0;
			break;
	}
}


#if 0
AVS_VOLTAGE_OPTIONS lookup_AVS_Code_Volt(AVS_VOLTAGE_CODES   code)
{
	{
		switch(code)
		{
			case (AVS_VOLT_CODE_0_747_MV) : return  AVS_VOLTS_747MV ;
			case (AVS_VOLT_CODE_0_758_MV) : return  AVS_VOLTS_758MV ;
			case (AVS_VOLT_CODE_0_770_MV) : return  AVS_VOLTS_770MV ;
			case (AVS_VOLT_CODE_0_782_MV) : return  AVS_VOLTS_782MV ;
			case (AVS_VOLT_CODE_0_793_MV) : return  AVS_VOLTS_793MV ;
			case (AVS_VOLT_CODE_0_805_MV) : return  AVS_VOLTS_805MV ;
			case (AVS_VOLT_CODE_0_817_MV) : return  AVS_VOLTS_817MV ;
			case (AVS_VOLT_CODE_0_828_MV) : return  AVS_VOLTS_828MV ;
			case (AVS_VOLT_CODE_0_840_MV) : return  AVS_VOLTS_840MV ;
			case (AVS_VOLT_CODE_0_852_MV) : return  AVS_VOLTS_852MV ;
			case (AVS_VOLT_CODE_0_863_MV) : return  AVS_VOLTS_863MV ;
			case (AVS_VOLT_CODE_0_875_MV) : return  AVS_VOLTS_875MV ;
			case (AVS_VOLT_CODE_0_887_MV) : return  AVS_VOLTS_887MV ;
			case (AVS_VOLT_CODE_0_898_MV) : return  AVS_VOLTS_898MV ;
			case (AVS_VOLT_CODE_0_910_MV) : return  AVS_VOLTS_910MV ;
			case (AVS_VOLT_CODE_0_922_MV) : return  AVS_VOLTS_922MV ;
			case (AVS_VOLT_CODE_0_933_MV) : return  AVS_VOLTS_933MV ;
			case (AVS_VOLT_CODE_0_945_MV) : return  AVS_VOLTS_945MV ;
			case (AVS_VOLT_CODE_0_957_MV) : return  AVS_VOLTS_957MV ;
			case (AVS_VOLT_CODE_0_968_MV) : return  AVS_VOLTS_968MV ;
			case (AVS_VOLT_CODE_0_980_MV) : return  AVS_VOLTS_980MV ;
			case (AVS_VOLT_CODE_0_992_MV) : return  AVS_VOLTS_992MV ;
			case (AVS_VOLT_CODE_1_003_MV) : return  AVS_VOLTS_1003MV;
			case (AVS_VOLT_CODE_1_015_MV) : return  AVS_VOLTS_1015MV;
			case (AVS_VOLT_CODE_1_027_MV) : return  AVS_VOLTS_1027MV;
			case (AVS_VOLT_CODE_1_038_MV) : return  AVS_VOLTS_1038MV;
			case (AVS_VOLT_CODE_1_050_MV) : return  AVS_VOLTS_1050MV;
			case (AVS_VOLT_CODE_1_062_MV) : return  AVS_VOLTS_1062MV;
			case (AVS_VOLT_CODE_1_073_MV) : return  AVS_VOLTS_1073MV;
			case (AVS_VOLT_CODE_1_085_MV) : return  AVS_VOLTS_1085MV;
			case (AVS_VOLT_CODE_1_097_MV) : return  AVS_VOLTS_1097MV;
			case (AVS_VOLT_CODE_1_108_MV) : return  AVS_VOLTS_1108MV;
			case (AVS_VOLT_CODE_1_120_MV) : return  AVS_VOLTS_1120MV;
			case (AVS_VOLT_CODE_1_132_MV) : return  AVS_VOLTS_1132MV;
			case (AVS_VOLT_CODE_1_143_MV) : return  AVS_VOLTS_1143MV;
			case (AVS_VOLT_CODE_1_155_MV) : return  AVS_VOLTS_1155MV;
			case (AVS_VOLT_CODE_1_167_MV) : return  AVS_VOLTS_1167MV;
			case (AVS_VOLT_CODE_1_178_MV) : return  AVS_VOLTS_1178MV;
			case (AVS_VOLT_CODE_1_190_MV) : return  AVS_VOLTS_1190MV;
			case (AVS_VOLT_CODE_1_202_MV) : return  AVS_VOLTS_1202MV;
			case (AVS_VOLT_CODE_1_213_MV) : return  AVS_VOLTS_1213MV;
			case (AVS_VOLT_CODE_1_225_MV) : return  AVS_VOLTS_1225MV;
			case (AVS_VOLT_CODE_1_237_MV) : return  AVS_VOLTS_1237MV;
			case (AVS_VOLT_CODE_1_248_MV) : return  AVS_VOLTS_1248MV;
			case (AVS_VOLT_CODE_1_260_MV) : return  AVS_VOLTS_1260MV;
			case (AVS_VOLT_CODE_1_272_MV) : return  AVS_VOLTS_1272MV;
			case (AVS_VOLT_CODE_1_283_MV) : return  AVS_VOLTS_1283MV;
			case (AVS_VOLT_CODE_1_295_MV) : return  AVS_VOLTS_1295MV;
			case (AVS_VOLT_CODE_1_307_MV) : return  AVS_VOLTS_1307MV;
			case (AVS_VOLT_CODE_1_318_MV) : return  AVS_VOLTS_1318MV;
			case (AVS_VOLT_CODE_1_330_MV) : return  AVS_VOLTS_1330MV;
			case (AVS_VOLT_CODE_1_342_MV) : return  AVS_VOLTS_1342MV;
			default:
				msg(MSG_ERROR, UTF_SOURCE_NAME, "ERROR: Invalide code =%0d\r\n", code);
				return 0;
				break;
		}
	}

}
#endif

/*------------------------------------------------------------------------
			reset_AVS
-------------------------------------------------------------------------*/
/**
 *
 * \brief  resets AVS
 *
 * \param [in] void
 *
 * \return [out] void
 *
 **/
void initAVS(void)
{
	// Turn on I2C so AVS can talk to regulators
	PMU_enable(ePMU_DEVICE_I2C);

	if (board_has_88PG870_regulator())
	{
		if (!((STATUS_OK == Enable_AVS_Mode_I2CReg88PG870()) && (STATUS_OK == Enable_AVS_Patch_I2CReg88PG870())))
			msg(MSG_DEBUG, UTF_SOURCE_NAME, "Failure Enabling AVS workaround on 88PG870\n");
	}

	//If Currently in Reset take out of Reset
	if(0 == AVS_CONTROL_RST_B_MASK_SHIFT(p_avs_regs->Control))
	{
	   p_avs_regs->Calibration = AVS_CALIBRATION_BG_CFG_REPLACE_VAL(p_avs_regs->Calibration , 3);
	   // p_avs_regs->Test = AVS_TEST_TP_CFG_REPLACE_VAL(p_avs_regs->Test, 4); //Debug: Send avs_vdd_sense1 to annalog test point
	   reset_AVS();
	}

	// set limits
	if( 0 == change_AVS_Voltage(AVS_VOLTS_1050MV) )
		msg(MSG_DEBUG, UTF_SOURCE_NAME,"AVS Init failed\r\n");

	//set DRO values - currently defaults
	p_avs_regs->DroMode1 = AVS_DROMODE1_SPEED_TARGET_REPLACE_VAL( p_avs_regs->DroMode1, 0xffff );
	p_avs_regs->DroMode1 = AVS_DROMODE1_DRO_COUNT_INTERVAL_REPLACE_VAL( p_avs_regs->DroMode1, 0x1e1 );
	p_avs_regs->DroMode2 = AVS_DROMODE2_AVG_REPLACE_VAL( p_avs_regs->DroMode2, 0 );

	reset_min_max();

}

uint32_t readAVSLimitflags(void)
{
	return (p_avs_regs->Status & 0xf);
}

/*------------------------------------------------------------------------
			readAVS_BG_Cal
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the Current Internal Bandgap Calibration value from the AVS status register
 *
 * \param [in] void
 *
 * \return [out] uint32_t BandGap Calibration Value
 *
 **/
uint32_t readAVS_BG_Cal(void)
{
	//exctract shifted bg_cal_value from status register
	return (AVS_STATUS_BG_CAL_VALUE_MASK_SHIFT(p_avs_regs->Status));
}

/*------------------------------------------------------------------------
			readAVSFastTravel
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the fast travel indication from the AVS status register
 *
 * \param [in] void
 *
 * \return [out] uint32_t fast travel indication
 *
 **/
uint32_t readAVSFastTravel(void)
{
	//exctract shifted fast travel indication from status register
	return (AVS_STATUS_FAST_TRAVEL_INDICATION_MASK_SHIFT(p_avs_regs->Status));
}

/*------------------------------------------------------------------------
		   readAVSAboveLimit
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the Above Limit indication from the AVS status register
 *
 * \param [in] void
 *
 * \return [out] uint32_t Above Limit indication
 *
 **/
uint32_t readAVSAboveLimit(void)
{
	//exctract shifted above limit from status register
	return (AVS_STATUS_ABOVE_LIMIT_MASK_SHIFT(p_avs_regs->Status));
}

/*------------------------------------------------------------------------
		   readAVSBelowLimit
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the Below Limit indication from the AVS status register
 *
 * \param [in] void
 *
 * \return [out] uint32_t Below Limit indication
 *
 **/
uint32_t readAVSBelowLimit(void)
{
	//exctract shifted Below limit from status register
	return (AVS_STATUS_BELOW_LIMIT_MASK_SHIFT(p_avs_regs->Status));
}

/*------------------------------------------------------------------------
		   readAVSDelta
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the AVS Delta from the AVS DeltaStatus register
 *
 * \param [in] void
 *
 * \return [out] uint32_t AVS Delta
 *
 **/
uint32_t readAVSDelta(void)
{
	//exctract shifted Below limit from DeltaStatus register
	return (AVS_DELTASTATUS_AVS_DELTA_MASK_SHIFT(p_avs_regs->DeltaStatus));
}

/*------------------------------------------------------------------------
		   readAVSMaxDelta
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the AVS Max Delta from the AVS DeltaStatus register
 *
 * \param [in] void
 *
 * \return [out] uint32_t AVS Max Delta
 *
 **/
uint32_t readAVSMaxDelta(void)
{
	//exctract shifted Below limit from DeltaStatus register
	return (AVS_DELTASTATUS_MAX_AVS_DELTA_MASK_SHIFT(p_avs_regs->DeltaStatus));
}

/*------------------------------------------------------------------------
		   readAVSMinDelta
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the AVS Min Delta from the AVS DeltaStatus register
 *
 * \param [in] void
 *
 * \return [out] uint32_t AVS Min Delta
 *
 **/
uint32_t readAVSMinDelta(void)
{
	//exctract shifted Below limit from DeltaStatus register
	return (AVS_DELTASTATUS_MIN_AVS_DELTA_MASK_SHIFT(p_avs_regs->DeltaStatus));
}


/*------------------------------------------------------------------------
			readCurrentDRO
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the Current DRO value from the AVS DRO status register
 *
 * \param [in] void
 *
 * \return [out] uint32_t DRO Value
 *
 **/
uint32_t readCurrentDRO(void)
{
	//read DRO Status Register
	return(p_avs_regs->DroStatus1);
}


/*------------------------------------------------------------------------
			readMinMaxDRO
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the Min and Max DRO values from the AVS DRO status register
 *
 * \param [in] void
 *
 * \return [out] uint32_t DRO Value
 *
 **/
uint32_t readMinMaxDRO(void)
{
	//read DRO Status Register
	return(p_avs_regs->DroStatus2);
}

/*------------------------------------------------------------------------
			readAVSstatus
-------------------------------------------------------------------------*/
/**
 *
 * \brief  Reads the Current DRO value from the AVS DRO status register
 *
 * \param [in] void
 *
 * \return [out] uint32_t DRO Value
 *
 **/
uint32_t readAVSstatus(void)
{
	//read DRO Status Register
	return(p_avs_regs->Status);
}

/*------------------------------------------------------------------------
			readEfuseDRO
-------------------------------------------------------------------------*/
/**
 *
 * \brief read tester DRO from eFuse
 *
 * \param [in]  void (none)
 *
 * \return [out] UTF PASS/FAIL
 *
 */
uint32_t readEfuseDRO(void)
{
	volatile BCMREG_REGS_t *BCM_BCM_reg = (void*) AP_BCM_BCMREG_BASE;
	uint32_t DRO_Parity_Bit = 0;
	uint32_t DRO_eFuse;

	DRO_eFuse = BCM_BCM_reg->Bank5_L2;
	DRO_Parity_Bit = (DRO_eFuse >> 16) & 1;
	DRO_eFuse &= 0xFFFF;

	if(DROParityCheck(DRO_eFuse) != DRO_Parity_Bit)
	{
	   return UTF_FAIL;
	}

	msg(MSG_DEBUG, UTF_SOURCE_NAME,"Efuse Bank0 = 0x%08x, g_DRO_eFuse = 0x%x\r\n", BCM_BCM_reg->Bank05, DRO_eFuse);

	return(DRO_eFuse);

}

/*------------------------------------------------------------------------
			DROParityCheck
-------------------------------------------------------------------------*/
/**
 *
 * \brief check the parity bit is correct for the efuse DRO
 *
 * \param [in]  uint32_t Efuse_DRO - the value read from the efuse
 *
 * \return [out] uint32_t DRO_Parity_Check (calculated value of parity bit)
 *
 */
uint32_t DROParityCheck(uint32_t Efuse_DRO)
{
	uint8_t i=0;
	uint32_t DRO_Parity_Check = 0;

	for(i=0; i<16; i++)
	{
		if((Efuse_DRO & 0x01) )//==0x01)
			DRO_Parity_Check +=1;

		Efuse_DRO >>= 1;
	}

	DRO_Parity_Check &= 0x01;
	msg(MSG_DEBUG, UTF_SOURCE_NAME,"DRO_Parity_Bit=%x\r\n", DRO_Parity_Check);
	return DRO_Parity_Check;
}

/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
