/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

//#define DEBUG

#define __APP_C_DEFINES
#include <stdint.h>
#include <stdbool.h>
#include "assert.h"
#include "error_types.h"
#include "regAddrs.h"
#include "minPrintf.h"
#include "board_types.h"
#include "rom_types.h"

#include "apb_top_regheaders.h"
#include "ID_utils.h"

#define ROM_BUILD_FIELD         (0xFFE00034)
#define DFT_ISLD_REV_ID_REG     (0xD401D800)
#define DFT_ISLD_GR2_REVA_VAL   (0x00010000)


static inline bool is_Gr2_RevA(void)
{
    volatile uint32_t *DFT_ISLD_rev_id_reg = (uint32_t*)DFT_ISLD_REV_ID_REG;
    
    return (DFT_ISLD_GR2_REVA_VAL == *DFT_ISLD_rev_id_reg);
}


inline void* get_APB_CONFIG_base(void)
{
    if (is_Gr2_RevA())
        return (void*)AP_AP_APB_APB_CONFIG_BASE;
    else
        return (void*)(AP_AP_APB_APB_CONFIG_BASE - 8);
}


inline void* get_CIU_base(void)
{
    if (is_Gr2_RevA())
        return (void*)AP_CIU_BASE_GR2_REVA;
    else
        return (void*)AP_CIU_BASE;
}


static rom_types_e get_gr2_rom_type(void)
{
    uint32_t *ROM_build_field_ptr = (uint32_t*) ROM_BUILD_FIELD;
    rom_types_e retval = invalid_ROM;

    if ((0x20000 == *ROM_build_field_ptr) || (0x30000 == *ROM_build_field_ptr))
        retval = eROM_RevA_bSPI_Aspl;

    else if ((0x30012 == *ROM_build_field_ptr) ||
            (0x40012 == *ROM_build_field_ptr) ||
            (0x50012 == *ROM_build_field_ptr))
        retval = eROM_RevB_bSPI_Aspl;

    else if (0x10006 == *ROM_build_field_ptr)
        retval = eROM_RevC_bSPI_Aspl;

    else if (is_RevA() && (0x17 == *ROM_build_field_ptr))
        retval = eROM_RevA_bSPI;

    else if (is_RevB() && (0x14 == *ROM_build_field_ptr))
        retval = eROM_RevB_bSPI;

    else
        switch (get_chip_rev())
        {
            case CHIP_REVC:
                retval = eROM_RevC;
                break;
            case CHIP_REVB1:
                retval = eROM_RevB1;
                break;
            case CHIP_REVA:
                retval = eROM_RevA;
                break;
        }
    return retval;
}


inline uint16_t get_chip_id(void)
{
    volatile uint32_t *CIU_chip_id_reg = (uint32_t*) get_CIU_base();
    
    return *CIU_chip_id_reg & 0xFFFF;
}


inline uint8_t get_chip_rev(void)
{
    volatile uint32_t *CIU_chip_id_reg = (uint32_t*) get_CIU_base();
    
    return ((*CIU_chip_id_reg) >> 16) & 0xFF;
}


inline bool is_FPGA(void)
{
    APB_CONFIG_REGS_t *apb_config_regs = (APB_CONFIG_REGS_t*)get_APB_CONFIG_base();
    uint32_t SQU_REV0 = apb_config_regs->SQU_REV0;

    // SQU_REV0 will read as zero on FPGAs.  Safe to read on Si.
    return (SQU_REV0 == 0);
}


inline bool is_Si(void)
{
    return (false == is_FPGA());
}


#define REV_C1_BIT  (0x00000001)

inline bool is_Gr2_RevC1(void)
{
    volatile APB_CONFIG_REGS_t *APB_config_reg = (void*) (AP_AP_APB_APB_CONFIG_BASE - 0xC);
    bool RevC1_fuse_bit = (1 == (APB_config_reg->Fuse_Bank5_R7 & REV_C1_BIT));

    return (is_Gr2() && is_RevC() && (true == RevC1_fuse_bit));
}


rom_types_e check_rom_type(void)
{
    rom_types_e retval = invalid_ROM;
    uint32_t    *ROM_build_field_ptr = (uint32_t*) ROM_BUILD_FIELD;

    if (is_Gr2())
    {
        if (is_Si())           // identify Si
            retval = get_gr2_rom_type();
        else if (is_FPGA())
        {
            if (is_RevA())    // identify ASIC_REV_A FPGA
                retval = eFPGA_RevA;
            else if (is_RevB() && (0x10 == *ROM_build_field_ptr)) // identify ASIC_REV_B 19_4 FPGA
                retval = eFPGA_RevB_19_16;
            else
                retval = invalid_ROM;
        }
    } else if (is_Ge2())
    {
        if (is_Si()) {         // identify Si
		if (is_Ge2_RevB())
			retval = eROM_Ge2_RevB;
		else
			retval = eROM_Ge2_RevA;
        }
	else if (is_FPGA() && (is_RevA()))   // identify Ge2 ASIC_REV_A FPGA
            retval = eFPGA_Ge2_RevA;
        else
            retval = invalid_ROM;
    }
    else
        retval = invalid_ROM;

    return retval;
}


board_types_e check_board_type(void)
{
    board_types_e retval = invalid_Brd;

    if (is_Si())
    {
        if (is_Gr2())
        {
            switch (get_chip_rev())
            {
                case CHIP_REVA:
                    retval = eBringUp_Brd_Gr2_RevA;
                    break;
                case CHIP_REVB:
                    retval = eBringUp_Brd_Gr2_RevB;
                    break;
                case CHIP_REVB1:
                    retval = eBringUp_Brd_Gr2_RevB1;
                    break;
                case CHIP_REVC:
                    retval = eBringUp_Brd_Gr2_RevC;
                    break;
                case CHIP_REVD:
                    retval = eBringUp_Brd_Gr2_RevD;
                    break;
            }
        } else if (is_Ge2())
        { 
            switch (get_chip_rev())
            {
                case CHIP_REVA:
                    retval = eBringUp_Brd_Ge2_RevA;
                    break;
                case CHIP_GE2_REVB:
                    retval = eBringUp_Brd_Ge2_RevB;
                    break;
                case CHIP_REVC:
                    retval = eBringUp_Brd_Ge2_RevC;
                    break;
                case CHIP_REVD:
                    retval = eBringUp_Brd_Ge2_RevD;
                    break;
            }
        } else
        {
            retval = invalid_Brd;
        }
    } else // FPGA
    {
        if (is_Gr2())
        {
            switch (get_chip_rev())
            {
                case CHIP_REVA:
                    retval = eFPGA_Brd_Gr2_RevA;
                    break;
                case CHIP_REVB:
                    retval = eFPGA_Brd_Gr2_RevB;
                    break;
                case CHIP_REVC:
                    retval = eFPGA_Brd_Gr2_RevC;
                    break;
                case CHIP_REVD:
                    retval = eFPGA_Brd_Gr2_RevD;
                    break;
            }
        } else if (is_Ge2())
        { 
            switch (get_chip_rev())
            {
                case CHIP_REVA:
                    retval = eFPGA_Brd_Ge2_RevA;
                    break;
                case CHIP_REVB:
                    retval = eFPGA_Brd_Ge2_RevB;
                    break;
                case CHIP_REVC:
                    retval = eFPGA_Brd_Ge2_RevC;
                    break;
                case CHIP_REVD:
                    retval = eFPGA_Brd_Ge2_RevD;
                    break;
            }
        } else
        {
            retval = invalid_Brd;
        }
    }
    return retval;
}
