/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2007-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/

#ifndef DWAPBUART_H
#define DWAPBUART_H

#include <stdint.h>

#define DBG_UART 0


/*
 * -----------------  UART based on DesignWare Cell  ---------------
 * "The DW_apb_uart is a programmable Universal Asynchronous 
 * Receiver/Transmitter (UART). This component is an AMBA 2.0-compliant 
 * Advanced Peripheral Bus (APB) slave device and is part of the family 
 * of DesignWare AMBA Synthesizable Components."
 *
 * Reference material on the DW APB UART is available at:
 * http://leda-design.com/products/designware/docs/iip/DW_apb_uart/latest/doc/dw_apb_uart_db.pdf
 *
 */
#define DWAPB_UART_RBR_OFST		0x00	// Receive Buffer Register (RO)
#define DWAPB_UART_THR_OFST		0x00	// Transmit Holding Register (WO)
#define DWAPB_UART_DLL_OFST		0x00	// Divisor Latch Low (R/W)
#define DWAPB_UART_DLH_OFST		0x04	// Divisor Latch High (R/W)
#define DWAPB_UART_IER_OFST		0x04	// Interrupt Enable Reg (R/W)
#define DWAPB_UART_IIR_OFST		0x08	// Interrupt Identification Reg (RO)
#define DWAPB_UART_FCR_OFST		0x08	// FIFO Control Reg (WO)
#define DWAPB_UART_LCR_OFST		0x0C	// Line Control Reg (R/W)
#define DWAPB_UART_MCR_OFST		0x10	// Modem Control Reg (R/W)
#define DWAPB_UART_LSR_OFST		0x14	// Line Status Reg (RO)
#define DWAPB_UART_MSR_OFST		0x18	// Modem Status Reg (RO)
#define DWAPB_UART_SCR_OFST		0x1C	// Scratchpad Reg (R/W)
#define DWAPB_UART_LPDLL_OFST	0x20	// Low Power Divisor Latch Low Reg (R/W)
#define DWAPB_UART_LPDLH_OFST	0x24	// Low Power Divisor Latch High Reg (R/W)
#define DWAPB_UART_SRBR_OFST	0x30	// Shadow Receive Buffer Reg (RO)
#define DWAPB_UART_STHR_OFST	0x30	// Shadow Transmit Holding Reg (WO)
#define DWAPB_UART_FAR_OFST		0x70	// FIFO Access Reg (R/W)
#define DWAPB_UART_TFR_OFST		0x74	// Transmit FIFO Read Reg (RO)
#define DWAPB_UART_RFW_OFST		0x74	// Receive FIFO Write Reg (WO)
#define DWAPB_UART_USR_OFST		0x7C	// UART Status Reg (RO)
#define DWAPB_UART_TFL_OFST		0x80	// Transmit FIFO Level (RO)
#define DWAPB_UART_RFL_OFST		0x84	// Receive FIFO Level (RO)
#define DWAPB_UART_SRR_OFST		0x88	// Software Reset Register (WO)
#define DWAPB_UART_SRTS_OFST	0x8C	// Shadow Reset Register (R/W)
#define DWAPB_UART_SBCR_OFST	0x90	// Shadow Break Control Reg (R/W)
#define DWAPB_UART_SDMAM_OFST	0x94	// Shadow DMA Mode (R/W)
#define DWAPB_UART_SFE_OFST		0x98	// Shadow FIFO Enable (R/W)
#define DWAPB_UART_SRT_OFST		0x9C	// Shadow RCVR Trigger (R/W)
#define DWAPB_UART_STET_OFST	0xA0	// Shadow TX Empty Trigger (R/W)
#define DWAPB_UART_HTX_OFST		0xA4	// Halt TX (R/W)
#define DWAPB_UART_DMASA_OFST	0xA8	// DMA Software Ack (WO)
#define DWAPB_UART_CPR_OFST		0xF4	// Component Param Reg (RO)
#define DWAPB_UART_UCV_OFST		0xF8	// UART Component Version (RO)
#define DWAPB_UART_CTR_OFST		0xFC	// Component Type Reg (RO)

typedef struct dwApbUartType {
	union {
		uint32_t	RBR;	//0x00	Receive Buffer Register (RO)
		uint32_t	THR;	//0x00	Transmit Holding Register (WO)
		uint32_t	DLL;	//0x00	Divisor Latch Low (R/W)
	} zero;
	union {
		uint32_t	DLH;	//0x04	Divisor Latch High (R/W)
		uint32_t	IER;	//0x04	Interrupt Enable Reg (R/W)
	} one;
	union {
		uint32_t	IIR;		//0x08	Interrupt Identification Reg (RO)
		uint32_t	FCR;		//0x08	FIFO Control Reg (WO)
	} two;
	uint32_t	LCR;		//0x0C	Line Control Reg (R/W)
	uint32_t	MCR;		//0x10	Modem Control Reg (R/W)
	uint32_t	LSR;		//0x14	Line Status Reg (RO)
	uint32_t	MSR;		//0x18	Modem Status Reg (RO)
	uint32_t	SCR;		//0x1C	Scratchpad Reg (R/W)
	uint32_t	LPDLL;		//0x20	Low Power Divisor Latch Low Reg (R/W)
	uint32_t	LPDLH;		//0x24	Low Power Divisor Latch High Reg (R/W)
	uint32_t	dummy1;     //0x28
	uint32_t	dummy2;		//0x2C
	union {
		uint32_t	SRBR;	//0x30	Shadow Receive Buffer Reg (RO)
		uint32_t	STHR;	//0x30	Shadow Transmit Holding Reg (WO)
	} shadow;
	/*	34 38 3c
		40 44 48 4c
		50 54 58 5c
		60 64 68 6c */
	uint32_t	dummy3[15]; //0x34 to 0x6c
	uint32_t	FAR;		//0x70	FIFO Access Reg (R/W)
	union {
		uint32_t	TFR;	//0x74	Transmit FIFO Read Reg (RO)
		uint32_t	RFW;	//0x74	Receive FIFO Write Reg (WO)
	} fifo;
	uint32_t  dummy4;		//0x78
	uint32_t	USR;		//0x7C	UART Status Reg (RO)
	uint32_t	TFL;		//0x80	Transmit FIFO Level (RO)
	uint32_t	RFL;		//0x84	Receive FIFO Level (RO)
	uint32_t	SRR;		//0x88	Software Reset Register (WO)
	uint32_t	SRTS;		//0x8C	Shadow Reset Register (R/W)
	uint32_t	SBCR;		//0x90	Shadow Break Control Reg (R/W)
	uint32_t	SDMAM;		//0x94	Shadow DMA Mode (R/W)
	uint32_t	SFE;		//0x98	Shadow FIFO Enable (R/W)
	uint32_t	SRT;		//0x9C	Shadow RCVR Trigger (R/W)
	uint32_t	STET;		//0xA0	Shadow TX Empty Trigger (R/W)
	uint32_t	HTX;		//0xA4	Halt TX (R/W)
	uint32_t	DMASA;		//0xA8	DMA Software Ack (WO)
	/*	ac
		b0 b4 b8 bc
		c0 c4 c8 cc
		d0 d4 d8 dc
		e0 e4 e8 ec
		f0 */
	uint32_t	dummy5[18]; //0xAc to 0xF0
	uint32_t	CPR;		//0xF4	Component Param Reg (RO)
	uint32_t	UCV;		//0xF8	UART Component Version (RO)
	uint32_t	CTR;		//0xFC	Component Type Reg (RO)
}dwApbUartType;


#define DWAPB_UART_RBR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_RBR_OFST))
#define DWAPB_UART_THR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_THR_OFST))
#define DWAPB_UART_DLL		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_DLL_OFST))
#define DWAPB_UART_DLH		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_DLH_OFST))
#define DWAPB_UART_IER		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_IER_OFST))
#define DWAPB_UART_IIR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_IIR_OFST))
#define DWAPB_UART_FCR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_FCR_OFST))
#define DWAPB_UART_LCR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_LCR_OFST))
#define DWAPB_UART_MCR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_MCR_OFST))
#define DWAPB_UART_LSR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_LSR_OFST))
#define DWAPB_UART_MSR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_MSR_OFST))
#define DWAPB_UART_SCR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_SCR_OFST))
#define DWAPB_UART_LPDLL	((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_LPDLL_OFST))
#define DWAPB_UART_LPDLH	((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_LPDLH_OFST))
#define DWAPB_UART_SRBR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_SRBR_OFST))
#define DWAPB_UART_STHR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_STHR_OFST))
#define DWAPB_UART_FAR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_FAR_OFST))
#define DWAPB_UART_TFR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_TFR_OFST))
#define DWAPB_UART_RFW		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_RFW_OFST))
#define DWAPB_UART_USR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_USR_OFST))
#define DWAPB_UART_TFL		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_TFL_OFST))
#define DWAPB_UART_RFL		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_RFL_OFST))
#define DWAPB_UART_SRR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_SRR_OFST))
#define DWAPB_UART_SRTS		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_SRTS_OFST))
#define DWAPB_UART_SBCR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_SBCR_OFST))
#define DWAPB_UART_SDMAM	((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_SDMAM_OFST))
#define DWAPB_UART_SFE		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_SFE_OFST))
#define DWAPB_UART_SRT		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_SRT_OFST))
#define DWAPB_UART_STET		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_STET_OFST))
#define DWAPB_UART_HTX		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_HTX_OFST))
#define DWAPB_UART_DMASA	((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_DMASA_OFST))
#define DWAPB_UART_CPR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_CPR_OFST))
#define DWAPB_UART_UCV		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_UCV_OFST))
#define DWAPB_UART_CTR		((volatile uint32_t *)(UART0_BASE+ DWAPB_UART_CTR_OFST))

/* Constants defined in our implementation */
#define APB_DATA_BUS_WIDTH	32
#define UART_FIFO_DEPTH		16

// Register bit pattern defines
/* RBR Receive Buffer register */
#define RBR_DATA     (0xFF<<0) /* Read Data */
// bits 7:0 are read only data, 31:8 are reserved & read as zero.

/* THR Transmit Holding register */
#define RBR_DATA     (0xFF<<0) /* Send Data */
// bits 7:0 are write only data, 31:8 are reserved & read as zero.

/* DLL Divsor Low Data register */
#define DLL_DATA		(0xFF<<0) /* Divisor Low Data */
// bits 7:0 are R/W data, 31:8 are reserved & read as zero.

/* DLH Divsor High Data register */
#define DLH_DATA		(0xFF<<0) /* Divisor High Data */
// bits 7:0 are R/W data, 31:8 are reserved & read as zero.

/* IER Interrupt Enable register */
#define IER_PTIME	(1<<7) // Prgmable THRE Int Mode Enable
#define IER_EDSSI	(1<<3) // Enable Modem Status Int
#define IER_ELSI		(1<<2) // Enable Receiver Line Status Int
#define IER_ETBEI	(1<<1) // Enable Transmit Holding Reg Empty Int
#define IER_ERBFI	(1<<0) // Enable Received Data Avail Int
// 31:8, 6:4 are reserved & read as zero.  All above bits reset to zero

/* IIR Interrupt Identification register */
#define IIR_FIFOS_MASK	(0x3<<6) // FIFO's Dis/Enabled Mask
#define IIR_FIFOSE	(0x3<<6) // FIFO's Enabled
#define IIR_FIFOSD	(0x0<<6) // FIFO's Disabled (Reset Val)

#define IIR_IID_MASK	(0xF<<0) // Int ID Mask
#define IIR_IID_MS	(0x0<<0) // Modem Status
#define IIR_IID_NI	(0x1<<0) // No Int (Reset Val)
#define IIR_IID_THRE	(0x2<<0) // THR Empty
#define IIR_IID_RDA	(0x4<<0) // Received Data Avail
#define IIR_IID_RLS	(0x6<<0) // Received Line Status
#define IIR_IID_BD	(0x7<<0) // Busy Detect
#define IIR_IID_CT	(0xC<<0) // Carrier Timeout
// 31:8, 5:4 are reserved & read as zero.


/* FCR FIFO Control register */
#define FCR_MASK	(0xFF<<0) // FCR Mask
#define FCR_RT_MASK	(0x3<<6) // RCVR Trigger Mask
#define FCR_RT_1CHR	(0x0<<6) // RCVR Trigger - 1 Char in FIFO (Reset Val)
#define FCR_RT_QTR	(0x1<<6) // RCVR Trigger - 1/4 full
#define FCR_RT_HALF	(0x2<<6) // RCVR Trigger - 1/2 full
#define FCR_RT_2LESS	(0x3<<6) // RCVR Trigger - 2 less than full


#define FCR_TET_MASK		(0x3<<4) // TX Empty Trigger Mask
#define FCR_TET_EMPTY	(0x0<<4) // TX Empty Trigger - FIFO Empty (Reset Val)
#define FCR_TET_2CHR		(0x1<<4) // TX Empty Trigger - 2 Chars in FIFO
#define FCR_TET_QTR		(0x2<<4) // TX Empty Trigger - 1/4 full
#define FCR_TET_HALF		(0x3<<4) // TX Empty Trigger - 1/2 full

#define FCR_TET_DMAM		(0x1<<3) // DMA Mode (Reset val 0)

#define FCR_XFIFORESET	(0x1<<2) // XMIT FIFO Reset (Reset val 0)

#define FCR_RFIFORESET	(0x1<<1) // RCV FIFO Reset (Reset val 0)

#define FCR_FIFOENBL	(0x1<<0) // FIFO Enable (Reset val 0)
// 31:8 are reserved & read as zero.


/* LCR Line Control register */
#define LCR_MASK		(0xFF<<0) // Line Control Reg Mask
#define LCR_DLAB		(0x1<<7)  // Divisor Latch Access Bit (Reset val 0)
#define LCR_BC		(0x1<<6)  // Break Control Bit (Reset val 0)
#define LCR_EPS		(0x1<<4)  // Even Parity Select Bit (Reset val 0)
#define LCR_PEN		(0x1<<3)  // Parity Enable Bit (Reset val 0)
#define LCR_STOP		(0x1<<2)  // Stop Bit (Reset val 0)
#define LCR_DLS_MASK	(0x3<<0)  // Data Length Select Mask
#define LCR_DLS_5B   (0x0<<0)  // 5 Bit Data (Reset val)
#define LCR_DLS_6B   (0x1<<0)  // 6 Bit Data
#define LCR_DLS_7B   (0x2<<0)  // 7 Bit Data
#define LCR_DLS_8B   (0x3<<0)  // 8 Bit Data
// 31:8,5 are reserved & read as zero.

/* MCR Modem Control register */
#define MCR_MASK		(0x7F<<0) // Modem Control Reg Mask
#define MCR_SIRE		(0x1<<6)  // SIR Mode Enable (Reset val 0)
#define MCR_AFCE		(0x1<<5)  // Auto Flow Control Enable (Reset val 0)
#define MCR_LB		(0x1<<4)  // Loop Back Enable Bit (Reset val 0)
#define MCR_OUT2		(0x1<<3)  // Out2 Bit (Reset val 0)
#define MCR_OUT1		(0x1<<2)  // Out1 Bit (Reset val 0)
#define MCR_RTS		(0x1<<1)  // RTS Bit (Reset val 0)
#define MCR_DTR		(0x1<<0)  // DTR Bit (Reset val 0)
// 31:7 are reserved & read as zero.

/* LSR Line Status register */
#define LSR_MASK		(0xFF<<0) // Line Status Reg Mask
#define LSR_RFE		(0x1<<7)  // Receiver FIFO Error Bit (Reset val 0)
#define LSR_TEMT		(0x1<<6)  // Xmit Empty Bit (Reset val 1)
#define LSR_THRE		(0x1<<5)  // Xmit Holding Reg Empty Bit (Reset val 1)
#define LSR_BI		(0x1<<4)  // Break Interrupt Bit (Reset val 0)
#define LSR_FE		(0x1<<3)  // Framing Error Bit (Reset val 0)
#define LSR_PE		(0x1<<2)  // Parity Error Bit (Reset val 0)
#define LSR_OE		(0x1<<1)  // Overrun Error Bit (Reset val 0)
#define LSR_DR		(0x1<<0)  // Data Ready Bit (Reset val 0)
// 31:8 are reserved & read as zero.

/* MSR Modem Status register */
#define MSR_MASK		(0xFF<<0) // Modem Status Reg Mask
#define MSR_DCD		(0x1<<7)  // Data Carrier Detect Bit (Reset val 0)
#define MSR_RI		(0x1<<6)  // Xmit Empty Bit (Reset val 0)
#define MSR_DSR		(0x1<<5)  // Data Set Ready Bit (Reset val 0)
#define MSR_CTS		(0x1<<4)  // Clear To Send Bit (Reset val 0)
#define MSR_DDCD		(0x1<<3)  // Delta Data Carrier Detect Bit (Reset val 0)
#define MSR_TERI		(0x1<<2)  // Trailing Edge of Ring Indicator Bit (Reset val 0)
#define MSR_DDSR		(0x1<<1)  // Delta Data Set Ready Bit (Reset val 0)
#define MSR_DCTS		(0x1<<0)  // Delta Clear To Send Bit (Reset val 0)
// 31:8 are reserved & read as zero.

/* SCR Scratchpad register */
#define SCR_MASK		(0xFF<<0) // Scratchpad Reg Mask
// 31:8 are reserved & read as zero.

/* LPDLL Low Power Divisor Latch Low register */
#define LPDLL_MASK	(0xFF<<0) // Low Power Divisor Latch Low Reg Mask
// 31:8 are reserved & read as zero.

/* LPDLH Low Power Divisor Latch High register */
#define LPDLH_MASK	(0xFF<<0) // Low Power Divisor Latch High Reg Mask
// 31:8 are reserved & read as zero.

/* SRBR Shadow Receive Buffer register */
#define SRBR_MASK	(0xFF<<0) // Shadow Receive Buffer Reg Mask
// 31:8 are reserved & read as zero.

/* SRBR Shadow Transmit Holding register */
#define STHR_MASK	(0xFF<<0) // Shadow Transmit Holding Reg Mask
// 31:8 are reserved & read as zero.

/* FAR FIFO Access register */
#define FAR_MASK	(0xFF<<0) // FIFO Access Reg Mask
// 31:8 are reserved & read as zero.

/* TFR Transmit FIFO Read register */
#define TFR_MASK	(0xFF<<0) // Transmit FIFO Read Reg Mask
// 31:8 are reserved & read as zero.

/* RFW Receive FIFO Write register */
#define RFW_MASK	(0x3FF<<0) // Receive FIFO Write Reg Mask
#define RFW_RFFE	(0x1<<9)   // Receive FIFO Framing Error (Reset val 0)
#define RFW_RFPE	(0x1<<8)   // Receive FIFO Parity Error (Reset val 0)
#define RFW_RFWD	(0xFF<<0)  // Receive FIFO Write Data (Reset val 0)
// 31:10 are reserved & read as zero.

/* USR UART Status register */
#define USR_MASK	(0x1F<<0) // UART Status Reg Mask
#define USR_RFF	(0x1<<4)  // Receive FIFO Full (Reset val 0)
#define USR_RFNE	(0x1<<3)  // Receive FIFO Not Empty (Reset val 0)
#define USR_TFE	(0x1<<2)  // Transmit FIFO Empty (Reset val 1)
#define USR_TFNF	(0x1<<1)  // Transmit FIFO Not Full (Reset val 1)
#define USR_BUSY	(0x1<<0)  // UART Busy (Reset val 0)
// 31:5 are reserved & read as zero.

/* Design Ware AMB UART DWAPB */
//void dwapb_uart_init( void );
//void dwapb_uart_out( char c );
//char dwapb_uart_in( void );
//void dwapb_interrupt( uint32_t intRegMask );
/* prototypes */
void dwApb_uart_init( int uartNum, int baudrate, int busSpd );
void dwApb_uart_out( int uartNum, uint8_t c );
void dwApb_uart_puts( int uartNum, uint8_t *s );
uint8_t dwApb_uart_in( int uartNum );
void dwApb_uart_enable_rcvr_int( int uartNum );
void dwApb_uart_disable_rcvr_int( int uartNum );
void dwApb_uart_interrupt ( int uartNum );
#endif
/* Used by vim and some versions of vi: set tabstop=4 shiftwidth=4: */
