/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2008-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/
/**
 * \file cpu.h
 *
 * \brief CPU device driver API.
 *
 * The common API used to interface to the CPU in an ASIC.
 *
 **/

#ifndef INC_CPU_H
#define INC_CPU_H

#include <stdint.h>
#include <stdbool.h>


#ifdef __cplusplus
extern "C" {
#endif

typedef enum
{
    CPU_MODE_USER,
    CPU_MODE_SUPERVISOR,
    CPU_MODE_INTERRUPT,
    CPU_MODE_OTHER
} CPU_MODE;

/**
 * \brief Initialize the cpu driver
 *
 * This function must be called before any other driver functions.
 **/
void cpu_init( void );

/**
 * \brief Get the current value of the CPU cycle counter
 * 
 * \return Cycle count in CPU clock ticks
 * 
 * This function returns the current value of the CPU cycle counter,
 * typically a free-running counter clocked by the CPU clock.
 **/
uint32_t cpu_get_ccount( void );

/** Cpu delay loop spins for at least given microseconds using cpu_get_ccount.
 *
 *  Uses cpu_get_count so time spent task switched out is accounted for,
 *  but the delay can be as much as a full task switch time greater than requested.
 *  Ideally should not be called for long intervals as it is a cpu spin loop.
 **/
void cpu_spin_delay(uint32_t delay_in_microseconds); 

/**
 * \brief Get the size of the cpu's data cache line in bytes
 * 
 * \return CPU data cache line size in bytes
 * 
 * This function returns the size of a cache line for the CPU
 * data cache.  This is useful for ensuring proper alignment
 * when utilizing the invalidate_region function.
 **/
uint32_t cpu_get_dcache_line_size( void );

/**
 * \brief Flush the entire data cache.
 *
 * Write everything in the data cache back to main memory.
 **/
void cpu_dcache_writeback_all( void );

/**
 * \brief Flush the data cache area starting at \a start_addr.
 *
 * \param start_addr The start address of the region of memory that will be
 *                   forced to be written back to main memory.
 *
 * \param size_in_bytes The size of the region of memory that will be forced
 *                      to be written back to main memory.
 *
 * Force the memory region defined by \a start_addr and \a size_in_bytes to be
 * written back to main memory if any of those memory locations were in
 * cache and needed to be written back to main memory (the data in cache was
 * "newer" than the data at the same address in main memory). This function
 * is typically used to force an area of memory to be updated with data that
 * might be in the cache before using DMA to pull data from that memory region.
 **/
void cpu_dcache_writeback_region( void *start_addr, uint32_t size_in_bytes );

/**
 * \brief Invalidate the entire data cache.
 *
 * Mark everything in the data cache as invalid.
 *
 * \warning This function invalidates everything in the data cache including
 *          dirty lines (data in the cache is "newer" than what's in memory)
 *          so that the "dirty" data will be lost, so use with extreme caution.
 **/
void cpu_dcache_invalidate_all( void );

void cpu_disable_dcache(void);

void cpu_enable_dcache(void);

void cpu_disable_icache(void);

void cpu_enable_icache(void);

/**
 * \brief Invalidate the data cache area starting at \a start_addr.
 *
 * \param start_addr The start address of the region of memory that will be
 *                   invalidated in the cache.
 *
 * \param size_in_bytes The size of the region of memory that will be forced
 *                      to be invalidated.
 *
 * Force the memory region defined by \a start_addr and \a size_in_bytes to be
 * marked as invalid. This is typically used on an area of memory that is
 * about to be filled by DMA process and you (the programmer) are confident
 * that the specified region of memory could not be in cache or has not been
 * updated since it was read into the cache. Exercise caution in using this
 * function because any changes to the data in cache (within the specified
 * memory region) will be discarded.
 *
 * \warning Because cache operations only operate on cache line boundaries,
 *          you should only call this function with a start address that is
 *          aligned with a cache line and a size that is a multiple
 *          of a cache line.  
 **/
void cpu_dcache_invalidate_region( void *start_addr, uint32_t size_in_bytes );

/**
 * \brief Get the size of the cpu's instruction cache line in bytes
 * 
 * \return CPU instruction cache line size in bytes
 * 
 * This function returns the size of a cache line for the CPU
 * instruction cache.  This is useful for ensuring proper alignment
 * when utilizing the invalidate_region function.
 **/
uint32_t cpu_get_icache_line_size( void );

/**
 * \brief Invalidate the entire instruction cache.
 *
 * Mark everything in the instruction cache as invalid.
 **/
void cpu_icache_invalidate_all( void );

/**
 * \brief Invalidate the instruction cache area starting at \a start_addr.
 *
 * \param start_addr The start address of the region of memory that will be
 *                   invalidated in the cache.
 *
 * \param size_in_bytes The size of the region of memory that will be forced
 *                      to be invalidated.
 *
 * \warning Because cache operations only operate on cache line boundaries,
 *          you should only call this function with a start address that is
 *          aligned with a cache line and a size that is a multiple
 *          of a cache line.  
 **/
void cpu_icache_invalidate_region( void *start_addr, uint32_t size_in_bytes );

/**
 * \brief Get the current operating mode of the CPU.
 *
 * \return The CPU operating mode.
 * 
 * The processor modes supported vary depending on the CPU.
 * Currently this function supports at least CPU_MODE_INTERRUPT
 * and one or more of the non-interrupt modes on all CPUs.
 *
 **/
CPU_MODE cpu_get_mode( void );

/**
 * \brief Globally disable all CPU interrupts.
 *
 * \return Current state of CPU interrupts.  The return value is an
 * opaque uint32_t that contains processor specific information 
 * regarding the state of the interrupts prior to the disable call.  
 * This value should be passed unmodified to the cpu_restore_interrupts 
 * call to return the interrupts to their previous posture.
 * 
 * \warning Use rarely and with extreme caution as this will disable ALL
 *          interrupts in the system until cpu_restore_interrupts is called.
 **/
uint32_t cpu_disable_interrupts( void );

/**
 * \brief Globally restore all CPU interrupts to their posture
 * prior to a preceeding call to cpu_disable_interrupts.
 * 
 * \param inter The interrupt state returned by a prior call to
 *              cpu_disable_interrupts.
 * 
 * \warning Use only after a call to cpu_disable_interrupts
 **/
void cpu_restore_interrupts( uint32_t previous_interrupt_state );

void enableMPU(void);
void disableMPU(void);
void startMPU(void);

#ifdef __cplusplus
}
#endif

#endif   // #ifndef INC_CPU_H

