 /*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#ifndef _APMU_TOP_REGSTRUCTS_H_
#define _APMU_TOP_REGSTRUCTS_H_

#include "hwincl.h"

//====================================================================
//
//Register File: Registers for AP PMU (APMU_top)
/** \brief The APMU provides various registers to control Power Management functions related to the
    Application Processing Subsystem of the SoC.*/
//
//====================================================================

//====================================================================
//
//Register File: APMU SRAM, Power Island control and miscellaneous registers (APMU_misc_reg)
/** \brief These registers provide control/status for the power islands, SRAMs, and other miscellaneous
    power related functions in the AP section of the chip.*/
//
//====================================================================

typedef struct APMU_MISC_REG_REGS_s
{
  volatile uint32_t A53Core0PwrIsland_PICR;  ///< 0x0 [R/W]: A53Core0 Power Island Configuration Register
  volatile uint32_t A53Core0PwrIsland_PISR;  ///< 0x4 [R]: A53Core0 Power Island Status Register
  volatile uint32_t A53Core1PwrIsland_PICR;  ///< 0x8 [R/W]: A53Core1 Power Island Configuration Register
  volatile uint32_t A53Core1PwrIsland_PISR;  ///< 0xc [R]: A53Core1 Power Island Status Register
  volatile uint32_t A53Core2PwrIsland_PICR;  ///< 0x10 [R/W]: A53Core2 Power Island Configuration Register
  volatile uint32_t A53Core2PwrIsland_PISR;  ///< 0x14 [R]: A53Core2 Power Island Status Register
  volatile uint32_t A53Core3PwrIsland_PICR;  ///< 0x18 [R/W]: A53Core3 Power Island Configuration Register
  volatile uint32_t A53Core3PwrIsland_PISR;  ///< 0x1c [R]: A53Core3 Power Island Status Register
  volatile uint32_t A53_L2PwrIsland_PICR;  ///< 0x20 [R/W]: A53 L2 Power Island Configuration Register
  volatile uint32_t A53_L2PwrIsland_PISR;  ///< 0x24 [R]: A53 L2 Power Island Status Register
  volatile uint32_t A53OtherPwrIsland_PICR;  ///< 0x28 [R/W]: A53 Other Power Island Configuration Register
  volatile uint32_t A53OtherPwrIsland_PISR;  ///< 0x2c [R]: A53 Other Power Island Status Register
  volatile uint32_t GPUPwrIsland_PICR;  ///< 0x30 [R/W]: GPU Power Island Configuration Register
  volatile uint32_t GPUPwrIsland_PISR;  ///< 0x34 [R]: GPU Power Island Status Register
  volatile uint32_t A53L2_SRAMPdwn;  ///< 0x38 [R/W]: A53 L2 Powerdown Control Register
  volatile uint32_t GC400LT_SRAMPdwn_SPCTL;  ///< 0x3c [R/W]: GC400LT SRAM Powerdown Control Register
  volatile uint32_t GC400LT_SRAMSpeedCtl_SRAMCTL;  ///< 0x40 [R/W]: GC400LT SRAM Speed Control Register
  volatile uint32_t GC400LT_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x44 [R/W]: GC400LT RF2P SRAMSpeed Control Register
  volatile uint32_t NAND_SRAMPdwn_SPCTL;  ///< 0x48 [R/W]: NAND SRAM Powerdown Control Register
  volatile uint32_t NAND_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x4c [R/W]: NAND RF2P SRAMSpeed Control Register
  volatile uint32_t SCCP_SRAMPdwn_SPCTL;  ///< 0x50 [R/W]: SCCP SRAM Powerdown Control Register
  volatile uint32_t SCCP_SRAMSpeedCtl_SRAMCTL;  ///< 0x54 [R/W]: SCCP SRAM Speed Control Register
  volatile uint32_t LANRX_SRAMPdwn_SPCTL;  ///< 0x58 [R/W]: LANRx SRAM Powerdown Control Register
  volatile uint32_t LANTX_SRAMPdwn_SPCTL;  ///< 0x5c [R/W]: LANTx SRAM Powerdown Control Register
  volatile uint32_t LAN_SRAMSpeedCtl_SRAMCTL;  ///< 0x60 [R/W]: LAN SRAM Speed Control Register
  volatile uint32_t LAN_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x64 [R/W]: LAN RF2P SRAMSpeed Control Register
  volatile uint32_t SATA_SRAMPdwn_SPCTL;  ///< 0x68 [R/W]: SATA SRAM Powerdown Control Register
  volatile uint32_t SATA_SRAMSpeedCtl_SRAMCTL;  ///< 0x6c [R/W]: SATA SRAM Speed Control Register
  volatile uint32_t SATA_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x70 [R/W]: SATA RF2P SRAMSpeed Control Register
  volatile uint32_t PCIe_dm2_SRAMPdwn_SPCTL;  ///< 0x74 [R/W]: PCIe_dm2 SRAM Powerdown Control Register
  volatile uint32_t PCIe_dm2_SRAMSpeedCtl_SRAMCTL;  ///< 0x78 [R/W]: PCIe_dm2 SRAM Speed Control Register
  volatile uint32_t PCIe_dm2_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x7c [R/W]: PCIe_dm2 RF2P SRAMSpeed Control Register
  volatile uint32_t PCIe_dm4_SRAMPdwn_SPCTL;  ///< 0x80 [R/W]: PCIe_dm4 SRAM Powerdown Control Register
  volatile uint32_t PCIe_dm4_SRAMSpeedCtl_SRAMCTL;  ///< 0x84 [R/W]: PCIe_dm4 SRAM Speed Control Register
  volatile uint32_t PCIe_dm4_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x88 [R/W]: PCIe_dm4 RF2P SRAMSpeed Control Register
  volatile uint32_t M2M0_SRAMPdwn_SPCTL;  ///< 0x8c [R/W]: M2M0 SRAM Powerdown Control Register
  volatile uint32_t M2M0_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x90 [R/W]: M2M0 RF2P SRAMSpeed Control Register
  volatile uint32_t M2M1_SRAMPdwn_SPCTL;  ///< 0x94 [R/W]: M2M1 SRAM Powerdown Control Register
  volatile uint32_t M2M1_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x98 [R/W]: M2M1 RF2P SRAMSpeed Control Register
  volatile uint32_t SDMMC_SRAMPdwn_SPCTL;  ///< 0x9c [R/W]: SDMMC SRAM Powerdown Control Register
  volatile uint32_t SDMMC_SRAMSpeedCtl_SRAMCTL;  ///< 0xa0 [R/W]: SDMMC SRAM Speed Control Register
  volatile uint32_t usb3d_SRAMPdwn_SPCTL;  ///< 0xa4 [R/W]: usb3d SRAM Powerdown Control Register
  volatile uint32_t usb3d_SRAMSpeedCtl_SRAMCTL;  ///< 0xa8 [R/W]: usb3d SRAM Speed Control Register
  volatile uint32_t usb3h_SRAMPdwn_SPCTL;  ///< 0xac [R/W]: usb3h SRAM Powerdown Control Register
  volatile uint32_t usb3h_SRAMSpeedCtl_SRAMCTL;  ///< 0xb0 [R/W]: usb3h SRAM Speed Control Register
  volatile uint32_t usb3h_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0xb4 [R/W]: usb3h RF2P SRAMSpeed Control Register
  volatile uint32_t usb3h_SRAMSpeedCtl_RF2PSRAMCTL1;  ///< 0xb8 [R/W]: usb3h RF2P SRAMSpeed Control Register 1
  volatile uint32_t usb3h_SRAMSpeedCtl_RF2PSRAMCTL2;  ///< 0xbc [R/W]: usb3h RF2P SRAMSpeed Control Register 2
  volatile uint32_t usbdm_SRAMPdwn_SPCTL;  ///< 0xc0 [R/W]: usbdm SRAM Powerdown Control Register
  volatile uint32_t usbdm_SRAMSpeedCtl_SRAMCTL;  ///< 0xc4 [R/W]: usbdm SRAM Speed Control Register
  volatile uint32_t usbh_SRAMPdwn_SPCTL;  ///< 0xc8 [R/W]: usbh SRAM Powerdown Control Register
  volatile uint32_t usbh_SRAMSpeedCtl_SRAMCTL;  ///< 0xcc [R/W]: usbh SRAM Speed Control Register
  volatile uint32_t lcd_SRAMPdwn_SPCTL;  ///< 0xd0 [R/W]: lcd SRAM Powerdown Control Register
  volatile uint32_t lcd_SRAMSpeedCtl_SRAMCTL;  ///< 0xd4 [R/W]: lcd SRAM Speed Control Register
  volatile uint32_t lcd_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0xd8 [R/W]: lcd RF2P SRAMSpeed Control Register
  volatile uint32_t PDMA0_SRAMPdwn_SPCTL;  ///< 0xdc [R/W]: PDMA0 SRAM Powerdown Control Register
  volatile uint32_t PDMA0_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0xe0 [R/W]: PDMA0 RF2P SRAMSpeed Control Register
  volatile uint32_t PDMA1_SRAMPdwn_SPCTL;  ///< 0xe4 [R/W]: PDMA1 SRAM Powerdown Control Register
  volatile uint32_t PDMA1_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0xe8 [R/W]: PDMA1 RF2P SRAMSpeed Control Register
  volatile uint32_t Coresight_SRAMPdwn_SPCTL;  ///< 0xec [R/W]: Coresight SRAM Powerdown Control Register
  volatile uint32_t Coresight_SRAMSpeedCtl_SRAMCTL;  ///< 0xf0 [R/W]: Coresight SRAM Speed Control Register
  volatile uint32_t SMMU_SRAMPdwn_SPCTL;  ///< 0xf4 [R/W]: SMMU SRAM Powerdown Control Register
  volatile uint32_t SMMU_SRAMSpeedCtl_SRAMCTL;  ///< 0xf8 [R/W]: SMMU SRAM Speed Control Register
  volatile uint32_t SQU_SRAMPdwn_SPCTL;  ///< 0xfc [R/W]: SQU SRAM Powerdown Control Register
  volatile uint32_t SQU_SRAMSpeedCtl_SRAMCTL;  ///< 0x100 [R/W]: SQU SRAM Speed Control Register
  volatile uint32_t MC5_SRAMPdwn_SPCTL;  ///< 0x104 [R/W]: MC5 SRAM Powerdown Control Register
  volatile uint32_t MC5_SRAMSpeedCtl_SRAMCTL;  ///< 0x108 [R/W]: MC5 SRAM Speed Control Register
  volatile uint32_t RedunRepairCtl;  ///< 0x10c [R/W]: Redundancy Repair Control Register
  volatile uint32_t RepairDone;  ///< 0x110 [R]: Redundancy Repair Done Status Register
  volatile uint32_t ClkDomIntEnReg1;  ///< 0x114 [R/W]: Clock Domain Interrupt Enable Register 1
  volatile uint32_t ClkDomIntPendReg1;  ///< 0x118 [R]: Clock Domain Interrupt Pending Register 1
  volatile uint32_t ClkDomIntAckReg1;  ///< 0x11c [W]: Clock Domain Interrupt Acknowledge Register 1
  volatile uint32_t ClkDomIntForceReg1;  ///< 0x120 [W]: Clock Domain Interrupt Force Register 1
  volatile uint32_t ClkDomIntEnReg2;  ///< 0x124 [R/W]: Clock Domain Interrupt Enable Register 2
  volatile uint32_t ClkDomIntPendReg2;  ///< 0x128 [R]: Clock Domain Interrupt Pending Register 2
  volatile uint32_t ClkDomIntAckReg2;  ///< 0x12c [W]: Clock Domain Interrupt Acknowledge Register 2
  volatile uint32_t ClkDomIntForceReg2;  ///< 0x130 [W]: Clock Domain Interrupt Force Register 2
  volatile uint32_t MiscIntEnReg;  ///< 0x134 [R/W]: Miscellaneous Interrupt Enable Register
  volatile uint32_t MiscIntPendReg;  ///< 0x138 [R]: Miscellaneous Interrupt Pending Register
  volatile uint32_t MiscIntAckReg;  ///< 0x13c [W]: Miscellaneous Interrupt Acknowledge Register
  volatile uint32_t MiscIntForceReg;  ///< 0x140 [W]: Miscellaneous Interrupt Force Register
  volatile uint32_t R4_AXI_ID;  ///< 0x144 [R/W]: R4 AXI ID register
  volatile uint32_t R4_AXI_MASK;  ///< 0x148 [R/W]: R4 AXI ID mask register
  volatile uint32_t ERR_AXI_ID;  ///< 0x14c [R]: Error AXI ID
  volatile uint32_t GENERIC_CTRL;  ///< 0x150 [R/W]: General Control Register
  volatile uint32_t WAKE_CLR;  ///< 0x154 [W]: Wakeup Clear Register
  volatile uint32_t REV0;  ///< 0x158 [R]: IP Tagging Revision 0
  volatile uint32_t REV1;  ///< 0x15c [R]: IP Tagging Revision 1
  volatile uint32_t reserved0[8];
  volatile uint32_t BCM_SRAMPdwn_SPCTL;  ///< 0x180 [R/W]: BCM SRAM Powerdown Control Register
  volatile uint32_t BCM_SRAMSpeedCtl_RF2PSRAMCTL;  ///< 0x184 [R/W]: BCM RF2P SRAMSpeed Control Register
  volatile uint32_t R4_SRAMSpeedCtl_SRAMCTL;  ///< 0x188 [R/W]: R4 SRAM Speed Control Register
  volatile uint32_t BCM_ROMCtl_RCTL;  ///< 0x18c [R/W]: BCM ROM Powerdown/Read Timing Control Register
} APMU_MISC_REG_REGS_t;

//====================================================================
//
//Register File: APMU Clock and Reset Control Registers (APMU_ClkRstGen)
/** \brief These registers provide control/status for the clocks/resets for all of the clock domains
    in the Application Processing section of the chip.*/
//
//====================================================================

typedef struct APMU_CLKRSTGEN_REGS_REVB_s
{
  volatile uint32_t CPUDbgClkConfig;  ///< 0x0 [R/W]: CPU Debug Clock Configuration Register
  volatile uint32_t CPUDbgClkStatus;  ///< 0x4 [R]: CPU Debug Clock Status Register
  volatile uint32_t TPIUClkConfig_ClkGenConfig;  ///< 0x8 [R/W]: TPIUClk ClkGen Configuration Register
  volatile uint32_t TPIUClkConfig_ClkGenStatus;  ///< 0xc [R]: TPIUClk ClkGen Status Register
  volatile uint32_t MCBusGuardConfig_ClkGenConfig;  ///< 0x10 [R/W]: MCBusGuard ClkGen Configuration Register
  volatile uint32_t MCBusGuardConfig_ClkGenStatus;  ///< 0x14 [R]: MCBusGuard ClkGen Status Register
  volatile uint32_t reserved0[2];
  volatile uint32_t DDRClkConfig; /*DDRClkConfig_ClkGenConfig*/  ///< 0x20 [R/W]: DDR Clock Configuration Register
  volatile uint32_t DDRClkStatus; /*DDRClkConfig_ClkGenStatus*/  ///< 0x24 [R]: DDR Clock Status Register
  volatile uint32_t I2CClkConfig_ClkGenConfig; /*I2C1ClkConfig_ClkGenConfig*/  ///< 0x28 [R/W]: I2C1Clk ClkGen Configuration Register
  volatile uint32_t I2CClkConfig_ClkGenStatus; /*I2C1ClkConfig_ClkGenStatus*/ ///< 0x2c [R]: I2C1Clk ClkGen Status Register
  volatile uint32_t I2C2ClkConfig_ClkGenConfig;  ///< 0x30 [R/W]: I2C2Clk ClkGen Configuration Register
  volatile uint32_t I2C2ClkConfig_ClkGenStatus;  ///< 0x34 [R]: I2C2Clk ClkGen Status Register
  volatile uint32_t I2C3ClkConfig_ClkGenConfig;  ///< 0x38 [R/W]: I2C3Clk ClkGen Configuration Register
  volatile uint32_t I2C3ClkConfig_ClkGenStatus;  ///< 0x3c [R]: I2C3Clk ClkGen Status Register
  volatile uint32_t I2C4ClkConfig_ClkGenConfig;  ///< 0x40 [R/W]: I2C4Clk ClkGen Configuration Register
  volatile uint32_t I2C4ClkConfig_ClkGenStatus;  ///< 0x44 [R]: I2C4Clk ClkGen Status Register
  volatile uint32_t I2C5ClkConfig_ClkGenConfig;  ///< 0x48 [R/W]: I2C5Clk ClkGen Configuration Register
  volatile uint32_t I2C5ClkConfig_ClkGenStatus;  ///< 0x4c [R]: I2C5Clk ClkGen Status Register
  volatile uint32_t I2C6ClkConfig_ClkGenConfig;  ///< 0x50 [R/W]: I2C6Clk ClkGen Configuration Register
  volatile uint32_t I2C6ClkConfig_ClkGenStatus;  ///< 0x54 [R]: I2C6Clk ClkGen Status Register
  volatile uint32_t SSPClkConfig_ClkGenConfig; /*SSP1ClkConfig_ClkGenConfig*/  ///< 0x58 [R/W]: SSP1Clk ClkGen Configuration Register
  volatile uint32_t SSPClkConfig_ClkGenStatus; /*SSP1ClkConfig_ClkGenStatus*/  ///< 0x5c [R]: SSP1Clk ClkGen Status Register
  volatile uint32_t SSP2ClkConfig_ClkGenConfig;  ///< 0x60 [R/W]: SSP2Clk ClkGen Configuration Register
  volatile uint32_t SSP2ClkConfig_ClkGenStatus;  ///< 0x64 [R]: SSP2Clk ClkGen Status Register
  volatile uint32_t SSP3ClkConfig_ClkGenConfig;  ///< 0x68 [R/W]: SSP3Clk ClkGen Configuration Register
  volatile uint32_t SSP3ClkConfig_ClkGenStatus;  ///< 0x6c [R]: SSP3Clk ClkGen Status Register
  volatile uint32_t APBusClkConfig_ClkGenConfig;  ///< 0x70 [R/W]: APBusClk ClkGen Configuration Register
  volatile uint32_t APBusClkConfig_ClkGenStatus;  ///< 0x74 [R]: APBusClk ClkGen Status Register
  volatile uint32_t APBusClkDiv2Config_ClkGenConfig;  ///< 0x78 [R/W]: APBusClkDiv2 ClkGen Configuration Register
  volatile uint32_t APBusClkDiv2Config_ClkGenStatus;  ///< 0x7c [R]: APBusClkDiv2 ClkGen Status Register
  volatile uint32_t APBusClk_APBConfig_ClkGenConfig;  ///< 0x80 [R/W]: APBusClk_APB ClkGen Configuration Register
  volatile uint32_t APBusClk_APBConfig_ClkGenStatus;  ///< 0x84 [R]: APBusClk_APB ClkGen Status Register
  volatile uint32_t APBusClk_LCDConfig_ClkGenConfig;  ///< 0x88 [R/W]: APBusClk_LCD ClkGen Configuration Register
  volatile uint32_t APBusClk_LCDConfig_ClkGenStatus;  ///< 0x8c [R]: APBusClk_LCD ClkGen Status Register
  volatile uint32_t APBusClk_CDMAConfig_ClkGenConfig;  ///< 0x90 [R/W]: APBusClk_CDMA ClkGen Configuration Register
  volatile uint32_t APBusClk_CDMAConfig_ClkGenStatus;  ///< 0x94 [R]: APBusClk_CDMA ClkGen Status Register
  volatile uint32_t APBusClk_SQUConfig_ClkGenConfig;  ///< 0x98 [R/W]: APBusClk_SQU ClkGen Configuration Register
  volatile uint32_t APBusClk_SQUConfig_ClkGenStatus;  ///< 0x9c [R]: APBusClk_SQU ClkGen Status Register
  volatile uint32_t APBusClk_IOWrapConfig_ClkGenConfig;  ///< 0xa0 [R/W]: APBusClk_IOWrap ClkGen Configuration Register
  volatile uint32_t APBusClk_IOWrapConfig_ClkGenStatus;  ///< 0xa4 [R]: APBusClk_IOWrap ClkGen Status Register
  volatile uint32_t APBusClk_LANConfig_ClkGenConfig;  ///< 0xa8 [R/W]: APBusClk_LAN ClkGen Configuration Register
  volatile uint32_t APBusClk_LANConfig_ClkGenStatus;  ///< 0xac [R]: APBusClk_LAN ClkGen Status Register
  volatile uint32_t APBusClk_USB2AConfig_ClkGenConfig;  ///< 0xb0 [R/W]: APBusClk_USB2A ClkGen Configuration Register
  volatile uint32_t APBusClk_USB2AConfig_ClkGenStatus;  ///< 0xb4 [R]: APBusClk_USB2A ClkGen Status Register
  volatile uint32_t APBusClk_USB2PHYConfig_ClkGenConfig;  ///< 0xb8 [R/W]: APBusClk_USB2PHY ClkGen Configuration Register
  volatile uint32_t APBusClk_USB2PHYConfig_ClkGenStatus;  ///< 0xbc [R]: APBusClk_USB2PHY ClkGen Status Register
  volatile uint32_t APBusClk_USB3DConfig_ClkGenConfig;  ///< 0xc0 [R/W]: APBusClk_USB3D ClkGen Configuration Register
  volatile uint32_t APBusClk_USB3DConfig_ClkGenStatus;  ///< 0xc4 [R]: APBusClk_USB3D ClkGen Status Register
  volatile uint32_t APBusClk_USB3PHYConfig_ClkGenConfig;  ///< 0xc8 [R/W]: APBusClk_USB3PHY ClkGen Configuration Register
  volatile uint32_t APBusClk_USB3PHYConfig_ClkGenStatus;  ///< 0xcc [R]: APBusClk_USB3PHY ClkGen Status Register
  volatile uint32_t APBusClk_PCIe_DM2Config_ClkGenConfig;  ///< 0xd0 [R/W]: APBusClk_PCIe_DM2 ClkGen Configuration Register
  volatile uint32_t APBusClk_PCIe_DM2Config_ClkGenStatus;  ///< 0xd4 [R]: APBusClk_PCIe_DM2 ClkGen Status Register
  volatile uint32_t APBusClk_PCIe_DM4Config_ClkGenConfig;  ///< 0xd8 [R/W]: APBusClk_PCIe_DM4 ClkGen Configuration Register
  volatile uint32_t APBusClk_PCIe_DM4Config_ClkGenStatus;  ///< 0xdc [R]: APBusClk_PCIe_DM4 ClkGen Status Register
  volatile uint32_t APBusClk_SATAConfig_ClkGenConfig;  ///< 0xe0 [R/W]: APBusClk_SATA ClkGen Configuration Register
  volatile uint32_t APBusClk_SATAConfig_ClkGenStatus;  ///< 0xe4 [R]: APBusClk_SATA ClkGen Status Register
  volatile uint32_t APBusClk_GPUConfig_ClkGenConfig;  ///< 0xe8 [R/W]: APBusClk_GPU ClkGen Configuration Register
  volatile uint32_t APBusClk_GPUConfig_ClkGenStatus;  ///< 0xec [R]: APBusClk_GPU ClkGen Status Register
  volatile uint32_t SMMUClkConfig_ClkGenConfig;  ///< 0xf0 [R/W]: SMMUClk ClkGen Configuration Register
  volatile uint32_t SMMUClkConfig_ClkGenStatus;  ///< 0xf4 [R]: SMMUClk ClkGen Status Register
  volatile uint32_t GPUBusClkConfig_ClkGenConfig;  ///< 0xf8 [R/W]: GPUBusClk ClkGen Configuration Register
  volatile uint32_t GPUBusClkConfig_ClkGenStatus;  ///< 0xfc [R]: GPUBusClk ClkGen Status Register
  volatile uint32_t GPUClkConfig_ClkGenConfig;  ///< 0x100 [R/W]: GPUClk ClkGen Configuration Register
  volatile uint32_t GPUClkConfig_ClkGenStatus;  ///< 0x104 [R]: GPUClk ClkGen Status Register
  volatile uint32_t SDMMCClkConfig_ClkGenConfig;  ///< 0x108 [R/W]: SDMMCClk ClkGen Configuration Register
  volatile uint32_t SDMMCClkConfig_ClkGenStatus;  ///< 0x10c [R]: SDMMCClk ClkGen Status Register
  volatile uint32_t QSPIClkConfig_ClkGenConfig;  ///< 0x110 [R/W]: QSPIClk ClkGen Configuration Register
  volatile uint32_t QSPIClkConfig_ClkGenStatus;  ///< 0x114 [R]: QSPIClk ClkGen Status Register
  volatile uint32_t LCDXClkConfig_ClkGenConfig;  ///< 0x118 [R/W]: LCDXClk ClkGen Configuration Register
  volatile uint32_t LCDXClkConfig_ClkGenStatus;  ///< 0x11c [R]: LCDXClk ClkGen Status Register
  volatile uint32_t LCDClkConfig;  ///< 0x120 [R/W]: LCD Clock Configuration Register
  volatile uint32_t LCDClkStatus;  ///< 0x124 [R]: LCD Clock Status Register
  volatile uint32_t UART0ClkConfig_ClkGenConfig; /*UART1ClkConfig_ClkGenConfig*/  ///< 0x128 [R/W]: UART1Clk ClkGen Configuration Register
  volatile uint32_t UART0ClkConfig_ClkGenStatus; /*UART1ClkConfig_ClkGenStatus*/  ///< 0x12c [R]: UART1Clk ClkGen Status Register
  volatile uint32_t UART0ClkFracDivCfg; /*UART1ClkFracDivCfg*/  ///< 0x130 [R/W]: UART1 Clock Fractional Divider Configuration Register
  volatile uint32_t UART1ClkConfig_ClkGenConfig; /*UART2ClkConfig_ClkGenConfig*/  ///< 0x134 [R/W]: UART2Clk ClkGen Configuration Register
  volatile uint32_t UART1ClkConfig_ClkGenStatus; /*UART2ClkConfig_ClkGenStatus*/  ///< 0x138 [R]: UART2Clk ClkGen Status Register
  volatile uint32_t UART1ClkFracDivCfg; /*UART2ClkFracDivCfg*/  ///< 0x13c [R/W]: UART2 Clock Fractional Divider Configuration Register
  volatile uint32_t UART2ClkConfig_ClkGenConfig; /*UART3ClkConfig_ClkGenConfig*/  ///< 0x140 [R/W]: UART3Clk ClkGen Configuration Register
  volatile uint32_t UART2ClkConfig_ClkGenStatus; /*UART3ClkConfig_ClkGenStatus*/  ///< 0x144 [R]: UART3Clk ClkGen Status Register
  volatile uint32_t UART2ClkFracDivCfg; /*UART3ClkFracDivCfg*/  ///< 0x148 [R/W]: UART3 Clock Fractional Divider Configuration Register
  volatile uint32_t UART3ClkConfig_ClkGenConfig; /*UART4ClkConfig_ClkGenConfig*/  ///< 0x14c [R/W]: UART4Clk ClkGen Configuration Register
  volatile uint32_t UART3ClkConfig_ClkGenStatus; /*UART4ClkConfig_ClkGenStatus*/  ///< 0x150 [R]: UART4Clk ClkGen Status Register
  volatile uint32_t UART3ClkFracDivCfg; /*UART4ClkFracDivCfg*/  ///< 0x154 [R/W]: UART4 Clock Fractional Divider Configuration Register
  volatile uint32_t LANMacClk2xConfig;  ///< 0x158 [R/W]: LAN Mac 2x Clock Configuration Register
  volatile uint32_t LANMacClk2xStatus;  ///< 0x15c [R]: LAN Mac 2x Clock Status Register
  volatile uint32_t LANRxClkGenConfig;  ///< 0x160 [R/W]: ClkGen Configuration Register
  volatile uint32_t LANRxClkGenStatus;  ///< 0x164 [R]: ClkGen Status Register
  volatile uint32_t APCPUMiscControl;  ///< 0x168 [R/W]: AP CPU Miscellaneous Control Register
  volatile uint32_t APCPUClkControl;  ///< 0x16c [R/W]: AP CPU Clock Control Register
  volatile uint32_t APCPUClkStatus;  ///< 0x170 [R]: AP CPU Clocks Status Register
  volatile uint32_t R4ClkStatus;  ///< 0x174 [R]: R4 CPU Clocks Status Register
  volatile uint32_t PCIeRstCtl;  ///< 0x178 [R/W]: PCIe Reset Control
} APMU_CLKRSTGEN_REGS_REVB_t;

typedef struct APMU_CLKRSTGEN_REGS_s
{
  volatile uint32_t CPUDbgClkConfig;  ///< 0x0 [R/W]: CPU Debug Clock Configuration Register
  volatile uint32_t CPUDbgClkStatus;  ///< 0x4 [R]: CPU Debug Clock Status Register
  volatile uint32_t TPIUClkConfig_ClkGenConfig;  ///< 0x8 [R/W]: TPIUClk ClkGen Configuration Register
  volatile uint32_t TPIUClkConfig_ClkGenStatus;  ///< 0xc [R]: TPIUClk ClkGen Status Register
  volatile uint32_t MCBusGuardConfig_ClkGenConfig;  ///< 0x10 [R/W]: MCBusGuard ClkGen Configuration Register
  volatile uint32_t MCBusGuardConfig_ClkGenStatus;  ///< 0x14 [R]: MCBusGuard ClkGen Status Register
  volatile uint32_t DDRClk2xConfig_ClkGenConfig;  ///< 0x18 [R/W]: DDRClk2x ClkGen Configuration Register
  volatile uint32_t DDRClk2xConfig_ClkGenStatus;  ///< 0x1c [R]: DDRClk2x ClkGen Status Register
  volatile uint32_t DDRClkConfig;  ///< 0x20 [R/W]: DDR Clock Configuration Register
  volatile uint32_t DDRClkStatus;  ///< 0x24 [R]: DDR Clock Status Register
  volatile uint32_t I2CClkConfig_ClkGenConfig;  ///< 0x28 [R/W]: I2CClk ClkGen Configuration Register
  volatile uint32_t I2CClkConfig_ClkGenStatus;  ///< 0x2c [R]: I2CClk ClkGen Status Register
  volatile uint32_t SSPClkConfig_ClkGenConfig;  ///< 0x30 [R/W]: SSPClk ClkGen Configuration Register
  volatile uint32_t SSPClkConfig_ClkGenStatus;  ///< 0x34 [R]: SSPClk ClkGen Status Register
  volatile uint32_t APBusClkConfig_ClkGenConfig;  ///< 0x38 [R/W]: APBusClk ClkGen Configuration Register
  volatile uint32_t APBusClkConfig_ClkGenStatus;  ///< 0x3c [R]: APBusClk ClkGen Status Register
  volatile uint32_t APBusClkDiv2Config_ClkGenConfig;  ///< 0x40 [R/W]: APBusClkDiv2 ClkGen Configuration Register
  volatile uint32_t APBusClkDiv2Config_ClkGenStatus;  ///< 0x44 [R]: APBusClkDiv2 ClkGen Status Register
  volatile uint32_t APBusClk_APBConfig_ClkGenConfig;  ///< 0x48 [R/W]: APBusClk_APB ClkGen Configuration Register
  volatile uint32_t APBusClk_APBConfig_ClkGenStatus;  ///< 0x4c [R]: APBusClk_APB ClkGen Status Register
  volatile uint32_t APBusClk_LCDConfig_ClkGenConfig;  ///< 0x50 [R/W]: APBusClk_LCD ClkGen Configuration Register
  volatile uint32_t APBusClk_LCDConfig_ClkGenStatus;  ///< 0x54 [R]: APBusClk_LCD ClkGen Status Register
  volatile uint32_t APBusClk_CDMAConfig_ClkGenConfig;  ///< 0x58 [R/W]: APBusClk_CDMA ClkGen Configuration Register
  volatile uint32_t APBusClk_CDMAConfig_ClkGenStatus;  ///< 0x5c [R]: APBusClk_CDMA ClkGen Status Register
  volatile uint32_t APBusClk_SQUConfig_ClkGenConfig;  ///< 0x60 [R/W]: APBusClk_SQU ClkGen Configuration Register
  volatile uint32_t APBusClk_SQUConfig_ClkGenStatus;  ///< 0x64 [R]: APBusClk_SQU ClkGen Status Register
  volatile uint32_t APBusClk_IOWrapConfig_ClkGenConfig;  ///< 0x68 [R/W]: APBusClk_IOWrap ClkGen Configuration Register
  volatile uint32_t APBusClk_IOWrapConfig_ClkGenStatus;  ///< 0x6c [R]: APBusClk_IOWrap ClkGen Status Register
  volatile uint32_t APBusClk_LANConfig_ClkGenConfig;  ///< 0x70 [R/W]: APBusClk_LAN ClkGen Configuration Register
  volatile uint32_t APBusClk_LANConfig_ClkGenStatus;  ///< 0x74 [R]: APBusClk_LAN ClkGen Status Register
  volatile uint32_t APBusClk_USB2AConfig_ClkGenConfig;  ///< 0x78 [R/W]: APBusClk_USB2A ClkGen Configuration Register
  volatile uint32_t APBusClk_USB2AConfig_ClkGenStatus;  ///< 0x7c [R]: APBusClk_USB2A ClkGen Status Register
  volatile uint32_t APBusClk_USB2PHYConfig_ClkGenConfig;  ///< 0x80 [R/W]: APBusClk_USB2PHY ClkGen Configuration Register
  volatile uint32_t APBusClk_USB2PHYConfig_ClkGenStatus;  ///< 0x84 [R]: APBusClk_USB2PHY ClkGen Status Register
  volatile uint32_t APBusClk_USB3DConfig_ClkGenConfig;  ///< 0x88 [R/W]: APBusClk_USB3D ClkGen Configuration Register
  volatile uint32_t APBusClk_USB3DConfig_ClkGenStatus;  ///< 0x8c [R]: APBusClk_USB3D ClkGen Status Register
  volatile uint32_t APBusClk_USB3HConfig_ClkGenConfig;  ///< 0x90 [R/W]: APBusClk_USB3H ClkGen Configuration Register
  volatile uint32_t APBusClk_USB3HConfig_ClkGenStatus;  ///< 0x94 [R]: APBusClk_USB3H ClkGen Status Register
  volatile uint32_t APBusClk_USB3PHYConfig_ClkGenConfig;  ///< 0x98 [R/W]: APBusClk_USB3PHY ClkGen Configuration Register
  volatile uint32_t APBusClk_USB3PHYConfig_ClkGenStatus;  ///< 0x9c [R]: APBusClk_USB3PHY ClkGen Status Register
  volatile uint32_t APBusClk_PCIe_DM2Config_ClkGenConfig;  ///< 0xa0 [R/W]: APBusClk_PCIe_DM2 ClkGen Configuration Register
  volatile uint32_t APBusClk_PCIe_DM2Config_ClkGenStatus;  ///< 0xa4 [R]: APBusClk_PCIe_DM2 ClkGen Status Register
  volatile uint32_t APBusClk_PCIe_DM4Config_ClkGenConfig;  ///< 0xa8 [R/W]: APBusClk_PCIe_DM4 ClkGen Configuration Register
  volatile uint32_t APBusClk_PCIe_DM4Config_ClkGenStatus;  ///< 0xac [R]: APBusClk_PCIe_DM4 ClkGen Status Register
  volatile uint32_t APBusClk_SATAConfig_ClkGenConfig;  ///< 0xb0 [R/W]: APBusClk_SATA ClkGen Configuration Register
  volatile uint32_t APBusClk_SATAConfig_ClkGenStatus;  ///< 0xb4 [R]: APBusClk_SATA ClkGen Status Register
  volatile uint32_t APBusClk_GPUConfig_ClkGenConfig;  ///< 0xb8 [R/W]: APBusClk_GPU ClkGen Configuration Register
  volatile uint32_t APBusClk_GPUConfig_ClkGenStatus;  ///< 0xbc [R]: APBusClk_GPU ClkGen Status Register
  volatile uint32_t SMMUClkConfig_ClkGenConfig;  ///< 0xc0 [R/W]: SMMUClk ClkGen Configuration Register
  volatile uint32_t SMMUClkConfig_ClkGenStatus;  ///< 0xc4 [R]: SMMUClk ClkGen Status Register
  volatile uint32_t GPUBusClkConfig_ClkGenConfig;  ///< 0xc8 [R/W]: GPUBusClk ClkGen Configuration Register
  volatile uint32_t GPUBusClkConfig_ClkGenStatus;  ///< 0xcc [R]: GPUBusClk ClkGen Status Register
  volatile uint32_t GPUClkConfig_ClkGenConfig;  ///< 0xd0 [R/W]: GPUClk ClkGen Configuration Register
  volatile uint32_t GPUClkConfig_ClkGenStatus;  ///< 0xd4 [R]: GPUClk ClkGen Status Register
  volatile uint32_t SDMMCClkConfig_ClkGenConfig;  ///< 0xd8 [R/W]: SDMMCClk ClkGen Configuration Register
  volatile uint32_t SDMMCClkConfig_ClkGenStatus;  ///< 0xdc [R]: SDMMCClk ClkGen Status Register
  volatile uint32_t QSPIClkConfig_ClkGenConfig;  ///< 0xe0 [R/W]: QSPIClk ClkGen Configuration Register
  volatile uint32_t QSPIClkConfig_ClkGenStatus;  ///< 0xe4 [R]: QSPIClk ClkGen Status Register
  volatile uint32_t LCDXClkConfig_ClkGenConfig;  ///< 0xe8 [R/W]: LCDXClk ClkGen Configuration Register
  volatile uint32_t LCDXClkConfig_ClkGenStatus;  ///< 0xec [R]: LCDXClk ClkGen Status Register
  volatile uint32_t LCDClkConfig;  ///< 0xf0 [R/W]: LCD Clock Configuration Register
  volatile uint32_t LCDClkStatus;  ///< 0xf4 [R]: LCD Clock Status Register
  volatile uint32_t UART0ClkConfig_ClkGenConfig;  ///< 0xf8 [R/W]: UART0Clk ClkGen Configuration Register
  volatile uint32_t UART0ClkConfig_ClkGenStatus;  ///< 0xfc [R]: UART0Clk ClkGen Status Register
  volatile uint32_t UART0ClkFracDivCfg;  ///< 0x100 [R/W]: UART0 Clock Fractional Divider Configuration Register
  volatile uint32_t UART1ClkConfig_ClkGenConfig;  ///< 0x104 [R/W]: UART1Clk ClkGen Configuration Register
  volatile uint32_t UART1ClkConfig_ClkGenStatus;  ///< 0x108 [R]: UART1Clk ClkGen Status Register
  volatile uint32_t UART1ClkFracDivCfg;  ///< 0x10c [R/W]: UART1 Clock Fractional Divider Configuration Register
  volatile uint32_t UART2ClkConfig_ClkGenConfig;  ///< 0x110 [R/W]: UART2Clk ClkGen Configuration Register
  volatile uint32_t UART2ClkConfig_ClkGenStatus;  ///< 0x114 [R]: UART2Clk ClkGen Status Register
  volatile uint32_t UART2ClkFracDivCfg;  ///< 0x118 [R/W]: UART2 Clock Fractional Divider Configuration Register
  volatile uint32_t UART3ClkConfig_ClkGenConfig;  ///< 0x11c [R/W]: UART3Clk ClkGen Configuration Register
  volatile uint32_t UART3ClkConfig_ClkGenStatus;  ///< 0x120 [R]: UART3Clk ClkGen Status Register
  volatile uint32_t UART3ClkFracDivCfg;  ///< 0x124 [R/W]: UART3 Clock Fractional Divider Configuration Register
  volatile uint32_t LANMacClk2xConfig;  ///< 0x128 [R/W]: LAN Mac 2x Clock Configuration Register
  volatile uint32_t LANMacClk2xStatus;  ///< 0x12c [R]: LAN Mac 2x Clock Status Register
  volatile uint32_t LANRxClkGenConfig;  ///< 0x130 [R/W]: ClkGen Configuration Register
  volatile uint32_t LANRxClkGenStatus;  ///< 0x134 [R]: ClkGen Status Register
  volatile uint32_t APCPUMiscControl;  ///< 0x138 [R/W]: AP CPU Miscellaneous Control Register
  volatile uint32_t APCPUClkControl;  ///< 0x13c [R/W]: AP CPU Clock Control Register
  volatile uint32_t APCPUClkStatus;  ///< 0x140 [R]: AP CPU Clocks Status Register
  volatile uint32_t R4ClkStatus;  ///< 0x144 [R]: R4 CPU Clocks Status Register
  volatile uint32_t PCIeRstCtl;  ///< 0x148 [R/W]: PCIe Reset Control
  volatile uint32_t reserved0[13];
  volatile uint32_t BCMClkConfig_ClkGenConfig;  ///< 0x180 [R/W]: BCMClk ClkGen Configuration Register
  volatile uint32_t BCMClkConfig_ClkGenStatus;  ///< 0x184 [R]: BCMClk ClkGen Status Register
  volatile uint32_t BCMCPUClkConfig_ClkGenConfig;  ///< 0x188 [R/W]: BCMCPUClk ClkGen Configuration Register
  volatile uint32_t BCMCPUClkConfig_ClkGenStatus;  ///< 0x18c [R]: BCMCPUClk ClkGen Status Register
} APMU_CLKRSTGEN_REGS_t;

//====================================================================
//
//Register File: APMU Timer Control Registers (APMU_timers)
/** \brief The APB TIMERS block provides programmable interval timers and watchdog. <BR>
The interval timers can be used to generate a repetitive period timer (for the RTO/S periodic tick interrupt), to time intervals (for switch presses, LED flashing, or fan speed), or for a watchdog reset. <BR>
Four identical programmable timers (Timer0, Timer1, Timer2, Timer3) are provided in the APB subsystem. Each timer generates a level interrupt when it reaches the final event, which can be programmed to be either a terminal count or an input pulse (on an external signal). When the final event is reached, the counter can restart at zero (continuous mode) or halt, depending on how it is configured. <BR>
Timer0 provides the watchdog function, and when enabled, the system will be reset if the watchdog timer reaches the final event. <BR>
The timer count is incremented on a selected input timebase. There are 8 choices of input timebases to select from: 1us, 10us, 100us, 1ms, 10ms and 100ms, the bus clock, and external input signal pulses. <BR>
To facilitate interval timing, the counter may be disabled to lock the count value that can then be read via the Timer Status Register. The counter is reset to zero and begins to increment when the Timer is re-enabled. <BR>
To assist system level functions, each timer can generate a signal that toggles when the final count is reached. The toggling signal is a periodic waveform that can be used within the SoC or be connected to the pins for further use by the system. <BR>*/
//
//====================================================================

typedef struct APMU_TIMERS_REGS_s
{
  volatile uint32_t TWR0;  ///< 0x0 [R/W]: Timer0 Watchdog Register
  volatile uint32_t TTCR0;  ///< 0x4 [R/W]: Timer0 Terminal Count Register
  volatile uint32_t TCR0;  ///< 0x8 [R/W]: Timer0 Control Register
  volatile uint32_t TSR0;  ///< 0xc [R]: Timer0 Status Register
  volatile uint32_t TISR;  ///< 0x10 [R]: Timers Interrupt Status Register
  volatile uint32_t TTCR1;  ///< 0x14 [R/W]: Timer1 Terminal Count Register
  volatile uint32_t TCR1;  ///< 0x18 [R/W]: Timer1 Control Register
  volatile uint32_t TSR1;  ///< 0x1c [R]: Timer1 Status Register
  volatile uint32_t TIAR;  ///< 0x20 [W]: Timers Interrupt Acknowledge Register
  volatile uint32_t TTCR2;  ///< 0x24 [R/W]: Timer2 Terminal Count Register
  volatile uint32_t TCR2;  ///< 0x28 [R/W]: Timer2 Control Register
  volatile uint32_t TSR2;  ///< 0x2c [R]: Timer2 Status Register
  volatile uint32_t reserved0;
  volatile uint32_t TTCR3;  ///< 0x34 [R/W]: Timer3 Terminal Count Register
  volatile uint32_t TCR3;  ///< 0x38 [R/W]: Timer3 Control Register
  volatile uint32_t TSR3;  ///< 0x3c [R]: Timer3 Status Register
  volatile uint32_t REV0;  ///< 0x40 [R]: IP Tagging Revision 0
  volatile uint32_t REV1;  ///< 0x44 [R]: IP Tagging Revision 1
} APMU_TIMERS_REGS_t;

#endif // APMU_TOP
 /*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#ifndef _APMU_TOP_REGMASKS_H_
#define _APMU_TOP_REGMASKS_H_

#include "hwincl.h"

//====================================================================
//
//Register File: Registers for AP PMU (APMU_top)
/** \brief The APMU provides various registers to control Power Management functions related to the
    Application Processing Subsystem of the SoC.*/
//
//====================================================================

//====================================================================
//
//Register File: APMU SRAM, Power Island control and miscellaneous registers (APMU_misc_reg)
/** \brief These registers provide control/status for the power islands, SRAMs, and other miscellaneous
    power related functions in the AP section of the chip.*/
//
//====================================================================

//====================================================================
//Register: A53Core0 Power Island Configuration Register (A53Core0PwrIsland_PICR)
/** \brief <p>The A53Core0 Power Island Configuration Register (PICR) and Power Island Status Register (PISR) contain fields used to control and monitor the A53Core0 Power Island. When power to this island is switched off the blocks in it are inactive and consume zero power. When the power is switched back on, the blocks get a complete hardware reset to return them to their default power-on state. </p>
        To switch off this island:<br/>
        <ul>
        <li>Before powering down the island, check the corresponding bit in the
        APCPUClkStatus.DbgNoPwrDwn field to ensure the core isn't in the process of 
        being debugged.</li>
        <li>Clear the DbgPwrdUp field to let the debugger know you are powering down the core.</li>
        <li>Disable clocks that go to the island using the corresponding ClkConfig registers.</li>
        <li>Clear the ISOB field to isolate the interface to the island.</li>
        <li>Clear both bits of the BULKPWRUP field.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR and verify that the BULKSTATUS field is 00.</li>
        <li>Clear the DISTPWRUP field.</li>
        </ul>
        <br />
        To switch this island back on:
        <ul>
        <li>Assert resets for all clock domains in the island via the ClkGenConfig
        register(s).</li>
        <li>Set the BULKPWRUP field to 01.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 01.</li>
        <li>Set the BULKPWRUP field to 11.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 11.</li>
        <li>Set the DISTPWRUP field to 1.</li>
        <li>Enable the clock(s) to the island in the ClkGenConfig register(s) for
        all clock domains on the island. This will propogate the asserted reset signal.</li>
        <li>Disable the clock(s) to the island in the ClkGenConfig register(s).</li>
        <li>Deassert the resets for all clock domains in the ClkGenConfig register(s).
        </li>
        <li>Set the ISOB field to 1 to de-isolate the island.</li>
        <li>Enable the clock(s) for the island in the ClkGenConfig register(s).</li>
        <li>Set the DbgPwrdUp field to let the debugger know the core is powered up.</li>
        </ul>
        Only switch one island at a time to avoid high surge currents which could cause a malfunction.*/
//====================================================================

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED1_MASK 0xfffffc00
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED1_SHIFT 10
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED1_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_MASK 0x200
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_SHIFT 9
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DBGPWRDUP_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_MASK 0x100
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_SHIFT 8
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PICR_DISTPWRUP_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED2_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_MASK 0x30
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_SHIFT 4
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PICR_BULKPWRUP_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED3_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PICR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PICR_ISOB_SHIFT))

//====================================================================
//Register: A53Core0 Power Island Status Register (A53Core0PwrIsland_PISR)
/** \brief The Power Island Status Register contains the current state of the power island controls.
      It is identical to the contents of the PICR except when the state is changing.*/
//====================================================================

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED1_MASK 0xfffffe00
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED1_SHIFT 9
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED1_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_DISTSTATUS_MASK 0x100
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_DISTSTATUS_SHIFT 8
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_DISTSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PISR_DISTSTATUS_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PISR_DISTSTATUS_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_DISTSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PISR_DISTSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PISR_DISTSTATUS_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED2_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_MASK 0x30
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_SHIFT 4
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PISR_BULKSTATUS_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED3_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PISR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_MASK) >> APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE0PWRISLAND_PISR_ISOB_SHIFT))

//====================================================================
//Register: A53Core1 Power Island Configuration Register (A53Core1PwrIsland_PICR)
/** \brief <p>The A53Core1 Power Island Configuration Register (PICR) and Power Island Status Register (PISR) contain fields used to control and monitor the A53Core1 Power Island. When power to this island is switched off the blocks in it are inactive and consume zero power. When the power is switched back on, the blocks get a complete hardware reset to return them to their default power-on state. </p>
        To switch off this island:<br/>
        <ul>
        <li>Before powering down the island, check the corresponding bit in the
        APCPUClkStatus.DbgNoPwrDwn field to ensure the core isn't in the process of 
        being debugged.</li>
        <li>Clear the DbgPwrdUp field to let the debugger know you are powering down the core.</li>
        <li>Disable clocks that go to the island using the corresponding ClkConfig registers.</li>
        <li>Clear the ISOB field to isolate the interface to the island.</li>
        <li>Clear both bits of the BULKPWRUP field.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR and verify that the BULKSTATUS field is 00.</li>
        <li>Clear the DISTPWRUP field.</li>
        </ul>
        <br />
        To switch this island back on:
        <ul>
        <li>Assert resets for all clock domains in the island via the ClkGenConfig
        register(s).</li>
        <li>Set the BULKPWRUP field to 01.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 01.</li>
        <li>Set the BULKPWRUP field to 11.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 11.</li>
        <li>Set the DISTPWRUP field to 1.</li>
        <li>Enable the clock(s) to the island in the ClkGenConfig register(s) for
        all clock domains on the island. This will propogate the asserted reset signal.</li>
        <li>Disable the clock(s) to the island in the ClkGenConfig register(s).</li>
        <li>Deassert the resets for all clock domains in the ClkGenConfig register(s).
        </li>
        <li>Set the ISOB field to 1 to de-isolate the island.</li>
        <li>Enable the clock(s) for the island in the ClkGenConfig register(s).</li>
        <li>Set the DbgPwrdUp field to let the debugger know the core is powered up.</li>
        </ul>
        Only switch one island at a time to avoid high surge currents which could cause a malfunction.*/
//====================================================================

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED1_MASK 0xfffffc00
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED1_SHIFT 10
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED1_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DBGPWRDUP_MASK 0x200
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DBGPWRDUP_SHIFT 9
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DBGPWRDUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DBGPWRDUP_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DBGPWRDUP_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DBGPWRDUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DBGPWRDUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DBGPWRDUP_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DISTPWRUP_MASK 0x100
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DISTPWRUP_SHIFT 8
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DISTPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DISTPWRUP_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DISTPWRUP_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DISTPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DISTPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PICR_DISTPWRUP_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED2_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_BULKPWRUP_MASK 0x30
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_BULKPWRUP_SHIFT 4
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_BULKPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PICR_BULKPWRUP_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PICR_BULKPWRUP_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PICR_BULKPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PICR_BULKPWRUP_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED3_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PICR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PICR_ISOB_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PICR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PICR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PICR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PICR_ISOB_SHIFT))

//====================================================================
//Register: A53Core1 Power Island Status Register (A53Core1PwrIsland_PISR)
/** \brief The Power Island Status Register contains the current state of the power island controls.
      It is identical to the contents of the PICR except when the state is changing.*/
//====================================================================

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED1_MASK 0xfffffe00
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED1_SHIFT 9
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED1_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_DISTSTATUS_MASK 0x100
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_DISTSTATUS_SHIFT 8
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_DISTSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PISR_DISTSTATUS_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PISR_DISTSTATUS_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_DISTSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PISR_DISTSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PISR_DISTSTATUS_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED2_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_BULKSTATUS_MASK 0x30
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_BULKSTATUS_SHIFT 4
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PISR_BULKSTATUS_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PISR_BULKSTATUS_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_BULKSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PISR_BULKSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PISR_BULKSTATUS_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED3_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PISR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE1PWRISLAND_PISR_ISOB_MASK) >> APMU_MISC_REG_A53CORE1PWRISLAND_PISR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53CORE1PWRISLAND_PISR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE1PWRISLAND_PISR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE1PWRISLAND_PISR_ISOB_SHIFT))

//====================================================================
//Register: A53Core2 Power Island Configuration Register (A53Core2PwrIsland_PICR)
/** \brief <p>The A53Core2 Power Island Configuration Register (PICR) and Power Island Status Register (PISR) contain fields used to control and monitor the A53Core2 Power Island. When power to this island is switched off the blocks in it are inactive and consume zero power. When the power is switched back on, the blocks get a complete hardware reset to return them to their default power-on state. </p>
        To switch off this island:<br/>
        <ul>
        <li>Before powering down the island, check the corresponding bit in the
        APCPUClkStatus.DbgNoPwrDwn field to ensure the core isn't in the process of 
        being debugged.</li>
        <li>Clear the DbgPwrdUp field to let the debugger know you are powering down the core.</li>
        <li>Disable clocks that go to the island using the corresponding ClkConfig registers.</li>
        <li>Clear the ISOB field to isolate the interface to the island.</li>
        <li>Clear both bits of the BULKPWRUP field.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR and verify that the BULKSTATUS field is 00.</li>
        <li>Clear the DISTPWRUP field.</li>
        </ul>
        <br />
        To switch this island back on:
        <ul>
        <li>Assert resets for all clock domains in the island via the ClkGenConfig
        register(s).</li>
        <li>Set the BULKPWRUP field to 01.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 01.</li>
        <li>Set the BULKPWRUP field to 11.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 11.</li>
        <li>Set the DISTPWRUP field to 1.</li>
        <li>Enable the clock(s) to the island in the ClkGenConfig register(s) for
        all clock domains on the island. This will propogate the asserted reset signal.</li>
        <li>Disable the clock(s) to the island in the ClkGenConfig register(s).</li>
        <li>Deassert the resets for all clock domains in the ClkGenConfig register(s).
        </li>
        <li>Set the ISOB field to 1 to de-isolate the island.</li>
        <li>Enable the clock(s) for the island in the ClkGenConfig register(s).</li>
        <li>Set the DbgPwrdUp field to let the debugger know the core is powered up.</li>
        </ul>
        Only switch one island at a time to avoid high surge currents which could cause a malfunction.*/
//====================================================================

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED1_MASK 0xfffffc00
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED1_SHIFT 10
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED1_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DBGPWRDUP_MASK 0x200
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DBGPWRDUP_SHIFT 9
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DBGPWRDUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DBGPWRDUP_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DBGPWRDUP_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DBGPWRDUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DBGPWRDUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DBGPWRDUP_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DISTPWRUP_MASK 0x100
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DISTPWRUP_SHIFT 8
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DISTPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DISTPWRUP_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DISTPWRUP_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DISTPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DISTPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PICR_DISTPWRUP_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED2_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_BULKPWRUP_MASK 0x30
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_BULKPWRUP_SHIFT 4
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_BULKPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PICR_BULKPWRUP_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PICR_BULKPWRUP_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PICR_BULKPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PICR_BULKPWRUP_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED3_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PICR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PICR_ISOB_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PICR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PICR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PICR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PICR_ISOB_SHIFT))

//====================================================================
//Register: A53Core2 Power Island Status Register (A53Core2PwrIsland_PISR)
/** \brief The Power Island Status Register contains the current state of the power island controls.
      It is identical to the contents of the PICR except when the state is changing.*/
//====================================================================

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED1_MASK 0xfffffe00
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED1_SHIFT 9
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED1_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_DISTSTATUS_MASK 0x100
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_DISTSTATUS_SHIFT 8
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_DISTSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PISR_DISTSTATUS_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PISR_DISTSTATUS_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_DISTSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PISR_DISTSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PISR_DISTSTATUS_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED2_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_BULKSTATUS_MASK 0x30
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_BULKSTATUS_SHIFT 4
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PISR_BULKSTATUS_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PISR_BULKSTATUS_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_BULKSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PISR_BULKSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PISR_BULKSTATUS_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED3_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PISR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE2PWRISLAND_PISR_ISOB_MASK) >> APMU_MISC_REG_A53CORE2PWRISLAND_PISR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53CORE2PWRISLAND_PISR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE2PWRISLAND_PISR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE2PWRISLAND_PISR_ISOB_SHIFT))

//====================================================================
//Register: A53Core3 Power Island Configuration Register (A53Core3PwrIsland_PICR)
/** \brief <p>The A53Core3 Power Island Configuration Register (PICR) and Power Island Status Register (PISR) contain fields used to control and monitor the A53Core3 Power Island. When power to this island is switched off the blocks in it are inactive and consume zero power. When the power is switched back on, the blocks get a complete hardware reset to return them to their default power-on state. </p>
        To switch off this island:<br/>
        <ul>
        <li>Before powering down the island, check the corresponding bit in the
        APCPUClkStatus.DbgNoPwrDwn field to ensure the core isn't in the process of 
        being debugged.</li>
        <li>Clear the DbgPwrdUp field to let the debugger know you are powering down the core.</li>
        <li>Disable clocks that go to the island using the corresponding ClkConfig registers.</li>
        <li>Clear the ISOB field to isolate the interface to the island.</li>
        <li>Clear both bits of the BULKPWRUP field.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR and verify that the BULKSTATUS field is 00.</li>
        <li>Clear the DISTPWRUP field.</li>
        </ul>
        <br />
        To switch this island back on:
        <ul>
        <li>Assert resets for all clock domains in the island via the ClkGenConfig
        register(s).</li>
        <li>Set the BULKPWRUP field to 01.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 01.</li>
        <li>Set the BULKPWRUP field to 11.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 11.</li>
        <li>Set the DISTPWRUP field to 1.</li>
        <li>Enable the clock(s) to the island in the ClkGenConfig register(s) for
        all clock domains on the island. This will propogate the asserted reset signal.</li>
        <li>Disable the clock(s) to the island in the ClkGenConfig register(s).</li>
        <li>Deassert the resets for all clock domains in the ClkGenConfig register(s).
        </li>
        <li>Set the ISOB field to 1 to de-isolate the island.</li>
        <li>Enable the clock(s) for the island in the ClkGenConfig register(s).</li>
        <li>Set the DbgPwrdUp field to let the debugger know the core is powered up.</li>
        </ul>
        Only switch one island at a time to avoid high surge currents which could cause a malfunction.*/
//====================================================================

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED1_MASK 0xfffffc00
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED1_SHIFT 10
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED1_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DBGPWRDUP_MASK 0x200
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DBGPWRDUP_SHIFT 9
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DBGPWRDUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DBGPWRDUP_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DBGPWRDUP_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DBGPWRDUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DBGPWRDUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DBGPWRDUP_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DISTPWRUP_MASK 0x100
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DISTPWRUP_SHIFT 8
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DISTPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DISTPWRUP_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DISTPWRUP_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DISTPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DISTPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PICR_DISTPWRUP_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED2_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_BULKPWRUP_MASK 0x30
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_BULKPWRUP_SHIFT 4
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_BULKPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PICR_BULKPWRUP_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PICR_BULKPWRUP_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PICR_BULKPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PICR_BULKPWRUP_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED3_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PICR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PICR_ISOB_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PICR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PICR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PICR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PICR_ISOB_SHIFT))

//====================================================================
//Register: A53Core3 Power Island Status Register (A53Core3PwrIsland_PISR)
/** \brief The Power Island Status Register contains the current state of the power island controls.
      It is identical to the contents of the PICR except when the state is changing.*/
//====================================================================

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED1_MASK 0xfffffe00
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED1_SHIFT 9
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED1_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_DISTSTATUS_MASK 0x100
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_DISTSTATUS_SHIFT 8
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_DISTSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PISR_DISTSTATUS_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PISR_DISTSTATUS_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_DISTSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PISR_DISTSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PISR_DISTSTATUS_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED2_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_BULKSTATUS_MASK 0x30
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_BULKSTATUS_SHIFT 4
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PISR_BULKSTATUS_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PISR_BULKSTATUS_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_BULKSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PISR_BULKSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PISR_BULKSTATUS_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED3_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PISR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53CORE3PWRISLAND_PISR_ISOB_MASK) >> APMU_MISC_REG_A53CORE3PWRISLAND_PISR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53CORE3PWRISLAND_PISR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53CORE3PWRISLAND_PISR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53CORE3PWRISLAND_PISR_ISOB_SHIFT))

//====================================================================
//Register: A53 L2 Power Island Configuration Register (A53_L2PwrIsland_PICR)
/** \brief <p>The A53 L2 Power Island Configuration Register (PICR) and Power Island Status Register (PISR) contain fields used to control and monitor the A53 L2 Power Island. When power to this island is switched off the blocks in it are inactive and consume zero power. When the power is switched back on, the blocks get a complete hardware reset to return them to their default power-on state. </p>
        To switch off this island:<br/>
        <ul>
        <li>Disable clocks that go to the island using the corresponding ClkConfig registers.</li>
        <li>Clear both bits of the BULKPWRUP field.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR and verify that the BULKSTATUS field is 00.</li>
        </ul>
        <br />
        To switch this island back on:
        <ul>
        <li>Assert resets for all clock domains in the island via the ClkGenConfig
        register(s).</li>
        <li>Set the BULKPWRUP field to 01.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 01.</li>
        <li>Set the BULKPWRUP field to 11.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 11.</li>
        <li>Enable the clock(s) to the island in the ClkGenConfig register(s) for
        all clock domains on the island. This will propogate the asserted reset signal.</li>
        <li>Disable the clock(s) to the island in the ClkGenConfig register(s).</li>
        <li>Deassert the resets for all clock domains in the ClkGenConfig register(s).
        </li>
        <li>Enable the clock(s) for the island in the ClkGenConfig register(s).</li>
        </ul>
        Only switch one island at a time to avoid high surge currents which could cause a malfunction.*/
//====================================================================

#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED1_MASK 0xfffffc00
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED1_SHIFT 10
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED1_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED2_MASK 0x200
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED2_SHIFT 9
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED2_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED3_MASK 0x100
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED3_SHIFT 8
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED3_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED4_MASK 0xc0
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED4_SHIFT 6
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED4_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED4_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED4_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED4_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_MASK 0x30
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_SHIFT 4
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PICR_BULKPWRUP_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED5_MASK 0xe
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED5_SHIFT 1
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED5_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED5_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED5_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED5_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED6_MASK 0x1
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED6_SHIFT 0
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED6_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED6_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED6_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PICR_RESERVED6_SHIFT))

//====================================================================
//Register: A53 L2 Power Island Status Register (A53_L2PwrIsland_PISR)
/** \brief The Power Island Status Register contains the current state of the power island controls.
      It is identical to the contents of the PICR except when the state is changing.*/
//====================================================================

#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED1_MASK 0xfffffe00
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED1_SHIFT 9
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED1_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED2_MASK 0x100
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED2_SHIFT 8
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED2_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED3_MASK 0xc0
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED3_SHIFT 6
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED3_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_MASK 0x30
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_SHIFT 4
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PISR_BULKSTATUS_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED4_MASK 0xe
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED4_SHIFT 1
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED4_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED4_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED4_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED4_SHIFT))

#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED5_MASK 0x1
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED5_SHIFT 0
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED5_MASK) >> APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED5_SHIFT)
#define  APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED5_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53_L2PWRISLAND_PISR_RESERVED5_SHIFT))

//====================================================================
//Register: A53 Other Power Island Configuration Register (A53OtherPwrIsland_PICR)
/** \brief <p>The A53 Other Power Island Configuration Register (PICR) and Power Island Status Register (PISR) contain fields used to control and monitor the A53 Other Power Island. When power to this island is switched off the blocks in it are inactive and consume zero power. When the power is switched back on, the blocks get a complete hardware reset to return them to their default power-on state. </p>
        To switch off this island:<br/>
        <ul>
        <li>Disable clocks that go to the island using the corresponding ClkConfig registers.</li>
        <li>Clear the ISOB field to isolate the interface to the island.</li>
        <li>Clear both bits of the BULKPWRUP field.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR and verify that the BULKSTATUS field is 00.</li>
        <li>Clear the DISTPWRUP field.</li>
        </ul>
        <br />
        To switch this island back on:
        <ul>
        <li>Assert resets for all clock domains in the island via the ClkGenConfig
        register(s).</li>
        <li>Set the BULKPWRUP field to 01.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 01.</li>
        <li>Set the BULKPWRUP field to 11.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 11.</li>
        <li>Set the DISTPWRUP field to 1.</li>
        <li>Enable the clock(s) to the island in the ClkGenConfig register(s) for
        all clock domains on the island. This will propogate the asserted reset signal.</li>
        <li>Disable the clock(s) to the island in the ClkGenConfig register(s).</li>
        <li>Deassert the resets for all clock domains in the ClkGenConfig register(s).
        </li>
        <li>Set the ISOB field to 1 to de-isolate the island.</li>
        <li>Enable the clock(s) for the island in the ClkGenConfig register(s).</li>
        </ul>
        Only switch one island at a time to avoid high surge currents which could cause a malfunction.*/
//====================================================================

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED1_MASK 0xfffffc00
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED1_SHIFT 10
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED1_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED2_MASK 0x200
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED2_SHIFT 9
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED2_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_DISTPWRUP_MASK 0x100
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_DISTPWRUP_SHIFT 8
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_DISTPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PICR_DISTPWRUP_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PICR_DISTPWRUP_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_DISTPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PICR_DISTPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PICR_DISTPWRUP_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED3_MASK 0xc0
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED3_SHIFT 6
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED3_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_BULKPWRUP_MASK 0x30
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_BULKPWRUP_SHIFT 4
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_BULKPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PICR_BULKPWRUP_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PICR_BULKPWRUP_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PICR_BULKPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PICR_BULKPWRUP_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED4_MASK 0xe
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED4_SHIFT 1
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED4_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED4_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED4_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PICR_RESERVED4_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PICR_ISOB_SHIFT))

//====================================================================
//Register: A53 Other Power Island Status Register (A53OtherPwrIsland_PISR)
/** \brief The Power Island Status Register contains the current state of the power island controls.
      It is identical to the contents of the PICR except when the state is changing.*/
//====================================================================

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED1_MASK 0xfffffe00
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED1_SHIFT 9
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED1_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_DISTSTATUS_MASK 0x100
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_DISTSTATUS_SHIFT 8
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_DISTSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PISR_DISTSTATUS_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PISR_DISTSTATUS_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_DISTSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PISR_DISTSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PISR_DISTSTATUS_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED2_MASK 0xc0
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED2_SHIFT 6
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED2_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_BULKSTATUS_MASK 0x30
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_BULKSTATUS_SHIFT 4
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PISR_BULKSTATUS_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PISR_BULKSTATUS_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_BULKSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PISR_BULKSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PISR_BULKSTATUS_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED3_MASK 0xe
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED3_SHIFT 1
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED3_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PISR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_MASK 0x1
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_SHIFT 0
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_MASK) >> APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_SHIFT)
#define  APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53OTHERPWRISLAND_PISR_ISOB_SHIFT))

//====================================================================
//Register: GPU Power Island Configuration Register (GPUPwrIsland_PICR)
/** \brief <p>The GPU Power Island Configuration Register (PICR) and Power Island Status Register (PISR) contain fields used to control and monitor the GPU Power Island. When power to this island is switched off the blocks in it are inactive and consume zero power. When the power is switched back on, the blocks get a complete hardware reset to return them to their default power-on state. </p>
        To switch off this island:<br/>
        <ul>
        <li>Disable clocks that go to the island using the corresponding ClkConfig registers.</li>
        <li>Clear the ISOB field to isolate the interface to the island.</li>
        <li>Clear both bits of the BULKPWRUP field.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR and verify that the BULKSTATUS field is 00.</li>
        </ul>
        <br />
        To switch this island back on:
        <ul>
        <li>Assert resets for all clock domains in the island via the ClkGenConfig
        register(s).</li>
        <li>Set the BULKPWRUP field to 01.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 01.</li>
        <li>Set the BULKPWRUP field to 11.</li>
        <li>Wait at least 200 us (no maximum).</li>
        <li>Read the PISR until the BULKSTATUS field is 11.</li>
        <li>Enable the clock(s) to the island in the ClkGenConfig register(s) for
        all clock domains on the island. This will propogate the asserted reset signal.</li>
        <li>Disable the clock(s) to the island in the ClkGenConfig register(s).</li>
        <li>Deassert the resets for all clock domains in the ClkGenConfig register(s).
        </li>
        <li>Set the ISOB field to 1 to de-isolate the island.</li>
        <li>Enable the clock(s) for the island in the ClkGenConfig register(s).</li>
        </ul>
        Only switch one island at a time to avoid high surge currents which could cause a malfunction.*/
//====================================================================

#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED1_MASK 0xfffffc00
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED1_SHIFT 10
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED1_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED2_MASK 0x200
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED2_SHIFT 9
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED2_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED3_MASK 0x100
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED3_SHIFT 8
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED3_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED4_MASK 0xc0
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED4_SHIFT 6
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED4_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED4_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED4_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED4_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_MASK 0x30
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_SHIFT 4
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PICR_BULKPWRUP_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED5_MASK 0xe
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED5_SHIFT 1
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED5_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED5_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED5_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PICR_RESERVED5_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_MASK 0x1
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_SHIFT 0
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PICR_ISOB_SHIFT))

//====================================================================
//Register: GPU Power Island Status Register (GPUPwrIsland_PISR)
/** \brief The Power Island Status Register contains the current state of the power island controls.
      It is identical to the contents of the PICR except when the state is changing.*/
//====================================================================

#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED1_MASK 0xfffffe00
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED1_SHIFT 9
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED1_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED2_MASK 0x100
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED2_SHIFT 8
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED2_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED2_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED2_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED3_MASK 0xc0
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED3_SHIFT 6
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED3_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED3_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED3_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_MASK 0x30
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_SHIFT 4
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PISR_BULKSTATUS_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED4_MASK 0xe
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED4_SHIFT 1
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED4_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED4_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED4_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PISR_RESERVED4_SHIFT))

#define  APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_MASK 0x1
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_SHIFT 0
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_MASK) >> APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_SHIFT)
#define  APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_MASK) | (((uint32_t)val) << APMU_MISC_REG_GPUPWRISLAND_PISR_ISOB_SHIFT))

//====================================================================
//Register: A53 L2 Powerdown Control Register (A53L2_SRAMPdwn)
/** \brief The fields of this register drive the low leakage power control signals of the A53 L2 SRAMs*/
//====================================================================

#define  APMU_MISC_REG_A53L2_SRAMPDWN_RESERVED1_MASK 0xfffffffe
#define  APMU_MISC_REG_A53L2_SRAMPDWN_RESERVED1_SHIFT 1
#define  APMU_MISC_REG_A53L2_SRAMPDWN_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53L2_SRAMPDWN_RESERVED1_MASK) >> APMU_MISC_REG_A53L2_SRAMPDWN_RESERVED1_SHIFT)
#define  APMU_MISC_REG_A53L2_SRAMPDWN_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53L2_SRAMPDWN_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53L2_SRAMPDWN_RESERVED1_SHIFT))

#define  APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_MASK 0x1
#define  APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_SHIFT 0
#define  APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_MASK) >> APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_SHIFT)
#define  APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_A53L2_SRAMPDWN_PDWN_SHIFT))

//====================================================================
//Register: GC400LT SRAM Powerdown Control Register (GC400LT_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the GC400LT
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: GC400LT SRAM Speed Control Register (GC400LT_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the GC400LT block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: GC400LT RF2P SRAMSpeed Control Register (GC400LT_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the GC400LT block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_GC400LT_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: NAND SRAM Powerdown Control Register (NAND_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the NAND
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_NAND_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: NAND RF2P SRAMSpeed Control Register (NAND_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the NAND block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_NAND_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: SCCP SRAM Powerdown Control Register (SCCP_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the SCCP
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_SCCP_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: SCCP SRAM Speed Control Register (SCCP_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the SCCP block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SCCP_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: LANRx SRAM Powerdown Control Register (LANRX_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the LANRx
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_LANRX_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: LANTx SRAM Powerdown Control Register (LANTX_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the LANTx
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_LANTX_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: LAN SRAM Speed Control Register (LAN_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the LAN block
        (least significant fields to most significant fields):
        RX SRAM*/
//====================================================================

#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_LAN_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: LAN RF2P SRAMSpeed Control Register (LAN_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the LAN block
        (least significant fields to most significant fields):
        TX SRAM*/
//====================================================================

#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_LAN_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: SATA SRAM Powerdown Control Register (SATA_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the SATA
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: SATA SRAM Speed Control Register (SATA_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the SATA block
        (least significant fields to most significant fields):
        128x24, 128x64, 32x32, 32x64*/
//====================================================================

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xffff0000
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 16
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK 0xc000
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT 14
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK 0x3000
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT 12
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: SATA RF2P SRAMSpeed Control Register (SATA_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the SATA block
        (least significant fields to most significant fields):
        64x32, 512x66*/
//====================================================================

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xffffff00
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 8
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SATA_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: PCIe_dm2 SRAM Powerdown Control Register (PCIe_dm2_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the PCIe_dm2
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: PCIe_dm2 SRAM Speed Control Register (PCIe_dm2_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the PCIe_dm2 block
        (least significant fields to most significant fields):
        Retry Buffer*/
//====================================================================

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: PCIe_dm2 RF2P SRAMSpeed Control Register (PCIe_dm2_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the PCIe_dm2 block
        (least significant fields to most significant fields):
        Receive queue heder, Receive queue data, RADMX Composition, GM Composition*/
//====================================================================

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xffff0000
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 16
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK 0xc000
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT 14
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK 0x3000
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT 12
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM2_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: PCIe_dm4 SRAM Powerdown Control Register (PCIe_dm4_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the PCIe_dm4
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: PCIe_dm4 SRAM Speed Control Register (PCIe_dm4_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the PCIe_dm4 block
        (least significant fields to most significant fields):
        Retry Buffer*/
//====================================================================

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: PCIe_dm4 RF2P SRAMSpeed Control Register (PCIe_dm4_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the PCIe_dm4 block
        (least significant fields to most significant fields):
        Receive queue heder, Receive queue data, RADMX Composition, GM Composition*/
//====================================================================

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xffff0000
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 16
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK 0xc000
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT 14
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK 0x3000
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT 12
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PCIE_DM4_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: M2M0 SRAM Powerdown Control Register (M2M0_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the M2M0
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M0_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: M2M0 RF2P SRAMSpeed Control Register (M2M0_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the M2M0 block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: M2M1 SRAM Powerdown Control Register (M2M1_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the M2M1
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M1_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: M2M1 RF2P SRAMSpeed Control Register (M2M1_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the M2M1 block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_M2M1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: SDMMC SRAM Powerdown Control Register (SDMMC_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the SDMMC
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_SDMMC_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: SDMMC SRAM Speed Control Register (SDMMC_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the SDMMC block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SDMMC_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: usb3d SRAM Powerdown Control Register (usb3d_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the usb3d
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: usb3d SRAM Speed Control Register (usb3d_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the usb3d block
        (least significant fields to most significant fields):
        RX, TX, Descriptor*/
//====================================================================

#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffff000
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 12
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) >> APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) >> APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) >> APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) >> APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3D_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: usb3h SRAM Powerdown Control Register (usb3h_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the usb3h
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: usb3h SRAM Speed Control Register (usb3h_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the usb3h block
        (least significant fields to most significant fields):
        slot_ep_cntxt, deqptr_dcs, trans_trb, ep_status_hs, ss_data_in, ep_status_ss, tx_data_buf_ram*/
//====================================================================

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xf0000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 28
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC6_MASK 0xc000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC6_SHIFT 26
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC6_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC6_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC6_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC6_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC6_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC6_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC6_MASK 0x3000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC6_SHIFT 24
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC6_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC6_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC6_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC6_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC6_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC6_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC5_MASK 0xc00000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC5_SHIFT 22
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC5_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC5_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC5_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC5_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC5_MASK 0x300000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC5_SHIFT 20
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC5_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC5_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC5_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC5_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC4_MASK 0xc0000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC4_SHIFT 18
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC4_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC4_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC4_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC4_MASK 0x30000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC4_SHIFT 16
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC4_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC4_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC4_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK 0xc000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT 14
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK 0x3000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT 12
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: usb3h RF2P SRAMSpeed Control Register (usb3h_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the usb3h block
        (least significant fields to most significant fields):
        ahb_addr_len, cmd_slot_state, cmd_input_cntxt, event_mem, er_seg_table, spipe, trans_que, ahb_out_fifo*/
//====================================================================

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC7_MASK 0xc0000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC7_SHIFT 30
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC7_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC7_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC7_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC7_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC7_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC7_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC7_MASK 0x30000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC7_SHIFT 28
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC7_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC7_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC7_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC7_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC7_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC7_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC6_MASK 0xc000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC6_SHIFT 26
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC6_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC6_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC6_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC6_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC6_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC6_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC6_MASK 0x3000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC6_SHIFT 24
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC6_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC6_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC6_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC6_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC6_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC6_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC5_MASK 0xc00000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC5_SHIFT 22
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC5_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC5_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC5_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC5_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC5_MASK 0x300000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC5_SHIFT 20
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC5_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC5_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC5_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC5_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC4_MASK 0xc0000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC4_SHIFT 18
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC4_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC4_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC4_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC4_MASK 0x30000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC4_SHIFT 16
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC4_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC4_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC4_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK 0xc000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT 14
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC3_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK 0x3000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT 12
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC3_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: usb3h RF2P SRAMSpeed Control Register 1 (usb3h_SRAMSpeedCtl_RF2PSRAMCTL1)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the usb3h block
        (least significant fields to most significant fields):
        ahb_in_fifo, seqnum_hs, hs_resp_queue, hs_data_in, hs_ctrl_fifo, hs_data_out, ss_resp_queue, ss_ctrl_in*/
//====================================================================

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC7_MASK 0xc0000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC7_SHIFT 30
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC7_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC7_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC7_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC7_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC7_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC7_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC7_MASK 0x30000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC7_SHIFT 28
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC7_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC7_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC7_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC7_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC7_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC7_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC6_MASK 0xc000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC6_SHIFT 26
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC6_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC6_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC6_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC6_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC6_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC6_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC6_MASK 0x3000000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC6_SHIFT 24
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC6_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC6_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC6_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC6_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC6_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC6_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC5_MASK 0xc00000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC5_SHIFT 22
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC5_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC5_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC5_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC5_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC5_MASK 0x300000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC5_SHIFT 20
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC5_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC5_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC5_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC5_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC5_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC5_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC4_MASK 0xc0000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC4_SHIFT 18
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC4_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC4_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC4_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC4_MASK 0x30000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC4_SHIFT 16
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC4_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC4_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC4_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC3_MASK 0xc000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC3_SHIFT 14
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC3_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC3_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC3_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC3_MASK 0x3000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC3_SHIFT 12
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC3_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC3_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC3_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC2_MASK 0xc00
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC2_SHIFT 10
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC2_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC2_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC2_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC2_MASK 0x300
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC2_SHIFT 8
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC2_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC2_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC2_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC1_MASK 0xc0
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC1_SHIFT 6
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC1_MASK 0x30
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC1_SHIFT 4
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC0_MASK 0xc
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC0_SHIFT 2
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC0_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC0_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_RTC0_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC0_MASK 0x3
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC0_SHIFT 0
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC0_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC0_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL1_WTC0_SHIFT))

//====================================================================
//Register: usb3h RF2P SRAMSpeed Control Register 2 (usb3h_SRAMSpeedCtl_RF2PSRAMCTL2)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the usb3h block
        (least significant fields to most significant fields):
        ss_ctrl_out, ss_data_out, seqnum_ss, que_ram, tx_sync_fifo_ds*/
//====================================================================

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RESERVED1_MASK 0xfff00000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RESERVED1_SHIFT 20
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RESERVED1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC4_MASK 0xc0000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC4_SHIFT 18
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC4_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC4_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC4_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC4_MASK 0x30000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC4_SHIFT 16
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC4_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC4_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC4_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC3_MASK 0xc000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC3_SHIFT 14
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC3_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC3_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC3_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC3_MASK 0x3000
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC3_SHIFT 12
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC3_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC3_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC3_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC2_MASK 0xc00
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC2_SHIFT 10
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC2_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC2_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC2_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC2_MASK 0x300
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC2_SHIFT 8
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC2_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC2_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC2_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC1_MASK 0xc0
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC1_SHIFT 6
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC1_MASK 0x30
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC1_SHIFT 4
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC1_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC1_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC1_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC0_MASK 0xc
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC0_SHIFT 2
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC0_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC0_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_RTC0_SHIFT))

#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC0_MASK 0x3
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC0_SHIFT 0
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC0_MASK) >> APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC0_SHIFT)
#define  APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USB3H_SRAMSPEEDCTL_RF2PSRAMCTL2_WTC0_SHIFT))

//====================================================================
//Register: usbdm SRAM Powerdown Control Register (usbdm_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the usbdm
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: usbdm SRAM Speed Control Register (usbdm_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the usbdm block
        (least significant fields to most significant fields):
        tx0, rx0, tx1, rx1*/
//====================================================================

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xffff0000
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 16
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK 0xc000
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT 14
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK 0x3000
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT 12
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBDM_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: usbh SRAM Powerdown Control Register (usbh_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the usbh
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBH_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: usbh SRAM Speed Control Register (usbh_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the usbh block
        (least significant fields to most significant fields):
        tx, rx*/
//====================================================================

#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xffffff00
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 8
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) >> APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) >> APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_USBH_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: lcd SRAM Powerdown Control Register (lcd_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the lcd
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: lcd SRAM Speed Control Register (lcd_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the lcd block
        (least significant fields to most significant fields):
        palette, gamma*/
//====================================================================

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xffffff00
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 8
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: lcd RF2P SRAMSpeed Control Register (lcd_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the lcd block
        (least significant fields to most significant fields):
        aff4gra, aff4yuv_y, aff4yuv_uv*/
//====================================================================

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffff000
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 12
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_LCD_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: PDMA0 SRAM Powerdown Control Register (PDMA0_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the PDMA0
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA0_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: PDMA0 RF2P SRAMSpeed Control Register (PDMA0_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the PDMA0 block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA0_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: PDMA1 SRAM Powerdown Control Register (PDMA1_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the PDMA1
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA1_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: PDMA1 RF2P SRAMSpeed Control Register (PDMA1_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the PDMA1 block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_PDMA1_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: Coresight SRAM Powerdown Control Register (Coresight_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the Coresight
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_CORESIGHT_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: Coresight SRAM Speed Control Register (Coresight_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the Coresight block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_CORESIGHT_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: SMMU SRAM Powerdown Control Register (SMMU_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the SMMU
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_SMMU_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: SMMU SRAM Speed Control Register (SMMU_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the SMMU block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SMMU_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: SQU SRAM Powerdown Control Register (SQU_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the SQU
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_SQU_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: SQU SRAM Speed Control Register (SQU_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the SQU block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_SQU_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: MC5 SRAM Powerdown Control Register (MC5_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the MC5
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_MC5_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: MC5 SRAM Speed Control Register (MC5_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the MC5 block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xffffff00
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 8
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) >> APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) >> APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_MC5_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: Redundancy Repair Control Register (RedunRepairCtl)
/** \brief The bits in this register allow the LPP to sequence through the redundancy repair process.*/
//====================================================================

#define  APMU_MISC_REG_REDUNREPAIRCTL_RESERVED1_MASK 0xfffffff8
#define  APMU_MISC_REG_REDUNREPAIRCTL_RESERVED1_SHIFT 3
#define  APMU_MISC_REG_REDUNREPAIRCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REDUNREPAIRCTL_RESERVED1_MASK) >> APMU_MISC_REG_REDUNREPAIRCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_REDUNREPAIRCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REDUNREPAIRCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_REDUNREPAIRCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_MASK 0x4
#define  APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_SHIFT 2
#define  APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_MASK) >> APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_SHIFT)
#define  APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_MASK) | (((uint32_t)val) << APMU_MISC_REG_REDUNREPAIRCTL_NRST_CA53_AXIM_SHIFT))

#define  APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_MASK 0x2
#define  APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_SHIFT 1
#define  APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_MASK) >> APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_SHIFT)
#define  APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_MASK) | (((uint32_t)val) << APMU_MISC_REG_REDUNREPAIRCTL_L2_REDUN_START_SHIFT))

#define  APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_MASK 0x1
#define  APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_SHIFT 0
#define  APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_MASK) >> APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_SHIFT)
#define  APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_MASK) | (((uint32_t)val) << APMU_MISC_REG_REDUNREPAIRCTL_NRST_L2_REDUN_SHIFT))

//====================================================================
//Register: Redundancy Repair Done Status Register (RepairDone)
/** \brief After reset, SRAMs with redundancy must go through a repair process. This register contains
      status bits for each SRAM with redundancy in the AP Subsystem. Firmware should ensure that
      the corresponding RepairDone status bit is set before doing anything that would access those
      SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_REPAIRDONE_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_REPAIRDONE_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_REPAIRDONE_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REPAIRDONE_RESERVED1_MASK) >> APMU_MISC_REG_REPAIRDONE_RESERVED1_SHIFT)
#define  APMU_MISC_REG_REPAIRDONE_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REPAIRDONE_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_REPAIRDONE_RESERVED1_SHIFT))

#define  APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_MASK 0x2
#define  APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_SHIFT 1
#define  APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_MASK) >> APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_SHIFT)
#define  APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_MASK) | (((uint32_t)val) << APMU_MISC_REG_REPAIRDONE_APL2_SRBIST_REPAIRDONE_SHIFT))

#define  APMU_MISC_REG_REPAIRDONE_AP_SRBIST_REPAIRDONE_MASK 0x1
#define  APMU_MISC_REG_REPAIRDONE_AP_SRBIST_REPAIRDONE_SHIFT 0
#define  APMU_MISC_REG_REPAIRDONE_AP_SRBIST_REPAIRDONE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REPAIRDONE_AP_SRBIST_REPAIRDONE_MASK) >> APMU_MISC_REG_REPAIRDONE_AP_SRBIST_REPAIRDONE_SHIFT)
#define  APMU_MISC_REG_REPAIRDONE_AP_SRBIST_REPAIRDONE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REPAIRDONE_AP_SRBIST_REPAIRDONE_MASK) | (((uint32_t)val) << APMU_MISC_REG_REPAIRDONE_AP_SRBIST_REPAIRDONE_SHIFT))

//====================================================================
//Register: Clock Domain Interrupt Enable Register 1 (ClkDomIntEnReg1)
/** \brief Bit order (high to low): lanmacclk2x, lanrxclk, uartclk_IRQ[4], lcdclk, lcdxclk,
      qspiclk, sdmmcclk, gpuclk, gpubusclk, smmuclk, bcmclk, apbusclkdiv2, apbusclk_gpu,
      apbusclk_sata, apbusclk_pcie_dm4, apbusclk_pcie_dm2, apbusclk_usb3phy, apbusclk_usb3h,
      apbusclk_usb3d, apbusclk_usb2phy, apbusclk_usb2a, apbusclk_lan, apbusclk_iowrap,
      apbusclk_squ, apbusclk_cdma, apbusclk_lcd, apbusclk_apb, sspclk, i2cclk.*/
//====================================================================

#define  APMU_MISC_REG_CLKDOMINTENREG1_INTENABLE_MASK 0xffffffff
#define  APMU_MISC_REG_CLKDOMINTENREG1_INTENABLE_SHIFT 0
#define  APMU_MISC_REG_CLKDOMINTENREG1_INTENABLE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTENREG1_INTENABLE_MASK) >> APMU_MISC_REG_CLKDOMINTENREG1_INTENABLE_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTENREG1_INTENABLE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTENREG1_INTENABLE_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTENREG1_INTENABLE_SHIFT))

//====================================================================
//Register: Clock Domain Interrupt Pending Register 1 (ClkDomIntPendReg1)
/** \brief This register indicates the pending status of the first 32 clock domain interrupts.
      See the ClkDomIntEnReg1 description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_CLKDOMINTPENDREG1_INTPEND_MASK 0xffffffff
#define  APMU_MISC_REG_CLKDOMINTPENDREG1_INTPEND_SHIFT 0
#define  APMU_MISC_REG_CLKDOMINTPENDREG1_INTPEND_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTPENDREG1_INTPEND_MASK) >> APMU_MISC_REG_CLKDOMINTPENDREG1_INTPEND_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTPENDREG1_INTPEND_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTPENDREG1_INTPEND_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTPENDREG1_INTPEND_SHIFT))

//====================================================================
//Register: Clock Domain Interrupt Acknowledge Register 1 (ClkDomIntAckReg1)
/** \brief Writing a '1' to a bit in this register acknowledges (clears) the corresponding interrupt.
      See the ClkDomIntEnReg1 description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_CLKDOMINTACKREG1_INTACK_MASK 0xffffffff
#define  APMU_MISC_REG_CLKDOMINTACKREG1_INTACK_SHIFT 0
#define  APMU_MISC_REG_CLKDOMINTACKREG1_INTACK_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTACKREG1_INTACK_MASK) >> APMU_MISC_REG_CLKDOMINTACKREG1_INTACK_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTACKREG1_INTACK_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTACKREG1_INTACK_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTACKREG1_INTACK_SHIFT))

//====================================================================
//Register: Clock Domain Interrupt Force Register 1 (ClkDomIntForceReg1)
/** \brief This register allows firmware to generate an interrupt for a particular clock domain.
      See the ClkDomIntEnReg1 description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_CLKDOMINTFORCEREG1_INTFORCE_MASK 0xffffffff
#define  APMU_MISC_REG_CLKDOMINTFORCEREG1_INTFORCE_SHIFT 0
#define  APMU_MISC_REG_CLKDOMINTFORCEREG1_INTFORCE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTFORCEREG1_INTFORCE_MASK) >> APMU_MISC_REG_CLKDOMINTFORCEREG1_INTFORCE_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTFORCEREG1_INTFORCE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTFORCEREG1_INTFORCE_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTFORCEREG1_INTFORCE_SHIFT))

//====================================================================
//Register: Clock Domain Interrupt Enable Register 2 (ClkDomIntEnReg2)
/** \brief This register allows firmware to enable or mask off the remaining clock domain interrupts
      generated by the clkgate modules in the APMU.
      Bit order (high to low): ddrclk, ddrclk2x, tpiuclk, cpudbgclk.*/
//====================================================================

#define  APMU_MISC_REG_CLKDOMINTENREG2_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_CLKDOMINTENREG2_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_CLKDOMINTENREG2_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTENREG2_RESERVED1_MASK) >> APMU_MISC_REG_CLKDOMINTENREG2_RESERVED1_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTENREG2_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTENREG2_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTENREG2_RESERVED1_SHIFT))

#define  APMU_MISC_REG_CLKDOMINTENREG2_INTENABLE_MASK 0xf
#define  APMU_MISC_REG_CLKDOMINTENREG2_INTENABLE_SHIFT 0
#define  APMU_MISC_REG_CLKDOMINTENREG2_INTENABLE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTENREG2_INTENABLE_MASK) >> APMU_MISC_REG_CLKDOMINTENREG2_INTENABLE_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTENREG2_INTENABLE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTENREG2_INTENABLE_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTENREG2_INTENABLE_SHIFT))

//====================================================================
//Register: Clock Domain Interrupt Pending Register 2 (ClkDomIntPendReg2)
/** \brief This register indicates the pending status of the remaining clock domain interrupts.
      See the ClkDomIntEnReg2 description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_CLKDOMINTPENDREG2_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_CLKDOMINTPENDREG2_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_CLKDOMINTPENDREG2_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTPENDREG2_RESERVED1_MASK) >> APMU_MISC_REG_CLKDOMINTPENDREG2_RESERVED1_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTPENDREG2_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTPENDREG2_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTPENDREG2_RESERVED1_SHIFT))

#define  APMU_MISC_REG_CLKDOMINTPENDREG2_INTPEND_MASK 0xf
#define  APMU_MISC_REG_CLKDOMINTPENDREG2_INTPEND_SHIFT 0
#define  APMU_MISC_REG_CLKDOMINTPENDREG2_INTPEND_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTPENDREG2_INTPEND_MASK) >> APMU_MISC_REG_CLKDOMINTPENDREG2_INTPEND_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTPENDREG2_INTPEND_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTPENDREG2_INTPEND_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTPENDREG2_INTPEND_SHIFT))

//====================================================================
//Register: Clock Domain Interrupt Acknowledge Register 2 (ClkDomIntAckReg2)
/** \brief Writing a '1' to a bit in this register acknowledges (clears) the corresponding interrupt.
      See the ClkDomIntEnReg2 description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_CLKDOMINTACKREG2_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_CLKDOMINTACKREG2_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_CLKDOMINTACKREG2_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTACKREG2_RESERVED1_MASK) >> APMU_MISC_REG_CLKDOMINTACKREG2_RESERVED1_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTACKREG2_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTACKREG2_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTACKREG2_RESERVED1_SHIFT))

#define  APMU_MISC_REG_CLKDOMINTACKREG2_INTACK_MASK 0xf
#define  APMU_MISC_REG_CLKDOMINTACKREG2_INTACK_SHIFT 0
#define  APMU_MISC_REG_CLKDOMINTACKREG2_INTACK_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTACKREG2_INTACK_MASK) >> APMU_MISC_REG_CLKDOMINTACKREG2_INTACK_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTACKREG2_INTACK_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTACKREG2_INTACK_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTACKREG2_INTACK_SHIFT))

//====================================================================
//Register: Clock Domain Interrupt Force Register 2 (ClkDomIntForceReg2)
/** \brief This register allows firmware to generate an interrupt for a particular clock domain.
      See the ClkDomIntEnReg2 description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_CLKDOMINTFORCEREG2_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_CLKDOMINTFORCEREG2_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_CLKDOMINTFORCEREG2_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTFORCEREG2_RESERVED1_MASK) >> APMU_MISC_REG_CLKDOMINTFORCEREG2_RESERVED1_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTFORCEREG2_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTFORCEREG2_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTFORCEREG2_RESERVED1_SHIFT))

#define  APMU_MISC_REG_CLKDOMINTFORCEREG2_INTFORCE_MASK 0xf
#define  APMU_MISC_REG_CLKDOMINTFORCEREG2_INTFORCE_SHIFT 0
#define  APMU_MISC_REG_CLKDOMINTFORCEREG2_INTFORCE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_CLKDOMINTFORCEREG2_INTFORCE_MASK) >> APMU_MISC_REG_CLKDOMINTFORCEREG2_INTFORCE_SHIFT)
#define  APMU_MISC_REG_CLKDOMINTFORCEREG2_INTFORCE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_CLKDOMINTFORCEREG2_INTFORCE_MASK) | (((uint32_t)val) << APMU_MISC_REG_CLKDOMINTFORCEREG2_INTFORCE_SHIFT))

//====================================================================
//Register: Miscellaneous Interrupt Enable Register (MiscIntEnReg)
/** \brief Bit order (high to low): MC_Access, Timer3, Timer2, Timer1, Timer0.*/
//====================================================================

#define  APMU_MISC_REG_MISCINTENREG_RESERVED1_MASK 0xffffffe0
#define  APMU_MISC_REG_MISCINTENREG_RESERVED1_SHIFT 5
#define  APMU_MISC_REG_MISCINTENREG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MISCINTENREG_RESERVED1_MASK) >> APMU_MISC_REG_MISCINTENREG_RESERVED1_SHIFT)
#define  APMU_MISC_REG_MISCINTENREG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MISCINTENREG_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_MISCINTENREG_RESERVED1_SHIFT))

#define  APMU_MISC_REG_MISCINTENREG_INTENABLE_MASK 0x1f
#define  APMU_MISC_REG_MISCINTENREG_INTENABLE_SHIFT 0
#define  APMU_MISC_REG_MISCINTENREG_INTENABLE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MISCINTENREG_INTENABLE_MASK) >> APMU_MISC_REG_MISCINTENREG_INTENABLE_SHIFT)
#define  APMU_MISC_REG_MISCINTENREG_INTENABLE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MISCINTENREG_INTENABLE_MASK) | (((uint32_t)val) << APMU_MISC_REG_MISCINTENREG_INTENABLE_SHIFT))

//====================================================================
//Register: Miscellaneous Interrupt Pending Register (MiscIntPendReg)
/** \brief This register indicates the pending status of the non-clock domain interrupts.
      See the MiscIntEnReg description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_MISCINTPENDREG_RESERVED1_MASK 0xffffffe0
#define  APMU_MISC_REG_MISCINTPENDREG_RESERVED1_SHIFT 5
#define  APMU_MISC_REG_MISCINTPENDREG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MISCINTPENDREG_RESERVED1_MASK) >> APMU_MISC_REG_MISCINTPENDREG_RESERVED1_SHIFT)
#define  APMU_MISC_REG_MISCINTPENDREG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MISCINTPENDREG_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_MISCINTPENDREG_RESERVED1_SHIFT))

#define  APMU_MISC_REG_MISCINTPENDREG_INTPEND_MASK 0x1f
#define  APMU_MISC_REG_MISCINTPENDREG_INTPEND_SHIFT 0
#define  APMU_MISC_REG_MISCINTPENDREG_INTPEND_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MISCINTPENDREG_INTPEND_MASK) >> APMU_MISC_REG_MISCINTPENDREG_INTPEND_SHIFT)
#define  APMU_MISC_REG_MISCINTPENDREG_INTPEND_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MISCINTPENDREG_INTPEND_MASK) | (((uint32_t)val) << APMU_MISC_REG_MISCINTPENDREG_INTPEND_SHIFT))

//====================================================================
//Register: Miscellaneous Interrupt Acknowledge Register (MiscIntAckReg)
/** \brief Writing a '1' to a bit in this register acknowledges (clears) the corresponding interrupt.
      See the MiscIntEnReg description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_MISCINTACKREG_RESERVED1_MASK 0xffffffe0
#define  APMU_MISC_REG_MISCINTACKREG_RESERVED1_SHIFT 5
#define  APMU_MISC_REG_MISCINTACKREG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MISCINTACKREG_RESERVED1_MASK) >> APMU_MISC_REG_MISCINTACKREG_RESERVED1_SHIFT)
#define  APMU_MISC_REG_MISCINTACKREG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MISCINTACKREG_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_MISCINTACKREG_RESERVED1_SHIFT))

#define  APMU_MISC_REG_MISCINTACKREG_INTACK_MASK 0x1f
#define  APMU_MISC_REG_MISCINTACKREG_INTACK_SHIFT 0
#define  APMU_MISC_REG_MISCINTACKREG_INTACK_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MISCINTACKREG_INTACK_MASK) >> APMU_MISC_REG_MISCINTACKREG_INTACK_SHIFT)
#define  APMU_MISC_REG_MISCINTACKREG_INTACK_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MISCINTACKREG_INTACK_MASK) | (((uint32_t)val) << APMU_MISC_REG_MISCINTACKREG_INTACK_SHIFT))

//====================================================================
//Register: Miscellaneous Interrupt Force Register (MiscIntForceReg)
/** \brief This register allows firmware to generate an interrupt for a particular interrupt.
      See the MiscIntEnReg description for bit mapping.*/
//====================================================================

#define  APMU_MISC_REG_MISCINTFORCEREG_RESERVED1_MASK 0xffffffe0
#define  APMU_MISC_REG_MISCINTFORCEREG_RESERVED1_SHIFT 5
#define  APMU_MISC_REG_MISCINTFORCEREG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MISCINTFORCEREG_RESERVED1_MASK) >> APMU_MISC_REG_MISCINTFORCEREG_RESERVED1_SHIFT)
#define  APMU_MISC_REG_MISCINTFORCEREG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MISCINTFORCEREG_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_MISCINTFORCEREG_RESERVED1_SHIFT))

#define  APMU_MISC_REG_MISCINTFORCEREG_INTFORCE_MASK 0x1f
#define  APMU_MISC_REG_MISCINTFORCEREG_INTFORCE_SHIFT 0
#define  APMU_MISC_REG_MISCINTFORCEREG_INTFORCE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_MISCINTFORCEREG_INTFORCE_MASK) >> APMU_MISC_REG_MISCINTFORCEREG_INTFORCE_SHIFT)
#define  APMU_MISC_REG_MISCINTFORCEREG_INTFORCE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_MISCINTFORCEREG_INTFORCE_MASK) | (((uint32_t)val) << APMU_MISC_REG_MISCINTFORCEREG_INTFORCE_SHIFT))

//====================================================================
//Register: R4 AXI ID register (R4_AXI_ID)
/** \brief The AXI ID for the R4 security/low power processor should be written into this register.
      This value is provided to special AXI BusGuards that are placed in front of the memory
      controller AXI slave ports when the memory controller has its clock gated. Any accesses
      from an AXI ID other than the R4 will generate an error interrupt. Accesses from the
      R4 are not considered to be an error, as they will be used instead to generate a fault
      that will trigger the R4 to wake the memory controller back up. The compare for the
      ID value in this register will be masked with the value in the R4AXIMASK register.*/
//====================================================================

#define  APMU_MISC_REG_R4_AXI_ID_RESERVED1_MASK 0xff000000
#define  APMU_MISC_REG_R4_AXI_ID_RESERVED1_SHIFT 24
#define  APMU_MISC_REG_R4_AXI_ID_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_AXI_ID_RESERVED1_MASK) >> APMU_MISC_REG_R4_AXI_ID_RESERVED1_SHIFT)
#define  APMU_MISC_REG_R4_AXI_ID_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_AXI_ID_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_AXI_ID_RESERVED1_SHIFT))

#define  APMU_MISC_REG_R4_AXI_ID_AXI_ID_MASK 0xffffff
#define  APMU_MISC_REG_R4_AXI_ID_AXI_ID_SHIFT 0
#define  APMU_MISC_REG_R4_AXI_ID_AXI_ID_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_AXI_ID_AXI_ID_MASK) >> APMU_MISC_REG_R4_AXI_ID_AXI_ID_SHIFT)
#define  APMU_MISC_REG_R4_AXI_ID_AXI_ID_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_AXI_ID_AXI_ID_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_AXI_ID_AXI_ID_SHIFT))

//====================================================================
//Register: R4 AXI ID mask register (R4_AXI_MASK)
/** \brief When comparing the value in the R4AXIID register with an incoming AXI transaction, the
      memory controller AXI shim will ignore the values of any bits that are set to 0 in
      this mask register.*/
//====================================================================

#define  APMU_MISC_REG_R4_AXI_MASK_RESERVED1_MASK 0xff000000
#define  APMU_MISC_REG_R4_AXI_MASK_RESERVED1_SHIFT 24
#define  APMU_MISC_REG_R4_AXI_MASK_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_AXI_MASK_RESERVED1_MASK) >> APMU_MISC_REG_R4_AXI_MASK_RESERVED1_SHIFT)
#define  APMU_MISC_REG_R4_AXI_MASK_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_AXI_MASK_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_AXI_MASK_RESERVED1_SHIFT))

#define  APMU_MISC_REG_R4_AXI_MASK_AXI_ID_MASK_MASK 0xffffff
#define  APMU_MISC_REG_R4_AXI_MASK_AXI_ID_MASK_SHIFT 0
#define  APMU_MISC_REG_R4_AXI_MASK_AXI_ID_MASK_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_AXI_MASK_AXI_ID_MASK_MASK) >> APMU_MISC_REG_R4_AXI_MASK_AXI_ID_MASK_SHIFT)
#define  APMU_MISC_REG_R4_AXI_MASK_AXI_ID_MASK_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_AXI_MASK_AXI_ID_MASK_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_AXI_MASK_AXI_ID_MASK_SHIFT))

//====================================================================
//Register: Error AXI ID (ERR_AXI_ID)
/** \brief If an AXI master other than the R4 low power processor (as indicated by the R4_AXI_ID and
      R4_AXI_MASK register values) tries to access the memory controller while its clock is gated
      off, an interrupt will be generated, and the AXI ID of the master that attempted the
      access will be available in this register.*/
//====================================================================

#define  APMU_MISC_REG_ERR_AXI_ID_RESERVED1_MASK 0xff000000
#define  APMU_MISC_REG_ERR_AXI_ID_RESERVED1_SHIFT 24
#define  APMU_MISC_REG_ERR_AXI_ID_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_ERR_AXI_ID_RESERVED1_MASK) >> APMU_MISC_REG_ERR_AXI_ID_RESERVED1_SHIFT)
#define  APMU_MISC_REG_ERR_AXI_ID_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_ERR_AXI_ID_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_ERR_AXI_ID_RESERVED1_SHIFT))

#define  APMU_MISC_REG_ERR_AXI_ID_AXI_ID_MASK 0xffffff
#define  APMU_MISC_REG_ERR_AXI_ID_AXI_ID_SHIFT 0
#define  APMU_MISC_REG_ERR_AXI_ID_AXI_ID_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_ERR_AXI_ID_AXI_ID_MASK) >> APMU_MISC_REG_ERR_AXI_ID_AXI_ID_SHIFT)
#define  APMU_MISC_REG_ERR_AXI_ID_AXI_ID_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_ERR_AXI_ID_AXI_ID_MASK) | (((uint32_t)val) << APMU_MISC_REG_ERR_AXI_ID_AXI_ID_SHIFT))

//====================================================================
//Register: General Control Register (GENERIC_CTRL)
/** \brief Miscellaneous power related control signals.*/
//====================================================================

#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED1_MASK 0xffe00000
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED1_SHIFT 21
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GENERIC_CTRL_RESERVED1_MASK) >> APMU_MISC_REG_GENERIC_CTRL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GENERIC_CTRL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_GENERIC_CTRL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_GENERIC_CTRL_MC_CS_OVERRIDE_MASK 0x1e0000
#define  APMU_MISC_REG_GENERIC_CTRL_MC_CS_OVERRIDE_SHIFT 17
#define  APMU_MISC_REG_GENERIC_CTRL_MC_CS_OVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GENERIC_CTRL_MC_CS_OVERRIDE_MASK) >> APMU_MISC_REG_GENERIC_CTRL_MC_CS_OVERRIDE_SHIFT)
#define  APMU_MISC_REG_GENERIC_CTRL_MC_CS_OVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GENERIC_CTRL_MC_CS_OVERRIDE_MASK) | (((uint32_t)val) << APMU_MISC_REG_GENERIC_CTRL_MC_CS_OVERRIDE_SHIFT))

#define  APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_MASK 0x10000
#define  APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_SHIFT 16
#define  APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_MASK) >> APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_SHIFT)
#define  APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_GENERIC_CTRL_MC_PHY_PWRDWN_SHIFT))

#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED2_MASK 0x8000
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED2_SHIFT 15
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GENERIC_CTRL_RESERVED2_MASK) >> APMU_MISC_REG_GENERIC_CTRL_RESERVED2_SHIFT)
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GENERIC_CTRL_RESERVED2_MASK) | (((uint32_t)val) << APMU_MISC_REG_GENERIC_CTRL_RESERVED2_SHIFT))

#define  APMU_MISC_REG_GENERIC_CTRL_FUSE_CLK_DIV_COUNT_DISABLE_MASK 0x4000
#define  APMU_MISC_REG_GENERIC_CTRL_FUSE_CLK_DIV_COUNT_DISABLE_SHIFT 14
#define  APMU_MISC_REG_GENERIC_CTRL_FUSE_CLK_DIV_COUNT_DISABLE_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GENERIC_CTRL_FUSE_CLK_DIV_COUNT_DISABLE_MASK) >> APMU_MISC_REG_GENERIC_CTRL_FUSE_CLK_DIV_COUNT_DISABLE_SHIFT)
#define  APMU_MISC_REG_GENERIC_CTRL_FUSE_CLK_DIV_COUNT_DISABLE_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GENERIC_CTRL_FUSE_CLK_DIV_COUNT_DISABLE_MASK) | (((uint32_t)val) << APMU_MISC_REG_GENERIC_CTRL_FUSE_CLK_DIV_COUNT_DISABLE_SHIFT))

#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED3_MASK 0x3fff
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED3_SHIFT 0
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_GENERIC_CTRL_RESERVED3_MASK) >> APMU_MISC_REG_GENERIC_CTRL_RESERVED3_SHIFT)
#define  APMU_MISC_REG_GENERIC_CTRL_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_GENERIC_CTRL_RESERVED3_MASK) | (((uint32_t)val) << APMU_MISC_REG_GENERIC_CTRL_RESERVED3_SHIFT))

//====================================================================
//Register: Wakeup Clear Register (WAKE_CLR)
/** \brief Provides signals to clear various wakeup sources.*/
//====================================================================

#define  APMU_MISC_REG_WAKE_CLR_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_WAKE_CLR_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_WAKE_CLR_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_WAKE_CLR_RESERVED1_MASK) >> APMU_MISC_REG_WAKE_CLR_RESERVED1_SHIFT)
#define  APMU_MISC_REG_WAKE_CLR_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_WAKE_CLR_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_WAKE_CLR_RESERVED1_SHIFT))

#define  APMU_MISC_REG_WAKE_CLR_SDH_WK_CLR_MASK 0xf
#define  APMU_MISC_REG_WAKE_CLR_SDH_WK_CLR_SHIFT 0
#define  APMU_MISC_REG_WAKE_CLR_SDH_WK_CLR_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_WAKE_CLR_SDH_WK_CLR_MASK) >> APMU_MISC_REG_WAKE_CLR_SDH_WK_CLR_SHIFT)
#define  APMU_MISC_REG_WAKE_CLR_SDH_WK_CLR_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_WAKE_CLR_SDH_WK_CLR_MASK) | (((uint32_t)val) << APMU_MISC_REG_WAKE_CLR_SDH_WK_CLR_SHIFT))

//====================================================================
//Register: IP Tagging Revision 0 (REV0)
/** \brief Provide means for firmware to read IP tagging information.
      This register of particular interest to firmware.*/
//====================================================================

#define  APMU_MISC_REG_REV0_MAJ_MASK 0xffff0000
#define  APMU_MISC_REG_REV0_MAJ_SHIFT 16
#define  APMU_MISC_REG_REV0_MAJ_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REV0_MAJ_MASK) >> APMU_MISC_REG_REV0_MAJ_SHIFT)
#define  APMU_MISC_REG_REV0_MAJ_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REV0_MAJ_MASK) | (((uint32_t)val) << APMU_MISC_REG_REV0_MAJ_SHIFT))

#define  APMU_MISC_REG_REV0_MID_MASK 0xffff
#define  APMU_MISC_REG_REV0_MID_SHIFT 0
#define  APMU_MISC_REG_REV0_MID_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REV0_MID_MASK) >> APMU_MISC_REG_REV0_MID_SHIFT)
#define  APMU_MISC_REG_REV0_MID_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REV0_MID_MASK) | (((uint32_t)val) << APMU_MISC_REG_REV0_MID_SHIFT))

//====================================================================
//Register: IP Tagging Revision 1 (REV1)
/** \brief Provide means for firmware to read IP tagging information.*/
//====================================================================

#define  APMU_MISC_REG_REV1_RESERVED1_MASK 0xffff0000
#define  APMU_MISC_REG_REV1_RESERVED1_SHIFT 16
#define  APMU_MISC_REG_REV1_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REV1_RESERVED1_MASK) >> APMU_MISC_REG_REV1_RESERVED1_SHIFT)
#define  APMU_MISC_REG_REV1_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REV1_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_REV1_RESERVED1_SHIFT))

#define  APMU_MISC_REG_REV1_CFG_MASK 0xffff
#define  APMU_MISC_REG_REV1_CFG_SHIFT 0
#define  APMU_MISC_REG_REV1_CFG_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_REV1_CFG_MASK) >> APMU_MISC_REG_REV1_CFG_SHIFT)
#define  APMU_MISC_REG_REV1_CFG_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_REV1_CFG_MASK) | (((uint32_t)val) << APMU_MISC_REG_REV1_CFG_SHIFT))

//====================================================================
//Register: BCM SRAM Powerdown Control Register (BCM_SRAMPdwn_SPCTL)
/** \brief The fields of this register drive the low leakage power control signals for the BCM
      block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_RESERVED1_SHIFT 2
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_RESERVED1_MASK) >> APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDLVMC_MASK 0x2
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDLVMC_SHIFT 1
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDLVMC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDLVMC_MASK) >> APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDLVMC_SHIFT)
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDLVMC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDLVMC_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDLVMC_SHIFT))

#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDWN_MASK) >> APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_SRAMPDWN_SPCTL_PDWN_SHIFT))

//====================================================================
//Register: BCM RF2P SRAMSpeed Control Register (BCM_SRAMSpeedCtl_RF2PSRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the BCM block's SRAMs.*/
//====================================================================

#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK 0xfffffff0
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT 4
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) >> APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) >> APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_SRAMSPEEDCTL_RF2PSRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: R4 SRAM Speed Control Register (R4_SRAMSpeedCtl_SRAMCTL)
/** \brief The fields of this register drive the RTC/WTC signals of the following SRAMS
          used in the R4 block
        (least significant fields to most significant fields):
        ICache Data, ICache Tag, DCache Data, DCache Tag, DCacheDirty*/
//====================================================================

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK 0xfff00000
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT 20
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC4_MASK 0xc0000
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC4_SHIFT 18
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC4_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC4_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC4_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC4_MASK 0x30000
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC4_SHIFT 16
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC4_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC4_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC4_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC4_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC4_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC4_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK 0xc000
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT 14
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC3_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK 0x3000
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT 12
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC3_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC3_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC3_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK 0xc00
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT 10
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC2_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK 0x300
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT 8
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC2_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC2_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC2_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK 0xc0
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT 6
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC1_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK 0x30
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT 4
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC1_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC1_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK 0xc
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT 2
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_RTC0_SHIFT))

#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK 0x3
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT 0
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) >> APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT)
#define  APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC0_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC0_MASK) | (((uint32_t)val) << APMU_MISC_REG_R4_SRAMSPEEDCTL_SRAMCTL_WTC0_SHIFT))

//====================================================================
//Register: BCM ROM Powerdown/Read Timing Control Register (BCM_ROMCtl_RCTL)
/** \brief The fields of this register drive the low leakage power control signals and the
      read timing controls for the BCM ROM.*/
//====================================================================

#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RESERVED1_MASK 0xffffffc0
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RESERVED1_SHIFT 6
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_ROMCTL_RCTL_RESERVED1_MASK) >> APMU_MISC_REG_BCM_ROMCTL_RCTL_RESERVED1_SHIFT)
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_ROMCTL_RCTL_RESERVED1_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_ROMCTL_RCTL_RESERVED1_SHIFT))

#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REF_MASK 0x30
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REF_SHIFT 4
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REF_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REF_MASK) >> APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REF_SHIFT)
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REF_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REF_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REF_SHIFT))

#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_MASK 0xe
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_SHIFT 1
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_MASK) >> APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_SHIFT)
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_ROMCTL_RCTL_RTC_SHIFT))

#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_PDWN_MASK 0x1
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_PDWN_SHIFT 0
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_PDWN_MASK_SHIFT(reg) (((reg) & APMU_MISC_REG_BCM_ROMCTL_RCTL_PDWN_MASK) >> APMU_MISC_REG_BCM_ROMCTL_RCTL_PDWN_SHIFT)
#define  APMU_MISC_REG_BCM_ROMCTL_RCTL_PDWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_MISC_REG_BCM_ROMCTL_RCTL_PDWN_MASK) | (((uint32_t)val) << APMU_MISC_REG_BCM_ROMCTL_RCTL_PDWN_SHIFT))

//====================================================================
//
//Register File: APMU Clock and Reset Control Registers (APMU_ClkRstGen)
/** \brief These registers provide control/status for the clocks/resets for all of the clock domains
    in the Application Processing section of the chip.*/
//
//====================================================================

//====================================================================
//Register: CPU Debug Clock Configuration Register (CPUDbgClkConfig)
/** \brief CPU Debug Clock domain Configuration Register.
      The maximum frequency for this clock is 400 MHz. It is important to ensure that the
      configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_AT_RSTN_MASK 0x4
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_AT_RSTN_SHIFT 2
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_AT_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_AT_RSTN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_AT_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_AT_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_AT_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_AT_RSTN_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_RSTN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKCONFIG_CPU_RSTN_SHIFT))

//====================================================================
//Register: CPU Debug Clock Status Register (CPUDbgClkStatus)
/** \brief CPU Debug Clock Status Register*/
//====================================================================

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_AT_RSTN_MASK 0x4
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_AT_RSTN_SHIFT 2
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_AT_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_AT_RSTN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_AT_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_AT_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_AT_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_AT_RSTN_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_RSTN_MASK) >> APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_CPUDBGCLKSTATUS_CPU_RSTN_SHIFT))

//====================================================================
//Register: TPIUClk ClkGen Configuration Register (TPIUClkConfig_ClkGenConfig)
/** \brief TPIUClk clock generator Configuration Register.
          The maximum frequency for this clock is 125 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK 0x1
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT 0
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED7_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT))

//====================================================================
//Register: TPIUClk ClkGen Status Register (TPIUClkConfig_ClkGenStatus)
/** \brief TPIUClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_TPIUCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: MCBusGuard ClkGen Configuration Register (MCBusGuardConfig_ClkGenConfig)
/** \brief MCBusGuard clock generator Configuration Register.
        This is not an actual clock generator, but instead provides a means to turn off the Memory Controller BusGuards before putting the memory controller to sleep.*/
//====================================================================

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: MCBusGuard ClkGen Status Register (MCBusGuardConfig_ClkGenStatus)
/** \brief MCBusGuard clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xf80000
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 19
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_HIDIV_MASK 0x70000
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xf800
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 11
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LODIV_MASK 0x700
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_MCBUSGUARDCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: DDRClk2x ClkGen Configuration Register (DDRClk2xConfig_ClkGenConfig)
/** \brief DDRClk2x clock generator Configuration Register.
          The maximum frequency for this clock is 1.6 GHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xf80000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 19
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_MASK 0x70000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xf800
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 11
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_MASK 0x700
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED7_MASK 0x1
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED7_SHIFT 0
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED7_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED7_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED7_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED7_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED7_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENCONFIG_RESERVED7_SHIFT))

//====================================================================
//Register: DDRClk2x ClkGen Status Register (DDRClk2xConfig_ClkGenStatus)
/** \brief DDRClk2x clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xf80000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 19
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_HIDIV_MASK 0x70000
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xf800
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 11
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LODIV_MASK 0x700
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLK2XCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: DDR Clock Configuration Register (DDRClkConfig)
/** \brief DDR Clock Configuration Register. The maximum frequency for this clock is 533 MHz.*/
//====================================================================

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED1_MASK 0x3fe00000
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED1_SHIFT 21
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_MASK 0x100000
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_SHIFT 20
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_DIVSEL_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED2_MASK 0xfff80
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED2_SHIFT 7
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED3_MASK 0x4
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED3_SHIFT 2
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_RSTN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x10000
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x100
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

//====================================================================
//Register: DDR Clock Status Register (DDRClkStatus)
/** \brief DDR Clock Status Register*/
//====================================================================

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED1_MASK 0xfe000000
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED1_SHIFT 25
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED2_MASK 0xe00000
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED2_SHIFT 21
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_DIVSEL_MASK 0x100000
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_DIVSEL_SHIFT 20
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_DIVSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_DIVSEL_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_DIVSEL_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_DIVSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_DIVSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_DIVSEL_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED3_MASK 0x80000
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED3_SHIFT 19
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_HIDIV_MASK 0x70000
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED4_MASK 0xf800
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED4_SHIFT 11
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LODIV_MASK 0x700
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_DDRCLKSTATUS_RSTN_MASK) >> APMU_CLKRSTGEN_DDRCLKSTATUS_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_DDRCLKSTATUS_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_DDRCLKSTATUS_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_DDRCLKSTATUS_RSTN_SHIFT))

//====================================================================
//Register: I2CClk ClkGen Configuration Register (I2CClkConfig_ClkGenConfig)
/** \brief I2CClk clock generator Configuration Register.
            The source of this clock generator is the APBusClk.
          The maximum frequency for this clock is 200 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7000000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 24
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK 0x1
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT 0
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED7_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT))

//====================================================================
//Register: I2CClk ClkGen Status Register (I2CClkConfig_ClkGenStatus)
/** \brief I2CClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x7000000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 24
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_I2CCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: SSPClk ClkGen Configuration Register (SSPClkConfig_ClkGenConfig)
/** \brief SSPClk clock generator Configuration Register.
            The source of this clock generator is the APBusClk.
          The maximum frequency for this clock is 200 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7000000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 24
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK 0x1
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT 0
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED7_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT))

//====================================================================
//Register: SSPClk ClkGen Status Register (SSPClkConfig_ClkGenStatus)
/** \brief SSPClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x7000000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 24
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SSPCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk ClkGen Configuration Register (APBusClkConfig_ClkGenConfig)
/** \brief APBusClk clock generator Configuration Register.
          The maximum frequency for this clock is 200 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0xf8
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED7_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED8_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED8_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED8_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED8_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED8_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED8_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED8_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENCONFIG_RESERVED8_SHIFT))

//====================================================================
//Register: APBusClk ClkGen Status Register (APBusClkConfig_ClkGenStatus)
/** \brief APBusClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK 0xf8
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED7_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED7_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED7_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED7_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED7_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED7_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED7_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED7_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED8_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED8_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED8_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED8_MASK) >> APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED8_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED8_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED8_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKCONFIG_CLKGENSTATUS_RESERVED8_SHIFT))

//====================================================================
//Register: APBusClkDiv2 ClkGen Configuration Register (APBusClkDiv2Config_ClkGenConfig)
/** \brief APBusClkDiv2 clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClkDiv2 ClkGen Status Register (APBusClkDiv2Config_ClkGenStatus)
/** \brief APBusClkDiv2 clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED1_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED1_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED2_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED2_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED3_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLKDIV2CONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

//====================================================================
//Register: APBusClk_APB ClkGen Configuration Register (APBusClk_APBConfig_ClkGenConfig)
/** \brief APBusClk_APB clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_APB ClkGen Status Register (APBusClk_APBConfig_ClkGenStatus)
/** \brief APBusClk_APB clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_APBCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_LCD ClkGen Configuration Register (APBusClk_LCDConfig_ClkGenConfig)
/** \brief APBusClk_LCD clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_LCD ClkGen Status Register (APBusClk_LCDConfig_ClkGenStatus)
/** \brief APBusClk_LCD clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LCDCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_CDMA ClkGen Configuration Register (APBusClk_CDMAConfig_ClkGenConfig)
/** \brief APBusClk_CDMA clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_CDMA ClkGen Status Register (APBusClk_CDMAConfig_ClkGenStatus)
/** \brief APBusClk_CDMA clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_CDMACONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_SQU ClkGen Configuration Register (APBusClk_SQUConfig_ClkGenConfig)
/** \brief APBusClk_SQU clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_SQU ClkGen Status Register (APBusClk_SQUConfig_ClkGenStatus)
/** \brief APBusClk_SQU clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SQUCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_IOWrap ClkGen Configuration Register (APBusClk_IOWrapConfig_ClkGenConfig)
/** \brief APBusClk_IOWrap clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_IOWrap ClkGen Status Register (APBusClk_IOWrapConfig_ClkGenStatus)
/** \brief APBusClk_IOWrap clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_IOWRAPCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_LAN ClkGen Configuration Register (APBusClk_LANConfig_ClkGenConfig)
/** \brief APBusClk_LAN clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_LAN ClkGen Status Register (APBusClk_LANConfig_ClkGenStatus)
/** \brief APBusClk_LAN clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_LANCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_USB2A ClkGen Configuration Register (APBusClk_USB2AConfig_ClkGenConfig)
/** \brief APBusClk_USB2A clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_USB2A ClkGen Status Register (APBusClk_USB2AConfig_ClkGenStatus)
/** \brief APBusClk_USB2A clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2ACONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_USB2PHY ClkGen Configuration Register (APBusClk_USB2PHYConfig_ClkGenConfig)
/** \brief APBusClk_USB2PHY clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            MASTERDOMAIN clock domain. Changes to this clock's frequency can be accomplished by
            writes to the MASTERDOMAIN ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the MASTERDOMAIN ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_USB2PHY ClkGen Status Register (APBusClk_USB2PHYConfig_ClkGenStatus)
/** \brief APBusClk_USB2PHY clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB2PHYCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_USB3D ClkGen Configuration Register (APBusClk_USB3DConfig_ClkGenConfig)
/** \brief APBusClk_USB3D clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_USB3D ClkGen Status Register (APBusClk_USB3DConfig_ClkGenStatus)
/** \brief APBusClk_USB3D clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3DCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_USB3H ClkGen Configuration Register (APBusClk_USB3HConfig_ClkGenConfig)
/** \brief APBusClk_USB3H clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            MASTERDOMAIN clock domain. Changes to this clock's frequency can be accomplished by
            writes to the MASTERDOMAIN ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the MASTERDOMAIN ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_USB3H ClkGen Status Register (APBusClk_USB3HConfig_ClkGenStatus)
/** \brief APBusClk_USB3H clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3HCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_USB3PHY ClkGen Configuration Register (APBusClk_USB3PHYConfig_ClkGenConfig)
/** \brief APBusClk_USB3PHY clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_USB3PHY ClkGen Status Register (APBusClk_USB3PHYConfig_ClkGenStatus)
/** \brief APBusClk_USB3PHY clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_USB3PHYCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_PCIe_DM2 ClkGen Configuration Register (APBusClk_PCIe_DM2Config_ClkGenConfig)
/** \brief APBusClk_PCIe_DM2 clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_PCIe_DM2 ClkGen Status Register (APBusClk_PCIe_DM2Config_ClkGenStatus)
/** \brief APBusClk_PCIe_DM2 clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM2CONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_PCIe_DM4 ClkGen Configuration Register (APBusClk_PCIe_DM4Config_ClkGenConfig)
/** \brief APBusClk_PCIe_DM4 clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: APBusClk_PCIe_DM4 ClkGen Status Register (APBusClk_PCIe_DM4Config_ClkGenStatus)
/** \brief APBusClk_PCIe_DM4 clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_PCIE_DM4CONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: APBusClk_SATA ClkGen Configuration Register (APBusClk_SATAConfig_ClkGenConfig)
/** \brief APBusClk_SATA clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RSTN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENCONFIG_RSTN_SHIFT))

//====================================================================
//Register: APBusClk_SATA ClkGen Status Register (APBusClk_SATAConfig_ClkGenStatus)
/** \brief APBusClk_SATA clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RSTN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_SATACONFIG_CLKGENSTATUS_RSTN_SHIFT))

//====================================================================
//Register: APBusClk_GPU ClkGen Configuration Register (APBusClk_GPUConfig_ClkGenConfig)
/** \brief APBusClk_GPU clock generator Configuration Register.
            This clock domain shares its frequency configuration, and is aligned, with the
            APBusClk clock domain. Changes to this clock's frequency can be accomplished by
            writes to the APBusClk ClkGenConfig register. The values in this clock domain's
            ClkGenStatus register will reflect the values in the APBusClk ClkGenConfig register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENCONFIG_RSTN_SHIFT))

//====================================================================
//Register: APBusClk_GPU ClkGen Status Register (APBusClk_GPUConfig_ClkGenStatus)
/** \brief APBusClk_GPU clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RSTN_MASK) >> APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APBUSCLK_GPUCONFIG_CLKGENSTATUS_RSTN_SHIFT))

//====================================================================
//Register: SMMUClk ClkGen Configuration Register (SMMUClkConfig_ClkGenConfig)
/** \brief SMMUClk clock generator Configuration Register.
          The maximum frequency for this clock is MAXFREQ. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RSTN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT))

//====================================================================
//Register: SMMUClk ClkGen Status Register (SMMUClkConfig_ClkGenStatus)
/** \brief SMMUClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0x6000000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 25
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0xe00000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 21
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe000
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 13
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RSTN_MASK) >> APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SMMUCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT))

//====================================================================
//Register: GPUBusClk ClkGen Configuration Register (GPUBusClkConfig_ClkGenConfig)
/** \brief GPUBusClk clock generator Configuration Register.
          The maximum frequency for this clock is 400 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT))

//====================================================================
//Register: GPUBusClk ClkGen Status Register (GPUBusClkConfig_ClkGenStatus)
/** \brief GPUBusClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RSTN_MASK) >> APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUBUSCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT))

//====================================================================
//Register: GPUClk ClkGen Configuration Register (GPUClkConfig_ClkGenConfig)
/** \brief GPUClk clock generator Configuration Register.
          The maximum frequency for this clock is 666.7 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENCONFIG_RSTN_SHIFT))

//====================================================================
//Register: GPUClk ClkGen Status Register (GPUClkConfig_ClkGenStatus)
/** \brief GPUClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RSTN_MASK 0x1
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT 0
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RSTN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RSTN_MASK) >> APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT)
#define  APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RSTN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RSTN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_GPUCLKCONFIG_CLKGENSTATUS_RSTN_SHIFT))

//====================================================================
//Register: SDMMCClk ClkGen Configuration Register (SDMMCClkConfig_ClkGenConfig)
/** \brief SDMMCClk clock generator Configuration Register.
          The maximum frequency for this clock is 200 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xf00000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 20
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xf000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 12
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

//====================================================================
//Register: SDMMCClk ClkGen Status Register (SDMMCClkConfig_ClkGenStatus)
/** \brief SDMMCClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0x6000000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 25
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0xf00000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 20
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0xf0000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xf000
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 12
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0xf00
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_SDMMCCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

//====================================================================
//Register: QSPIClk ClkGen Configuration Register (QSPIClkConfig_ClkGenConfig)
/** \brief QSPIClk clock generator Configuration Register.
          The maximum frequency for this clock is 200 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED7_MASK 0x1
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT 0
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED7_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED7_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED7_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENCONFIG_RESERVED7_SHIFT))

//====================================================================
//Register: QSPIClk ClkGen Status Register (QSPIClkConfig_ClkGenStatus)
/** \brief QSPIClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_QSPICLKCONFIG_CLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: LCDXClk ClkGen Configuration Register (LCDXClkConfig_ClkGenConfig)
/** \brief LCDXClk clock generator Configuration Register.
          The maximum frequency for this clock is 100 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xe00000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 21
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xe000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 13
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

//====================================================================
//Register: LCDXClk ClkGen Status Register (LCDXClkConfig_ClkGenStatus)
/** \brief LCDXClk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0x6000000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 25
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0xe00000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 21
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xe000
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 13
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDXCLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

//====================================================================
//Register: LCD Clock Configuration Register (LCDClkConfig)
/** \brief LCD Clock Controls.*/
//====================================================================

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED2_MASK 0x7ffff00
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED2_SHIFT 8
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED3_MASK 0x80
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED3_SHIFT 7
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED4_MASK 0x4
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED4_SHIFT 2
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED5_MASK 0x1
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED5_SHIFT 0
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKCONFIG_RESERVED5_SHIFT))

//====================================================================
//Register: LCD Clock Status Register (LCDClkStatus)
/** \brief LCD Clock Status Register. Note that this clock is derived from the LCDXCLK. The
      read-only fields for PreDiv, Divider count values, and such come from the LCDXCLK
      control register. The ratio select field comes from a control register in the
      LCD block itself.*/
//====================================================================

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RATIOSEL_MASK 0x6000000
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RATIOSEL_SHIFT 25
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RATIOSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_RATIOSEL_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_RATIOSEL_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RATIOSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_RATIOSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_RATIOSEL_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED1_MASK 0xe00000
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED1_SHIFT 21
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_HIDIV_MASK 0x1f0000
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED2_MASK 0xe000
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED2_SHIFT 13
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LODIV_MASK 0x1f00
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED3_MASK 0x1
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED3_SHIFT 0
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LCDCLKSTATUS_RESERVED3_SHIFT))

//====================================================================
//Register: UART0Clk ClkGen Configuration Register (UART0ClkConfig_ClkGenConfig)
/** \brief UART0Clk clock generator Configuration Register.
          The maximum frequency for this clock is 125 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x4000000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 26
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x3000000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xfc0000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 18
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x30000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xfc00
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 10
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x300
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

//====================================================================
//Register: UART0Clk ClkGen Status Register (UART0ClkConfig_ClkGenStatus)
/** \brief UART0Clk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0x4000000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 26
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x3000000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0xfc0000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 18
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x30000
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xfc00
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 10
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x300
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

//====================================================================
//Register: UART0 Clock Fractional Divider Configuration Register (UART0ClkFracDivCfg)
/** \brief This register is used to configure the UART0 clock generator's fractional divider. The 
      equation for determining the numerator and denominator is <br/><br/>
      numerator/denominator = input frequency/(2*ouput frequency) <br/><br/>
      The maximum output frequency the divider is capable of producing is one half of the
      input frequency; therefore, setting the numerator to be less than or equal to the
      denominator will result in a divide by 2 of the input frequency.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_MASK 0xffff0000
#define  APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_SHIFT 16
#define  APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_MASK) >> APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_NUMERATOR_SHIFT))

#define  APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_MASK 0xffff
#define  APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_SHIFT 0
#define  APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_MASK) >> APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_SHIFT)
#define  APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART0CLKFRACDIVCFG_DENOMINATOR_SHIFT))

//====================================================================
//Register: UART1Clk ClkGen Configuration Register (UART1ClkConfig_ClkGenConfig)
/** \brief UART1Clk clock generator Configuration Register.
          The maximum frequency for this clock is 125 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x4000000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 26
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x3000000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xfc0000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 18
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x30000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xfc00
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 10
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x300
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

//====================================================================
//Register: UART1Clk ClkGen Status Register (UART1ClkConfig_ClkGenStatus)
/** \brief UART1Clk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0x4000000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 26
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x3000000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0xfc0000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 18
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x30000
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xfc00
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 10
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x300
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

//====================================================================
//Register: UART1 Clock Fractional Divider Configuration Register (UART1ClkFracDivCfg)
/** \brief This register is used to configure the UART1 clock generator's fractional divider. The 
      equation for determining the numerator and denominator is <br/><br/>
      numerator/denominator = input frequency/(2*ouput frequency) <br/><br/>
      The maximum output frequency the divider is capable of producing is one half of the
      input frequency; therefore, setting the numerator to be less than or equal to the
      denominator will result in a divide by 2 of the input frequency.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_MASK 0xffff0000
#define  APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_SHIFT 16
#define  APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_MASK) >> APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_NUMERATOR_SHIFT))

#define  APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_MASK 0xffff
#define  APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_SHIFT 0
#define  APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_MASK) >> APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_SHIFT)
#define  APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART1CLKFRACDIVCFG_DENOMINATOR_SHIFT))

//====================================================================
//Register: UART2Clk ClkGen Configuration Register (UART2ClkConfig_ClkGenConfig)
/** \brief UART2Clk clock generator Configuration Register.
          The maximum frequency for this clock is 125 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x4000000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 26
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x3000000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xfc0000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 18
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x30000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xfc00
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 10
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x300
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

//====================================================================
//Register: UART2Clk ClkGen Status Register (UART2ClkConfig_ClkGenStatus)
/** \brief UART2Clk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0x4000000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 26
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x3000000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0xfc0000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 18
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x30000
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xfc00
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 10
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x300
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

//====================================================================
//Register: UART2 Clock Fractional Divider Configuration Register (UART2ClkFracDivCfg)
/** \brief This register is used to configure the UART2 clock generator's fractional divider. The 
      equation for determining the numerator and denominator is <br/><br/>
      numerator/denominator = input frequency/(2*ouput frequency) <br/><br/>
      The maximum output frequency the divider is capable of producing is one half of the
      input frequency; therefore, setting the numerator to be less than or equal to the
      denominator will result in a divide by 2 of the input frequency.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_MASK 0xffff0000
#define  APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_SHIFT 16
#define  APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_MASK) >> APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_NUMERATOR_SHIFT))

#define  APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_MASK 0xffff
#define  APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_SHIFT 0
#define  APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_MASK) >> APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_SHIFT)
#define  APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART2CLKFRACDIVCFG_DENOMINATOR_SHIFT))

//====================================================================
//Register: UART3Clk ClkGen Configuration Register (UART3ClkConfig_ClkGenConfig)
/** \brief UART3Clk clock generator Configuration Register.
          The maximum frequency for this clock is 125 MHz. It is important to ensure that the
          configuration settings for this clock result in an output frequency less than this value.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK 0x4000000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT 26
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK 0x3000000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK 0xfc0000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT 18
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_MASK 0x30000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK 0xfc00
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT 10
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_MASK 0x300
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENCONFIG_RESERVED6_SHIFT))

//====================================================================
//Register: UART3Clk ClkGen Status Register (UART3ClkConfig_ClkGenStatus)
/** \brief UART3Clk clock generator Status Register.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK 0x4000000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT 26
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK 0x3000000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK 0xfc0000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT 18
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_HIDIV_MASK 0x30000
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK 0xfc00
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT 10
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LODIV_MASK 0x300
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKCONFIG_CLKGENSTATUS_RESERVED4_SHIFT))

//====================================================================
//Register: UART3 Clock Fractional Divider Configuration Register (UART3ClkFracDivCfg)
/** \brief This register is used to configure the UART3 clock generator's fractional divider. The 
      equation for determining the numerator and denominator is <br/><br/>
      numerator/denominator = input frequency/(2*ouput frequency) <br/><br/>
      The maximum output frequency the divider is capable of producing is one half of the
      input frequency; therefore, setting the numerator to be less than or equal to the
      denominator will result in a divide by 2 of the input frequency.*/
//====================================================================

#define  APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_MASK 0xffff0000
#define  APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_SHIFT 16
#define  APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_MASK) >> APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_NUMERATOR_SHIFT))

#define  APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_MASK 0xffff
#define  APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_SHIFT 0
#define  APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_MASK) >> APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_SHIFT)
#define  APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_UART3CLKFRACDIVCFG_DENOMINATOR_SHIFT))

//====================================================================
//Register: LAN Mac 2x Clock Configuration Register (LANMacClk2xConfig)
/** \brief <p>LAN MAC 2x Clock Configuration Register. The LAN MAC Clock clock generator
      always uses the System PLL as its reference clock. The System PLL must be programmed to
      run at either 2.5 GHz, or 1 GHz for proper LAN operation. </p>
      <p>Note that the LAN MAC clock will always run at half the frequency that is
      programmed for this clock.</p>*/
//====================================================================

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED1_MASK 0x3e000000
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED1_SHIFT 25
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_MASK 0x1000000
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_SHIFT 24
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_SRCFREQ_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED2_MASK 0xffff80
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED2_SHIFT 7
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED3_MASK 0x4
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED3_SHIFT 2
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XCONFIG_RESERVED4_SHIFT))

//====================================================================
//Register: LAN Mac 2x Clock Status Register (LANMacClk2xStatus)
/** \brief LAN MAC 2x Clock Status Register. Note that the LAN MAC clock will always
      run at half the frequency that is programmed for this clock.*/
//====================================================================

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIV_MASK 0xf8000000
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIV_SHIFT 27
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIV_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIV_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIV_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED1_MASK 0x6000000
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED1_SHIFT 25
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_SRCFREQ_MASK 0x1000000
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_SRCFREQ_SHIFT 24
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_SRCFREQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_SRCFREQ_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_SRCFREQ_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_SRCFREQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_SRCFREQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_SRCFREQ_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED2_MASK 0xc00000
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED2_SHIFT 22
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_HIDIV_MASK 0x3f0000
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_HIDIV_SHIFT 16
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_HIDIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_HIDIV_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_HIDIV_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_HIDIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_HIDIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_HIDIV_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED3_MASK 0xc000
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED3_SHIFT 14
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LODIV_MASK 0x3f00
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LODIV_SHIFT 8
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LODIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LODIV_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LODIV_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LODIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LODIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LODIV_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIVEN_MASK 0x4
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIVEN_SHIFT 2
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIVEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIVEN_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIVEN_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIVEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIVEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_PREDIVEN_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED4_MASK 0x1
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED4_SHIFT 0
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANMACCLK2XSTATUS_RESERVED4_SHIFT))

//====================================================================
//Register: ClkGen Configuration Register (LANRxClkGenConfig)
/** \brief PMU ClkGen Configuration Register*/
//====================================================================

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACKOVERRIDE_MASK 0x80000000
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACKOVERRIDE_SHIFT 31
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACKOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACKOVERRIDE_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACKOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACKOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACKOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACKOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACTIVEOVERRIDE_MASK 0x40000000
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACTIVEOVERRIDE_SHIFT 30
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACTIVEOVERRIDE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACTIVEOVERRIDE_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACTIVEOVERRIDE_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACTIVEOVERRIDE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACTIVEOVERRIDE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_ACTIVEOVERRIDE_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED1_MASK 0x38000000
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED1_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED2_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED3_MASK 0xff0000
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED3_SHIFT 16
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED3_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED4_MASK 0xff00
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED4_SHIFT 8
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED4_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED5_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKONREQIRQEN_MASK 0x40
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKONREQIRQEN_SHIFT 6
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKONREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKONREQIRQEN_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKONREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKONREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKONREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKONREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKOFFREQIRQEN_MASK 0x20
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKOFFREQIRQEN_SHIFT 5
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKOFFREQIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKOFFREQIRQEN_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKOFFREQIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKOFFREQIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKOFFREQIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKOFFREQIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQDENIEDIRQEN_MASK 0x10
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT 4
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQDENIEDIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQDENIEDIRQEN_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQDENIEDIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQDENIEDIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQDENIEDIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQSUCCESSIRQEN_MASK 0x8
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT 3
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQSUCCESSIRQEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQSUCCESSIRQEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQSUCCESSIRQEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_LPREQSUCCESSIRQEN_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED6_MASK 0x4
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED6_SHIFT 2
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED6_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED7_MASK 0x1
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED7_SHIFT 0
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED7_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED7_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED7_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED7_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED7_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENCONFIG_RESERVED7_SHIFT))

//====================================================================
//Register: ClkGen Status Register (LANRxClkGenStatus)
/** \brief PMU ClkGen Status Register*/
//====================================================================

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED1_MASK 0xf8000000
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED1_SHIFT 27
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED2_MASK 0x6000000
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED2_SHIFT 25
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_SRCSEL_MASK 0x1000000
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_SRCSEL_SHIFT 24
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_SRCSEL_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED3_MASK 0xff0000
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED3_SHIFT 16
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED3_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED4_MASK 0xff00
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED4_SHIFT 8
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED4_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_MASK 0x80
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_SHIFT 7
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKGATED_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKONREQIRQ_MASK 0x40
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKONREQIRQ_SHIFT 6
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKONREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKONREQIRQ_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKONREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKONREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKONREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKONREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKOFFREQIRQ_MASK 0x20
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKOFFREQIRQ_SHIFT 5
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKOFFREQIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKOFFREQIRQ_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKOFFREQIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKOFFREQIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKOFFREQIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKOFFREQIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQDENIEDIRQ_MASK 0x10
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQDENIEDIRQ_SHIFT 4
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQDENIEDIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQDENIEDIRQ_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQDENIEDIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQDENIEDIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQDENIEDIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQDENIEDIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQSUCCESSIRQ_MASK 0x8
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT 3
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQSUCCESSIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQSUCCESSIRQ_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQSUCCESSIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQSUCCESSIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_LPREQSUCCESSIRQ_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED5_MASK 0x4
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED5_SHIFT 2
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED5_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKEN_MASK 0x2
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKEN_SHIFT 1
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKEN_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKEN_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_CLKEN_SHIFT))

#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED6_MASK 0x1
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED6_SHIFT 0
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED6_MASK) >> APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_LANRXCLKGENSTATUS_RESERVED6_SHIFT))

//====================================================================
//Register: AP CPU Miscellaneous Control Register (APCPUMiscControl)
/** \brief The bits in this register are used to control the resets going to the AP CPU subsystem.*/
//====================================================================

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED1_MASK 0xf0000000
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED1_SHIFT 28
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED1_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUFIQ_MASK 0xf000000
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUFIQ_SHIFT 24
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUFIQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUFIQ_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUFIQ_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUFIQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUFIQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUFIQ_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUIRQ_MASK 0xf00000
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUIRQ_SHIFT 20
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUIRQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUIRQ_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUIRQ_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUIRQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUIRQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_NMASKCPUIRQ_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED2_MASK 0x80000
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED2_SHIFT 19
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED2_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK 0x40000
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_SHIFT 18
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUAT_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_MASK 0x20000
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_SHIFT 17
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_CPUL2RSTDISABLE_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_MASK 0x10000
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_SHIFT 16
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUL2_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED3_MASK 0xff00
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED3_SHIFT 8
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED3_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK 0xf0
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_SHIFT 4
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPU_SHIFT))

#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK 0xf
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_SHIFT 0
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK) >> APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_SHIFT)
#define  APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUMISCCONTROL_NRSTCPUPOR_SHIFT))

//====================================================================
//Register: AP CPU Clock Control Register (APCPUClkControl)
/** \brief The bits in this register are fed to the CPU sub-systems clock gen block.*/
//====================================================================

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED1_MASK 0xe0000000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED1_SHIFT 29
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED1_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ACPINACTIVE_MASK 0x10000000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ACPINACTIVE_SHIFT 28
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ACPINACTIVE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_ACPINACTIVE_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_ACPINACTIVE_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ACPINACTIVE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_ACPINACTIVE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_ACPINACTIVE_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_MASK 0x8000000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_SHIFT 27
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLK_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_MASK 0x4000000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_SHIFT 26
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_L2RAMDUMMYCLKSEL_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2FLUSHREQ_MASK 0x2000000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2FLUSHREQ_SHIFT 25
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2FLUSHREQ_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_L2FLUSHREQ_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_L2FLUSHREQ_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_L2FLUSHREQ_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_L2FLUSHREQ_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_L2FLUSHREQ_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_PCLKENDBG_DIV_MASK 0x1f00000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_PCLKENDBG_DIV_SHIFT 20
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_PCLKENDBG_DIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_PCLKENDBG_DIV_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_PCLKENDBG_DIV_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_PCLKENDBG_DIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_PCLKENDBG_DIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_PCLKENDBG_DIV_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED2_MASK 0x80000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED2_SHIFT 19
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED2_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_MASK 0x70000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_SHIFT 16
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_ATCLKEN_DIV_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED3_MASK 0x8000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED3_SHIFT 15
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED3_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED3_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED3_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED3_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED3_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED3_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_PERIPHCLK_DIV_MASK 0x7000
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_PERIPHCLK_DIV_SHIFT 12
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_PERIPHCLK_DIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_PERIPHCLK_DIV_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_PERIPHCLK_DIV_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_PERIPHCLK_DIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_PERIPHCLK_DIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_PERIPHCLK_DIV_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED4_MASK 0x800
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED4_SHIFT 11
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED4_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED4_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED4_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED4_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED4_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED4_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_MASK 0x700
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_SHIFT 8
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_ACLKENM_DIV_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED5_MASK 0x80
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED5_SHIFT 7
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED5_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED5_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED5_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED5_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED5_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED5_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_MASK 0x70
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_SHIFT 4
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_COREAPCKL_DIV_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_MASK 0xc
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_SHIFT 2
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_CORECLK_SRCSEL_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED6_MASK 0x2
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED6_SHIFT 1
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED6_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED6_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED6_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED6_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED6_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_RESERVED6_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_MASK 0x1
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_SHIFT 0
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_MASK) >> APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKCONTROL_CPUCLKGEN_STOP_SHIFT))

//====================================================================
//Register: AP CPU Clocks Status Register (APCPUClkStatus)
/** \brief Various CPU clock status fields.*/
//====================================================================

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED1_MASK 0xfc000000
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED1_SHIFT 26
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2FLUSHDONE_MASK 0x2000000
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2FLUSHDONE_SHIFT 25
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2FLUSHDONE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_L2FLUSHDONE_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_L2FLUSHDONE_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2FLUSHDONE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_L2FLUSHDONE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_L2FLUSHDONE_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2QACTIVE_MASK 0x1000000
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2QACTIVE_SHIFT 24
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2QACTIVE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_L2QACTIVE_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_L2QACTIVE_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2QACTIVE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_L2QACTIVE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_L2QACTIVE_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_MASK 0xf00000
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_SHIFT 20
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_DBGNOPWRDOWN_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUSMPEN_MASK 0xf0000
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUSMPEN_SHIFT 16
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUSMPEN_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUSMPEN_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUSMPEN_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUSMPEN_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUSMPEN_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUSMPEN_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUQACTIVE_MASK 0xf000
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUQACTIVE_SHIFT 12
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUQACTIVE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUQACTIVE_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUQACTIVE_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUQACTIVE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUQACTIVE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_CPUQACTIVE_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFE_MASK 0xf00
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFE_SHIFT 8
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFE_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFE_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFE_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_MASK 0xf0
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_SHIFT 4
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_STANDBYWFI_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2STANDBYWFI_MASK 0x8
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2STANDBYWFI_SHIFT 3
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2STANDBYWFI_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_L2STANDBYWFI_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_L2STANDBYWFI_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_L2STANDBYWFI_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_L2STANDBYWFI_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_L2STANDBYWFI_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED2_MASK 0x4
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED2_SHIFT 2
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED2_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED2_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED2_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED2_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED2_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_RESERVED2_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_MASK 0x2
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_SHIFT 1
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKGENIDLE_SHIFT))

#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKCHANGE_MASK 0x1
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKCHANGE_SHIFT 0
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKCHANGE_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKCHANGE_MASK) >> APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKCHANGE_SHIFT)
#define  APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKCHANGE_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKCHANGE_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_APCPUCLKSTATUS_CLKCHANGE_SHIFT))

//====================================================================
//Register: R4 CPU Clocks Status Register (R4ClkStatus)
/** \brief R4 clock status.*/
//====================================================================

#define  APMU_CLKRSTGEN_R4CLKSTATUS_RESERVED1_MASK 0xfffffffe
#define  APMU_CLKRSTGEN_R4CLKSTATUS_RESERVED1_SHIFT 1
#define  APMU_CLKRSTGEN_R4CLKSTATUS_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_R4CLKSTATUS_RESERVED1_MASK) >> APMU_CLKRSTGEN_R4CLKSTATUS_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_R4CLKSTATUS_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_R4CLKSTATUS_RESERVED1_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_R4CLKSTATUS_RESERVED1_SHIFT))

#define  APMU_CLKRSTGEN_R4CLKSTATUS_STANDBYWFI_MASK 0x1
#define  APMU_CLKRSTGEN_R4CLKSTATUS_STANDBYWFI_SHIFT 0
#define  APMU_CLKRSTGEN_R4CLKSTATUS_STANDBYWFI_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_R4CLKSTATUS_STANDBYWFI_MASK) >> APMU_CLKRSTGEN_R4CLKSTATUS_STANDBYWFI_SHIFT)
#define  APMU_CLKRSTGEN_R4CLKSTATUS_STANDBYWFI_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKRSTGEN_R4CLKSTATUS_STANDBYWFI_MASK) | (((uint32_t)val) << APMU_CLKRSTGEN_R4CLKSTATUS_STANDBYWFI_SHIFT))

//====================================================================
//Register: PCIe Reset Control (PCIeRstCtl)
//====================================================================

#define  APMU_CLKRSTGEN_PCIERSTCTL_RESERVED1_MASK 0xfffffffc
#define  APMU_CLKRSTGEN_PCIERSTCTL_RESERVED1_SHIFT 2
#define  APMU_CLKRSTGEN_PCIERSTCTL_RESERVED1_MASK_SHIFT(reg) (((reg) & APMU_CLKRSTGEN_PCIERSTCTL_RESERVED1_MASK) >> APMU_CLKRSTGEN_PCIERSTCTL_RESERVED1_SHIFT)
#define  APMU_CLKRSTGEN_PCIERSTCTL_RESERVED1_REPLACE_VAL(reg,val) (((reg) & ~APMU_CLKR