// SPDX-License-Identifier: GPL-2.0
/*
 * techpoint techpoint lib
 *
 * Copyright (C) 2021 Rockchip Electronics Co., Ltd.
 */

#include "techpoint_tp9930.h"
#include "techpoint_dev.h"

static __maybe_unused const struct regval common_setting_148M_1080p_25fps_regs[] = {
	{ 0x40, 0x04 },
	{ 0x02, 0x44 },
	{ 0x05, 0x00 },
	{ 0x06, 0x32 },
	{ 0x07, 0x80 },
	{ 0x08, 0x00 },
	{ 0x09, 0x24 },
	{ 0x0a, 0x48 },
	{ 0x0b, 0xc0 },
	{ 0x0c, 0x03 },
	{ 0x0d, 0x73 },
	{ 0x10, 0x00 },
	{ 0x11, 0x40 },
	{ 0x12, 0x40 },
	{ 0x13, 0x00 },
	{ 0x14, 0x00 },
	{ 0x15, 0x01 },
	{ 0x16, 0xf0 },
	{ 0x17, 0x80 },
	{ 0x18, 0x29 },
	{ 0x19, 0x38 },
	{ 0x1a, 0x47 },
	{ 0x1c, 0x0a },
	{ 0x1d, 0x50 },
	{ 0x20, 0x3c },
	{ 0x21, 0x46 },
	{ 0x22, 0x36 },
	{ 0x23, 0x3c },
	{ 0x24, 0x04 },
	{ 0x25, 0xfe },
	{ 0x26, 0x0d },
	{ 0x27, 0x2d },
	{ 0x28, 0x00 },
	{ 0x29, 0x48 },
#if TECHPOINT_TEST_PATTERN
	{ 0x2a, 0x3c },
#else
	{ 0x2a, 0x30 },
#endif
	{ 0x2b, 0x60 },
	{ 0x2c, 0x3a },
	{ 0x2d, 0x54 },
	{ 0x2e, 0x40 },
	{ 0x30, 0xa5 },
	{ 0x31, 0x86 },
	{ 0x32, 0xfb },
	{ 0x33, 0x60 },
	{ 0x35, 0x05 },
	{ 0x36, 0xca },
	{ 0x38, 0x00 },
	{ 0x39, 0x1c },
	{ 0x3a, 0x32 },
	{ 0x3b, 0x26 },
	{ 0x40, 0x00 },
	{ 0x34, 0x10 },
	{ 0x40, 0x01 },
	{ 0x34, 0x11 },
	{ 0x40, 0x02 },
	{ 0x34, 0x12 },
	{ 0x40, 0x03 },
	{ 0x34, 0x13 },
	{ 0x4f, 0x03 },
	{ 0x50, 0xb2 },
	{ 0x52, 0xf6 },
	{ 0xf1, 0x04 },
	{ 0xf2, 0x77 },
	{ 0xf3, 0x77 },
	{ 0xf5, 0xf0 },
	{ 0xf6, 0x10 },
	{ 0xf8, 0x54 },
	{ 0xfa, 0x88 },
	{ 0xfb, 0x88 },
	{ 0x4d, 0x07 },
	{ 0x4e, 0x05 },
};

static __maybe_unused const struct regval common_setting_148M_1080p_30fps_regs[] = {
	{ 0x40, 0x04 },
	{ 0x02, 0x44 },
	{ 0x07, 0x80 },
	{ 0x0b, 0xc0 },
	{ 0x0c, 0x03 },
	{ 0x0d, 0x72 },
	{ 0x10, 0x00 },
	{ 0x11, 0x40 },
	{ 0x12, 0x40 },
	{ 0x13, 0x00 },
	{ 0x14, 0x00 },
	{ 0x15, 0x01 },
	{ 0x16, 0xf0 },
	{ 0x17, 0x80 },
	{ 0x18, 0x29 },
	{ 0x19, 0x38 },
	{ 0x1a, 0x47 },
	{ 0x1c, 0x08 },
	{ 0x1d, 0x98 },
	{ 0x20, 0x38 },
	{ 0x21, 0x46 },
	{ 0x22, 0x36 },
	{ 0x23, 0x3c },
	{ 0x24, 0x04 },
	{ 0x25, 0xfe },
	{ 0x26, 0x0d },
	{ 0x27, 0x2d },
	{ 0x28, 0x00 },
	{ 0x29, 0x48 },
#if TECHPOINT_TEST_PATTERN
	{ 0x2a, 0x3c },
#else
	{ 0x2a, 0x30 },
#endif
	{ 0x2b, 0x60 },
	{ 0x2c, 0x3a },
	{ 0x2d, 0x54 },
	{ 0x2e, 0x40 },
	{ 0x30, 0xa5 },
	{ 0x31, 0x95 },
	{ 0x32, 0xe0 },
	{ 0x33, 0x60 },
	{ 0x35, 0x45 },
	{ 0x36, 0xca },
	{ 0x38, 0x00 },
	{ 0x39, 0x1c },
	{ 0x3a, 0x32 },
	{ 0x3b, 0x26 },
	{ 0x40, 0x00 },
	{ 0x34, 0x10 },
	{ 0x40, 0x01 },
	{ 0x34, 0x11 },
	{ 0x40, 0x02 },
	{ 0x34, 0x12 },
	{ 0x40, 0x03 },
	{ 0x34, 0x13 },
	{ 0x4f, 0x03 },
	{ 0x50, 0xA3 },
	{ 0x52, 0xE7 },
	{ 0xf1, 0x04 },
	{ 0xf2, 0x77 },
	{ 0xf3, 0x77 },
	{ 0xf4, 0x00 },
	{ 0xf5, 0xf0 },
	{ 0xf6, 0x10 },
	{ 0xf8, 0x54 },
	{ 0xfa, 0x99 },
	{ 0xfb, 0x99 },
	{ 0x4d, 0x07 },
	{ 0x4e, 0x05 },
};

static __maybe_unused const struct regval common_setting_148M_720p_25fps_regs[] = {
	{ 0x40, 0x04 },
	{ 0x02, 0x4e },
	{ 0x05, 0x00 },
	{ 0x06, 0x32 },
	{ 0x07, 0xc0 },
	{ 0x08, 0x00 },
	{ 0x09, 0x24 },
	{ 0x0a, 0x48 },
	{ 0x0b, 0xc0 },
	{ 0x0c, 0x13 },
	{ 0x0d, 0x71 },
	{ 0x0e, 0x00 },
	{ 0x0f, 0x00 },
	{ 0x10, 0x00 },
	{ 0x11, 0x40 },
	{ 0x12, 0x40 },
	{ 0x13, 0x00 },
	{ 0x14, 0x00 },
	{ 0x15, 0x13 },
	{ 0x16, 0x16 },
	{ 0x17, 0x00 },
	{ 0x18, 0x19 },
	{ 0x19, 0xd0 },
	{ 0x1a, 0x25 },
	{ 0x1b, 0x00 },
	{ 0x1c, 0x07 },
	{ 0x1d, 0xbc },
	{ 0x1e, 0x60 },
	{ 0x1f, 0x06 },
	{ 0x20, 0x40 },
	{ 0x21, 0x46 },
	{ 0x22, 0x36 },
	{ 0x23, 0x3c },
	{ 0x24, 0x04 },
	{ 0x25, 0xfe },
	{ 0x26, 0x01 },
	{ 0x27, 0x2d },
	{ 0x28, 0x00 },
	{ 0x29, 0x48 },
#if TECHPOINT_TEST_PATTERN
	{ 0x2a, 0x3c },
#else
	{ 0x2a, 0x30 },
#endif
	{ 0x2b, 0x60 },
	{ 0x2c, 0x3a },
	{ 0x2d, 0x5a },
	{ 0x2e, 0x40 },
	{ 0x2f, 0x06 },
	{ 0x30, 0x9e },
	{ 0x31, 0x20 },
	{ 0x32, 0x01 },
	{ 0x33, 0x90 },
	{ 0x35, 0x25 },
	{ 0x36, 0xca },
	{ 0x37, 0x00 },
	{ 0x38, 0x00 },
	{ 0x39, 0x18 },
	{ 0x3a, 0x32 },
	{ 0x3b, 0x26 },
	{ 0x3c, 0x00 },
	{ 0x3d, 0x60 },
	{ 0x3e, 0x00 },
	{ 0x3f, 0x00 },

	{ 0x40, 0x00 },
	{ 0x34, 0x10 },
	{ 0x40, 0x01 },
	{ 0x34, 0x11 },
	{ 0x40, 0x02 },
	{ 0x34, 0x12 },
	{ 0x40, 0x03 },
	{ 0x34, 0x13 },

	{ 0x4f, 0x03 },
	{ 0x50, 0xA3 },
	{ 0x52, 0xE7 },
	{ 0xf1, 0x04 },
	{ 0xf2, 0x77 },
	{ 0xf3, 0x77 },
	{ 0xf4, 0x00 },
	{ 0xf5, 0xff },
	{ 0xf6, 0x10 },
	{ 0xf8, 0x54 },
	{ 0xfa, 0x99 },
	{ 0xfb, 0x99 },

	{ 0x4d, 0x07 },
	{ 0x4e, 0x05 },
};

static __maybe_unused const struct regval common_setting_148M_720p_30fps_regs[] = {
	{ 0x40, 0x04 },
	{ 0x02, 0x4e },
	{ 0x05, 0x00 },
	{ 0x06, 0x32 },
	{ 0x07, 0xc0 },
	{ 0x08, 0x00 },
	{ 0x09, 0x24 },
	{ 0x0a, 0x48 },
	{ 0x0b, 0xc0 },
	{ 0x0c, 0x13 },
	{ 0x0d, 0x70 },
	{ 0x0e, 0x00 },
	{ 0x0f, 0x00 },
	{ 0x10, 0x00 },
	{ 0x11, 0x40 },
	{ 0x12, 0x40 },
	{ 0x13, 0x00 },
	{ 0x14, 0x00 },
	{ 0x15, 0x13 },
	{ 0x16, 0x16 },
	{ 0x17, 0x00 },
	{ 0x18, 0x19 },
	{ 0x19, 0xd0 },
	{ 0x1a, 0x25 },
	{ 0x1b, 0x00 },
	{ 0x1c, 0x06 },
	{ 0x1d, 0x72 },
	{ 0x1e, 0x60 },
	{ 0x1f, 0x06 },
	{ 0x20, 0x40 },
	{ 0x21, 0x46 },
	{ 0x22, 0x36 },
	{ 0x23, 0x3c },
	{ 0x24, 0x04 },
	{ 0x25, 0xfe },
	{ 0x26, 0x01 },
	{ 0x27, 0x2d },
	{ 0x28, 0x00 },
	{ 0x29, 0x48 },
#if TECHPOINT_TEST_PATTERN
	{ 0x2a, 0x3c },
#else
	{ 0x2a, 0x30 },
#endif
	{ 0x2b, 0x60 },
	{ 0x2c, 0x3a },
	{ 0x2d, 0x5a },
	{ 0x2e, 0x40 },
	{ 0x2f, 0x06 },
	{ 0x30, 0x9d },
	{ 0x31, 0xca },
	{ 0x32, 0x01 },
	{ 0x33, 0xd0 },
	{ 0x35, 0x25 },
	{ 0x36, 0xca },
	{ 0x37, 0x00 },
	{ 0x38, 0x00 },
	{ 0x39, 0x18 },
	{ 0x3a, 0x32 },
	{ 0x3b, 0x26 },
	{ 0x3c, 0x00 },
	{ 0x3d, 0x60 },
	{ 0x3e, 0x00 },
	{ 0x3f, 0x00 },

	{ 0x40, 0x00 },
	{ 0x34, 0x10 },
	{ 0x40, 0x01 },
	{ 0x34, 0x11 },
	{ 0x40, 0x02 },
	{ 0x34, 0x12 },
	{ 0x40, 0x03 },
	{ 0x34, 0x13 },

	{ 0x4f, 0x03 },
	{ 0x50, 0xA3 },
	{ 0x52, 0xE7 },
	{ 0xf1, 0x04 },
	{ 0xf2, 0x77 },
	{ 0xf3, 0x77 },
	{ 0xf4, 0x00 },
	{ 0xf5, 0xff },
	{ 0xf6, 0x10 },
	{ 0xf8, 0x54 },
	{ 0xfa, 0x99 },
	{ 0xfb, 0x99 },

	{ 0x4d, 0x07 },
	{ 0x4e, 0x05 },
};

static struct techpoint_video_modes supported_modes[] = {
#if DEF_1080P
	{
	 .bus_fmt = MEDIA_BUS_FMT_UYVY8_2X8,
	 .width = 1920,
	 .height = 1080,
	 .max_fps = {
		     .numerator = 10000,
		     .denominator = 250000,
		     },
	 .link_freq_value = TP9930_LINK_FREQ_297M,
	 .common_reg_list = common_setting_148M_1080p_25fps_regs,
	 .common_reg_size = ARRAY_SIZE(common_setting_148M_1080p_25fps_regs),
	  },
	{
	 .bus_fmt = MEDIA_BUS_FMT_UYVY8_2X8,
	 .width = 1920,
	 .height = 1080,
	 .max_fps = {
		     .numerator = 10000,
		     .denominator = 300000,
		     },
	 .link_freq_value = TP9930_LINK_FREQ_297M,
	 .common_reg_list = common_setting_148M_1080p_30fps_regs,
	 .common_reg_size = ARRAY_SIZE(common_setting_148M_1080p_30fps_regs),
	  },
#endif
	{
	 .bus_fmt = MEDIA_BUS_FMT_UYVY8_2X8,
	 .width = 1280,
	 .height = 720,
	 .max_fps = {
		     .numerator = 10000,
		     .denominator = 250000,
		     },
	 .link_freq_value = TP9930_LINK_FREQ_148M5,
	 .common_reg_list = common_setting_148M_720p_25fps_regs,
	 .common_reg_size = ARRAY_SIZE(common_setting_148M_720p_25fps_regs),
	  },
	{
	 .bus_fmt = MEDIA_BUS_FMT_UYVY8_2X8,
	 .width = 1280,
	 .height = 720,
	 .max_fps = {
		     .numerator = 10000,
		     .denominator = 300000,
		     },
	 .link_freq_value = TP9930_LINK_FREQ_148M5,
	 .common_reg_list = common_setting_148M_720p_30fps_regs,
	 .common_reg_size = ARRAY_SIZE(common_setting_148M_720p_30fps_regs),
	  }
};

int tp9930_initialize(struct techpoint *techpoint)
{
	int array_size = 0;
	struct i2c_client *client = techpoint->client;
	struct device *dev = &client->dev;

	techpoint->video_modes_num = ARRAY_SIZE(supported_modes);
	array_size =
	    sizeof(struct techpoint_video_modes) * techpoint->video_modes_num;
	techpoint->video_modes = devm_kzalloc(dev, array_size, GFP_KERNEL);
	memcpy(techpoint->video_modes, supported_modes, array_size);

	techpoint->cur_video_mode = &techpoint->video_modes[0];

	return 0;
}

int tp9930_do_reset_pll(struct i2c_client *client)
{
	u8 val_0x44 = 0, val_0x43 = 0, val_0xf4 = 0;

	techpoint_read_reg(client, 0x43, &val_0x43);
	techpoint_write_reg(client, 0x43, val_0x43 | 0x40);
	techpoint_read_reg(client, 0x44, &val_0x44);
	techpoint_write_reg(client, 0x44, val_0x44 | 0x40);

	techpoint_read_reg(client, 0xf4, &val_0xf4);
	techpoint_write_reg(client, 0xf4, val_0xf4 | 0x80);
	msleep(10);

	techpoint_write_reg(client, 0x43, val_0x43);
	techpoint_write_reg(client, 0x44, val_0x44);

	return 0;
}

int tp9930_pll_reset(struct i2c_client *client)
{
	techpoint_write_reg(client, 0x40, 0x00);
	// output disable
	techpoint_write_reg(client, 0x4d, 0x00);
	techpoint_write_reg(client, 0x4e, 0x00);
	// PLL reset
	tp9930_do_reset_pll(client);

	techpoint_write_reg(client, 0x40, 0x04);
	techpoint_write_reg(client, 0x3b, 0x20);
	techpoint_write_reg(client, 0x3d, 0xe0);
	techpoint_write_reg(client, 0x3d, 0x60);
	techpoint_write_reg(client, 0x3b, 0x25);
	techpoint_write_reg(client, 0x40, 0x40);
	techpoint_write_reg(client, 0x7a, 0x20);
	techpoint_write_reg(client, 0x3c, 0x20);
	techpoint_write_reg(client, 0x3c, 0x00);
	techpoint_write_reg(client, 0x7a, 0x25);
	techpoint_write_reg(client, 0x40, 0x00);

#if DEF_1080P
#if 1				// 25FPS
	techpoint_write_reg(client, 0x44, 0x07);
	techpoint_write_reg(client, 0x43, 0x17);
	techpoint_write_reg(client, 0x45, 0x09);
	techpoint_write_reg(client, 0xf4, 0xa0);
#else
	techpoint_write_reg(client, 0x44, 0x17);
	techpoint_write_reg(client, 0x43, 0x12);
	techpoint_write_reg(client, 0x45, 0x09);
#endif
#else
	techpoint_write_reg(client, 0x44, 0x17);
	techpoint_write_reg(client, 0x43, 0x17);
	techpoint_write_reg(client, 0x45, 0x09);
#endif

	return 0;
}

int tp9930_set_decoder_mode(struct i2c_client *client, int ch, int status)
{
	u8 val = 0;

	techpoint_write_reg(client, PAGE_REG, ch);
	techpoint_read_reg(client, 0x26, &val);
	if (status)
		val |= 0x1;
	else
		val &= ~0x1;
	techpoint_write_reg(client, 0x26, val);

	return 0;
}

int tp9930_get_channel_input_status(struct i2c_client *client, u8 ch)
{
	u8 val = 0;

	techpoint_write_reg(client, PAGE_REG, ch);
	techpoint_read_reg(client, INPUT_STATUS_REG, &val);
	dev_dbg(&client->dev, "input_status ch %d : %x\n", ch, val);

#if 0				// inaccuracy
	return (val & INPUT_STATUS_MASK) ? 0 : 1;
#else
	return (val == INPUT_STATUS_MATCH) ? 1 : 0;
#endif
}

int tp9930_get_all_input_status(struct i2c_client *client, u8 *detect_status)
{
	u8 val = 0, i;

	for (i = 0; i < PAD_MAX; i++) {
		techpoint_write_reg(client, PAGE_REG, i);
		techpoint_read_reg(client, INPUT_STATUS_REG, &val);
		detect_status[i] = tp9930_get_channel_input_status(client, i);
	}

	return 0;
}

int tp9930_set_channel_reso(struct i2c_client *client, int ch,
			    enum techpoint_support_reso reso)
{
	int val = reso;

	switch (val) {
	case TECHPOINT_S_RESO_1080P_30:
		dev_err(&client->dev, "set channel %d 1080P_30, TBD", ch);
		break;
	case TECHPOINT_S_RESO_1080P_25:
		dev_err(&client->dev, "set channel %d 1080P_25, TBD", ch);
		break;
	case TECHPOINT_S_RESO_720P_30:
		dev_err(&client->dev, "set channel %d 720P_30, TBD", ch);
		break;
	case TECHPOINT_S_RESO_720P_25:
		dev_err(&client->dev, "set channel %d 720P_25, TBD", ch);
		break;
	default:
#if DEF_1080P
		dev_err(&client->dev,
			"set channel %d UNSUPPORT, default 1080P_25, TBD", ch);
#else
		dev_err(&client->dev,
			"set channel %d UNSUPPORT, default 720P_25, TBD", ch);
#endif
		break;
	}

	return 0;
}

int tp9930_get_channel_reso(struct i2c_client *client, int ch)
{
	u8 detect_fmt = 0xff;
	u8 reso = 0xff;

	techpoint_write_reg(client, 0x40, ch);
	techpoint_read_reg(client, 0x03, &detect_fmt);
	reso = detect_fmt & 0x7;

	switch (reso) {
	case TP9930_CVSTD_1080P_30:
		dev_err(&client->dev, "detect channel %d 1080P_30", ch);
		return TECHPOINT_S_RESO_1080P_30;
		break;
	case TP9930_CVSTD_1080P_25:
		dev_err(&client->dev, "detect channel %d 1080P_25", ch);
		return TECHPOINT_S_RESO_1080P_25;
		break;
	case TP9930_CVSTD_720P_30:
		dev_err(&client->dev, "detect channel %d 720P_30", ch);
		return TECHPOINT_S_RESO_720P_30;
		break;
	case TP9930_CVSTD_720P_25:
		dev_err(&client->dev, "detect channel %d 720P_25", ch);
		return TECHPOINT_S_RESO_720P_25;
		break;
	case TP9930_CVSTD_720P_60:
	case TP9930_CVSTD_720P_50:
	default:
#if DEF_1080P
		dev_err(&client->dev,
			"detect channel %d UNSUPPORT, default 1080P_25", ch);
		return TECHPOINT_S_RESO_1080P_25;
#else
		dev_err(&client->dev,
			"detect channel %d UNSUPPORT, default 720P_25", ch);
		return TECHPOINT_S_RESO_720P_25;
#endif
		break;
	}

	return reso;
}
