/*
 * An I2C driver for the PCF85063 RTC
 * Copyright 2014 Rose Technology
 *
 * Author: Søren Andersen <san@rosetechnology.dk>
 * Maintainers: http://www.nslu2-linux.org/
 *
 * based on the other drivers in this same directory.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/i2c.h>
#include <linux/bcd.h>
#include <linux/rtc.h>
#include <linux/module.h>
#include <linux/gpio.h>
#include <linux/of_platform.h>
#include <linux/of_gpio.h>

/*
 * Information for this driver was pulled from the following datasheets.
 *
 *  http://www.nxp.com/documents/data_sheet/PCF85063A.pdf
 *  http://www.nxp.com/documents/data_sheet/PCF85063TP.pdf
 *
 *  PCF85063A -- Rev. 6 — 18 November 2015
 *  PCF85063TP -- Rev. 4 — 6 May 2015
*/

#define PCF85063_REG_CTRL1		0x00 /* status */
#define PCF85063_REG_CTRL1_STOP		BIT(5)
#define PCF85063_REG_CTRL2		0x01

#define PCF85063_REG_SC			0x04 /* datetime */
#define PCF85063_REG_SC_OS		0x80
#define PCF85063_REG_MN			0x05
#define PCF85063_REG_HR			0x06
#define PCF85063_REG_DM			0x07
#define PCF85063_REG_DW			0x08
#define PCF85063_REG_MO			0x09
#define PCF85063_REG_YR			0x0A
#define PCF85063_REG_ALARM_SC	0x0B

static struct i2c_driver pcf85063_driver;
static struct i2c_client *g_client;
int g_min=-1,g_hour=-1,g_mday=-1,g_wday=-1,alarm_int=0;

struct pcf85063 {
	struct device *dev;
	struct i2c_client *i2c;
};

static void gpio_set_input(struct device_node *np,int gpio,char *name)
{
	int ret=0;
	enum  of_gpio_flags flag;
	
	gpio = of_get_named_gpio_flags(np, name, 0,&flag);
	ret = gpio_request(gpio, name);
	if (ret != 0) {
		printk("%s request %s error\n", __func__,name);
		return ;
	}
	else	
		printk("%s set input \n",name);
	gpio_direction_input(gpio); 	
	gpio_export(gpio, true);		
}


static int pcf85063_stop_clock(struct i2c_client *client, u8 *ctrl1)
{
	int rc;
	u8 reg;

	rc = i2c_smbus_read_byte_data(client, PCF85063_REG_CTRL1);
	if (rc < 0) {
		dev_err(&client->dev, "Failing to stop the clock\n");
		return -EIO;
	}

	/* stop the clock */
	reg = rc | PCF85063_REG_CTRL1_STOP;

	rc = i2c_smbus_write_byte_data(client, PCF85063_REG_CTRL1, reg);
	if (rc < 0) {
		dev_err(&client->dev, "Failing to stop the clock\n");
		return -EIO;
	}

	*ctrl1 = reg;

	return 0;
}

static int pcf85063_start_clock(struct i2c_client *client, u8 ctrl1)
{
	int rc;

	/* start the clock */
	ctrl1 &= ~PCF85063_REG_CTRL1_STOP;

	rc = i2c_smbus_write_byte_data(client, PCF85063_REG_CTRL1, ctrl1);
	if (rc < 0) {
		dev_err(&client->dev, "Failing to start the clock\n");
		return -EIO;
	}

	return 0;
}

static int pcf85063_reset_clock(struct i2c_client *client)
{
	int rc;

	printk("%s\n",__FUNCTION__);
	rc = i2c_smbus_write_byte_data(client, PCF85063_REG_CTRL1, 0x58);
	if (rc < 0) {
		dev_err(&client->dev, "Failing to reset the clock\n");
		return -EIO;
	}

	return 0;
}

static int pcf85063_rtc_read_time(struct device *dev, struct rtc_time *tm)
{
	struct i2c_client *client = to_i2c_client(dev);
	int rc;
	u8 regs[7];

	/*
	 * while reading, the time/date registers are blocked and not updated
	 * anymore until the access is finished. To not lose a second
	 * event, the access must be finished within one second. So, read all
	 * time/date registers in one turn.
	 */
	rc = i2c_smbus_read_i2c_block_data(client, PCF85063_REG_SC,
					   sizeof(regs), regs);
	if (rc != sizeof(regs)) {
		dev_err(&client->dev, "date/time register read error\n");
		return -EIO;
	}

	/* if the clock has lost its power it makes no sense to use its time */
	//if (regs[0] & PCF85063_REG_SC_OS) {
	//	dev_warn(&client->dev, "Power loss detected, invalid time\n");
	//	return -EINVAL;
	//}

	tm->tm_sec = bcd2bin(regs[0] & 0x7F);
	tm->tm_min = bcd2bin(regs[1] & 0x7F);
	tm->tm_hour = bcd2bin(regs[2] & 0x3F); /* rtc hr 0-23 */
	tm->tm_mday = bcd2bin(regs[3] & 0x3F);
	tm->tm_wday = regs[4] & 0x07;
	tm->tm_mon = bcd2bin(regs[5] & 0x1F) - 1; /* rtc mn 1-12 */
	tm->tm_year = bcd2bin(regs[6]);
	tm->tm_year += 100;
	
	if((tm->tm_mon+1)>12 || tm->tm_mday>31 || tm->tm_hour > 24 || tm->tm_min > 60 || tm->tm_sec > 60)
	{
		printk("%s: Read RTC time error , reset RTC \n",__FUNCTION__);
		pcf85063_reset_clock(client);
		
		rc = i2c_smbus_read_i2c_block_data(client, PCF85063_REG_SC,
						   sizeof(regs), regs);
		if (rc != sizeof(regs)) {
			dev_err(&client->dev, "date/time register read error\n");
			return -EIO;
		}
		tm->tm_sec = bcd2bin(regs[0] & 0x7F);
		tm->tm_min = bcd2bin(regs[1] & 0x7F);
		tm->tm_hour = bcd2bin(regs[2] & 0x3F); /* rtc hr 0-23 */
		tm->tm_mday = bcd2bin(regs[3] & 0x3F);
		tm->tm_wday = regs[4] & 0x07;
		tm->tm_mon = bcd2bin(regs[5] & 0x1F) - 1; /* rtc mn 1-12 */
		tm->tm_year = bcd2bin(regs[6]);
		tm->tm_year += 100;
		printk("%s: Read RTC time again\n",__FUNCTION__);
		
	}	
	printk("%s: year:%d,mon:%d,mday:%d,hour:%d,min:%d,sec:%d\n",__FUNCTION__,tm->tm_year+1900,tm->tm_mon+1,tm->tm_mday,tm->tm_hour,tm->tm_min,tm->tm_sec);

	return 0;
}

static int pcf85063_rtc_set_time(struct device *dev, struct rtc_time *tm)
{
	struct i2c_client *client = to_i2c_client(dev);
	int rc;
	u8 regs[7];
	u8 ctrl1;

	if ((tm->tm_year < 100) || (tm->tm_year > 199))
		return -EINVAL;

	/*
	 * to accurately set the time, reset the divider chain and keep it in
	 * reset state until all time/date registers are written
	 */
	rc = pcf85063_stop_clock(client, &ctrl1);
	if (rc != 0)
		return rc;

	/* hours, minutes and seconds */
	regs[0] = bin2bcd(tm->tm_sec) & 0x7F; /* clear OS flag */

	regs[1] = bin2bcd(tm->tm_min);
	regs[2] = bin2bcd(tm->tm_hour);

	/* Day of month, 1 - 31 */
	regs[3] = bin2bcd(tm->tm_mday);

	/* Day, 0 - 6 */
	regs[4] = tm->tm_wday & 0x07;

	/* month, 1 - 12 */
	regs[5] = bin2bcd(tm->tm_mon + 1);

	/* year and century */
	regs[6] = bin2bcd(tm->tm_year - 100);
	
	printk("%s: year:%d,mon:%d,mday:%d,hour:%d,min:%d,sec:%d\n",__FUNCTION__,tm->tm_year+1900,tm->tm_mon+1,tm->tm_mday,tm->tm_hour,tm->tm_min,tm->tm_sec);
	/* write all registers at once */
	rc = i2c_smbus_write_i2c_block_data(client, PCF85063_REG_SC,
					    sizeof(regs), regs);
	if (rc < 0) {
		dev_err(&client->dev, "date/time register write error\n");
		return rc;
	}

	/*
	 * Write the control register as a separate action since the size of
	 * the register space is different between the PCF85063TP and
	 * PCF85063A devices.  The rollover point can not be used.
	 */
	rc = pcf85063_start_clock(client, ctrl1);
	if (rc != 0)
		return rc;

	return 0;
}

static int pcf85063_rtc_set_alarm_enable(int enable)
{
	u8 control2_reg=0x80;
	int rc,i;
	u8 regs[5];
	int sec=0;
	
	if(enable==1)
		control2_reg=0x80;
	else
		control2_reg=0x00;
	
	rc = i2c_smbus_write_byte_data(g_client, PCF85063_REG_CTRL2, control2_reg);
	if (rc < 0) {
		dev_err(&g_client->dev, "Failing to set AIE enabled\n");
		return -EIO;
	}
	
	if(enable==1)
	{
		if(g_min<0 || g_hour<0 || g_mday<0 || g_wday<0)
			return -2;
		
		/* hours, minutes and seconds */
		regs[0] = bin2bcd(sec); 

		regs[1] = bin2bcd(g_min);
		regs[2] = bin2bcd(g_hour);

		/* Day of month, 1 - 31 */
		regs[3] = bin2bcd(g_mday);

		/* Day, 0 - 6 */
		regs[4] = g_wday;

		/* write all registers at once */
		rc = i2c_smbus_write_i2c_block_data(g_client, PCF85063_REG_ALARM_SC,
							sizeof(regs), regs);
		if (rc < 0) {
			dev_err(&g_client->dev, "date/time alarm register write error\n");
			return rc;
		}
	}
	//printk("%s :success \n",__func__);
	
	//for(i=0;i<5;i++)
	//{
	//	printk("%s :regs[%d]=0x%x\n",__func__,i,regs[i]);
	//}

	return 0;
}


static int pcf85063_rtc_get(void)
{
	int rc,i;
	u8 regs[16];


	rc = i2c_smbus_read_i2c_block_data(g_client, PCF85063_REG_CTRL1,
					   sizeof(regs), regs);
	if (rc != sizeof(regs)) {
		dev_err(&g_client->dev, "all register read error\n");
		return -EIO;
	}

	for(i=0;i<16;i++)
	{
		printk("%s :regs[%d]=0x%x\n",__func__,i,regs[i]);
	}

	return 0;
}
static ssize_t get_all_value(struct device *dev, struct device_attribute *attr, const char *buf,size_t count)
{
	pcf85063_rtc_get();
	return count;
}
static DEVICE_ATTR(salarmread,S_IRUSR,get_all_value,NULL);

static ssize_t set_alarm_enable(struct device *dev, struct device_attribute *attr, const char *buf,size_t count)
{
	int ret;
	int val = simple_strtoul(buf, NULL, 10);	
	ret=pcf85063_rtc_set_alarm_enable(val);
	if(ret !=0)
		printk("pcf85063_rtc_set_alarm_enable failed = %d\n",ret);
	return count;
}
static DEVICE_ATTR(salarm,S_IWUSR,NULL,set_alarm_enable);

static ssize_t set_alarm_min(struct device *dev, struct device_attribute *attr, const char *buf,size_t count)
{
	g_min=simple_strtoul(buf, NULL, 10);	
	return count;
}
static DEVICE_ATTR(salarmmin,S_IWUSR,NULL,set_alarm_min);

static ssize_t set_alarm_hour(struct device *dev, struct device_attribute *attr, const char *buf,size_t count)
{
	g_hour=simple_strtoul(buf, NULL, 10);	
	return count;
}
static DEVICE_ATTR(salarmhour,S_IWUSR,NULL,set_alarm_hour);

static ssize_t set_alarm_mday(struct device *dev, struct device_attribute *attr, const char *buf,size_t count)
{
	g_mday=simple_strtoul(buf, NULL, 10);	
	return count;
}
static DEVICE_ATTR(salarmmday,S_IWUSR,NULL,set_alarm_mday);

static ssize_t set_alarm_wday(struct device *dev, struct device_attribute *attr, const char *buf,size_t count)
{
	g_wday=simple_strtoul(buf, NULL, 10);	
	return count;
}
static DEVICE_ATTR(salarmwday,S_IWUSR,NULL,set_alarm_wday);

static const struct rtc_class_ops pcf85063_rtc_ops = {
	.read_time	= pcf85063_rtc_read_time,
	.set_time	= pcf85063_rtc_set_time
};

static int pcf85063_probe(struct i2c_client *client,
				const struct i2c_device_id *id)
{
	struct rtc_device *rtc;
	struct pcf85063 *pcf85063;
	int err;
	dev_dbg(&client->dev, "%s\n", __func__);
	pcf85063 = devm_kzalloc(&client->dev,sizeof(*pcf85063), GFP_KERNEL);
	if (!pcf85063) {
		return -ENOMEM;
	}	
	pcf85063->i2c = client;
	pcf85063->dev = &client->dev;
	g_client=client;
	struct device_node *pcf85063_node = pcf85063->dev->of_node;
	
	device_create_file(&client->dev, &dev_attr_salarmread);
	device_create_file(&client->dev, &dev_attr_salarm);
	device_create_file(&client->dev, &dev_attr_salarmmin);
	device_create_file(&client->dev, &dev_attr_salarmhour);
	device_create_file(&client->dev, &dev_attr_salarmmday);
	device_create_file(&client->dev, &dev_attr_salarmwday);

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C))
		return -ENODEV;

	err = i2c_smbus_read_byte_data(client, PCF85063_REG_CTRL1);
	if (err < 0) {
		dev_err(&client->dev, "RTC chip is not present\n");
		return err;
	}
	//gpio_set_input(pcf85063_node,alarm_int,"alarm");
	
	rtc = devm_rtc_device_register(&client->dev,
				       pcf85063_driver.driver.name,
				       &pcf85063_rtc_ops, THIS_MODULE);

	return PTR_ERR_OR_ZERO(rtc);
}

static const struct i2c_device_id pcf85063_id[] = {
	{ "pcf85063", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, pcf85063_id);

#ifdef CONFIG_OF
static const struct of_device_id pcf85063_of_match[] = {
	{ .compatible = "nxp,pcf85063" },
	{}
};
MODULE_DEVICE_TABLE(of, pcf85063_of_match);
#endif

static struct i2c_driver pcf85063_driver = {
	.driver		= {
		.name	= "rtc-pcf85063",
		.of_match_table = of_match_ptr(pcf85063_of_match),
	},
	.probe		= pcf85063_probe,
	.id_table	= pcf85063_id,
};

module_i2c_driver(pcf85063_driver);

MODULE_AUTHOR("Søren Andersen <san@rosetechnology.dk>");
MODULE_DESCRIPTION("PCF85063 RTC driver");
MODULE_LICENSE("GPL");
