// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2018 Fuzhou Rockchip Electronics Co., Ltd
 */

#include <linux/module.h>
#include <linux/rtc.h>
#include <linux/time.h>
#include <linux/timer.h>
#include <linux/of_platform.h>

static struct timespec begtime;
static unsigned long time;
static unsigned long alarm_time;
static int alarm_irq_enable;
static struct timer_list timer_list;
static struct rtc_device *fake_rtc;

static unsigned long timespec_to_ulong(struct timespec *ts)
{
	return ts->tv_nsec < NSEC_PER_SEC / 2 ? ts->tv_sec : ts->tv_sec + 1;
}

static void get_uptime(struct timespec *ts)
{
	getrawmonotonic(ts);
}

static int fake_rtc_read_time(struct device *dev, struct rtc_time *tm)
{
	struct timespec now, diff;

	get_uptime(&now);
	diff = timespec_sub(now, begtime);

	rtc_time_to_tm(time + timespec_to_ulong(&diff), tm);

	return rtc_valid_tm(tm);
}

static int fake_rtc_set_time(struct device *dev, struct rtc_time *tm)
{
	get_uptime(&begtime);
	rtc_tm_to_time(tm, &time);

	return 0;
}

static int fake_rtc_alarm_irq_enable(struct device *dev,
					unsigned int enabled)
{
	if (enabled)
		alarm_irq_enable = 1;
	else
		alarm_irq_enable = 0;

	return 0;
}

static int fake_rtc_read_alarm(struct device *dev, struct rtc_wkalrm *alm)
{
	if (!alarm_time) {
		alm->enabled = true;

		rtc_time_to_tm(alarm_time, &alm->time);
	}

	return 0;
}

static void fake_rtc_timer(unsigned long data)
{
	struct rtc_device *fake_rtc = (struct rtc_device *)data;

	if (alarm_irq_enable)
		rtc_update_irq(fake_rtc, 1, RTC_IRQF | RTC_AF);
}

static void fake_rtc_set_wakeup_time(unsigned long time)
{
	alarm_irq_enable = 1;
	timer_list.expires = jiffies + HZ * time;
	add_timer(&timer_list);
}

static int fake_rtc_set_alarm(struct device *dev, struct rtc_wkalrm *alm)
{
	struct timeval time;
	unsigned long local_time;
	unsigned long alarm_secs;
	int ret;

	alarm_irq_enable = 0;
	del_timer_sync(&timer_list);

	if (alm->enabled) {
		ret = rtc_tm_to_time(&alm->time, &alarm_secs);
		if (ret)
			return ret;

		do_gettimeofday(&time);
		local_time = time.tv_sec;

		alarm_time = alarm_secs;

		if (alarm_secs >= local_time) {
			alarm_secs = alarm_secs - local_time;
			fake_rtc_set_wakeup_time(alarm_secs);
			pr_debug("system will wakeup %lus later\n", alarm_secs);
		}
	} else {
		alarm_time = 0;
	}

	return 0;
}

static const struct rtc_class_ops fake_rtc_ops = {
	.read_time	= fake_rtc_read_time,
	.set_time	= fake_rtc_set_time,
	.alarm_irq_enable	= fake_rtc_alarm_irq_enable,
	.read_alarm		= fake_rtc_read_alarm,
	.set_alarm		= fake_rtc_set_alarm,
};

#ifdef CONFIG_PM_SLEEP
static int fake_rtc_suspend(struct device *dev)
{
	return 0;
}

static int fake_rtc_resume(struct device *dev)
{
	return 0;
}
#endif

static SIMPLE_DEV_PM_OPS(fake_rtc_pm_ops,
	fake_rtc_suspend, fake_rtc_resume);

static int fake_rtc_probe(struct platform_device *pdev)
{
	struct rtc_time tm = {
		.tm_wday = 1,
		.tm_year = 118,
		.tm_mon = 0,
		.tm_mday = 1,
		.tm_hour = 12,
		.tm_min = 0,
		.tm_sec = 0,
	};

	get_uptime(&begtime);
	fake_rtc_set_time(&pdev->dev, &tm);

	device_init_wakeup(&pdev->dev, 1);

	fake_rtc = devm_rtc_device_register(&pdev->dev,
				pdev->name, &fake_rtc_ops, THIS_MODULE);
	if (IS_ERR(fake_rtc))
		return PTR_ERR(fake_rtc);

	setup_timer(&timer_list, fake_rtc_timer, (unsigned long)fake_rtc);
	dev_info(&pdev->dev, "loaded; begtime is %lu, time is %lu\n",
		 timespec_to_ulong(&begtime), time);

	return 0;
}

static const struct of_device_id rtc_dt_ids[] = {
	{ .compatible = "rtc-fake" },
	{},
};

struct platform_driver fake_rtc_driver = {
	.driver		= {
		.name	= "rtc-fake",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(rtc_dt_ids),
		.pm	= &fake_rtc_pm_ops,
	},
	.probe		= fake_rtc_probe,
};

module_platform_driver(fake_rtc_driver);

MODULE_AUTHOR("jesse.huang@rock-chips.com");
MODULE_DESCRIPTION("FAKE RTC driver");
MODULE_LICENSE("GPL");

