/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Power supply driver for batteryless.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/power_supply.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/vermagic.h>

enum batteryless_id {
	BATTERYLESS_AC,
	BATTERYLESS_BATTERY,
	BATTERYLESS_USB,
	BATTERYLESS_POWER_NUM,
};

static int ac_online			= 1;
static int usb_online			= 0;
static int battery_status		= POWER_SUPPLY_STATUS_CHARGING;
static int battery_health		= POWER_SUPPLY_HEALTH_UNKNOWN;
static int battery_present		= 0;
static int battery_technology		= POWER_SUPPLY_TECHNOLOGY_LION;
static int battery_capacity		= 100;
static int battery_voltage		= 3300;
static int battery_charge_count		= 10000;

static bool module_initialized;

static int batteryless_get_ac_property(struct power_supply *psy,
				      enum power_supply_property psp,
				      union power_supply_propval *val)
{
	switch (psp) {
	case POWER_SUPPLY_PROP_ONLINE:
		val->intval = ac_online;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int batteryless_get_usb_property(struct power_supply *psy,
				      enum power_supply_property psp,
				      union power_supply_propval *val)
{
	switch (psp) {
	case POWER_SUPPLY_PROP_ONLINE:
		val->intval = usb_online;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int batteryless_get_battery_property(struct power_supply *psy,
					   enum power_supply_property psp,
					   union power_supply_propval *val)
{
	switch (psp) {
	case POWER_SUPPLY_PROP_MODEL_NAME:
		val->strval = "battery";
		break;
	case POWER_SUPPLY_PROP_MANUFACTURER:
		val->strval = "Linux";
		break;
	case POWER_SUPPLY_PROP_SERIAL_NUMBER:
		val->strval = UTS_RELEASE;
		break;
	case POWER_SUPPLY_PROP_STATUS:
		val->intval = battery_status;
		break;
	case POWER_SUPPLY_PROP_CHARGE_TYPE:
		val->intval = POWER_SUPPLY_CHARGE_TYPE_FAST;
		break;
	case POWER_SUPPLY_PROP_HEALTH:
		val->intval = battery_health;
		break;
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = battery_present;
		break;
	case POWER_SUPPLY_PROP_TECHNOLOGY:
		val->intval = battery_technology;
		break;
	case POWER_SUPPLY_PROP_CAPACITY_LEVEL:
		val->intval = POWER_SUPPLY_CAPACITY_LEVEL_NORMAL;
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
	case POWER_SUPPLY_PROP_CHARGE_NOW:
		val->intval = battery_capacity;
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN:
	case POWER_SUPPLY_PROP_CHARGE_FULL:
		val->intval = 100;
		break;
	case POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG:
	case POWER_SUPPLY_PROP_TIME_TO_FULL_NOW:
		val->intval = 3600;
		break;
	case POWER_SUPPLY_PROP_TEMP:
		val->intval = 26;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		val->intval = battery_voltage;
		break;
	case POWER_SUPPLY_PROP_CHARGE_COUNTER:
		val->intval = battery_charge_count;
		break;
	default:
		pr_info("%s: some properties deliberately report errors.\n",
			__func__);
		return -EINVAL;
	}
	return 0;
}

static enum power_supply_property batteryless_ac_props[] = {
	POWER_SUPPLY_PROP_ONLINE,
};

static enum power_supply_property batteryless_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_CHARGE_TYPE,
	POWER_SUPPLY_PROP_HEALTH,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN,
	POWER_SUPPLY_PROP_CHARGE_FULL,
	POWER_SUPPLY_PROP_CHARGE_NOW,
	POWER_SUPPLY_PROP_CAPACITY,
	POWER_SUPPLY_PROP_CAPACITY_LEVEL,
	POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG,
	POWER_SUPPLY_PROP_TIME_TO_FULL_NOW,
	POWER_SUPPLY_PROP_MODEL_NAME,
	POWER_SUPPLY_PROP_MANUFACTURER,
	POWER_SUPPLY_PROP_SERIAL_NUMBER,
	POWER_SUPPLY_PROP_TEMP,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CHARGE_COUNTER,
};

static char *batteryless_ac_supplied_to[] = {
	"battery",
};

static struct power_supply *batteryless_supplies[BATTERYLESS_POWER_NUM];

static const struct power_supply_desc batteryless_desc[] = {
	[BATTERYLESS_AC] = {
		.name = "ac",
		.type = POWER_SUPPLY_TYPE_MAINS,
		.properties = batteryless_ac_props,
		.num_properties = ARRAY_SIZE(batteryless_ac_props),
		.get_property = batteryless_get_ac_property,
	},
	[BATTERYLESS_BATTERY] = {
		.name = "battery",
		.type = POWER_SUPPLY_TYPE_BATTERY,
		.properties = batteryless_battery_props,
		.num_properties = ARRAY_SIZE(batteryless_battery_props),
		.get_property = batteryless_get_battery_property,
	},
	[BATTERYLESS_USB] = {
		.name = "usb",
		.type = POWER_SUPPLY_TYPE_USB,
		.properties = batteryless_ac_props,
		.num_properties = ARRAY_SIZE(batteryless_ac_props),
		.get_property = batteryless_get_usb_property,
	},
};

static const struct power_supply_config batteryless_configs[] = {
	{
		/* ac */
		.supplied_to = batteryless_ac_supplied_to,
		.num_supplicants = ARRAY_SIZE(batteryless_ac_supplied_to),
	}, {
		/* battery */
	}, {
		/* usb */
		.supplied_to = batteryless_ac_supplied_to,
		.num_supplicants = ARRAY_SIZE(batteryless_ac_supplied_to),
	},
};

static int __init batteryless_init(void)
{
	int i;
	int ret;

	BUILD_BUG_ON(BATTERYLESS_POWER_NUM != ARRAY_SIZE(batteryless_supplies));
	BUILD_BUG_ON(BATTERYLESS_POWER_NUM != ARRAY_SIZE(batteryless_configs));

	for (i = 0; i < ARRAY_SIZE(batteryless_supplies); i++) {
		batteryless_supplies[i] = power_supply_register(NULL,
						&batteryless_desc[i],
						&batteryless_configs[i]);
		if (IS_ERR(batteryless_supplies[i])) {
			pr_err("%s: failed to register %s\n", __func__,
				batteryless_desc[i].name);
			ret = PTR_ERR(batteryless_supplies[i]);
			goto failed;
		}
	}

	module_initialized = true;
	return 0;
failed:
	while (--i >= 0)
		power_supply_unregister(batteryless_supplies[i]);
	return ret;
}
module_init(batteryless_init);

static void __exit batteryless_exit(void)
{
	int i;

	/* Let's see how we handle changes... */
	ac_online = 0;
	usb_online = 0;
	battery_status = POWER_SUPPLY_STATUS_DISCHARGING;
	for (i = 0; i < ARRAY_SIZE(batteryless_supplies); i++)
		power_supply_changed(batteryless_supplies[i]);
	pr_info("%s: 'changed' event sent, sleeping for 10 seconds...\n",
		__func__);
	ssleep(10);

	for (i = 0; i < ARRAY_SIZE(batteryless_supplies); i++)
		power_supply_unregister(batteryless_supplies[i]);

	module_initialized = false;
}
module_exit(batteryless_exit);

MODULE_DESCRIPTION("Power supply driver for batteryless");
MODULE_LICENSE("GPL");
