/* test Linux i2c driver by accessing eeprom with 16bit address (e.g. 24LC64)
 * test modes:
 * - write stdin to eeprom
 * - dump eeprom to stdout
 * - write pattern to eeprom and verify
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>

#define PAGE_LEN 32  // 24LC64

static void usage(const char *prog)
{
	fprintf(stderr,
		"usage: %s dev addr offset length {read|write|rw}\n"
		"	read : dump eeprom to stdout\n"
		"	write: write stdin to eeprom\n"
		"	rw   : write pattern and verify\n"
		"  dev   : device, e.g. /dev/i2c-0\n"
		"  addr  : eeprom address (7 bit, e.g. 0x28)\n"
		"  offset: offset from start of eeprom\n"
		"  length: number of bytes\n", prog);
	exit(1);
}

static void dump(int fd, __u16 addr, __u16 offset, __u16 length)
{
	__u8 abuf[2] = { offset >> 8, offset & 0xff };
	__u8 buf[length];
	struct i2c_msg msgs[2] = {
		{ .addr = addr, .flags = 0, .len = 2, .buf = abuf },
		{ .addr = addr, .flags = I2C_M_RD, .len = length, .buf = buf },
	};
	struct i2c_rdwr_ioctl_data iod = {
		.msgs = msgs,
		.nmsgs = 2
	};
	int rc;

	rc = ioctl(fd, I2C_RDWR, &iod);
	if (rc != 2) {
		fprintf(stderr, "error: I2C_RDWR: %d, %m %d\n", rc, errno);
		return;
	}
	rc = write(1, buf, length);
	if (rc != length)
		fprintf(stderr, "error: write stdout: %m %d\n", errno);
}

static void undump(int fd, __u16 addr, __u16 offset, __u16 length)
{
	__u8 buf[PAGE_LEN + 2];
	struct i2c_msg msgs[1] = {
		{ .addr = addr, .flags = 0, .len = PAGE_LEN + 2, .buf = buf },
	};
	struct i2c_rdwr_ioctl_data iod = {
		.msgs = msgs,
		.nmsgs = 1
	};
	int rc, len = PAGE_LEN;

again:
	buf[0] = offset >> 8;
	buf[1] = offset & 0xff;
	if (len > length)
		len = length;
	length -= len;
	offset += len;
	rc = read(0, buf + 2, len);
	if (rc <= 0) {
		fprintf(stderr, "error: read stdin %m %d\n", errno);
		return;
	} else if (rc < len) {
		fprintf(stderr, "note: only got %d bytes\n", rc);
		msgs[0].len = rc + 2;
	}
	rc = ioctl(fd, I2C_RDWR, &iod);
	if (rc != 1) {
		fprintf(stderr, "error: I2C_RDWR: %d, %m %d\n", rc, errno);
		return;
	}
	usleep(10000);
	if (length)
		goto again;
}

static void run_test(int fd, __u16 addr, __u16 offset, __u16 length)
{
	__u8 abuf[2] = { offset >> 8, offset & 0xff };
	__u8 buf[length + 2];
	__u8 cmp;
	struct i2c_msg msgs[2] = {
		{ .addr = addr, .flags = 0, .len = PAGE_LEN + 2, .buf = buf },
		{ .addr = addr, .flags = I2C_M_RD, .len = length, .buf = buf },
	};
	struct i2c_rdwr_ioctl_data iod = {
		.msgs = msgs,
		.nmsgs = 1
	};
	int rc, i, j = 0, len = PAGE_LEN, wlen = length;

again:
	buf[0] = offset >> 8;
	buf[1] = offset & 0xff;
	if (len > wlen)
		len = wlen;
	wlen -= len;
	offset += len;
	for (i = 0; i < len; i++, j++)
		buf[i + 2] = "PIXELWORKS"[j % 10] | (j % 3 ? 0x20 : 0);
	rc = ioctl(fd, I2C_RDWR, &iod);
	if (rc != 1) {
		fprintf(stderr, "error: I2C_RDWR w: %d, %m %d\n", rc, errno);
		return;
	}
	usleep(10000);
	if (wlen)
		goto again;

	msgs[0].len = 2;
	msgs[0].buf = abuf;
	iod.nmsgs = 2;
	rc = ioctl(fd, I2C_RDWR, &iod);
	if (rc != 2) {
		fprintf(stderr, "error: I2C_RDWR r: %d, %m %d\n", rc, errno);
		return;
	}
	for (i = 0; i < length; i++) {
		cmp = "PIXELWORKS"[i % 10] | (i % 3 ? 0x20 : 0);
		if (buf[i] != cmp)
			fprintf(stderr, "verify fail at 0x%04x: %02x != %02x\n",
				i, buf[i], cmp);
	}
}

int main(int argc, char *argv[])
{
	unsigned long addr, offset, len;
	int fd;

	if (argc != 6)
		usage(argv[0]);
	addr = strtoul(argv[2], NULL, 0);
	offset = strtoul(argv[3], NULL, 0);
	len = strtoul(argv[4], NULL, 0);
	if ((addr <= 0) || (addr >= 0x7f) || (len <= 0)
	    || (offset < 0) || (offset > 0xffff))
		usage(argv[0]);
	addr <<= 1;
	fd = open(argv[1], O_RDWR);
	if (fd == -1) {
		fprintf(stderr, "error: cannot open '%s': %m %d\n", argv[1], errno);
		return 1;
	}
	if (!strcmp(argv[5], "read"))
		dump(fd, addr, offset, len);
	else if (!strcmp(argv[5], "write"))
		undump(fd, addr, offset, len);
	else if (!strcmp(argv[5], "rw"))
		run_test(fd, addr, offset, len);
	else
		usage(argv[0]);
	close(fd);
	return 0;
}
