/*
 * Pixelworks TopazEH USB2 PHY driver
 * based on phy-pxlw-usb.c
 *
 * Copyright (C) 2016 Pixelworks, Inc.
 * Copyright (C) 2014 Marvell Technology Group Ltd.
 *
 * Antoine Tenart <antoine.tenart@free-electrons.com>
 * Jisheng Zhang <jszhang@marvell.com>
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2. This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/phy/phy.h>


struct phy_pxlw_usb2_priv {
	struct clk *clk;
};


static int phy_pxlw_usb2_power_on(struct phy *phy)
{
	struct phy_pxlw_usb2_priv *priv = phy_get_drvdata(phy);

	pr_info("%s:%d\n", __func__, __LINE__);
	clk_prepare_enable(priv->clk);
	return 0;
}

static int phy_pxlw_usb2_power_off(struct phy *phy)
{
	struct phy_pxlw_usb2_priv *priv = phy_get_drvdata(phy);

	pr_info("%s:%d\n", __func__, __LINE__);
	clk_disable_unprepare(priv->clk);
	return 0;
}

static const struct phy_ops phy_pxlw_usb2_ops = {
	.power_on	= phy_pxlw_usb2_power_on,
	.power_off	= phy_pxlw_usb2_power_off,
	.owner		= THIS_MODULE,
};

static const struct of_device_id phy_pxlw_usb2_of_match[] = {
	{ .compatible = "pxlw,topazeh-usb2-phy", },
	{ },
};
MODULE_DEVICE_TABLE(of, phy_pxlw_usb2_of_match);

static int phy_pxlw_usb2_probe(struct platform_device *pdev)
{
	struct phy_pxlw_usb2_priv *priv;
	struct phy *phy;
	struct phy_provider *phy_provider;

	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	phy = devm_phy_create(&pdev->dev, NULL, &phy_pxlw_usb2_ops);
	if (IS_ERR(phy)) {
		dev_err(&pdev->dev, "failed to create PHY\n");
		return PTR_ERR(phy);
	}

	priv->clk = devm_clk_get(&pdev->dev, NULL);
	if (IS_ERR(priv->clk)) {
		dev_err(&pdev->dev, "failed to get clock\n");
		return PTR_ERR(priv->clk);
	}

	phy_set_drvdata(phy, priv);

	phy_provider = devm_of_phy_provider_register(&pdev->dev,
						     of_phy_simple_xlate);
	return PTR_ERR_OR_ZERO(phy_provider);
}

static struct platform_driver phy_pxlw_usb2_driver = {
	.probe	= phy_pxlw_usb2_probe,
	.driver	= {
		.name		= "phy-pxlw-usb2",
		.of_match_table	= phy_pxlw_usb2_of_match,
	},
};
module_platform_driver(phy_pxlw_usb2_driver);

MODULE_DESCRIPTION("Pixelworks TopazEH USB 2.0 PHY driver");
MODULE_LICENSE("GPL");
