/*
 * Evatronix NAND Controller
 *
 * Copyright 2012 Pixelworks, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/bitops.h>
#include <linux/clk.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/nand_ecc.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/dma-direction.h>

static int nand_dbg;
module_param(nand_dbg,  int, 0644);
MODULE_PARM_DESC(nand_dbg, "nand debug level ");

static int cpu_fifo;
module_param(cpu_fifo,  int, 0644);
MODULE_PARM_DESC(cpu_fifo, "CPU or DMA access FIFO");

static int polling;
module_param(polling,  int, 0644);
MODULE_PARM_DESC(polling, "polling or interrupt");

#define		HWECC	1
#define		NoECC	0
#define		SWECC	-1

static int hwecc = HWECC;

module_param(hwecc, int, 0644);
MODULE_PARM_DESC(hwecc, "HW ECC option");

static int hwecc_verify;
module_param(hwecc_verify, int, 0644);
MODULE_PARM_DESC(hwecc_verify, "HW ECC verify");

/*only for SG DMA verification,only write, 2(1 page), 4(0-3 page)*/

static int sg_verify;
module_param(sg_verify, int, 0644);
MODULE_PARM_DESC(sg_verify, "Scatter gather DMA verify");

#define NAND_PRINTK(level, fmt, ...) do { \
	if (nand_dbg >= level) \
		printk(KERN_DEBUG "CAST NAND %s:%d: "fmt, __func__, \
		__LINE__, ## __VA_ARGS__); \
	} while (0)

/* NAND device chip timing, refer to HY27UF082G2b unit ns*/
#define TADL        70
#define TCCS        TADL
#define TRHW        100
#define TWHR        60
#define TRR         20
#define TWB         100
#define TRWH        10
#define TRWP        12

/* NAND device chip timing,refer to HY27UF082G2b units us*/
#define TR          25
#define TBRES       2000
#define TPROG       700
#define TRST        500

#define REG_COMMAND		0x00
#define REG_CONTROL		0x04
#define REG_STATUS		0x08
#define REG_STATUS_MASK		0x0C
#define REG_INT_MASK		0x10
#define REG_INT_STATUS		0x14
#define REG_ECC_CTRL		0x18
#define REG_ECC_OFFSET		0x1C
#define REG_ECC_STAT		0x20
#define REG_ADDR0_COL		0x24
#define REG_ADDR0_ROW		0x28
#define REG_ADDR1_COL		0x2C
#define REG_ADDR1_ROW		0x30
#define REG_PROTECT		0x34
#define REG_FIFO_DATA		0x38
#define REG_DATA_REG		0x3C
#define REG_DATA_REG_SIZE	0x40
#define REG_DEV0_PTR		0x44
#define REG_DEV1_PTR		0x48
#define REG_DEV2_PTR		0x4C
#define REG_DEV3_PTR		0x50
#define REG_DEV4_PTR		0x54
#define REG_DEV5_PTR		0x58
#define REG_DEV6_PTR		0x5C
#define REG_DEV7_PTR		0x60
#define REG_DMA_ADDR_L		0x64
#define REG_DMA_ADD_H		0x68
#define REG_DMA_CNT		0x6C
#define REG_DMA_CTRL		0x70
#define REG_MEM_CTRL		0x80
#define REG_DATA_SIZE		0x84
#define REG_TIMINGS_ASYN	0x88
#define REG_TIMINGS_SYN		0x8C
#define REG_TIME_SEQ_0		0x90
#define REG_TIME_SEQ_1		0x94
#define REG_TIME_GEN_SEQ_0	0x98
#define REG_TIME_GEN_SEQ_1	0x9C
#define REG_TIME_GEN_SEQ_2	0xA0
#define REG_FIFO_INIT		0xB0
#define REG_FIFO_STATE		0xB4
#define REG_GEN_SEQ_CTRL	0xB8
#define REG_MLUN		0xBC
#define REG_DEV0_SIZE		0xC0
#define REG_DEV1_SIZE		0xC4
#define REG_DEV2_SIZE		0xC8
#define REG_DEV3_SIZE		0xCC
#define REG_DEV4_SIZE		0xD0
#define REG_DEV5_SIZE		0xD4
#define REG_DEV6_SIZE		0xD8
#define REG_DEV7_SIZE		0xDC
#define REG_SS_CCNT0		0xE0
#define REG_SS_CCNT1		0xE4
#define REG_SS_SCNT		0xE8
#define REG_SS_ADDR_DEV_CTRL	0xEC
#define REG_SS_CMD0		0xF0
#define REG_SS_CMD1		0xF4
#define REG_SS_CMD2		0xF8
#define REG_SS_CMD3		0xFC
#define REG_SS_ADDR		0x100
#define REG_SS_MSEL		0x104
#define REG_SS_REQ		0x108
#define REG_SS_BRK		0x10C
#define REG_DMA_TLVL		0x114
#define REG_AES_CTRL		0x118
#define REG_AES_DATAW		0x11C
#define REG_AES_SVECT		0x120
#define REG_CMD_MARK		0x124
#define REG_LUN_STATUS_0	0x128
#define REG_LUN_STATUS_1	0x12C
#define REG_TIMINGS_TOGGLE	0x130
#define REG_TIME_GEN_SEQ_3	0x134
#define REG_CNE_MASK		0x13C
#define REG_CNE_VAL		0x140
#define REG_CNE_CTRL		0x144
#define REG_INTERNAL_STATUS	0x148

#define SEQ_0	0x00
#define SEQ_2	0x22
#define SEQ_1	0x21
#define SEQ_4	0x24
#define SEQ_6	0x26
#define SEQ_10	0x2a
#define SEQ_12	0x0C
#define SEQ_13	0x0d
#define SEQ_14	0x0e

#define I_SIU 0
#define I_DMA 1
#define D_FIFO 0
#define D_DATA 1

#define DMA_WORK_MODE_SFR	0
#define DMA_WORK_MODE_SG	1

#define DMA_BURST_LEN_4BEAT	0
#define DMA_BURST_LEN_8BEAT	4
#define DMA_BURST_LEN_16BEAT	5	/*default:from verification owner*/

	/*000:incrementing precise burst of precisely four transfers
	001:stream burst(address constant)
	010:single transfer(address increment)
	011:burst of unspecified length(address increment)
	100:incrementing precise burst of precisely eight transfers
	101:incrementing precise burst of precisely sixteen transfers*/

#define mkcmd(seq, input, dsel, cmd0, cmd1, cmd2) \
	((seq) | ((input) << 6) | ((dsel) << 7) | \
	((cmd0) << 8) | ((cmd1) << 16) | ((cmd2) << 24))


enum eva_dma_ops {
	DMANONE = 0,
	DMAREAD = 1,
	DMAWRITE = 2,
};

struct evtr_nand_ctrl {
	struct nand_chip chip;

	struct device *dev;
	struct clk *clk;
	int irq;
	void __iomem *base;

	int d_fifo;
	u32 bbuf;
	int bbuf_bytes;
	int opsdelay;
	int swecc_buf_frag;

	enum eva_dma_ops dma;	/*0, no dma, 1, read, 2, write*/
	bool polling;

	struct completion evt_dma_comp;
	dma_addr_t dma_wt_paddr, dma_rd_paddr;
	u8 *dma_rd_vaddr;
	u8 *dma_cache[2];
	u32 dma_cp_len[2];

	dma_addr_t dma_sg_paddr;
	u8 *dma_sg_vaddr;
	dma_addr_t dma_swecc_paddr;
	u8 *dma_swecc_vaddr;
};

struct evtr_dma_sg_desc {
	u32	flag;
	u32	length;
	dma_addr_t buf;
	dma_addr_t next;
	u8 *bufvaddr;
};

/* 2048-byte page size with 1-bit correction
 * (4 bytes ECC per 512 byte, for SLC flash)
 */
static struct nand_ecclayout evtr_ecc_layout = {
	.eccbytes = 16,
	.eccpos = {
		48, 49, 50, 51, 52, 53, 54, 55,
		56, 57, 58, 59, 60, 61, 62, 63,
	},
	.oobfree = { {2, 46} },
};

static struct nand_ecclayout evtr_sw_ecc_layout = {
	.eccbytes = 24,
	.eccpos = {
		13, 14, 15,  8,  9, 10, 29, 30,
		31, 24, 25, 26, 45, 46, 47, 40,
		41, 42, 61, 62, 63, 56, 57, 58,
	},
	.oobfree = { {2, 6}, {11, 2}, {16, 8}, {27, 2},
		{32, 8}, {43, 2}, {48, 8}, {59, 2},
	},
};

static inline u32 evtr_reg_read(struct evtr_nand_ctrl *priv, int reg)
{
	return readl(priv->base + reg);
}

static inline void evtr_reg_write(struct evtr_nand_ctrl *priv, int reg, u32 val)
{
	writel(val, priv->base + reg);
}

static inline void evtr_hwecc_disable(struct evtr_nand_ctrl *priv)
{
	u32 ctrl = 0;

	if (hwecc == HWECC) {
		ctrl = evtr_reg_read(priv, REG_CONTROL);
		ctrl &= (~BIT(5));
		evtr_reg_write(priv, REG_CONTROL, ctrl);
	}
}

static inline void evtr_hwecc_enable(struct evtr_nand_ctrl *priv)
{
	u32 ctrl = 0;

	if (hwecc == HWECC) {
		ctrl = evtr_reg_read(priv, REG_CONTROL);
		ctrl |= BIT(5);
		evtr_reg_write(priv, REG_CONTROL, ctrl);
	}
}

static int evtr_dma_config(struct mtd_info *mtd, struct nand_chip *chip,
	dma_addr_t addr, int len)
{
	struct evtr_nand_ctrl *priv = chip->priv;
	bool dmaready = (evtr_reg_read(priv, REG_DMA_CTRL) & BIT(0)) == BIT(0);

	if (!dmaready) {
		dev_err(priv->dev, "DMA not ready\n");
		return -EIO;
	}

	NAND_PRINTK(2, "DMA trg\n");

	evtr_reg_write(priv, REG_DMA_CNT, len);
	evtr_reg_write(priv, REG_DMA_ADDR_L, addr);

	evtr_reg_write(priv, REG_DMA_CTRL,
		       ((DMA_WORK_MODE_SFR << 5) | (DMA_BURST_LEN_16BEAT << 2) | BIT(7)));

	return 0;
}

static void evtr_dma_xfer(struct mtd_info *mtd, dma_addr_t addr, int len,
			enum dma_data_direction dma_dir)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;

	dma_sync_single_for_device(priv->dev, addr, len, dma_dir);

	evtr_dma_config(mtd, chip, addr, len);
}

static void evtr_dma_release(struct mtd_info *mtd, dma_addr_t addr,
	int len, enum dma_data_direction dma_dir)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;

	dma_sync_single_for_cpu(priv->dev, addr, len, dma_dir);

	dma_unmap_single(priv->dev, addr, len, dma_dir);
}

static void evtr_sg_desc_init(struct evtr_nand_ctrl *priv,
			      struct evtr_dma_sg_desc *desc,
			      int desc_count, int len)
{
/*
00:Execute current descriptor and go to next descriptor on the list.
01:Execute current descriptor and end the transfer.
10:Execute current descriptor and link to another descriptor.
11:Execute current descriptor and end the transfer.
*/
	int i;
	struct evtr_dma_sg_desc *evt_desc = desc;
	int buflen = len/desc_count;
	u8 *bufvaddr;
	dma_addr_t bufpaddr;

	if (desc_count & 1) {
		NAND_PRINTK(0, "SG DMA count must be even\n");
		return;
	}

	NAND_PRINTK(2, "des paddr %08x , desc len %08x ",
		    priv->dma_sg_paddr, sizeof(struct evtr_dma_sg_desc));

	for (i = 0; i < desc_count; i++, evt_desc++) {
		if (i < (desc_count-1)) {
			evt_desc->flag = 0x02;
			evt_desc->next = (dma_addr_t)((u8 *)(priv->dma_sg_paddr)
					+ sizeof(struct evtr_dma_sg_desc)*(i+1));
		} else {
			evt_desc->flag = 0x03;
			evt_desc->next = (dma_addr_t)NULL;
		}

		bufvaddr = dma_alloc_coherent(priv->dev, buflen,
					      &bufpaddr, GFP_KERNEL);

		evt_desc->buf = bufpaddr;

		evt_desc->length = buflen;

		evt_desc->bufvaddr = bufvaddr;

		NAND_PRINTK(1, "DESC %d flg %08x, len %08x buf paddr %08x,next paddr %08x ",
			    i, evt_desc->flag, evt_desc->length,\
			    evt_desc->buf, evt_desc->next);

		memset(bufvaddr, (0x31+i), buflen);
	}
}

static void evtr_sg_desc_deinit(struct evtr_nand_ctrl *priv,
	struct evtr_dma_sg_desc *desc, int desc_count, int len)
{
	int i;
	struct evtr_dma_sg_desc *evt_desc = desc;

	NAND_PRINTK(2, "des paddr %08x , desc len %08x ",
		    priv->dma_sg_paddr, sizeof(struct evtr_dma_sg_desc));

	for (i = 0; i < desc_count; i++, evt_desc++)
		dma_free_coherent(priv->dev,
				  sg_verify * sizeof(struct evtr_dma_sg_desc),
				  evt_desc->bufvaddr, evt_desc->buf);
}

static int evtr_sg_dma_config(struct nand_chip *chip)
{
	struct evtr_nand_ctrl *priv = chip->priv;
	bool dmaready = (evtr_reg_read(priv, REG_DMA_CTRL) & BIT(0)) == BIT(0);

	if (!dmaready)
		return -EIO;

	NAND_PRINTK(2, "sg paddr %08x vaddr %p", priv->dma_sg_paddr,
		    priv->dma_sg_vaddr);
	evtr_reg_write(priv, REG_DMA_ADDR_L, priv->dma_sg_paddr);

	evtr_reg_write(priv, REG_DMA_CTRL,
		       ((DMA_WORK_MODE_SG << 5) | (DMA_BURST_LEN_16BEAT << 2) | BIT(7)));

	return 0;
}

static int evtr_wait(struct mtd_info *mtd, struct nand_chip *chip)
{
	struct evtr_nand_ctrl *priv = chip->priv;
	volatile int cnt = 0;
	u32 status;

	for (; cnt < priv->opsdelay; cnt++) {
		status = evtr_reg_read(priv, REG_STATUS);
		NAND_PRINTK(2, "%ld ", status & BIT(0));
		if (status & BIT(0)) {
			/*printk("%d ", cnt);*/
			return NAND_STATUS_READY;
		} else
			udelay(1);
	}
	NAND_PRINTK(1, "timeout");
	return NAND_STATUS_FAIL;
}

static void evtr_multi_sg_handle(struct evtr_nand_ctrl *priv, int sg_count)
{
	int i;
	u32 temp;

	for (i = 0; i < sg_count-1; i++) {
		init_completion(&priv->evt_dma_comp);
		temp = evtr_reg_read(priv, REG_DMA_CTRL);
		temp &= (~BIT(7));
		evtr_reg_write(priv, REG_DMA_CTRL, temp);

		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_12, (cpu_fifo == 0) ? I_DMA : I_SIU, D_FIFO,
				     NAND_CMD_SEQIN, NAND_CMD_PAGEPROG, 0));

		temp = evtr_reg_read(priv, REG_DMA_CTRL);
		temp |= BIT(7);
		evtr_reg_write(priv, REG_DMA_CTRL, temp);

		wait_for_completion(&priv->evt_dma_comp);
		NAND_PRINTK(2, "page %d write ok", i+2);
	}
}

static void evtr_cmdfunc(struct mtd_info *mtd,
			unsigned int command,
			int column, int page_addr)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	u32	temp;
	int	ret;

	priv->d_fifo = 1;
	priv->dma = DMANONE;
	priv->opsdelay = chip->chip_delay;

	NAND_PRINTK(2, "CMD %02x column %08x page_addr %08x \n",
		    command, column, page_addr);

	switch (command) {
	case NAND_CMD_READ0:

		evtr_reg_write(priv, REG_FIFO_INIT, 1);

		priv->bbuf_bytes = 0;

		evtr_reg_write(priv, REG_ADDR0_COL, column);
		evtr_reg_write(priv, REG_ADDR0_ROW, page_addr);
		if ((hwecc == SWECC) || (cpu_fifo == 1))
			evtr_reg_write(priv, REG_DATA_SIZE, mtd->writesize + mtd->oobsize);
		else
			evtr_reg_write(priv, REG_DATA_SIZE, mtd->writesize);

		//enable hwecc after REG_DATA_SIZE to avoid PG_SZ_ERR_INT_FL error interrupt.
		evtr_hwecc_enable(priv);

		if (cpu_fifo == 1) {
			evtr_reg_write(priv, REG_COMMAND,
			mkcmd(SEQ_10, I_SIU, D_FIFO,
			      NAND_CMD_READ0, 0, NAND_CMD_READSTART));
		} else {
			priv->dma = DMAREAD;
		}
		break;

	case NAND_CMD_READOOB:
		evtr_reg_write(priv, REG_FIFO_INIT, 1);
		evtr_hwecc_disable(priv);
		priv->bbuf_bytes = 0;

		evtr_reg_write(priv, REG_DATA_SIZE, mtd->oobsize);
		evtr_reg_write(priv, REG_ADDR0_COL, mtd->writesize);
		evtr_reg_write(priv, REG_ADDR0_ROW, page_addr);
		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_10, I_SIU, D_FIFO,
				     NAND_CMD_READ0, 0, NAND_CMD_READSTART));
		break;

	case NAND_CMD_READID:
		evtr_reg_write(priv, REG_FIFO_INIT, 1);
		evtr_hwecc_disable(priv);
		priv->bbuf_bytes = 0;

		evtr_reg_write(priv, REG_DATA_SIZE, 5);
		evtr_reg_write(priv, REG_ADDR0_COL, 0);
		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_1, I_SIU, D_FIFO, NAND_CMD_READID, 0, 0));
		break;

	case NAND_CMD_ERASE1:
		evtr_reg_write(priv, REG_ADDR0_ROW, page_addr);
		break;

	case NAND_CMD_ERASE2:
		priv->opsdelay = TBRES; /*tbers, max 2 ms*/
		evtr_hwecc_disable(priv);
		evtr_reg_write(priv, REG_DATA_SIZE, 0);

		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_14, I_SIU, D_FIFO,
				     NAND_CMD_ERASE1, NAND_CMD_ERASE2, 0));
		break;

	case NAND_CMD_SEQIN:
		if (hwecc == SWECC && column == (int)(mtd->writesize))
			cpu_fifo = 1;

		evtr_reg_write(priv, REG_FIFO_INIT, 1);

		if (cpu_fifo == 0)
			priv->dma = DMAWRITE;

		priv->bbuf_bytes = 0;

		if (sg_verify >= 4) {
			temp = evtr_reg_read(priv, REG_CONTROL);
			temp |= BIT(16);
			evtr_reg_write(priv, REG_CONTROL, temp);
		}

		if (hwecc == SWECC) {
			if (column == mtd->writesize)
				evtr_reg_write(priv, REG_DATA_SIZE, mtd->oobsize);
			else {
				if (cpu_fifo == 0)
					evtr_reg_write(priv, REG_DATA_SIZE,
						       mtd->writesize + mtd->oobsize);
				else
					evtr_reg_write(priv, REG_DATA_SIZE, mtd->writesize);
			}
		} else
			evtr_reg_write(priv, REG_DATA_SIZE, mtd->writesize);

		//enable hwecc after REG_DATA_SIZE to avoid PG_SZ_ERR_INT_FL error interrupt.
		evtr_hwecc_enable(priv);

		evtr_reg_write(priv, REG_ADDR0_COL, column);

		evtr_reg_write(priv, REG_ADDR0_ROW, page_addr);

		break;

	case NAND_CMD_PAGEPROG:
		priv->opsdelay = TPROG; /*tprog, max 700us*/
		evtr_hwecc_enable(priv);

		//priv->dma must be set before issue REG_COMMAND.
		//If not, the irq handle maybe called with pri->dma=0,
		//whick cause time out in race condition.
		if (cpu_fifo == 0)
			priv->dma = DMAWRITE;
		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_12, (cpu_fifo == 0) ? I_DMA : I_SIU, D_FIFO,
				     NAND_CMD_SEQIN, NAND_CMD_PAGEPROG, 0));

		if (cpu_fifo == 0) {
			if (polling == 0) {
				NAND_PRINTK(3, " wait for complete\n");
				ret = wait_for_completion_timeout(&priv->evt_dma_comp, HZ/10);

				if (!ret) {
					dev_err(priv->dev, "DMA write timeout at page %u c %08x\n",
						evtr_reg_read(priv, REG_ADDR0_ROW),
						evtr_reg_read(priv, REG_DMA_CTRL));
					nand_dbg = 0;
					break;
				}

				if (sg_verify >= 4) {
					evtr_multi_sg_handle(priv, sg_verify);
				} else {
					if (sg_verify == 0)
						evtr_dma_release(mtd, priv->dma_wt_paddr,
								 mtd->writesize,
								 DMA_TO_DEVICE);
				}
			} else {
				evtr_wait(mtd, chip);
				evtr_dma_release(mtd, priv->dma_wt_paddr,
						 mtd->writesize, DMA_TO_DEVICE);
			}
		}
		udelay(1);/*wait twb 100 before check ready/busy bit as device spec*/
		break;

	case NAND_CMD_STATUS:

		evtr_hwecc_disable(priv);
		priv->d_fifo = 0;
		priv->bbuf_bytes = 0;
		evtr_reg_write(priv, REG_DATA_REG_SIZE, 0);
		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_4, I_SIU, D_DATA, NAND_CMD_STATUS, 0, 0));
		udelay(1);/*check status wait twhr 60 + trea 20 as device spec */
		break;

	case NAND_CMD_RESET:
		priv->opsdelay = TRST;
		evtr_hwecc_disable(priv);
		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_0, 0, 0, NAND_CMD_RESET, 0, 0));
		evtr_wait(mtd, chip);
		break;
	default:
		dev_err(priv->dev, "unsupported command %d\n", command);
	}
}



static void evtr_prep_write_buf(struct mtd_info *mtd, const u8 *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	dma_addr_t	dmaaddr;

	if (polling)
		return;

	init_completion(&priv->evt_dma_comp);

	if (sg_verify >= 2)
		evtr_sg_dma_config(chip);
	else {
		priv->dma_cache[1] = (u8 *)buf;
		priv->dma_cp_len[1] = 0;
		if (!virt_addr_valid(buf)) {
			priv->dma_cp_len[1] = len;
			if (chip->buffers) {
				priv->dma_cache[1] =
					&(chip->buffers->databuf[mtd->writesize / 2]);
				memcpy((void *)priv->dma_cache[1], (void *)buf, len);
			} else
				dev_err(priv->dev, "chip->buffers is null\n");

			NAND_PRINTK(3, "dma_cache=%p\n", priv->dma_cache[1]);
		}
		dmaaddr = dma_map_single(priv->dev, (void *)priv->dma_cache[1], len,
					 DMA_TO_DEVICE);

		if (dma_mapping_error(priv->dev, dmaaddr)) {
			dev_err(priv->dev, "dma_map error\n");
			return;
		}
		priv->dma_wt_paddr = dmaaddr;

		evtr_dma_xfer(mtd, dmaaddr, len, DMA_TO_DEVICE);
	}
}

static void evtr_write_buf(struct mtd_info *mtd, const u8 *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	u32 val;
	u32 cnt = 0;

	NAND_PRINTK(2, "buf %p len %d\n", buf, len);

	if (DMANONE == priv->dma) {
		while (len > 0) {
			while ((evtr_reg_read(priv, REG_FIFO_STATE) & BIT(1))
				&& (cnt++ < 1000))
				udelay(1);
			if (cnt >= 1000)
				dev_err(priv->dev, "WRITE FIFO Full Timeout! len = %d\n", len);
			memcpy((u8 *)&val, buf, min(len, 4));

			evtr_reg_write(priv, REG_FIFO_DATA, val);
			len -= 4;
			buf += 4;
		}
	}
}

static void evtr_write_buf_swecc(struct mtd_info *mtd, const u8 *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	u32 cpu_fifo_tmp;

	/*write oob only*/
	if (len == mtd->oobsize && priv->swecc_buf_frag == 0) {
		priv->dma = DMANONE;
		cpu_fifo_tmp = cpu_fifo;
		cpu_fifo = 1;
		if (cpu_fifo_tmp == 1) {
			evtr_reg_write(priv, REG_ADDR0_COL, mtd->writesize);
			evtr_reg_write(priv, REG_DATA_SIZE, mtd->oobsize);
		}
		evtr_write_buf(mtd, buf, len);
		cpu_fifo = cpu_fifo_tmp;
		return;
	}
	if (cpu_fifo == 0) {
		/*write main+oob area*/
		if (len == mtd->writesize) {
			priv->swecc_buf_frag = 1;
			memcpy(priv->dma_swecc_vaddr, buf, mtd->writesize);
		} else if (len == mtd->oobsize) {
			if (priv->swecc_buf_frag == 1) {
				init_completion(&priv->evt_dma_comp);
				memcpy(priv->dma_swecc_vaddr + mtd->writesize,
				       buf, mtd->oobsize);
				evtr_dma_config(mtd, chip, priv->dma_swecc_paddr,
						mtd->writesize + mtd->oobsize);
				priv->swecc_buf_frag = 0;
			} else
				NAND_PRINTK(0, "sw ecc write buf error\n");
		} else
			NAND_PRINTK(0, "sw ecc write buf length error\n");
	} else {
		priv->dma = DMANONE;
		NAND_PRINTK(2, "sw ecc SIU mode, write buf len=%d\n", len);
		memcpy(chip->buffers->databuf, buf, len);
		evtr_write_buf(mtd, buf, len);
		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_12, I_SIU, D_FIFO,
				     NAND_CMD_SEQIN, NAND_CMD_PAGEPROG, 0));

	}
}

static uint8_t evtr_read_byte(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	u32 val;

	if (priv->bbuf_bytes) {
		val = priv->bbuf;
		priv->bbuf_bytes--;
	} else {
		if (priv->d_fifo)
			while (evtr_reg_read(priv, REG_FIFO_STATE) & BIT(0))
				;
		else
			while (!(evtr_reg_read(priv, REG_STATUS) & BIT(10)))
				;
		val = evtr_reg_read(priv, priv->d_fifo ?
				    REG_FIFO_DATA : REG_DATA_REG);
		priv->bbuf_bytes = 3;
	}
	priv->bbuf = val >> 8;

	NAND_PRINTK(2, "%02x ", val&0xff);

	return val & 0xff;
}

static void evtr_prep_read_buf(struct mtd_info *mtd, u8 *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	dma_addr_t	dmaaddr;

	if (priv->dma == DMAREAD && cpu_fifo == 0) {
		init_completion(&priv->evt_dma_comp);
		priv->dma_cache[0] = buf;
		priv->dma_cp_len[0] = 0;
		if (!virt_addr_valid(buf)) {
			priv->dma_cp_len[0] = len;
			if (chip->buffers)
				priv->dma_cache[0] = chip->buffers->databuf;
			else
				dev_err(priv->dev, "chip->buffers is null\n");

			NAND_PRINTK(3, "dma_cache=%p\n", priv->dma_cache[0]);
		}
		dmaaddr = dma_map_single(priv->dev, priv->dma_cache[0],
					 len, DMA_FROM_DEVICE);

		if (dma_mapping_error(priv->dev, dmaaddr)) {
			printk(KERN_DEBUG "%s map error!!\n", __func__);

			dev_err(priv->dev, "dma map error\n");
			return;
		}

		priv->dma_rd_paddr = dmaaddr;
		priv->dma_rd_vaddr = buf;

		evtr_dma_xfer(mtd, dmaaddr, len, DMA_FROM_DEVICE);
	}
}
static void evtr_read_buf(struct mtd_info *mtd, u8 *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	u32 val;
	u32 cnt = 0;

	NAND_PRINTK(3, "len  %d\n", len);

	if (priv->dma == DMAREAD && cpu_fifo == 0)
		return;

	while (len > 0) {
			while ((evtr_reg_read(priv, REG_FIFO_STATE) & BIT(0))
				&& (cnt++ < 1000))
				udelay(1);
			if (cnt >= 1000)
				dev_err(priv->dev, "READ FIFO Empty Timeout! len = %d\n", len);

		val = evtr_reg_read(priv, REG_FIFO_DATA);

		memcpy(buf, (u8 *)&val, min(len, 4));
		len -= 4;
		buf += 4;
	}
}

static void evtr_read_buf_swecc(struct mtd_info *mtd, u8 *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	uint8_t *dp = priv->dma_swecc_vaddr;
	u32 cpu_fifo_tmp;
	int ret;

	/*read oob only*/
	if (len == mtd->oobsize && priv->swecc_buf_frag == 0) {
		priv->dma = DMANONE;
		cpu_fifo_tmp = cpu_fifo;
		cpu_fifo = 1;
		evtr_wait(mtd, chip);
		evtr_read_buf(mtd, buf, len);
		cpu_fifo = cpu_fifo_tmp;
		return;
	}

	if (cpu_fifo == 0) {
		/*read main+oob area*/
		if (len == mtd->writesize) {
			init_completion(&priv->evt_dma_comp);
			priv->swecc_buf_frag = 1;

			evtr_dma_config(mtd, chip, priv->dma_swecc_paddr,
					mtd->writesize + mtd->oobsize);
			evtr_reg_write(priv, REG_COMMAND,
				       mkcmd(SEQ_10, priv->dma == DMAREAD ? I_DMA : I_SIU,
					     D_FIFO, NAND_CMD_READ0, 0, NAND_CMD_READSTART));

			ret = wait_for_completion_timeout(&priv->evt_dma_comp, HZ / 10);

			if (!ret) {
				dev_err(priv->dev, "read DMA time out\n");
				return;
			}
		}

		if (len == mtd->writesize)
			memcpy(buf, priv->dma_swecc_vaddr, mtd->writesize);
		else if (len == mtd->oobsize) {
			if (priv->swecc_buf_frag) {
				dp += mtd->writesize;
				/*copy oob area to buf*/
				memcpy(buf, dp,  mtd->oobsize);
				priv->swecc_buf_frag = 0;
			} else
				NAND_PRINTK(0, "sw ecc read buf error\n");
		} else
			NAND_PRINTK(0, "sw ecc read buf length error\n");
	} else {
		evtr_wait(mtd, chip);
		evtr_read_buf(mtd, buf, len);
	}
}

static int evtr_dev_ready(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct evtr_nand_ctrl *priv = chip->priv;
	u32 status;

	status = evtr_reg_read(priv, REG_STATUS);
	NAND_PRINTK(3, "status %d ", status);

	return status & BIT(0);
}

static int evtr_read_page(struct mtd_info *mtd,
			  struct nand_chip *chip,
			  uint8_t *buf, int oob_required, int page)
{
	int status = 0, ret;
	struct evtr_nand_ctrl *priv = chip->priv;
	u32 stat;

	NAND_PRINTK(2, "buf[0] %02x buf[1] %02x\n", buf[0], buf[1]);

	evtr_reg_write(priv, REG_ECC_STAT, 0);
	if (cpu_fifo == 0 && priv->dma == DMAREAD) {
		evtr_prep_read_buf(mtd, buf, mtd->writesize);
		evtr_reg_write(priv, REG_COMMAND,
			       mkcmd(SEQ_10, priv->dma == DMAREAD ? I_DMA : I_SIU,
				     D_FIFO, NAND_CMD_READ0, 0, NAND_CMD_READSTART));

		if (polling == 0) {
			ret = wait_for_completion_timeout(&priv->evt_dma_comp, HZ / 10);

			if (!ret) {
				dev_err(priv->dev, "wait completition time out\n");
				return -1;
			}
			evtr_dma_release(mtd, priv->dma_rd_paddr,
					 mtd->writesize, DMA_FROM_DEVICE);
		} else {
			evtr_wait(mtd, chip);
			evtr_dma_release(mtd, priv->dma_rd_paddr,
					 mtd->writesize, DMA_FROM_DEVICE);
		}
	} else {
		status = evtr_wait(mtd, chip);
		udelay(1); /*wait trr 20ns after check ready bit as dev spec*/
		evtr_read_buf(mtd, buf, mtd->writesize);
	}


	// bit error catch and report
	stat = evtr_reg_read(priv, REG_ECC_STAT);
	if (stat & (BIT(0) | BIT(8))) {
		/* check for empty pages with bitflips */
		uint8_t *ecc_code = chip->buffers->ecccode;
		uint32_t *eccpos = chip->ecc.layout->eccpos;
		int i, res;

		chip->ecc.read_oob(mtd, chip, page);
		for (i = 0; i < chip->ecc.total; i++)
			ecc_code[i] = chip->oob_poi[eccpos[i]];
		res = nand_check_erased_ecc_chunk(buf, chip->ecc.size,
						  ecc_code, chip->ecc.total,
						  NULL, 0,
						  chip->ecc.strength);
		NAND_PRINTK(3, "ECC stat %08x -> %d\n", stat, res);
		if (res >= 0)
			return 0;
	}

	if (stat & BIT(8)) {
		pr_err("%s:%d: ECC UNCORRECTABLE\n", __func__, __LINE__);
		mtd->ecc_stats.failed++;
	}
	else if (stat & BIT(0)) {
		pr_err("%s:%d: ECC corrected\n", __func__, __LINE__);
		mtd->ecc_stats.corrected++;
	}
	return 0;
}

static int evtr_write_page(struct mtd_info *mtd,
			   struct nand_chip *chip,
			   const uint8_t *buf,
			   int oob_required, int page)
{
	struct evtr_nand_ctrl *priv = chip->priv;

	NAND_PRINTK(2, "buf[0] %02x buf[1] %02x\n", buf[0], buf[1]);

	if (priv->dma == DMAWRITE && cpu_fifo == 0)
		evtr_prep_write_buf(mtd, buf, mtd->writesize);
	else
		evtr_write_buf(mtd, buf, mtd->writesize);
	return 0;
}

static void evtr_select_chip(struct mtd_info *mtd, int chipnr)
{
	/* no op, only 1 chip supported */
}

static irqreturn_t evtr_nand_isr(int irq, void *dev_id)
{
	struct evtr_nand_ctrl *priv = dev_id;
	u32 status = evtr_reg_read(priv, REG_INT_STATUS);

	NAND_PRINTK(2, "status %08x\n", status);
	evtr_reg_write(priv, REG_INT_STATUS, 0);

	if (status & BIT(3)) {
		if ((priv->dma == DMAREAD) || (priv->dma == DMAWRITE)) {
			if ((priv->dma == DMAREAD) && (priv->dma_cp_len[0] > 0)
			    && priv->dma_cache[0]) {
				memcpy((void *)priv->dma_rd_vaddr,
				       (void *)priv->dma_cache[0],
				       priv->dma_cp_len[0]);
			}
			priv->dma = DMANONE;
			complete(&priv->evt_dma_comp);
			NAND_PRINTK(2, "1");
		}

		NAND_PRINTK(2, "DMA transfer complete\n");
	} else {
		printk(KERN_ERR "Line %d, NAND status %08x\n", __LINE__, status);
		nand_dbg = 0;
	}
	return IRQ_HANDLED;

}

static int evtr_chip_read_oob(struct mtd_info *mtd,
			      struct nand_chip *chip, int page)
{
	struct evtr_nand_ctrl *priv = chip->priv;

	u8 buf[64];
	int i;

	priv->dma = DMANONE;

	evtr_hwecc_disable(priv);

	chip->cmdfunc(mtd, NAND_CMD_READOOB, 0, page);

	NAND_PRINTK(3, "buf %p length %d\n", chip->oob_poi, mtd->oobsize);

	evtr_wait(mtd, chip);
	udelay(1); /*wait trr 20ns after check ready bit as device spec*/

	if (hwecc_verify) {
		evtr_read_buf(mtd, buf, 64);
		NAND_PRINTK(3, "HWECC_VERIFY\n");

		for (i = 0; i < 64; i++)
			printk(KERN_DEBUG "%02x ", buf[i]);

		/* nand_base transfer_oob will omit 2 bytes*/
		memcpy(chip->oob_poi, &buf[46], 18);
	} else {
		evtr_read_buf(mtd, chip->oob_poi, mtd->oobsize);
	}
	#if 0
	for (i = 0; i < mtd->oobsize; i++)
		printk(KERN_DEBUG "%02x ", chip->oob_poi[i]);
	#endif

	return 0;
}

int evtr_chip_write_oob(struct mtd_info *mtd, struct nand_chip *chip,
			int page)
{
	struct evtr_nand_ctrl *priv = chip->priv;
	int status = 0, i;
	const uint8_t *buf = chip->oob_poi + 2;
	u8 fakecc[16];

	int length = chip->ecc.layout->oobfree[0].length ;
	int oob_offset = chip->ecc.layout->oobfree[0].offset;
	int eccposi = chip->ecc.layout->eccpos[0];

	u32 fifolen = ((u32)length + 3) & 0xfffffffc;

	evtr_hwecc_disable(priv);

	NAND_PRINTK(3, "buf %p length %d, offset %d eccposi %d\n",
		    buf, length, oob_offset, eccposi);

	evtr_reg_write(priv, REG_FIFO_INIT, 1);
	priv->opsdelay = TPROG; /*tprog, max 700us*/
	priv->bbuf_bytes = 0;
	priv->dma = DMANONE;

	evtr_reg_write(priv, REG_ADDR0_ROW, page);

	if (hwecc_verify) {
		memcpy(fakecc, &buf[0], 16);
		evtr_reg_write(priv, REG_ADDR0_COL, mtd->writesize + eccposi);
		evtr_reg_write(priv, REG_DATA_SIZE, 16);
		evtr_write_buf(mtd, fakecc, 16);
	} else {
		evtr_reg_write(priv, REG_ADDR0_COL, mtd->writesize + oob_offset);
		evtr_reg_write(priv, REG_DATA_SIZE, fifolen);
		evtr_write_buf(mtd, buf, fifolen);
	}

	evtr_reg_write(priv, REG_COMMAND,
		       mkcmd(SEQ_12, I_SIU, D_FIFO,
			     NAND_CMD_SEQIN, NAND_CMD_PAGEPROG, 0));

	udelay(1);/*wait twb 100ns befor check reday/busy bit as device spec*/
	status = evtr_wait(mtd, chip);

	if (hwecc_verify) {
		NAND_PRINTK(3, "write oob start\n");

		for (i = 0; i < 16; i++)
			printk("%02x ", fakecc[i]);
	}

	return status & NAND_STATUS_FAIL ? -EIO : 0;

}

static void evtr_nand_dma_deinit(struct evtr_nand_ctrl *priv)
{
	struct mtd_info *mtd = nand_to_mtd(&priv->chip);

	if (priv->dma_sg_vaddr && priv->dma_sg_paddr) {
		if (sg_verify > 2)
			evtr_sg_desc_deinit(priv, (struct evtr_dma_sg_desc *)(priv->dma_sg_vaddr),
					    sg_verify, mtd->writesize);
		else if (sg_verify == 2)
			evtr_sg_desc_deinit(priv, (struct evtr_dma_sg_desc *)(priv->dma_sg_vaddr),
					    sg_verify, mtd->writesize / 2);
		else
			printk("%s wrong sg number\n", __func__);

		dma_free_coherent(priv->dev, sg_verify * sizeof(struct evtr_dma_sg_desc),
				  priv->dma_sg_vaddr, priv->dma_sg_paddr);
	}

	if (priv->dma_swecc_paddr && priv->dma_swecc_vaddr)
		dma_free_coherent(priv->dev, mtd->writesize + mtd->oobsize,
				  priv->dma_swecc_vaddr, priv->dma_swecc_paddr);
}


static int evtr_nand_probe(struct platform_device *pdev)
{
	struct evtr_nand_ctrl *priv = NULL;
	struct resource *iomem;
	const char *part_probe_types[] = { "ofpart", NULL };
	struct nand_chip *chip = NULL;
	struct mtd_info *mtd;
	long clk_rate, slot_time;
	int ret = 0;
	u32 temp;
	int tccs, tadl, trhw, twhr, trr, twb, trwh, trwp;

	iomem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!iomem)
		return -ENOMEM;
	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	chip = &priv->chip;
	mtd = nand_to_mtd(chip);

	priv->dev = &pdev->dev;
	priv->base = ioremap(iomem->start, resource_size(iomem));
	if (!priv->base) {
		dev_err(priv->dev, "ioremap failed\n");
		ret = -ENOMEM;
		goto err;
	}
	priv->irq = platform_get_irq(pdev,  0);
	if (priv->irq) {
		ret = request_irq(priv->irq, evtr_nand_isr, 0,
				  "evatronix_nand", priv);
		if (ret) {
			priv->irq = 0;
			dev_err(priv->dev, "ioremap failed\n");
			goto err;
		}
	}

	priv->clk = clk_get(&pdev->dev, "nand_clk");
	if (IS_ERR(priv->clk)) {
		priv->clk = NULL;
		dev_err(&pdev->dev, "nand_clk not found\n");
		ret = -ENODEV;
		goto err;
	}

	clk_set_rate(priv->clk, 80000000);
	clk_prepare_enable(priv->clk);

	clk_rate = clk_get_rate(priv->clk);

	slot_time = 1000 * 1000 * 1000 / clk_rate;

	dev_set_drvdata(priv->dev, priv);

	mtd->name = "nand";
	mtd->priv = chip;
	mtd->dev.parent = &pdev->dev;
	nand_set_flash_node(chip, pdev->dev.of_node);

	chip->read_byte = evtr_read_byte;
	chip->write_buf = (hwecc == SWECC) ? evtr_write_buf_swecc : evtr_write_buf;
	chip->read_buf = (hwecc == SWECC) ? evtr_read_buf_swecc : evtr_read_buf;
	chip->select_chip = evtr_select_chip;
	chip->cmdfunc = evtr_cmdfunc;
	chip->dev_ready = evtr_dev_ready;
	chip->waitfunc = evtr_wait;
	chip->options = NAND_CACHEPRG;

	chip->bbt_options = NAND_BBT_USE_FLASH;
	chip->priv = priv;

	chip->ecc.read_page = evtr_read_page;
	chip->ecc.write_page = evtr_write_page;
	chip->ecc.strength = 1;
	chip->ecc.size = (hwecc == SWECC) ? 256 : 512;
	chip->ecc.bytes = (hwecc == SWECC) ? 3 : 4;
	chip->ecc.layout = (hwecc == SWECC) ? &evtr_sw_ecc_layout : &evtr_ecc_layout;
	chip->ecc.write_oob = evtr_chip_write_oob;
	chip->ecc.read_oob = evtr_chip_read_oob;
	chip->ecc.mode = (hwecc == SWECC) ? NAND_ECC_SOFT : NAND_ECC_HW;

	evtr_reg_write(priv, REG_CONTROL, 0);

	evtr_reg_write(priv, REG_INT_STATUS, 0);

	/* FPGA domain B 30M, ASIC domain B 80M */
	/* timing setting equals value+1 */
	/* TCCS(sync mode), TADL(70n), TRHW(100n), TWHR(60n) */
	/* TCCS >= TADL*/
	tccs = (int)(TCCS/slot_time);
	tadl = (int)(TADL/slot_time);
	trhw = (int)(TRHW/slot_time);
	twhr = (int)(TWHR/slot_time);

	evtr_reg_write(priv, REG_TIME_SEQ_0,
		(tccs | (tadl << 8) | (trhw << 16) | (twhr << 24)));

	/* TRR (20n), TWB(100n) */
	trr = (int)(TRR/slot_time);
	twb = (int)(TWB/slot_time);

	evtr_reg_write(priv, REG_TIME_SEQ_1,
		(twb | (trr << 8)));

	trwh = (int)(TRWH/slot_time);
	trwp = (int)(TRWP/slot_time);

	/* TRWH 10n, TRWP 12n */
	evtr_reg_write(priv, REG_TIMINGS_ASYN,
		(trwp | (trwh << 4)));

	/* flush FIFO */
	evtr_reg_write(priv, REG_FIFO_INIT, 1);

	/*HY27UF082G2b spec,  25 us*/
	chip->chip_delay = TR;

	/* first scan to find the device and get the page size */
	if (nand_scan_ident(mtd, 1, NULL)) {
		ret = -ENXIO;
		goto err;
	}

	evtr_reg_write(priv, REG_FIFO_INIT, 1);

	/* device Write protection*/
	evtr_reg_write(priv, REG_MEM_CTRL, BIT(8));

	/* store ECC bytes at the end of spare area */
	temp = mtd->writesize + mtd->oobsize -
		((mtd->writesize >> 9) << 2);
	evtr_reg_write(priv, REG_ECC_OFFSET, temp);

	/* block size */
	temp = ((mtd->erasesize / mtd->writesize) >> 5);
	temp = temp == 1 ? 0 : (temp == 2 ? 1 : (temp == 4 ? 2 : 3));

	/* async mode, 8bit, big block, 512 byte ECC */
	evtr_reg_write(priv, REG_CONTROL, (1 << 1) | (temp << 6));

	/* ECC correct bit 2, error threshold */
	if (hwecc == HWECC)
		evtr_reg_write(priv, REG_ECC_CTRL, (2 << 8));
	else if (hwecc == SWECC) {
		polling = 0;
		priv->dma_swecc_vaddr = dma_alloc_coherent(priv->dev,
			mtd->writesize + mtd->oobsize,
			&priv->dma_swecc_paddr, GFP_KERNEL);
		NAND_PRINTK(2, "hwecc %d eccmode %d dma_swecc_paddr %08x  dma_swecc_vaddr %08x \n",
			hwecc, chip->ecc.mode, priv->dma_swecc_paddr,
			(unsigned int)priv->dma_swecc_vaddr);
	}

	/* sg dma only verify on inetrrupt */
	if (sg_verify >= 2) {
		priv->dma_sg_vaddr = dma_alloc_coherent(priv->dev,
			sg_verify * sizeof(struct evtr_dma_sg_desc),
			&priv->dma_sg_paddr, GFP_KERNEL);

		NAND_PRINTK(2, "dma desc vaddr %p  phy addr %08x",
			priv->dma_sg_vaddr, priv->dma_sg_paddr);

		/* page 0,1,2,3*/
		/* sg test case 1, 1 page write trigger >=4 page write*/
		if (sg_verify > 2) {
			evtr_sg_desc_init(priv,
				(struct evtr_dma_sg_desc *)(priv->dma_sg_vaddr),
				sg_verify, mtd->writesize * sg_verify);
		} else {
			/* sg test case 2, 2 dma block write into 1 page*/
			evtr_sg_desc_init(priv,
				(struct evtr_dma_sg_desc *)(priv->dma_sg_vaddr),
				sg_verify, mtd->writesize);
		}

		polling = 0;
		cpu_fifo = 0;
	}

	/* 3 option, cpu/polling, dma/polling, dma/interrupt */
	if (cpu_fifo)
		polling = 1;

	/* Enalbe interrupt for dev0 busy/rdy, ECC, DMA interrupt */
	if (!polling) {
		evtr_reg_write(priv, REG_INT_MASK,
			       BIT(0) | BIT(3) | BIT(4) | BIT(6) | BIT(16));
		evtr_reg_write(priv, REG_INT_STATUS, 0);
		temp = evtr_reg_read(priv, REG_CONTROL);
		temp |= BIT(4);
		evtr_reg_write(priv, REG_CONTROL, temp);
	}

	/* second phase scan */
	if (nand_scan_tail(mtd)) {
		ret = -ENXIO;
		goto err;
	}

	mtd_device_parse_register(mtd, part_probe_types,
				  NULL, NULL, 0);
	return 0;

err:
	nand_release(mtd);
	if (priv->clk) {
		clk_disable_unprepare(priv->clk);
		clk_put(priv->clk);
	}
	if (priv->irq)
		free_irq(priv->irq, priv);
	if (priv->base)
		iounmap(priv->base);
	dev_set_drvdata(priv->dev, NULL);
	return ret;
}

static int evtr_nand_remove(struct platform_device *pdev)
{
	struct evtr_nand_ctrl *priv = dev_get_drvdata(&pdev->dev);
	struct mtd_info *mtd = nand_to_mtd(&priv->chip);

	nand_release(mtd);
	clk_disable_unprepare(priv->clk);
	clk_put(priv->clk);
	if (priv->irq)
		free_irq(priv->irq, priv);

	evtr_nand_dma_deinit(priv);

	iounmap(priv->base);
	dev_set_drvdata(priv->dev, NULL);
	return 0;
}

static const struct of_device_id evtr_nand_match[] = {
	{
		.compatible = "evatronix,nand-des-6h",
	},
	{}
};

static struct platform_driver evtr_nand_driver = {
	.driver = {
		.name	= "evatronix,nand-des-6h",
		.owner = THIS_MODULE,
		.of_match_table = evtr_nand_match,
	},
	.probe	= evtr_nand_probe,
	.remove	= evtr_nand_remove,
};
module_platform_driver(evtr_nand_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Pixelworks");
MODULE_DESCRIPTION("Evatronix NAND Flash Controller");
