/*
 * Cast Inc. / Evatronix SDIO-HOST
 * Copyright (C) 2012 Pixelworks Inc.
 *
 * based on sdhci-pci.c - SDHCI on PCI bus interface
 *
 *  Copyright (C) 2005-2008 Pierre Ossman, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * Thanks to the following companies for their support:
 *
 *     - JMicron (hardware and technical support)
 */

#include <linux/err.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/clk.h>
#include <linux/bitops.h>
#include <linux/platform_device.h>
#include <linux/mmc/host.h>
#include <linux/mmc/card.h>

#include "sdhci.h"

/* constroller sepecific registers */
#define REG_CONFIG		0x00
#  define CONFIG_MMC_DDR(x)	(1 << (28 + (x)))	/* slot 0...3 */
#  define CONFIG_MMC_8BIT(x)	(1 << (24 + (x)))	/* slot 0...3 */
#  define CONFIG_SLOT_EXISTS(x)	(1 << (16 + (x)))	/* slot 0...3 */
#  define CONFIG_RESET_ALL	(1 << 0)
#define REG_DEBOUNCE		0x04
#define REG_BUS_CONFIG		0x08
#define SDHCI_REG_OFFSET	0x100

/* FIXME */
#define MAX_SLOTS 1


struct sdhci_evatronix_slot {
	struct sdhci_evatronix	*ctrl;
	struct sdhci_host	*host;
	unsigned int		slot_num;
};

struct sdhci_evatronix {
	struct platform_device		*pdev;
	void __iomem			*ioaddr;
	struct clk			*clk;
	int				num_slots;
	struct sdhci_evatronix_slot	*slots[MAX_SLOTS];
};


static struct sdhci_ops sdhci_evatronix_ops = {
	.set_clock		= sdhci_set_clock,
	.set_bus_width		= sdhci_set_bus_width,
	.reset			= sdhci_reset,
	.set_uhs_signaling	= sdhci_set_uhs_signaling,
};

static int sdhci_evatronix_probe(struct platform_device *pdev)
{
	struct sdhci_evatronix *ctrl;
	struct sdhci_evatronix_slot *slot;
	struct sdhci_host *host;
	struct device_node *np = pdev->dev.of_node;
	struct resource *iomem;
	u32 config, pbl;
	bool nonremovable;
	long clk_rate;
	int i, irq;
	int rc = 0;

	iomem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!iomem) {
		rc = -ENOMEM;
		goto err;
	}

	ctrl = kzalloc(sizeof(*ctrl), GFP_KERNEL);
	if (!ctrl) {
		rc = -ENOMEM;
		goto err;
	}
	ctrl->pdev = pdev;

	irq = platform_get_irq(pdev, 0);
	if (!request_mem_region(iomem->start, resource_size(iomem),
				dev_name(&pdev->dev))) {
		dev_err(&pdev->dev, "iomem is busy\n");
		rc = -EBUSY;
		goto err_free_ctrl;
	}
	ctrl->ioaddr = ioremap(iomem->start, resource_size(iomem));
	if (!ctrl->ioaddr) {
		dev_err(&pdev->dev, "ioremap failed\n");
		rc = -ENOMEM;
		goto err_release_iomem;
	}

	ctrl->clk = clk_get(&pdev->dev, "sdio_clk");
	if (IS_ERR(ctrl->clk)) {
		dev_err(&pdev->dev, "sdio_clk not found\n");
		rc = PTR_ERR(ctrl->clk);
		goto err_no_clk;
	}
	clk_prepare_enable(ctrl->clk);
	clk_rate = clk_get_rate(ctrl->clk);
	/* configure 20msec debounce time */
	writel(clk_rate / 50, ctrl->ioaddr + REG_DEBOUNCE);

	nonremovable = of_property_read_bool(np, "non-removable");
	if (of_property_read_u32(np, "dma-burst-length", &pbl))
		pbl = 4;
	writel(fls(pbl), ctrl->ioaddr + REG_BUS_CONFIG);

	/* reset bus mode bits left over from boot sequence */
	writel(0, ctrl->ioaddr + REG_CONFIG);

	config = readl(ctrl->ioaddr + REG_CONFIG);
	for (i = 0; i < MAX_SLOTS; i++) {
		if (!(config & CONFIG_SLOT_EXISTS(i)))
			continue;
		host = sdhci_alloc_host(&pdev->dev, sizeof(*slot));
		if (IS_ERR(host)) {
			rc = PTR_ERR(host);
			goto err_free_hosts;
		}
		host->hw_name = dev_name(&pdev->dev);
		host->irq = irq;
		host->ioaddr = ctrl->ioaddr + (i + 1) * SDHCI_REG_OFFSET;
		host->ops = &sdhci_evatronix_ops;
		if (readl(host->ioaddr + SDHCI_CAPABILITIES) &
			SDHCI_CAN_DO_8BIT)
			host->mmc->caps |= MMC_CAP_8_BIT_DATA;
		if (nonremovable)
			host->mmc->caps |= MMC_CAP_NONREMOVABLE;
		host->quirks = SDHCI_QUIRK_32BIT_DMA_ADDR;
		slot = sdhci_priv(host);
		slot->ctrl = ctrl;
		slot->host = host;
		slot->slot_num = i;
		rc = sdhci_add_host(host);
		if (rc) {
			sdhci_free_host(host);
			goto err_free_hosts;
		}
		ctrl->slots[i] = slot;
		ctrl->num_slots++;
	}
	platform_set_drvdata(pdev, ctrl);
	return 0;

err_free_hosts:
	clk_disable_unprepare(ctrl->clk);
	clk_put(ctrl->clk);
err_no_clk:
	for (i = 0; i < ctrl->num_slots; i++) {
		sdhci_remove_host(ctrl->slots[i]->host, 1);
		sdhci_free_host(ctrl->slots[i]->host);
	}
	iounmap(ctrl->ioaddr);
err_release_iomem:
	release_mem_region(iomem->start, resource_size(iomem));
err_free_ctrl:
	kfree(ctrl);
err:
	return rc;
}

static int sdhci_evatronix_remove(struct platform_device *pdev)
{
	struct sdhci_evatronix *ctrl = platform_get_drvdata(pdev);
	struct resource *iomem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	int i;

	clk_disable_unprepare(ctrl->clk);
	clk_put(ctrl->clk);
	for (i = 0; i < ctrl->num_slots; i++) {
		sdhci_remove_host(ctrl->slots[i]->host, 0);
		sdhci_free_host(ctrl->slots[i]->host);
	}
	iounmap(ctrl->ioaddr);
	release_mem_region(iomem->start, resource_size(iomem));
	platform_set_drvdata(pdev, NULL);
	kfree(ctrl);
	return 0;
}

static const struct of_device_id sdhci_evatronix_of_match[] = {
	{ .compatible = "evatronix,sdio-host" },
	{ }
};
MODULE_DEVICE_TABLE(of, sdhci_evatronix_of_match);

#ifdef CONFIG_PM
static int sdhci_evatronix_suspend(struct device *dev)
{
	struct sdhci_evatronix *ctrl = dev_get_drvdata(dev);
	int i;
	int ret = 0;

	for (i = 0; i < ctrl->num_slots; i++)
		if (ctrl->slots[i]) {
			ret = sdhci_suspend_host(ctrl->slots[i]->host);
			if (ret)
				goto err_sdhci_evatronix_suspend;
		}

	return 0;

err_sdhci_evatronix_suspend:
	while (--i >= 0)
		sdhci_resume_host(ctrl->slots[i]->host);
	return ret;
}

static int sdhci_evatronix_resume(struct device *dev)
{
	struct sdhci_evatronix *ctrl = dev_get_drvdata(dev);
	int i;
	int ret = 0;

	for (i = 0; i < ctrl->num_slots && ret >= 0; i++)
		if (ctrl->slots[i])
			ret = sdhci_resume_host(ctrl->slots[i]->host);

	return ret;
}
#else
#define	sdhci_evatronix_suspend NULL
#define	sdhci_evatronix_resume NULL
#endif

static SIMPLE_DEV_PM_OPS(sdhc_evatronix_pm_ops,
	sdhci_evatronix_suspend, sdhci_evatronix_resume);

static struct platform_driver sdhci_evatronix_driver = {
	.driver = {
		.name = "sdhci-evatronix",
		.owner = THIS_MODULE,
		.of_match_table = sdhci_evatronix_of_match,
		.pm = &sdhc_evatronix_pm_ops,
	},
	.probe = sdhci_evatronix_probe,
	.remove = sdhci_evatronix_remove,
};

module_platform_driver(sdhci_evatronix_driver);

MODULE_DESCRIPTION("Cast Inc. / Evatronix SDIO-HOST");
MODULE_AUTHOR("Pixelworks Inc.");
MODULE_LICENSE("GPL v2");
