/*
   -------------------------------------------------------------------------
   pw2wire-hw-algo.c two wire driver algorithms for Pixelworks Ruby adapters

   Antson Hu, Pixelworks, Inc.
   HZhou, Pixelworks Japan, Inc.

   Copyright 2007-2010 Pixelworks Inc.

   ---------------------------------------------------------------------------
   This file was highly leveraged from pw2wire-hw-algo.c, which was created
   by Antson Hu:


     Copyright (C) 1995-1997 Simon G. Vogl
                   1998-2000 Hans Berglund

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/* ------------------------------------------------------------------------- */

/* With some changes from Kysti Mlkki <kmalkki@cc.hut.fi> and
   Frodo Looijaard <frodol@dds.nl> ,and also from Martin Bailey
   <mbailey@littlefeet-inc.com> */

#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/platform_device.h>
#include <linux/i2c.h>
#include <linux/clk.h>
#include <linux/pm_runtime.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/device.h>
#include <asm/uaccess.h>

#include <mach/platform.h>

#include "types.h"
#include "ral.h"
#include "2wire_v1_ral.h"
#include "pxlw/pw2wire.h"

// FIXME: these should be removed
#define I2C_M_NO_STOP   0x100   /* Force no stop */
#define I2C_M_FORCE_ACK 0x200   /* Force to send ACK when READ */


//Flags to indicate interrupt status
#define INT_STATUS_NOERR    0x00    //Interrupt initial status
#define INT_STATUS_ERR      0x01    //Interrupt status of error
#define INT_STATUS_STOP     0x02    //Interrupt status of normal stop
#define INT_STATUS_NACK     0x04    //Interrupt status of NACK
#define INT_STATUS_READ_FIN 0x08    //Interrupt status of byte read done

//State of processing
#define STATE_SEND      0x00        //State to send commands
#define STATE_CHECK     0x01        //State to check results


enum {
	CMD_START,
	CMD_STOP,
	CMD_WRITE,
	CMD_READ,
	CMD_SNDACK = 7,
	CMD_INVALID
};

struct pw2wire_command {
	unsigned char cmd;
	unsigned char data;
};


struct pw2wire {
	struct platform_device *pdev;
	int irq;
	void *regs;
	struct i2c_adapter adapter;
	struct clk *clk;

	BlockInfo blkinfo;			// Information from user
	int clock_mode;				// Clock speed 1:high 0:low
	unsigned char rthreshold;		// Read FIFO threshold
	unsigned char wthreshold;		// Write FIFO threshold
	unsigned char c2wire_autohold;
	spinlock_t irqlock;			// Lock between interrupt and user space

	unsigned long interrupt_status;		// Interrupt status
        unsigned long err_flag;                 // Driver error information
        unsigned long status;                   // HW bus state


	struct pw2wire_command *cmdbuf;		// Command to be send to MWFIFO
	int cmdIndex;				// Index to current command
	int cmdIndexLast;			// Last index of command
	int cmdIndexMax;			// Max index of command
	unsigned char *recvbuf;			// Receive buffer
	int recvcount;				// Receive counter
	int recvmax;				// Receive max bytes
	int readcmdcnt;				// Count of read commands in WFIFO

	wait_queue_head_t wq;				// Wait queue
};


static int pw2wire_xtal_type = 1;        //XTAL TYPE (0-27M, 1-24M)
static int pw2wire_timeout_en = 0;       //TIMEOUT ENABLE(0-Ignore timeout, 1-Timeout enabled)
static int pw2wire_timeout = 4800000;   //TIMEOUT value at 1-Timeout enabled
static int pw2wire_stop_monitor = 500;    //Timeout value to monitor STOP condition in msec 
module_param_named(xtal_type, pw2wire_xtal_type, int, S_IRUGO|S_IWUSR);
module_param_named(timeout_en, pw2wire_timeout_en, int, S_IRUGO|S_IWUSR);
module_param_named(timeout, pw2wire_timeout, int, S_IRUGO|S_IWUSR);
module_param_named(monitor, pw2wire_stop_monitor, int, S_IRUGO|S_IWUSR);


struct pw2wire_speedconfig {
	u32   speed_hz;
	u16   value;
};

static const struct pw2wire_speedconfig _speedconfig[] =
{/*
	{ 100000, 0x00 },
	{  25000, 0x01 },
	{  50000, 0x02 },
	{  75000, 0x03 },
	{ 150000, 0x04 },
	{ 200000, 0x05 },
	{ 250000, 0x06 },
	{ 275000, 0x07 },
	{ 300000, 0x08 },
	{ 325000, 0x09 },
	{ 350000, 0x0A },
	{ 375000, 0x0B },
	{ 400000, 0x0C },
	{ 600000, 0x0D },
	{ 800000, 0x0E },
	{1000000, 0x0F },
*/
        {FREQ_100KHZ, 0x00},
        {FREQ_25KHZ,  0x01},
        {FREQ_50KHZ,  0x02},
        {FREQ_75KHZ,  0x03},
        {FREQ_150KHZ, 0x04},
        {FREQ_200KHZ, 0x05},
        {FREQ_250KHZ, 0x06},
        {FREQ_275KHZ, 0x07},
        {FREQ_300KHZ, 0x08},
        {FREQ_325KHZ, 0x09},
        {FREQ_350KHZ, 0x0A},
        {FREQ_375KHZ, 0x0B},
        {FREQ_400KHZ, 0x0C},
        {FREQ_600KHZ, 0x0D},
        {FREQ_800KHZ, 0x0E},
        {FREQ_1000KHZ, 0x0F},
        {FREQ_DEFAULT, 0x00},       //Last item as default setting

};
#define NUM_OF_SPEEDITEMS ARRAY_SIZE(_speedconfig)

static int get_clock_mode(u32 speed_hz)
{
	int i;
	int ret = 0; // default is 100kHz

	for (i = 0; i < NUM_OF_SPEEDITEMS; i++) {
		if (_speedconfig[i].speed_hz == speed_hz) {
			ret = _speedconfig[i].value;
			break;
		}
	}
	return ret;
}

static void pw2wire_master_enable(struct pw2wire *pw2wire)
{
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MASTER_EN(0x1));
}

static void pw2wire_master_disable(struct pw2wire *pw2wire)
{
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MASTER_EN(0x0));
}

static void pw2wire_disable_interrupt(struct pw2wire *pw2wire)
{
	TW_V1_INTEN_SET(&pw2wire->blkinfo,
			MNACK(0x0),
			MBYRCVD(0x0),
			MARBLOST(0x0),
			MBYCDONE(0x0),
			MRFLIM(0x0),
			MWFLIM(0x0),
			MILLCMD(0x0),
			MHOLD(0x0),
			SNACK(0x0),
			SBYRCVD(0x0),
			SWRREQ(0x0),
			SRDREQ(0x0),
			SRFLIM(0x0),
			SWFLIM(0x0),
			SWFLOCKED(0x0),
			SGENCALL(0x0),
			SHOLD(0x0),
			START(0x0),
			STOP(0x0));
}

static void pw2wire_enable_interrupt(struct pw2wire *pw2wire, int rwflg)
{
	if (rwflg == 0) {	//Receive
		TW_V1_INTEN_SET(&pw2wire->blkinfo,
				MNACK(0x1),
				MBYRCVD(0x1),
				MARBLOST(0x1),
				MBYCDONE(0x1),
				MILLCMD(0x1),
				STOP(0x1));
	} else {
		TW_V1_INTEN_SET(&pw2wire->blkinfo,
				MNACK(0x1),
				MARBLOST(0x1),
				MBYCDONE(0x1),
				MILLCMD(0x1),
				STOP(0x1));
	}
}

static void pw2wire_master_reset(struct pw2wire *pw2wire)
{
	//Reset control logic
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MASTER_RST(0x1));
	//Reset write FIFO
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MWFRST(0x1));
	//Reset read FIFO
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MRFRST(0x1));
	//Clear halt status
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,CLRHLT(0x1));

	//Clear all interrupt status
	TW_V1_INTCLR_SET(&pw2wire->blkinfo,
			 MNACK(0x1),
			 MBYRCVD(0x1),
			 MARBLOST(0x1),
			 MBYCDONE(0x1),
			 MRFLIM(0x1),
			 MWFLIM(0x1),
			 MILLCMD(0x1),
			 MHOLD(0x1),
			 SNACK(0x1),
			 SBYRCVD(0x1),
			 SWRREQ(0x1),
			 SRDREQ(0x1),
			 SRFLIM(0x1),
			 SWFLIM(0x1),
			 SWFLOCKED(0x1),
			 SGENCALL(0x1),
			 SHOLD(0x1),
			 START(0x1),
			 STOP(0x1));
}

static void pw2wire_master_init(struct pw2wire *pw2wire)
{
	pw2wire_master_disable(pw2wire);
	pw2wire_master_reset(pw2wire);

#if defined(AUTO_HOLD)
	// Enable master with auto-hold and local start/stop
	TW_V1_MCTL_SET(&pw2wire->blkinfo,
		       LCLSTST(0x1),
		       AUTOHOLD(0x1));
#else
	// Enable master with local start/stop
	TW_V1_MCTL_SET(&pw2wire->blkinfo,
		       LCLSTST(0x1),
		       AUTOHOLD(0x0));
#endif

	//Write FIFO threshold. Remaining entries 9->8 will cause INT
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MWFTHR(0x3));
	//Read FIFO threshold. 16 entries full will cause INT
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MRFTHR(0x0));

	/* Set speed */
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,SCLSPEED(pw2wire->clock_mode));

	/* Clear all interrupt status */
	TW_V1_INTCLR_SET(&pw2wire->blkinfo,
			 MNACK(0x1),
			 MBYRCVD(0x1),
			 MARBLOST(0x1),
			 MBYCDONE(0x1),
			 MRFLIM(0x1),
			 MWFLIM(0x1),
			 MILLCMD(0x1),
			 MHOLD(0x1),
			 SNACK(0x1),
			 SBYRCVD(0x1),
			 SWRREQ(0x1),
			 SRDREQ(0x1),
			 SRFLIM(0x1),
			 SWFLIM(0x1),
			 SWFLOCKED(0x1),
			 SGENCALL(0x1),
			 SHOLD(0x1),
			 START(0x1),
			 STOP(0x1));

	pw2wire_master_enable(pw2wire);
}

static int pw2wire_process_fifo(struct pw2wire *pw2wire,
				unsigned short flag, unsigned long timeout)
{
	struct pw2wire_command *current_cmd = NULL;
	unsigned short fifodepth;
	unsigned short fifopending;
	unsigned short rdfifodepth;
	unsigned short rdfifopending;
	unsigned short i;
	int ret   = 0;
	int loop  = 1;
	int state = STATE_SEND;
	struct timeval current_time;
	struct timeval target_time;
	unsigned long irqflags;

	if (flag & I2C_M_RD)
		pw2wire_enable_interrupt(pw2wire, 0);
	else
		pw2wire_enable_interrupt(pw2wire, 1);

	while (loop) {
		switch (state)
		{
		case STATE_SEND:     //Sending command
			//Get FIFO info
			TW_V1_MFIFOSTAT_GET(&pw2wire->blkinfo, WFNE(fifopending), WFSZ(fifodepth),
								RFNE(rdfifopending), RFSZ(rdfifodepth));

			//Check if fifo is full
			if (fifopending == fifodepth || rdfifopending == rdfifodepth)
				//Yield
				cond_resched();
			else {
				spin_lock_irqsave(&pw2wire->irqlock, irqflags);
				TW_V1_MFIFOSTAT_GET(&pw2wire->blkinfo, WFNE(fifopending), WFSZ(fifodepth),
									RFNE(rdfifopending), RFSZ(rdfifodepth));
				for (i = fifopending; i < fifodepth; i++) {
					current_cmd = (struct pw2wire_command *)&pw2wire->cmdbuf[pw2wire->cmdIndex];
					// Check current command
					if (CMD_READ == current_cmd->cmd) {
						// For read command, control the level of WFIFO
						if( i - fifopending >= rdfifodepth - rdfifopending || pw2wire->readcmdcnt == rdfifodepth) {
							// Try again to wait for empty room
							dev_dbg(&pw2wire->pdev->dev, "i=%d, w(%d,%d),r(%d,%d)\n",
									i, fifopending, fifodepth, rdfifopending, rdfifodepth);
							spin_unlock_irqrestore(&pw2wire->irqlock, irqflags);
							do
							{
								// Wait until WFIFO becomes empty
								pw2wire->interrupt_status &= ~INT_STATUS_READ_FIN;
								wait_event_timeout(pw2wire->wq, INT_STATUS_READ_FIN & pw2wire->interrupt_status ,msecs_to_jiffies(30));
								TW_V1_MFIFOSTAT_GET1(&pw2wire->blkinfo, WFNE(fifopending));
							}
							while(fifopending > i/2);
							spin_lock_irqsave(&pw2wire->irqlock, irqflags);
							break;
						}
						pw2wire->readcmdcnt++;
					}
					//Advance the command index
					pw2wire->cmdIndex++;
					
					//Send one command
					dev_dbg(&pw2wire->pdev->dev, "TX (%d, %d)\n",
						current_cmd->cmd, current_cmd->data);
					TW_V1_MWFIFO_SET(&pw2wire->blkinfo, WDATA(current_cmd->data), CMD(current_cmd->cmd));
					if (pw2wire->cmdIndex == pw2wire->cmdIndexLast) { //Last command
						//Finish writing
						state = STATE_CHECK;
						do_gettimeofday(&target_time);
						target_time.tv_usec += timeout;
						//printk(KERN_ERR "timeout: %d\n", timeout);
						while (target_time.tv_usec >= USEC_PER_SEC) {
							target_time.tv_usec -= USEC_PER_SEC;
							target_time.tv_sec += 1;
						}
						break;
					}
					//If received NACK, then stop the communication
					if (pw2wire->interrupt_status & INT_STATUS_NACK) {
						state = STATE_CHECK;
						break;
					}
				}
				spin_unlock_irqrestore(&pw2wire->irqlock, irqflags);
			}
			//Check if error happened
			if (pw2wire->interrupt_status & INT_STATUS_ERR)
				state = STATE_CHECK; //Error status is set in handler
			//If received NACK, then stop the communication
			else if (pw2wire->interrupt_status & INT_STATUS_NACK)
				state = STATE_CHECK;
			break;
		case STATE_CHECK:     //Waiting for result
			//Check if error happened
			if (pw2wire->interrupt_status & INT_STATUS_ERR) {
				dev_dbg(&pw2wire->pdev->dev, "Error happened.\n");
				ret = -EIO;
				loop = 0;   //Error status is set in handler
				break;
			}
			// If received NACK, then force to send STOP.
			if (pw2wire->interrupt_status & INT_STATUS_NACK) {
				dev_dbg(&pw2wire->pdev->dev, "Got NACK\n");
				current_cmd->cmd = CMD_STOP;
				current_cmd->data = 0;
			       ret = -EIO;
#if defined(AUTO_HOLD)
			       TW_V1_MCTL_SET1(&pw2wire->blkinfo,MWFRST(0x1));
			       TW_V1_MCTL_SET1(&pw2wire->blkinfo,MRFRST(0x1));
				//Send stop command
				dev_dbg(&pw2wire->pdev->dev, "TX (%d, %d)\n",
					current_cmd->cmd, current_cmd->data);
				TW_V1_MWFIFO_SET(&pw2wire->blkinfo,
						WDATA(current_cmd->data),
						CMD(current_cmd->cmd));
#endif
			}
			//Check if last command is STOP
			if (current_cmd->cmd == CMD_STOP) {

				if (pw2wire->interrupt_status & INT_STATUS_STOP) {
					dev_dbg(&pw2wire->pdev->dev, "Got STOP\n");
					loop = 0;
			       } else if (pw2wire->interrupt_status & INT_STATUS_NACK) {
				       dev_dbg(&pw2wire->pdev->dev, "After NACK, not stop\n");
				       loop = 0;
				} else {
					wait_event_timeout(pw2wire->wq,pw2wire->interrupt_status,msecs_to_jiffies(timeout / 1000));

					//Wait until STOP appears
					do_gettimeofday(&current_time);
					if( timeval_compare( &current_time, &target_time ) >= 0 &&
					    (pw2wire->interrupt_status & INT_STATUS_STOP) == 0)
					{
						loop = 0;
						ret = -ETIMEDOUT;
						dev_dbg(&pw2wire->pdev->dev,
							"Time out Can't get STOP\n");
					}
				}
			} else {
				//If last command is not a STOP, continue to process next message packet
				loop = 0;
			}
			break;
		}
	}
	pw2wire_disable_interrupt(pw2wire);
	return ret;
}

static void set_cmdbuf(struct pw2wire *pw2wire,
		       unsigned char cmd, unsigned char data)
{
	struct pw2wire_command *cmdbuf;
	int i;

	i = pw2wire->cmdIndexLast;
	cmdbuf = pw2wire->cmdbuf;

	cmdbuf[i].cmd  = cmd;
	cmdbuf[i].data = data;
	pw2wire->cmdIndexLast++;
}

static void pw2wire_parse_message(struct pw2wire *pw2wire,
				  struct i2c_msg *pmsg, int lastflg)
{
	unsigned char addr;
	unsigned short i;
	unsigned char *buf;

	//Check START/RESTART condition
	if ((pmsg->flags & I2C_M_NOSTART) != I2C_M_NOSTART) {
		//Set START
		set_cmdbuf(pw2wire, CMD_START, 0);

		//Set ADDR
		if (pmsg->flags & I2C_M_TEN) {
			//10 bit address
			addr = 0xf0 | ((pmsg->addr >> 7) & 0x03);
			set_cmdbuf(pw2wire, CMD_WRITE, addr);

			set_cmdbuf(pw2wire, CMD_WRITE, pmsg->addr & 0x7f);

			if (pmsg->flags & I2C_M_RD) {
				addr |= 0x01;

				//Set START
				set_cmdbuf(pw2wire, CMD_START, 0);

				//Set address
				set_cmdbuf(pw2wire, CMD_WRITE, addr);
			}
		} else {
			//7 bit address
			addr = pmsg->addr << 1;
			if (pmsg->flags & I2C_M_RD)
				addr |= 0x01;

			if (pmsg->flags & I2C_M_REV_DIR_ADDR)
				addr ^= 1;
			//Set address
			set_cmdbuf(pw2wire, CMD_WRITE, addr);
		}
	}

	//Set read or write data
	if(pmsg->flags & I2C_M_RD) {
		//Set read command
		for (i = 0; i < pmsg->len; i++)
			set_cmdbuf(pw2wire, CMD_READ, 0);
		//ACK and STOP
		if (lastflg) {
			if (!(pmsg->flags & I2C_M_NO_RD_ACK)) {
				if (pmsg->flags & I2C_M_FORCE_ACK)
					set_cmdbuf(pw2wire, CMD_SNDACK, 0); //Force to send ACK
				else
					set_cmdbuf(pw2wire, CMD_SNDACK, 1);  //NACK
			}

			if (!( pmsg->flags & I2C_M_NO_STOP))
				set_cmdbuf(pw2wire, CMD_STOP, 0);
		} else
			set_cmdbuf(pw2wire, CMD_SNDACK, 0);  //ACK
		//Set recv buffer for interrupt handler
		pw2wire->recvbuf   = pmsg->buf;
		pw2wire->recvcount = 0;
		pw2wire->recvmax   = pmsg->len;
		pw2wire->readcmdcnt = 0;
	} else {
		buf = pmsg->buf;

		//Set write command
		for (i = 0; i < pmsg->len; i++) {
			set_cmdbuf(pw2wire, CMD_WRITE, *buf);
			buf++;
		}
		//STOP
		if (lastflg) {
			if (!( pmsg->flags & I2C_M_NO_STOP))
				set_cmdbuf(pw2wire, CMD_STOP, 0);
		}
	}
}


static int pw2wire_xfer_msg(struct pw2wire *pw2wire,
			    struct i2c_msg *msg, int last)
{
	int ret;

	pw2wire->interrupt_status = INT_STATUS_NOERR;

	// Allocate command buffer for message
	pw2wire->cmdIndexMax = msg->len + 8;
	pw2wire->cmdIndex = 0;
	pw2wire->cmdIndexLast = 0;
	pw2wire->cmdbuf = kzalloc(pw2wire->cmdIndexMax *
				  sizeof(struct pw2wire_command), GFP_KERNEL);
	if (!pw2wire->cmdbuf)
		return -ENOMEM;

	// Parse input message into FIFO command
	pw2wire_parse_message(pw2wire, msg, last);

	// Process FIFO command
	ret = pw2wire_process_fifo(pw2wire, msg->flags, pw2wire_stop_monitor*1000);

	// Release cmd buffer
	kfree(pw2wire->cmdbuf);

	if (ret)
		// Failed
		pw2wire_master_reset(pw2wire);

	return ret;
}


static int pw2wire_xfer(struct i2c_adapter *adapter,
			   struct i2c_msg *msgs, int num)
{
	struct pw2wire *pw2wire = i2c_get_adapdata(adapter);
	int i, ret;

	if (!num)
		return 0;

	pm_runtime_get_sync(&pw2wire->pdev->dev);

	// Check first msg only. Don't initialize H/W if special flags are set.
	if (!(msgs[0].flags & (I2C_M_NO_STOP | I2C_M_FORCE_ACK | I2C_M_NO_RD_ACK)))
		pw2wire_master_init(pw2wire);

	for (i = 0; i < num; i++) {
		ret = pw2wire_xfer_msg(pw2wire, &msgs[i], (i + 1) == num);
		if (ret) {
			dev_dbg(&pw2wire->pdev->dev, "returning error %d\n", ret);
			pm_runtime_put(&pw2wire->pdev->dev);
			return ret;
		}
	}
	pm_runtime_put(&pw2wire->pdev->dev);
	dev_dbg(&pw2wire->pdev->dev, "returning OK %d\n", num);
	return num; // Number of message sent
}

static u32 pw2wire_func(struct i2c_adapter *ppw2wire_adapter)
{
	return I2C_FUNC_I2C | I2C_FUNC_10BIT_ADDR | I2C_FUNC_PROTOCOL_MANGLING | I2C_FUNC_SMBUS_EMUL;
}

/* Implements device specific ioctls.  Higher level ioctls can
 * be found in i2c-core.c and are typical of any i2c controller (specifying
 * slave address, timeouts, etc).
 * Locally supported ioctls are:
 *      PW2WIRE_WAITTIME
 *      PW2WIRE_CONFIGURE
 *      PW2WIRE_SPEED
 *      PW2WIRE_RESET
 *      PW2WIRE_STATUS
 */
static int pw2wire_algo_control(struct i2c_adapter *adapter,
    unsigned int cmd, unsigned long arg)
{
    struct pw2wire *pw2wire = i2c_get_adapdata(adapter);        
    struct pw2wire_confinfo confinfo;
    struct pw2wire_fifoinfo fifoinfo;
   // struct pw2wire_timeoutconfinfo timeoutconfinfo;
    int ret = 0;
    uint16 pw2wirespeed;

    //Lock
//    mutex_lock(&pw2wire->transfer_lock);
    switch(cmd)
    {
        case PW2WIRE_CONFIGURE:
            memset(&confinfo,0,sizeof(struct pw2wire_confinfo));
            if(copy_from_user(&confinfo, (struct pw2wire_confinfo *)arg, sizeof(struct pw2wire_confinfo)))
            {
                pw2wire->status = STATUS_ERR;
                pw2wire->err_flag |= ERR_REQUEST_RESOURCE;
                dev_err(&pw2wire->pdev->dev, "ioctl PW2WIRE_CONFIGURE failed.\n");
                ret = -EFAULT;
                break;
            }
            pw2wire->clock_mode = get_clock_mode(confinfo.u2wire_speed);
            pw2wire->rthreshold = confinfo.rthreshold;
            pw2wire->wthreshold = confinfo.wthreshold;
            pw2wire->c2wire_autohold = confinfo.c2wire_autohold;
            break;
        case PW2WIRE_SPEED:
            if(copy_from_user(&pw2wirespeed, (uint16 *)arg, sizeof(uint16)))
            {
                pw2wire->status = STATUS_ERR;
                pw2wire->err_flag |= ERR_REQUEST_RESOURCE;
                dev_err(&pw2wire->pdev->dev, "ioctl PW2WIRE_SPEED failed.\n");
                ret = -EFAULT;
                break;
            }
            pw2wire->clock_mode = get_clock_mode(pw2wirespeed);
            break;
        case PW2WIRE_RESET:
            //Reset HW
            pw2wire_master_reset(pw2wire);

            pw2wire->status = STATUS_OK;
            pw2wire->err_flag = ERR_NO_ERR;
            break;
        case PW2WIRE_STATUS:
            memset(&fifoinfo,0,sizeof(struct pw2wire_fifoinfo));
            //Get FIFO status
            TW_V1_MFIFOSTAT_GET(&pw2wire->blkinfo,
                                RFSZ(fifoinfo.rfifodepth),
                                WFSZ(fifoinfo.wfifodepth),
                                RFNE(fifoinfo.rfifocnt),
                                WFNE(fifoinfo.wfifocnt));
            fifoinfo.bussts= pw2wire->status;
            fifoinfo.errflag = pw2wire->err_flag;
            if (copy_to_user((void __user *) arg, &fifoinfo, sizeof(struct pw2wire_fifoinfo)))
            {
                pw2wire->status = STATUS_ERR;
                pw2wire->err_flag |= ERR_REQUEST_RESOURCE;
                dev_err(&pw2wire->pdev->dev, "ioctl W2WIRE_STATUS failed.\n");
                ret = -EFAULT;
            }
            break;       
        } 
    //Unlock
     //mutex_unlock(&pw2wire->transfer_lock);

    return ret;
}

static struct i2c_algorithm pw2wire_algo = {
	.master_xfer    = pw2wire_xfer,
        .algo_control   = pw2wire_algo_control,
	.functionality  = pw2wire_func,
};


/*
 * This is the interrupt handler
 */
static irqreturn_t pw2wire_handler(int this_irq, void *dev_id)
{
	struct pw2wire *pw2wire = (struct pw2wire *)dev_id;
	unsigned long interrupt_status;
	unsigned char read_fifo_len;
	unsigned char read_cnt;

	TW_V1_INTSTAT_GET(&pw2wire->blkinfo, ALL(interrupt_status));
	//printk(KERN_ERR "interrupt_status[0x%x]\n", interrupt_status);
	//Check error
	if( interrupt_status & TW_V1_INTSTAT_m_STOP() )
	{
		pw2wire->interrupt_status |= INT_STATUS_STOP;
		TW_V1_INTCLR_SET(&pw2wire->blkinfo, STOP(0x1));
	}
	if (interrupt_status & TW_V1_INTSTAT_m_MARBLOST() ||
	    interrupt_status & TW_V1_INTSTAT_m_MILLCMD()) {
		pw2wire->interrupt_status |= INT_STATUS_ERR;
		if (interrupt_status & TW_V1_INTSTAT_m_MARBLOST())
			TW_V1_INTCLR_SET(&pw2wire->blkinfo, MARBLOST(0x1));

		if (interrupt_status & TW_V1_INTSTAT_m_MILLCMD())
			TW_V1_INTCLR_SET(&pw2wire->blkinfo, MILLCMD(0x1));
	}
	if (interrupt_status & TW_V1_INTSTAT_m_MNACK()) {
	       // reset read and write FIFO
	       TW_V1_MCTL_SET1(&pw2wire->blkinfo,MWFRST(0x1));
	       TW_V1_MCTL_SET1(&pw2wire->blkinfo,MRFRST(0x1));
	       pw2wire->interrupt_status |= INT_STATUS_NACK;
	       if (interrupt_status & TW_V1_INTSTAT_m_MNACK())
		       TW_V1_INTCLR_SET(&pw2wire->blkinfo, MNACK(0x1));
	}
	if (interrupt_status & TW_V1_INTSTAT_m_MBYCDONE())
		TW_V1_INTCLR_SET(&pw2wire->blkinfo, MBYCDONE(0x1));
	if (interrupt_status & TW_V1_INTSTAT_m_MBYRCVD()) {
		pw2wire->interrupt_status |= INT_STATUS_READ_FIN;
		TW_V1_INTCLR_SET(&pw2wire->blkinfo, MBYRCVD(0x1));
		TW_V1_MFIFOSTAT_GET1(&pw2wire->blkinfo,RFNE(read_fifo_len));
		for (read_cnt = 0; read_cnt < read_fifo_len; read_cnt++) {
			//Save the byte
			if (pw2wire->recvcount < pw2wire->recvmax) {
				TW_V1_MRFIFO_GET(&pw2wire->blkinfo, ALL(pw2wire->recvbuf[pw2wire->recvcount]));
				pw2wire->recvcount++;
				pw2wire->readcmdcnt--;
			}
		}
	}

	// Clear all interrupt status
	TW_V1_INTCLR_SET(&pw2wire->blkinfo,
			 MRFLIM(0x1),
			 MWFLIM(0x1),
			 MHOLD(0x1),
			 SNACK(0x1),
			 SBYRCVD(0x1),
			 SWRREQ(0x1),
			 SRDREQ(0x1),
			 SRFLIM(0x1),
			 SWFLIM(0x1),
			 SWFLOCKED(0x1),
			 SGENCALL(0x1),
			 SHOLD(0x1));

	wake_up(&pw2wire->wq);

	return IRQ_HANDLED;
}


static int pw2wire_probe(struct platform_device *pdev)
{
	struct pw2wire *pw2wire;
	struct i2c_adapter *adapter = NULL;
	struct resource *res;
	struct device_node *np = pdev->dev.of_node;
	int result = 0;
	u32 i2c_id;
	u32 speed_hz;

	if (of_property_read_u32(np, "i2c-id", &i2c_id)) {
		dev_err(&pdev->dev, "i2c-id is not set\n");
		return -EINVAL;
	}
	if (of_property_read_u32(np, "clock-frequency", &speed_hz)) {
		dev_err(&pdev->dev, "clock-frequency is not set\n");
		return -EINVAL;
	}
	printk(KERN_INFO "pw2wire: %s%d, clock %uHz\n",
	       pdev->name, i2c_id, speed_hz);

	pw2wire = devm_kzalloc(&pdev->dev, sizeof(struct pw2wire), GFP_KERNEL);
	if (!pw2wire)
		return -ENOMEM;

	pw2wire->pdev = pdev;
	pw2wire->clock_mode = get_clock_mode(speed_hz);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "pw2wire_probe - no memory resource.\n");
		return -ENOENT;
	}
	if (!devm_request_mem_region(&pdev->dev, res->start,
				     resource_size(res), res->name)) {
		dev_err(&pdev->dev, "can't get I/O mem address 0x%x\n", res->start);
		return -EBUSY;
	}
	pw2wire->regs = devm_ioremap(&pdev->dev, res->start, resource_size(res));
	if (!pw2wire->regs) {
		dev_err(&pdev->dev, "can't map I/O mem address 0x%x\n", res->start);
		return -ENOMEM;
	}

	pw2wire->clk = clk_get(&pdev->dev, NULL);
	if (IS_ERR(pw2wire->clk)) {
		dev_err(&pdev->dev, "can't find clock\n");
		return PTR_ERR(pw2wire->clk);
	}
	clk_prepare_enable(pw2wire->clk);

	RAL_INIT(&pw2wire->blkinfo,(uint32)pw2wire->regs, NULL);
	pw2wire_master_reset(pw2wire);
	pw2wire_master_enable(pw2wire);
	TW_V1_CRYSTAL_FREQ_SET1(&pw2wire->blkinfo,XTAL_TYPE(pw2wire_xtal_type));
	TW_V1_CRYSTAL_FREQ_SET1(&pw2wire->blkinfo,TIMEOUT_EN(pw2wire_timeout_en));
        TW_V1_TIMEOUT_SET1(&pw2wire->blkinfo,TIMEOUT_VAL(pw2wire_timeout));

	pw2wire->irq = platform_get_irq(pdev, 0);
	if (pw2wire->irq < 0) {
		dev_err(&pdev->dev, "pw2wire_probe - no IO resource.\n");
		result = pw2wire->irq;
		goto err;
	}

	if (pw2wire->irq) {
		result = devm_request_irq(&pdev->dev, pw2wire->irq, pw2wire_handler, 0,
					  dev_name(&pdev->dev), pw2wire);
		if (result) {
			dev_err(&pdev->dev, "can't get assigned irq %d\n", pw2wire->irq);
			goto err;
		}
	}

	init_waitqueue_head(&pw2wire->wq);

	spin_lock_init(&pw2wire->irqlock);

	platform_set_drvdata(pdev, pw2wire);

	adapter = &pw2wire->adapter;
	strlcpy(adapter->name, "pw2wire adapter", sizeof(adapter->name));
	adapter->owner = THIS_MODULE;
	adapter->algo = &pw2wire_algo;
	adapter->nr = i2c_id;
	adapter->dev.parent = &pdev->dev;
	i2c_set_adapdata(adapter, pw2wire);

	result = i2c_add_numbered_adapter(adapter);
	if (result) {
		dev_err(&pdev->dev, "pw2wire_probe - failed to add adapter.\n");
		goto err;
	}

	pm_runtime_enable(&pdev->dev);
	dev_info(&pdev->dev, "hw 2wire probe OK.\n");
	return 0;

err:
	clk_disable_unprepare(pw2wire->clk);
	clk_put(pw2wire->clk);
	return result;
}

static int pw2wire_remove(struct platform_device *pdev)
{
	struct pw2wire *pw2wire = NULL;

	pw2wire = platform_get_drvdata(pdev);
	i2c_del_adapter(&pw2wire->adapter);
	platform_set_drvdata(pdev, NULL);

	pw2wire_master_reset(pw2wire);
	pw2wire_master_disable(pw2wire);
	clk_disable_unprepare(pw2wire->clk);
	clk_put(pw2wire->clk);
	return 0;
}

#ifdef CONFIG_PM
static int pw2wire_suspend(struct device *dev)
{
	struct pw2wire *pw2wire = dev_get_drvdata(dev);
	clk_disable(pw2wire->clk);
	return 0;
}

static int pw2wire_resume(struct device *dev)
{
	struct pw2wire *pw2wire = dev_get_drvdata(dev);
	clk_enable(pw2wire->clk);
	return 0;
}
#endif

static struct dev_pm_ops pw2wire_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(pw2wire_suspend, pw2wire_resume)
	SET_RUNTIME_PM_OPS(pw2wire_suspend, pw2wire_resume, NULL)
};


static struct of_device_id pw2wire_of_match[]={
	{ .compatible = "pixelworks,two_wire_v1" },
	{ },
};

static struct platform_driver pw2wire_driver = {
	.remove     = pw2wire_remove,
	.driver     = {
		.name       = "pw2wire",
		.owner = THIS_MODULE,
		.of_match_table = pw2wire_of_match,
		.pm = &pw2wire_pm_ops,
	}
};

static int __init pw2wire_init(void)
{
	return platform_driver_probe(&pw2wire_driver, pw2wire_probe);
}
subsys_initcall(pw2wire_init);

static void __exit pw2wire_exit(void)
{
	platform_driver_unregister(&pw2wire_driver);
}
module_exit(pw2wire_exit);

MODULE_AUTHOR("Pixelworks, Inc.");
MODULE_DESCRIPTION("Pixelworks two_wire master driver");
MODULE_LICENSE("GPL");
