/*
 * SoC Solutions IPC AMBA Timer
 * based on linux/arch/arm/common/timer-sp.c
 * rewrite based on linux/drivers/clocksource/timer-sp.c
 *
 *  Copyright (C) 1999 - 2003 ARM Limited
 *  Copyright (C) 2000 Deep Blue Solutions Ltd
 *  Copyright (C) 2012-2016 Pixelworks Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/clk.h>
#include <linux/clocksource.h>
#include <linux/clockchips.h>
#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/sched_clock.h>

#include <mach/ipc_timer.h>


static void __iomem *sched_clock_base;

static u64 ipctimer_read(void)
{
	return ~readl_relaxed(sched_clock_base + IPC_TMR_VALUE);
}

static void __init ipctimer_clocksource_and_sched_clock_init(void __iomem *base,
							     const char *name,
							     long rate)
{
	/* setup timer 0 as free-running clocksource */
	writel(0, base + IPC_TMR_CONTROL);
	writel(0xffffffff, base + IPC_TMR_LOAD);
	writel(0xffffffff, base + IPC_TMR_VALUE);
	writel(IPC_TMR_CONTROL_PRESCALE_NONE |
	       IPC_TMR_CONTROL_MODE_FREERUNNING | IPC_TMR_CONTROL_ENABLE,
	       base + IPC_TMR_CONTROL);

	clocksource_mmio_init(base + IPC_TMR_VALUE, name,
		rate, 200, 32, clocksource_mmio_readl_down);

	sched_clock_base = base;
	sched_clock_register(ipctimer_read, 32, rate);
}


static void __iomem *clkevt_base;
static unsigned long clkevt_reload;

/*
 * IRQ handler for the timer
 */
static irqreturn_t ipctimer_timer_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *evt = dev_id;

	/* clear the interrupt */
	writel(1, clkevt_base + IPC_TMR_CLEAR);

	evt->event_handler(evt);

	return IRQ_HANDLED;
}

static int ipctimer_set_periodic(struct clock_event_device *evt)
{
	unsigned long ctrl = 0;

	writel(ctrl, clkevt_base + IPC_TMR_CONTROL);
	writel(clkevt_reload, clkevt_base + IPC_TMR_LOAD);
	ctrl = IPC_TMR_CONTROL_MODE_PERIODIC | IPC_TMR_CONTROL_ENABLE;
	writel(ctrl, clkevt_base + IPC_TMR_CONTROL);
	return 0;
}

static int ipctimer_shutdown(struct clock_event_device *evt)
{
	writel(0L, clkevt_base + IPC_TMR_CONTROL);
	return 0;
}

static int ipctimer_set_next_event(unsigned long next,
	struct clock_event_device *evt)
{
	unsigned long ctrl = readl(clkevt_base + IPC_TMR_CONTROL);

	writel(next, clkevt_base + IPC_TMR_LOAD);
	/* note: timer generates irqs only in periodic mode */
	writel(ctrl | IPC_TMR_CONTROL_ENABLE | IPC_TMR_CONTROL_MODE_PERIODIC,
	       clkevt_base + IPC_TMR_CONTROL);
	return 0;
}

static struct clock_event_device ipctimer_clockevent = {
	.features		= CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT,
	.set_state_periodic	= ipctimer_set_periodic,
	.set_state_oneshot	= ipctimer_shutdown,
	.set_state_shutdown	= ipctimer_shutdown,
	.tick_resume		= ipctimer_shutdown,
	.set_next_event		= ipctimer_set_next_event,
	.rating			= 300,
};

static struct irqaction ipctimer_timer_irq = {
	.name		= "timer",
	.flags		= IRQF_TIMER | IRQF_IRQPOLL,
	.handler	= ipctimer_timer_interrupt,
	.dev_id		= &ipctimer_clockevent,
};

void __init ipctimer_clockevents_init(void __iomem *base, unsigned int irq,
				      const char *name, long rate)
{
	struct clock_event_device *evt = &ipctimer_clockevent;

	clkevt_base = base;
	clkevt_reload = DIV_ROUND_CLOSEST(rate, HZ);
	evt->name = name;
	evt->irq = irq;

	setup_irq(irq, &ipctimer_timer_irq);
	clockevents_config_and_register(evt, rate, 0xf, 0xffffffff);
}

static void __init ipctimer_of_init(struct device_node *np)
{
	void __iomem *base;
	int irq;
	bool clkevt, clksrc;
	struct clk *clk;
	long rate;
	int err;

	clkevt = of_property_read_bool(np, "clockevents");
	clksrc = of_property_read_bool(np, "clocksource");
	if (!clkevt && !clksrc)
		return;

	base = of_iomap(np, 0);
	if (WARN_ON(!base))
		return;
	irq = irq_of_parse_and_map(np, 0);
	if (WARN_ON(irq <= 0)) {
		iounmap(base);
		return;
	}

	clk = of_clk_get(np, 0);
	if (WARN_ON(IS_ERR(clk)))
		return;
	err = clk_prepare(clk);
	if (err) {
		pr_err("ipctimer: %s clock failed to prepare: %d\n", np->name, err);
		clk_put(clk);
		return;
	}

	err = clk_enable(clk);
	if (err) {
		pr_err("ipctimer: %s clock failed to enable: %d\n", np->name, err);
		clk_unprepare(clk);
		clk_put(clk);
		return;
	}

	rate = clk_get_rate(clk);
	if (rate < 0) {
		pr_err("ipctimer: %s clock failed to get rate: %ld\n", np->name, rate);
		clk_disable(clk);
		clk_unprepare(clk);
		clk_put(clk);
		return;
	}

	if (clksrc)
		ipctimer_clocksource_and_sched_clock_init(base, np->name, rate);
	if (clkevt)
		ipctimer_clockevents_init(base, irq, np->name, rate);
}
CLOCKSOURCE_OF_DECLARE(ipctimer, "socsolutions,ipc-timer", ipctimer_of_init);
