/*
 * Copyright (C) 2013 Pixelworks, Inc.
 * based on linux-3.6.3 clk-fixed-factor.c:
 * Copyright (C) 2011 Sascha Hauer, Pengutronix <s.hauer@pengutronix.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Standard functionality for the common clock API.
 */
#include <linux/module.h>
#include <linux/clk-provider.h>
#include <linux/slab.h>
#include <linux/err.h>
#include <linux/clk.h>
#include "clk-fixed-div.h"

/*
 * DOC: basic fixed divider clock that cannot gate
 *
 * Traits of this clock:
 * prepare - clk_prepare only ensures that parents are prepared
 * enable - clk_enable only ensures that parents are enabled
 * rate - rate is fixed.  clk->rate = parent->rate / div
 * parent - fixed parent.  No clk_set_parent support
 */

#define to_clk_fixed_divider(_hw) container_of(_hw, struct clk_fixed_divider, hw)

static unsigned long clk_divider_recalc_rate(struct clk_hw *hw,
               unsigned long parent_rate)
{
       struct clk_fixed_divider *fix = to_clk_fixed_divider(hw);

       return parent_rate * fix->div;
}

static long clk_divider_round_rate(struct clk_hw *hw, unsigned long rate,
                               unsigned long *prate)
{
       struct clk_fixed_divider *fix = to_clk_fixed_divider(hw);

       if (__clk_get_flags(hw->clk) & CLK_SET_RATE_PARENT) {
               unsigned long best_parent;

               best_parent = rate * fix->div;
               *prate = clk_round_rate(clk_get_parent(hw->clk),
                               best_parent);
       }

       return *prate / fix->div;
}

static int clk_divider_set_rate(struct clk_hw *hw, unsigned long rate,
				unsigned long parent_rate)
{
       return 0;
}

struct clk_ops clk_fixed_divider_ops = {
       .round_rate = clk_divider_round_rate,
       .set_rate = clk_divider_set_rate,
       .recalc_rate = clk_divider_recalc_rate,
};

struct clk *clk_register_fixed_divider(struct device *dev, const char *name,
               const char *parent_name, unsigned long flags,
               unsigned int div)
{
       struct clk_fixed_divider *fix;
       struct clk *clk;
       struct clk_init_data init;

       fix = kmalloc(sizeof(*fix), GFP_KERNEL);
       if (!fix) {
               pr_err("%s: could not allocate fixed divider clk\n", __func__);
               return NULL;
       }

       /* struct clk_fixed_divider assignments */
       fix->div = div;

       if (parent_name) {
               fix->parent[0] = kstrdup(parent_name, GFP_KERNEL);
               if (!fix->parent[0])
                       goto out;
       }

       init.name = name;
       init.ops = &clk_fixed_divider_ops;
       init.parent_names = &parent_name;
       init.num_parents = 1;
       init.flags = flags;
       fix->hw.init = &init;

       clk = clk_register(dev, &fix->hw);
       if (clk)
               return clk;

out:
       kfree(fix->parent[0]);
       kfree(fix);
       return NULL;
}

static void __init pxlw_clk_fixed_div_init(struct device_node *np)
{
	struct clk *clk;
	const char *name = np->name;
	const char *parent_name;
	u32 div;
	int rc;

	of_property_read_string(np, "clock-output-names", &name);
	parent_name = of_clk_get_parent_name(np, 0);

	rc = of_property_read_u32(np, 0, &div);
	if (rc) {
		pr_err("%s: can't get registers\n", np->name);
		return;
	}
	clk = clk_register_fixed_divider(NULL, name, parent_name, 0, div);
	if (!IS_ERR(clk))
		of_clk_add_provider(np, of_clk_src_simple_get, clk);
}
CLK_OF_DECLARE(pxlw_clk_fixed_div, "pxlw,clk_fixed_div", pxlw_clk_fixed_div_init);
