/*
 *  Copyright (C) 2012,2013 Pixelworks, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/suspend.h>
#include <linux/syscore_ops.h>
#include <linux/io.h>
#include <linux/kobject.h>
#include <linux/sysfs.h>
#include <asm/cacheflush.h>
#include <asm/suspend.h>
#include <asm/idmap.h>
#include <asm/system_misc.h>
#include <mach/hardware.h>

extern void topazeh_s2ram(void);
extern int topazeh_suspend_fn(unsigned long arg);
extern u32 topazeh_s2ram_size;

unsigned long topazeh_suspend_itcm_phys;
unsigned long topazeh_suspend_dtcm_phys;
static void *topazeh_suspend_itcm;
static void *topazeh_suspend_dtcm;

#define SUSPEND_ITCM_ADDR 0x20000000
#define SUSPEND_DTCM_ADDR 0x20010000


static int topazeh_pm_suspend(void)
{
	printk(KERN_DEBUG "topazeh_pm_suspend\n");
	return 0;
}

static void topazeh_pm_resume(void)
{
	printk(KERN_DEBUG "topazeh_pm_resume\n");
}

static struct syscore_ops topazeh_pm_syscore_ops = {
	.suspend	= topazeh_pm_suspend,
	.resume		= topazeh_pm_resume,
};


static u8 *topazeh_mode6_lp_code;
static u32 topazeh_mode6_lp_code_len;
static u32 topazeh_mode6_lp_code_alloc;

/* pointer to LP image at reserved vector location in boot image */
#define topazeh_mode6_lp_code_ptr 0x14

struct topazeh_mode6_lp_code_hdr {
	u32	res_b_instr;
	u32	magic;
#define TOPAZEH_LP_MAGIC 0x4d49504c /* "LPIM" */
	u32	itcm_addr;
	u32	itcm_len;	/* including this hdr */
	u32	dtcm_addr;
	u32	dtcm_len;
};

static ssize_t topazeh_lp_img_read(struct file *filp, struct kobject *kobj,
				    struct bin_attribute *bin_attr,
				    char *buf, loff_t off, size_t count)
{
	if (off >= topazeh_mode6_lp_code_len)
		return 0;
	if (count > topazeh_mode6_lp_code_len - off)
		count = topazeh_mode6_lp_code_len - off;
	memcpy(buf, topazeh_mode6_lp_code + off, count);
	return count;
}

static ssize_t topazeh_lp_img_write(struct file *filp, struct kobject *kobj,
				     struct bin_attribute *bin_attr,
				     char *buf, loff_t off, size_t count)
{
	if (!off) {
		kfree(topazeh_mode6_lp_code);
		topazeh_mode6_lp_code = NULL;
		topazeh_mode6_lp_code_len = 0;
		topazeh_mode6_lp_code_alloc = 0;
	}
	if (off + count > topazeh_mode6_lp_code_alloc) {
		u8 *new = kmalloc(round_up(off + count, 8192), GFP_KERNEL);
		if (!new)
			return -ENOMEM;
		topazeh_mode6_lp_code_alloc = round_up(off + count, 8192);
		memcpy(new, topazeh_mode6_lp_code, topazeh_mode6_lp_code_len);
		kfree(topazeh_mode6_lp_code);
		topazeh_mode6_lp_code = new;
	}
	memcpy(topazeh_mode6_lp_code + off, buf, count);
	topazeh_mode6_lp_code_len += count;
	return count;
}

static struct bin_attribute lp_img_attribute = {
	.attr = {
		.name = "lp_img",
		.mode = 0600,
	},
	.read = topazeh_lp_img_read,
	.write = topazeh_lp_img_write,
};

static __init int topazeh_pm_mode6_init(void)
{
	int rc = sysfs_create_bin_file(kernel_kobj, &lp_img_attribute);
	if (rc)
		printk(KERN_ERR "topazeh_pm_mode6_init: sysfs err %d\n", rc);
	return 0;
}
late_initcall(topazeh_pm_mode6_init);


static int topazeh_suspend_valid(suspend_state_t state)
{
	return state == PM_SUSPEND_MEM || state == PM_SUSPEND_STANDBY;
}

static u32 topazeh_tcm_get_size(u32 bank, int itcm)
{
	u32 tcm, tcm_size;

	asm("mcr p15, 0, %0, c9, c2, 0" : : "r" (bank));
	if (itcm)
		asm("mrc p15, 0, %0, c9, c1, 1" : "=r" (tcm));
	else
		asm("mrc p15, 0, %0, c9, c1, 0" : "=r" (tcm));
	tcm_size = (tcm >> 2) & 0xf;
	if ((tcm_size < 3) || (tcm_size > 6))
		tcm_size = 0;
	else
		tcm_size = 1 << (tcm_size - 1 + 10);
	return tcm_size;
}

static void topazeh_tcm_enable(u32 bank, int itcm, u32 addr)
{
	u32 tcm = addr | 1;

	asm("mcr p15, 0, %0, c9, c2, 0" : : "r" (bank));
	if (itcm)
		asm("mcr p15, 0, %0, c9, c1, 1" : : "r" (tcm));
	else
		asm("mcr p15, 0, %0, c9, c1, 0" : : "r" (tcm));
}

static int topazeh_suspend_begin(suspend_state_t state)
{
	u32 itcm_addr, dtcm_addr, itcm_size, dtcm_size;
	struct topazeh_mode6_lp_code_hdr *h;
	u32 off;

	printk(KERN_DEBUG "topazeh_suspend_begin\n");

	dtcm_size = topazeh_tcm_get_size(0, 0);
	dtcm_size += topazeh_tcm_get_size(1, 0);
	itcm_size = topazeh_tcm_get_size(0, 1);
	itcm_size += topazeh_tcm_get_size(1, 1);

	if (state == PM_SUSPEND_STANDBY) {
		if (!topazeh_mode6_lp_code
		    || topazeh_mode6_lp_code_len < topazeh_mode6_lp_code_ptr + 4)
			return -ENOENT;
		off = readl(topazeh_mode6_lp_code + topazeh_mode6_lp_code_ptr);
		if (!off || (off & 3) || off + sizeof(struct topazeh_mode6_lp_code_hdr)
					> topazeh_mode6_lp_code_len)
			return -EINVAL;
		h = (struct topazeh_mode6_lp_code_hdr *)(topazeh_mode6_lp_code + off);
		if (h->magic != TOPAZEH_LP_MAGIC
		    || h->itcm_len + off > itcm_size
		    || h->dtcm_len > dtcm_size
		    || h->itcm_len + h->dtcm_len + off > topazeh_mode6_lp_code_len)
			return -ENOEXEC;
		itcm_addr = h->itcm_addr;
		dtcm_addr = h->dtcm_addr;
	} else {
		if (topazeh_s2ram_size > itcm_size) {
			printk(KERN_ERR "topazeh_suspend_begin: ITCM too small"
			       "(%d < %u)\n", itcm_size, topazeh_s2ram_size);
			return -ENOMEM;
		}
		itcm_addr = SUSPEND_ITCM_ADDR;
		dtcm_addr = SUSPEND_DTCM_ADDR;
	}
	topazeh_suspend_itcm_phys = itcm_addr;
	topazeh_suspend_dtcm_phys = dtcm_addr;

	topazeh_suspend_itcm = __arm_ioremap_exec(itcm_addr, itcm_size, 0);
	if (!topazeh_suspend_itcm) {
		printk(KERN_ERR "topazeh_suspend_begin: can't map ITCM\n");
		return -ENOMEM;
	}

	topazeh_suspend_dtcm = ioremap(dtcm_addr, dtcm_size);
	if (!topazeh_suspend_dtcm) {
		printk(KERN_ERR "topazeh_suspend_begin: can't map DTCM\n");
		iounmap(topazeh_suspend_itcm);
		return -ENOMEM;
	}
	memset(topazeh_suspend_itcm, '\0', itcm_size);
	memset(topazeh_suspend_dtcm, '\0', dtcm_size);
	return 0;
}

static int topazeh_suspend_prepare(void)
{
	printk(KERN_DEBUG "topazeh_suspend_prepare\n");
	topazeh_tcm_enable(0, 1, topazeh_suspend_itcm_phys);
	topazeh_tcm_enable(1, 1, topazeh_suspend_itcm_phys
			    + topazeh_tcm_get_size(0, 1));
	topazeh_tcm_enable(0, 0, topazeh_suspend_dtcm_phys);
	topazeh_tcm_enable(1, 0, topazeh_suspend_dtcm_phys
			    + topazeh_tcm_get_size(0, 0));
	return 0;
}


static int topazeh_suspend_enter(suspend_state_t state)
{
	struct topazeh_mode6_lp_code_hdr *h;
	u32 off;
	int rc;

	printk(KERN_DEBUG "topazeh_suspend_enter\n");
	if (state == PM_SUSPEND_STANDBY) {
		/* go to mode 6 (EUP standby): load LP code into
		 * ITCM and DTCM, disable L1 cache and MMU and execute LP code
		 */
		off = readl(topazeh_mode6_lp_code + topazeh_mode6_lp_code_ptr);
		h = (struct topazeh_mode6_lp_code_hdr *)(topazeh_mode6_lp_code + off);
		memcpy(topazeh_suspend_itcm, topazeh_mode6_lp_code, off + h->itcm_len);
		memcpy(topazeh_suspend_dtcm,
		       topazeh_mode6_lp_code + off + h->itcm_len, h->dtcm_len);
		writel(0x4c4f5750 /* "LOWP" */, __io_address(STATIC_REGS_BASE));
		soft_restart(topazeh_suspend_itcm_phys);
		/* never get here */
	}

	/* mode 5 (suspend to RAM) */
	/* copy the suspend code which enables DRAM self-refreshto ITCM */
	memcpy(topazeh_suspend_itcm, topazeh_s2ram, topazeh_s2ram_size);

	/* disable L2 cache */
	outer_flush_all();
	outer_disable();

	/* enable identity mapping so MMU can be turned off */
	setup_mm_for_reboot();
	/* Clean and invalidate caches */
	flush_cache_all();
	/* Turn off caching */
	cpu_proc_fin();
	/* Push out any further dirty data, and ensure cache is empty */
	flush_cache_all();

	/* suspend, this calls topazeh_suspend_fn using the idmap */
	rc = cpu_suspend(virt_to_phys(cpu_resume),
			 (void *)virt_to_phys(topazeh_suspend_fn));

	/* reenable L2 cache */
	outer_resume();
	printk(KERN_DEBUG "topazeh_suspend_enter: woken up (rc %d)\n", rc);
	return rc;
}

static void topazeh_suspend_finish(void)
{
	printk(KERN_DEBUG "topazeh_suspend_finish\n");
}

static void topazeh_suspend_end(void)
{
	printk(KERN_DEBUG "topazeh_suspend_end\n");
	if (topazeh_suspend_itcm) {
		iounmap(topazeh_suspend_itcm);
		topazeh_suspend_itcm = NULL;
	}
	if (topazeh_suspend_dtcm) {
		iounmap(topazeh_suspend_dtcm);
		topazeh_suspend_dtcm = NULL;
	}
}

static const struct platform_suspend_ops topazeh_pm_ops = {
	.valid = topazeh_suspend_valid,
	.begin = topazeh_suspend_begin,
	.prepare = topazeh_suspend_prepare,
	.enter = topazeh_suspend_enter,
	.finish = topazeh_suspend_finish,
	.end = topazeh_suspend_end,
};

static __init int topazeh_pm_syscore_init(void)
{
	suspend_set_ops(&topazeh_pm_ops);
	register_syscore_ops(&topazeh_pm_syscore_ops);
	return 0;
}
arch_initcall(topazeh_pm_syscore_init);
