/*
 *  Copyright (C) 2007-2012 Pixelworks, Inc.
 *  Copyright (C) 1999 - 2003 ARM Limited
 *  Copyright (C) 2000 Deep Blue Solutions Ltd
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/init.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/stmmac.h>
#include <linux/amba/pl08x.h>
#include <linux/amba/pl061.h>
#include <linux/amba/serial.h>
#include <linux/aug0603.h>

#include <asm/system.h>
#include <asm/sizes.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/hardware/timer-sp.h>
#include <asm/hardware/vic.h>
#include <asm/hardware/pl080.h>
#include <asm/mach-types.h>
#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/time.h>
#include <asm/mach/map.h>

#include <mach/hardware.h>
#include <mach/pwgpio.h>

#include "core.h"


#define WATCHDOG_UNLOCK 0x1ACCE551
#define WATCHDOG_LOCK 0xc00
#define WATCHDOG_ITCR 0xf00
#define WATCHDOG_ITOP 0xf04

#define RGMIIMUXCTRL		0x5c
#define RGMIIDIVSEL_SHIFT       2
#define RGMIIDIVSEL_MASK        3


/* the STATIC_REGS are preserved across watchdog reset, the value is
 * used by the boot loader to decide whether to reboot or go into
 * low power sleep
 */
static void topaz_enter_low_power_sleep(void)
{
	writel(0x4c4f5750 /* "LOWP" */, __io_address(STATIC_REGS_BASE));
	writel(WATCHDOG_UNLOCK, __io_address(WATCHDOG_BASE + WATCHDOG_LOCK));
	writel(1, __io_address(WATCHDOG_BASE + WATCHDOG_ITCR));
	writel(1, __io_address(WATCHDOG_BASE + WATCHDOG_ITOP));
}

void topaz_restart(char mode, const char *cmd)
{
	/* perform reset using watchdog */
	writel(0x52454254 /* "REBT" */, __io_address(STATIC_REGS_BASE));
	writel(WATCHDOG_UNLOCK, __io_address(WATCHDOG_BASE + WATCHDOG_LOCK));
	writel(1, __io_address(WATCHDOG_BASE + WATCHDOG_ITCR));
	writel(1, __io_address(WATCHDOG_BASE + WATCHDOG_ITOP));
}


static void topaz_fix_mac_speed(void *priv, unsigned int speed)
{
	u32 rgmii_ctrl;
	rgmii_ctrl = readl(IO_ADDRESS(SYS_BASE + RGMIIMUXCTRL));
	rgmii_ctrl &= ~(RGMIIDIVSEL_MASK << RGMIIDIVSEL_SHIFT);
	switch (speed) {
		case 1000:
			rgmii_ctrl |= (0 << RGMIIDIVSEL_SHIFT);
			break;
		case 100:
			rgmii_ctrl |= (3 << RGMIIDIVSEL_SHIFT);
			break;
		case 10:
			rgmii_ctrl |= (2 << RGMIIDIVSEL_SHIFT);
			break;
		default:
			printk(KERN_WARNING "fix mac speed error!\n");
			return;
	}
	writel(rgmii_ctrl, IO_ADDRESS(SYS_BASE + RGMIIMUXCTRL));
}

static struct stmmac_mdio_bus_data stmmacemdio_pdata;
static struct plat_stmmacenet_data stmmacenet_pdata = {
	.phy_addr = -1,	/* use first PHY found */
	.fix_mac_speed = topaz_fix_mac_speed,
	.has_gmac = 1,
	.pmt = 1,
	.pbl = 32,
	.mdio_bus_data = &stmmacemdio_pdata,
};


#define TOPAZ_GPIO(n) \
static struct pl061_platform_data topaz_gpio ## n ## _pdata = { \
	.gpio_base = n * 8, \
	.irq_base = GPIO_IRQ_BASE + n * 8, \
}

TOPAZ_GPIO(0);
TOPAZ_GPIO(1);
TOPAZ_GPIO(2);
TOPAZ_GPIO(3);
TOPAZ_GPIO(4);
TOPAZ_GPIO(5);
TOPAZ_GPIO(6);
TOPAZ_GPIO(7);
TOPAZ_GPIO(8);
TOPAZ_GPIO(9);

#define OF_TOPAZ_GPIO(n) \
	OF_DEV_AUXDATA("arm,pl061", GPIO ## n ## _BASE, "gpio" #n, &topaz_gpio ## n ##_pdata)

static struct of_dev_auxdata topaz_dt_auxdata_lookup[] __initdata = {
	/* assign a sane name to UARTs which is used to look up the clock */
	OF_DEV_AUXDATA("arm,pl011", UART0_BASE, "uart0", &topaz_uart0_plat_data),
	OF_DEV_AUXDATA("arm,pl011", UART1_BASE, "uart1", &topaz_uart1_plat_data),
	OF_DEV_AUXDATA("st,spear600-gmac", ETH0_BASE, "eth0", &stmmacenet_pdata),
	OF_DEV_AUXDATA("arm,pl080", DMA8CH0_BASE, "dma8ch0", &topaz_dma8ch_pdata),
	OF_DEV_AUXDATA("aurora,au-g0603", SPI0_BASE, "spi0", &topaz_aug0603_pdata),
	OF_TOPAZ_GPIO(0),
	OF_TOPAZ_GPIO(1),
	OF_TOPAZ_GPIO(2),
	OF_TOPAZ_GPIO(3),
	OF_TOPAZ_GPIO(4),
	OF_TOPAZ_GPIO(5),
	OF_TOPAZ_GPIO(6),
	OF_TOPAZ_GPIO(7),
	OF_TOPAZ_GPIO(8),
	OF_TOPAZ_GPIO(9),
	{ }
};

/* for backwards compatibility we have one platform GPIO device
 * per real GPIO device to handle the ioctl API
 */
#define TOPAZ_GPIO_PDEV(n) \
static struct pwgpio_pdata topaz_gpio ## n ## _pdevdata = { \
	.id = n, .gpio_base = n * 8, .num_gpio = 8 }; \
static struct platform_device topaz_gpio ## n ## _pdev = { \
	.name = "pwgpio", \
	.id = n, \
	.dev = { \
		.platform_data = &topaz_gpio ## n ## _pdevdata, \
	} \
}

TOPAZ_GPIO_PDEV(0);
TOPAZ_GPIO_PDEV(1);
TOPAZ_GPIO_PDEV(2);
TOPAZ_GPIO_PDEV(3);
TOPAZ_GPIO_PDEV(4);
TOPAZ_GPIO_PDEV(5);
TOPAZ_GPIO_PDEV(6);
TOPAZ_GPIO_PDEV(7);
TOPAZ_GPIO_PDEV(8);
TOPAZ_GPIO_PDEV(9);

static void topaz_gpio_init(void)
{
	platform_device_register(&topaz_gpio0_pdev);
	platform_device_register(&topaz_gpio1_pdev);
	platform_device_register(&topaz_gpio2_pdev);
	platform_device_register(&topaz_gpio3_pdev);
	platform_device_register(&topaz_gpio4_pdev);
	platform_device_register(&topaz_gpio5_pdev);
	platform_device_register(&topaz_gpio6_pdev);
	platform_device_register(&topaz_gpio7_pdev);
	platform_device_register(&topaz_gpio8_pdev);
	platform_device_register(&topaz_gpio9_pdev);
}

void __init topaz_init(void)
{
	pm_power_off = topaz_enter_low_power_sleep;
	of_platform_populate(NULL, of_default_bus_match_table,
			     topaz_dt_auxdata_lookup, NULL);
	topaz_gpio_init();
}

static const struct of_device_id vic_of_match[] __initconst = {
	{ .compatible = "arm,pl192-vic", .data = vic_of_init, },
	{ }
};

static void __init topaz_init_irq(void)
{
	of_irq_init(vic_of_match);
}

/*
 * system peripherals mapped early during boot
 */
#define IO_DESC(base, size) { .virtual = IO_ADDRESS(base), \
        .pfn = __phys_to_pfn(base), \
        .length = size, \
        .type = MT_DEVICE }

static struct map_desc topaz_io_desc[] __initdata = {
	IO_DESC(0xff000000, SZ_16M), // map all IO
};

void __init topaz_map_io(void)
{
	iotable_init(topaz_io_desc, ARRAY_SIZE(topaz_io_desc));
}

/* We use both 32bit timers of Timer0:
 * Timer 0.0: clockevent source for hrtimers
 * Timer 0.1: clocksource for generic timekeeping
 */
static void __init topaz_timer_init(void)
{
	int irq;
	struct device_node *np;
	void __iomem *timer_base;
	extern void topaz_clk_init(void);

	topaz_clk_init();

	np = of_find_compatible_node(NULL, NULL, "arm,sp804");
	timer_base = of_iomap(np, 0);
	BUG_ON(!timer_base);
	irq = irq_of_parse_and_map(np, 0);

	sp804_clocksource_and_sched_clock_init(timer_base + 0x20, "timer1");
	sp804_clockevents_init(timer_base, irq, "timer0");
}

static const char *topaz_dt_match[] = {
	"pixelworks,topaz-eval",
	NULL,
};

DT_MACHINE_START(PIXELWORKS, "Pixelworks Topaz")
	.atag_offset	= 0x100,
	.map_io		= topaz_map_io,
	.init_irq	= topaz_init_irq,
	.handle_irq	= vic_handle_irq,
	.init_time	= topaz_timer_init,
	.init_machine	= topaz_init,
	.dt_compat	= topaz_dt_match,
	.restart	= topaz_restart,
MACHINE_END
