/* -------------------------------------------------------------------------- *\
   driver for watch dog timer

   Copyright (c) 2007-2012 Pixelworks Inc.

   --------------------------------------------------------------------------
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\* -------------------------------------------------------------------------- */
//#define DEBUG

#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/of_platform.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/watchdog.h>
#include <linux/ioport.h>
#include <linux/io.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/clk.h>
#include <linux/bitops.h>
#include <linux/uaccess.h>
#include <linux/timex.h>
#include <asm/irq.h>

#include <pxlw/pwwdog.h>

/* register definitions */
#define WDOG_LOAD      0x00
#define WDOG_VALUE     0x04
#define WDOG_CONTROL   0x08
#define   WDOG_CONTROL_TMEN            BIT(7)
#define   WDOG_CONTROL_INTEN           BIT(6)
#define   WDOG_CONTROL_RESEN           BIT(5)
#define   WDOG_CONTROL_PRESCALE_1      (0 << 2)
#define   WDOG_CONTROL_PRESCALE_16     (1 << 2)
#define   WDOG_CONTROL_PRESCALE_256    (2 << 2)
#define   WDOG_CONTROL_PRESCALE_32     (3 << 2)
#define   WDOG_CONTROL_PRESCALE_128    (4 << 2)
#define   WDOG_CONTROL_PRESCALE_1024   (5 << 2)
#define   WDOG_CONTROL_PRESCALE_4096   (6 << 2)
#define WDOG_KICK      0x0c
#define WDOG_RSTOCC    0x10
#define WDOG_CLEAR     0x14
#define WDOG_INTVAL    0x18
#define WDOG_LOCK      0x1c

#define UNLOCK_CODE 0x1ACCE551
#define RELOAD_CODE 0xA5A5


#define WDT_DEFAULT_TIME       5       /* seconds */

static int wdt_time = WDT_DEFAULT_TIME;
static int nowayout = WATCHDOG_NOWAYOUT;

module_param(wdt_time, int, 0);
MODULE_PARM_DESC(wdt_time, "Watchdog time in seconds. (default="
                __MODULE_STRING(WDT_DEFAULT_TIME) ")");

module_param(nowayout, int, 0);
MODULE_PARM_DESC(nowayout, "Watchdog cannot be stopped once started (default="
                __MODULE_STRING(WATCHDOG_NOWAYOUT) ")");


struct pwwdt {
       struct device *dev;
       void __iomem *regs;
       int irq;
       struct clk *clk;

       struct miscdevice misc;

       u16 load_value;
       u8 prescale;
       wait_queue_head_t wq;
       int bite;
       unsigned long busy;
       int crash_system;
};


/*
 * Disable the watchdog.
 */
static void pwwdt_stop(struct pwwdt *pwwdt)
{
       writel(UNLOCK_CODE, pwwdt->regs + WDOG_LOCK);
       writel(0, pwwdt->regs + WDOG_CONTROL);
       writel(0, pwwdt->regs + WDOG_CLEAR);
       writel(0, pwwdt->regs + WDOG_LOCK);
       dev_dbg(pwwdt->dev, "stopped\n");
}

/*
 * Enable and reset the watchdog.
 */
static void pwwdt_start(struct pwwdt *pwwdt)
{
       writel(UNLOCK_CODE, pwwdt->regs + WDOG_LOCK);
       writel(pwwdt->load_value, pwwdt->regs + WDOG_LOAD);
       writel(WDOG_CONTROL_TMEN | WDOG_CONTROL_INTEN | WDOG_CONTROL_RESEN
              | pwwdt->prescale, pwwdt->regs + WDOG_CONTROL);
       writel(0, pwwdt->regs + WDOG_LOCK);
       dev_dbg(pwwdt->dev, "started\n");
}

/*
 * Reload the watchdog timer.
 */
static void pwwdt_reload(struct pwwdt *pwwdt)
{
       writel(RELOAD_CODE, pwwdt->regs + WDOG_KICK);
       dev_dbg(pwwdt->dev, "reloaded by PID %d (%s)\n",
              current->pid, current->comm);
}

static void pwwdt_calc_ps_load(struct pwwdt *pwwdt)
{
       unsigned long freq = clk_get_rate(pwwdt->clk);
       // FIXME: find optimal prescale
       pwwdt->prescale = WDOG_CONTROL_PRESCALE_4096;
       pwwdt->load_value = wdt_time * freq / 4096;
}

/*
 * Change the watchdog time interval.
 */
static int pwwdt_settimeout(struct pwwdt *pwwdt, int new_time)
{
       if (new_time <= 0)
               return -EINVAL;

       /* Set new watchdog time.
        * It will be used when pwwdt_start() is called.
        */
       wdt_time = new_time;
       pwwdt_calc_ps_load(pwwdt);
       return 0;
}


/*
 * Watchdog device is opened, and watchdog starts running.
 */
static int pwwdt_open(struct inode *inode, struct file *file)
{
       struct miscdevice *misc = file->private_data;
       struct pwwdt *pwwdt = container_of(misc, struct pwwdt, misc);

       if (test_and_set_bit(0, &pwwdt->busy))
               return -EBUSY;

       dev_dbg(pwwdt->dev, "open\n");
       init_waitqueue_head(&pwwdt->wq);
       pwwdt->bite = 0;
       file->private_data = pwwdt;

//     pwwdt_start(pwwdt);
       return nonseekable_open(inode, file);
}

/*
 * Close the watchdog device.
 * If CONFIG_WATCHDOG_NOWAYOUT is NOT defined then the watchdog is also
 *  disabled.
 */
static int pwwdt_release(struct inode *inode, struct file *file)
{
       struct pwwdt *pwwdt = file->private_data;

       if (!nowayout)
               pwwdt_stop(pwwdt);

       clear_bit(0, &pwwdt->busy);
       return 0;
}

/*
 * Reload the watchdog whenever device is written to.
 */
static ssize_t pwwdt_write(struct file *file, const char *data,
                          size_t len, loff_t *ppos)
{
       struct pwwdt *pwwdt = file->private_data;

       pwwdt_reload(pwwdt);
       return len;
}


static struct watchdog_info pwwdt_info = {
       .identity       = "pw watchdog",
       .options        = WDIOF_SETTIMEOUT | WDIOF_KEEPALIVEPING,
};

static long pwwdt_ioctl(struct file *file, unsigned int cmd,
                       unsigned long arg)
{
       struct pwwdt *pwwdt = file->private_data;
       void __user *argp = (void __user *)arg;
       int __user *p = argp;
       int new_value;

       switch (cmd) {
       case WDIOC_KEEPALIVE:
               pwwdt_reload(pwwdt);
               return 0;

       case WDIOC_WAITATGATE:
               dev_dbg(pwwdt->dev, "WAITATGATE begin.\n");
	       wait_event_interruptible(pwwdt->wq, pwwdt->bite);
	       pwwdt->bite = 0;
               dev_dbg(pwwdt->dev, "WAITATGATE end.\n");
               return 0;
       case WDIOC_GETSUPPORT:
               if (copy_to_user(argp, &pwwdt_info, sizeof(pwwdt_info)))
                       return -EFAULT;
               return 0;

       case WDIOC_SETTIMEOUT:
               if (get_user(new_value, p))
                       return -EFAULT;
               if (pwwdt_settimeout(pwwdt, new_value))
                       return -EINVAL;
               /* Enable new time value */
               pwwdt_start(pwwdt);
               /* Return current value */
               return put_user(wdt_time, p);

       case WDIOC_GETTIMEOUT:
               return put_user(wdt_time, p);

       case WDIOC_GETSTATUS:
       case WDIOC_GETBOOTSTATUS:
               return put_user(0, p);

       case WDIOC_SETOPTIONS:
               if (get_user(new_value, p))
                       return -EFAULT;
               if (new_value & WDIOS_DISABLECARD)
                       pwwdt_stop(pwwdt);
               if (new_value & WDIOS_ENABLECARD)
                       pwwdt_start(pwwdt);
               return 0;
       case WDIOC_CRASHSYSTEM:
               if (get_user(new_value, p))
                       return -EFAULT;
               pwwdt->crash_system = 0;
               if (new_value)
                       pwwdt->crash_system = 1;
               return 0;
       default:
               return -ENOTTY;
       }
}

static const struct file_operations pwwdt_fops = {
       .owner          = THIS_MODULE,
       .llseek         = no_llseek,
       .unlocked_ioctl = pwwdt_ioctl,
       .open           = pwwdt_open,
       .release        = pwwdt_release,
       .write          = pwwdt_write,
};

static irqreturn_t pwwdt_interrupt(int irq, void *dev_id)
{
       struct pwwdt *pwwdt = dev_id;

       writel(0, pwwdt->regs + WDOG_CLEAR);
       if (pwwdt->crash_system)
               panic("pwwdt irq: crash_system\n");
       pwwdt->bite = 1;
       wake_up_interruptible(&pwwdt->wq);
       return IRQ_HANDLED;
}

static int pwwdt_probe(struct platform_device *pdev)
{
       struct pwwdt *pwwdt;
       struct resource *res;
       struct device *dev = &pdev->dev;
       int ret = 0;

       dev_dbg(&pdev->dev, "probe\n");
       pwwdt = devm_kzalloc(dev, sizeof(struct pwwdt), GFP_KERNEL);
       if (!pwwdt)
               return -ENOMEM;
       pwwdt->dev = dev;
       res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
       if (res == NULL) {
               dev_err(dev, "no memory resource specified\n");
               return -ENOENT;
       }
       if (!devm_request_mem_region(dev, res->start,
                                    resource_size(res), pdev->name)) {
               dev_err(dev, "failed to get memory region\n");
               return -EBUSY;
       }
       pwwdt->regs = devm_ioremap(dev, res->start, resource_size(res));
       if (pwwdt->regs == 0) {
               dev_err(dev, "failed to ioremap() region\n");
               return -ENOMEM;
       }
       pwwdt->irq = platform_get_irq(pdev, 0);
       if (pwwdt->irq <= 0) {
               dev_err(dev, "no irq resource specified\n");
               return -ENOENT;
       }

       pwwdt->clk = of_clk_get(dev->of_node, 0);
       if (IS_ERR(pwwdt->clk)) {
               dev_err(dev, "can't find pwwdt clock\n");
               return PTR_ERR(pwwdt->clk);
       }
       /* select slowest possible clock to support long WDOG timeout */
       clk_set_rate(pwwdt->clk, 0);
       dev_info(dev, "using clock rate %luHz\n", clk_get_rate(pwwdt->clk));
       clk_prepare_enable(pwwdt->clk);
       pwwdt_calc_ps_load(pwwdt);

       writel(UNLOCK_CODE, pwwdt->regs + WDOG_LOCK);
       writel(0, pwwdt->regs + WDOG_CONTROL);
       writel(0, pwwdt->regs + WDOG_CLEAR);
       writel(0, pwwdt->regs + WDOG_INTVAL);

       ret = devm_request_irq(dev, pwwdt->irq, pwwdt_interrupt,
                              0, pdev->name, pwwdt);
       if (ret != 0) {
               dev_err(dev, "failed to install irq (%d)\n", ret);
               goto err_clk;
       }
       platform_set_drvdata(pdev, pwwdt);

       pwwdt->misc.minor = WATCHDOG_MINOR,
       pwwdt->misc.name = "pwwdog",
       pwwdt->misc.fops = &pwwdt_fops,
       ret = misc_register(&pwwdt->misc);
       if (ret) {
               dev_err(dev, "cannot register miscdev on minor=%d (%d)\n",
                       WATCHDOG_MINOR, ret);
               return ret;
       }

       printk(KERN_INFO "PW Watchdog Timer (%d seconds%s)\n",
              wdt_time, nowayout ? ", nowayout" : "");
       return 0;

err_clk:
       clk_disable_unprepare(pwwdt->clk);
       clk_put(pwwdt->clk);
       return ret;
}

static int __exit pwwdt_remove(struct platform_device *pdev)
{
       struct pwwdt *pwwdt = platform_get_drvdata(pdev);

       misc_deregister(&pwwdt->misc);
       clk_disable_unprepare(pwwdt->clk);
       clk_put(pwwdt->clk);
       platform_set_drvdata(pdev, NULL);
       return 0;
}

#ifdef CONFIG_PM
static int pwwdt_suspend(struct device *dev)
{
       struct pwwdt *pwwdt = dev_get_drvdata(dev);

       pwwdt_stop(pwwdt);
       clk_disable(pwwdt->clk);
       return 0;
}

static int pwwdt_resume(struct device *dev)
{
       struct pwwdt *pwwdt = dev_get_drvdata(dev);

       clk_enable(pwwdt->clk);
       if (pwwdt->busy)
               pwwdt_start(pwwdt);
       return 0;
}
#endif

static struct dev_pm_ops pwwdt_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(pwwdt_suspend, pwwdt_resume)
};


static struct of_device_id pwwdt_of_match[] = {
       { .compatible = "pixelworks,wdogv2" },
       { },
};

static struct platform_driver pwwdt_driver = {
       .probe          = pwwdt_probe,
       .remove         = __exit_p(pwwdt_remove),
       .driver         = {
               .name   = "pwwdog",
               .owner  = THIS_MODULE,
               .of_match_table = pwwdt_of_match,
	       .pm     = &pwwdt_pm_ops,
       },
};
module_platform_driver(pwwdt_driver);

MODULE_AUTHOR("Pixelworks, Inc.");
MODULE_DESCRIPTION("Driver for Pixelworks watchdog timer v2");
MODULE_LICENSE("GPL");
MODULE_ALIAS_MISCDEV(WATCHDOG_MINOR);
