/* -------------------------------------------------------------------------- *\
   pwirq.c -- ioctl API for userspace irq handling driver

   Copyright (c) 2007-2012 Pixelworks Inc.

   --------------------------------------------------------------------------
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\* -------------------------------------------------------------------------- */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/miscdevice.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/wait.h>
#include <linux/spinlock.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/delay.h>

#include <pxlw/pwirq.h>

MODULE_AUTHOR ("Pixelworks Inc.");
MODULE_LICENSE("GPL");


#define MAX_IRQ NR_IRQS
#define MAX_IRQ_LONGS (MAX_IRQ / sizeof(unsigned long))

// misc device minor number
#define PWIRQ_MISC_MINOR 254

#define DEBUG
#ifdef DEBUG
static int pwirq_debug;
module_param_named(debug, pwirq_debug, int, 0644);
MODULE_PARM_DESC(debug, "debug level (0 off, 1 errors, 2 trace, 3 full)");

#define dprintk(level, fmt, ...) do { \
       if (pwirq_debug >= level) \
               printk(KERN_DEBUG "PWIRQ: %s:%d: " fmt, __FUNCTION__, __LINE__, ## __VA_ARGS__); \
       } while (0)
#else
#define dprintk(...) do { } while (0)
#endif


struct pwirq_data {
       wait_queue_head_t wq;
       unsigned long irq_set[MAX_IRQ_LONGS];
};

// bitsets for irq status
// be careful to access them in the right order!
static unsigned long irq_set[MAX_IRQ_LONGS];
static unsigned long irq_enabled[MAX_IRQ_LONGS];
static unsigned long irq_pending[MAX_IRQ_LONGS];
static struct rt_mutex pwirq_mutex;
static DEFINE_SPINLOCK(pwirq_lock);

// irq names for /proc/interrupts
static char irq_name[MAX_IRQ][PWIRQ_NAME_LEN];


// called with irqs disabled
static irqreturn_t pwirq_interrupt(int irq, void *dev_id)
{
       struct pwirq_data *d = dev_id;

       BUG_ON(irq == 0); // Linux numbers irqs from 1
       irq--;

       dprintk(3, "irq %d\n", irq);
       WARN_ON(!test_bit(irq, irq_enabled));
       dprintk(3, "disable_irq(%d)\n", irq);
       disable_irq_nosync(irq + 1); // reenabled by PWIRQ_END

       spin_lock(&pwirq_lock);
       // note: must use disable_irq_nosync() here since disable_irq() will
       // wait for the irq handler to finish -> deadlock (at least in newer
       // kernels, but it was always documented behaviour)

       set_bit(irq, irq_pending);
       spin_unlock(&pwirq_lock);

       wake_up(&d->wq);
       return IRQ_HANDLED;
}


static int pwirq_open(struct inode *inode, struct file *file)
{
       struct pwirq_data *d;

       dprintk(2, "\n");
       d = kzalloc(sizeof(*d), GFP_KERNEL);
       if (!d)
               return -ENOMEM;
       init_waitqueue_head(&d->wq);
       file->private_data = d;
       return 0;
}

static int pwirq_release(struct inode *inode, struct file *file)
{
       struct pwirq_data *d = file->private_data;
       int i;

       dprintk(2, "\n");
       for (i = 0; i < MAX_IRQ; i++) {
               if (test_bit(i, d->irq_set)) {
                       dprintk(2, "unset irq %u %s\n", i, irq_name[i]);
                       free_irq(i + 1, d);
                       clear_bit(i, irq_enabled);
                       clear_bit(i, irq_pending);
                       clear_bit(i, irq_set);
               }
       }
       file->private_data = NULL;
       kfree(d);
       return 0;
}

// get lowest pending irq
static int get_pending_irq(struct pwirq_data *d, u32 *irq)
{
       int i, found = 0;
       unsigned long pending[MAX_IRQ_LONGS];
       unsigned long flags;

       spin_lock_irqsave(&pwirq_lock, flags);
       for (i = 0; i< MAX_IRQ_LONGS; i++)
               pending[i] = irq_pending[i] & d->irq_set[i];
       spin_unlock_irqrestore(&pwirq_lock, flags);

       i = find_first_bit(pending, MAX_IRQ);
       if (i < MAX_IRQ) {
               *irq = i;
               found = 1;
               dprintk(3, "got irq %d\n", i);
       }
       return found;
}

static long pwirq_ioctl(struct file *filp, unsigned cmd, unsigned long arg)
{
       struct pwirq_data *d = filp->private_data;
       unsigned long flags;
       int rc = 0;

       dprintk(3, "cmd %08x arg %08lx d %p\n", cmd, arg, d);

       if (rt_mutex_lock_interruptible(&pwirq_mutex))
               return -ERESTARTSYS;

       switch(cmd) {
       case PWIRQ_SET:
       {
               struct pwirq_configuration conf;

               if (copy_from_user(&conf, (void *)arg, sizeof(conf))) {
                       rc = -EFAULT;
                       goto err;
               }
               dprintk(2, "PWIRQ_SET %u %.*s\n", conf.irq, PWIRQ_NAME_LEN, conf.name);
               if (conf.irq >= MAX_IRQ) {
                       rc = -EINVAL;
                       goto err;
               }
               if (test_bit(conf.irq, irq_set)) {
                       rc = -EBUSY;
                       goto err;
               }
               strlcpy(irq_name[conf.irq], conf.name, PWIRQ_NAME_LEN);
               set_bit(conf.irq, d->irq_set);
               set_bit(conf.irq, irq_set);
               set_bit(conf.irq, irq_enabled);
               WARN_ON(test_bit(conf.irq, irq_pending));
               rc = request_irq(conf.irq + 1, pwirq_interrupt,
                                0, irq_name[conf.irq], d);
               if (rc) {
                       clear_bit(conf.irq, d->irq_set);
                       clear_bit(conf.irq, irq_set);
                       clear_bit(conf.irq, irq_enabled);
               }
               break;
       }
       case PWIRQ_UNSET:
       {
               u32 irq;

               if (get_user(irq, (u32 *)arg)) {
                       rc = -EFAULT;
                       goto err;
               }
               dprintk(2, "PWIRQ_UNSET %u %s\n", irq, irq_name[irq]);
               if (irq >= MAX_IRQ) {
                       rc = -EINVAL;
                       goto err;
               }
               if (!test_bit(irq, d->irq_set)) {
                       rc = -ESRCH;
                       goto err;
               }
               BUG_ON(!test_bit(irq, irq_set));
               free_irq(irq + 1, d);
               clear_bit(irq, irq_pending);
               clear_bit(irq, d->irq_set);
               clear_bit(irq, irq_enabled);
               clear_bit(irq, irq_set);
               break;
       }
       case PWIRQ_DISABLE:
       {
               u32 irq;

               if (get_user(irq, (u32 *)arg)) {
                       rc = -EFAULT;
                       goto err;
               }
               dprintk(2, "PWIRQ_DISABLE %u %s\n", irq, irq_name[irq]);
               if (irq >= MAX_IRQ) {
                       rc = -EINVAL;
                       goto err;
               }
               if (!test_bit(irq, d->irq_set)) {
                       rc = -ENOENT;
                       goto err;
               }
               BUG_ON(!test_bit(irq, irq_set));
               if (!test_bit(irq, irq_enabled)) {
                       rc = -ESRCH;
                       goto err;
               }
               spin_lock_irqsave(&pwirq_lock, flags);
               if (!test_bit(irq, irq_pending)) {
                       dprintk(3, "disable_irq(%d)\n", irq);
                       disable_irq(irq + 1);
               } else
                       dprintk(3, "no disable_irq, irq %d is pending\n", irq);
               spin_unlock_irqrestore(&pwirq_lock, flags);
               clear_bit(irq, irq_enabled);
               break;
       }
       case PWIRQ_ENABLE:
       {
               u32 irq;

               if (get_user(irq, (u32 *)arg)) {
                       rc = -EFAULT;
                       goto err;
               }
               dprintk(2, "PWIRQ_ENABLE %u %s\n", irq, irq_name[irq]);
               if (irq >= MAX_IRQ) {
                       rc = -EINVAL;
                       goto err;
               }
               if (!test_bit(irq, d->irq_set)) {
                       rc = -ENOENT;
                       goto err;
               }
               BUG_ON(!test_bit(irq, irq_set));
               if (test_bit(irq, irq_enabled)) {
                       rc = -EBUSY;
                       goto err;
               }
               set_bit(irq, irq_enabled);
               if (!test_bit(irq, irq_pending)) {
                       dprintk(3, "enable_irq(%d)\n", irq);
                       enable_irq(irq + 1);
               } else
                       dprintk(3, "defer enable_irq, irq %d is pending\n", irq);
               break;
       }
       case PWIRQ_WAIT:
       {
               u32 irq;

               dprintk(3, "PWIRQ_WAIT\n");
again:
               // drop the mutex while waiting
               rt_mutex_unlock(&pwirq_mutex);
               if (wait_event_interruptible(d->wq, get_pending_irq(d, &irq))) {
                       dprintk(1, "PWIRQ_WAIT interrupted irq: %d\n", irq);
                       return -ERESTARTSYS;
               }
               // there is a small chance for a PWIRQ_UNSET or PWIRQ_DISABLE to come in here...
               if (rt_mutex_lock_interruptible(&pwirq_mutex)) {
                       dprintk(3, "PWIRQ_WAIT mutex lock interrupted IRQ %d\n", irq);
                       return -ERESTARTSYS;
               }
               // ... so we have to re-check the irq for validity
               if (!test_bit(irq, irq_enabled)) {
                       dprintk(3, "PWIRQ_WAIT got irq %u %s but irq is now disabled -> wait again\n",
                               irq, irq_name[irq]);
                       goto again;
               }
               dprintk(3, "PWIRQ_WAIT got irq %u %s\n", irq, irq_name[irq]);
               if (copy_to_user((void *)arg, &irq, sizeof(irq))) {
                       rc = -EFAULT;
                       goto err;
               }
               break;
       }
       case PWIRQ_END:
       {
               u32 irq;

               if (get_user(irq, (u32 *)arg)) {
                       rc = -EFAULT;
                       goto err;
               }
               dprintk(3, "PWIRQ_END %u %s\n", irq, irq_name[irq]);
               if (irq >= MAX_IRQ) {
                       rc = -EINVAL;
                       goto err;
               }
               if (!test_bit(irq, d->irq_set)) {
                       rc = -ENOENT;
                       goto err;
               }
               BUG_ON(!test_bit(irq, irq_set));
               if (!test_bit(irq, irq_pending)) {
                       rc = -ESRCH;
                       // leave with irq enabled
               }
               clear_bit(irq, irq_pending);
               if (test_bit(irq, irq_enabled)) {
                       dprintk(3, "enable_irq(%d)\n", irq);
                       enable_irq(irq + 1);
               } else
                       dprintk(3, "no enable_irq, irq %d has been disabled\n", irq);
               break;
       }
       default:
               rc = -ENOTTY;
       }

err:
       rt_mutex_unlock(&pwirq_mutex);
       return rc;
}


struct file_operations pwirq_fops =
{
       .owner = THIS_MODULE,
       .open = pwirq_open,
       .release = pwirq_release,
       .unlocked_ioctl= pwirq_ioctl,
};

static struct miscdevice pwirq_miscdev = {
       .minor = PWIRQ_MISC_MINOR,
       .name = "pwirq",
       .fops = &pwirq_fops,
};


static int __init pwirq_init(void)
{
       int rc;

       printk(KERN_INFO "PWIRQ driver loaded\n");
       rc = misc_register(&pwirq_miscdev);
       if (rc)
               printk(KERN_ERR "PWIRQ: misc_register failed\n");
       else
               rt_mutex_init(&pwirq_mutex);
       return rc;
}

static __exit void pwirq_cleanup(void)
{
       printk(KERN_INFO "PWIRQ driver removed\n");
       misc_deregister(&pwirq_miscdev);
}

module_init(pwirq_init);
module_exit(pwirq_cleanup);
