/*
 * pwi2s.c -- Pixelworks I2S driver for ALSA
 *
 * Copyright (C) 2010 Pixelworks Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Note: The ALSA driver cannot work before appcode has set the
 * DMA buffers using the PWI2S_SET_BUFFER_CONFIG ioctl on the
 * hwdep device (/dev/hwC0D0).  If the hwdep device is closed
 * (e.g. when appcode exits), or when appcode sets the buffer size
 * to zero using PWI2S_SET_BUFFER_CONFIG, then DMA will stop
 * and the driver returns to the unconfigured state.
 * If appcode enables audio delay mode using the
 * PWI2S_SET_DELAY_MODE, stream plaback will preempt the
 * delay mode and reenable it once playback finishes.
 */


#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/jiffies.h>
#include <asm/irq.h>

#include <sound/core.h>
#include <sound/initval.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/hwdep.h>

#include <mach/platform.h>

#include "pwi2s.h"


MODULE_AUTHOR("Pixelworks Inc.");
MODULE_DESCRIPTION("Pixelworks I2S ALSA Driver");
MODULE_LICENSE("GPL");


#define DEBUG 1
#ifdef DEBUG
static int pwi2s_slice = 0;
static int pwi2s_debug = 2;
static int pwi2s_dummy_write = 0;
module_param_named(debug, pwi2s_debug, int, 0644);
MODULE_PARM_DESC(debug, "debug level");
module_param_named(dummy_write, pwi2s_dummy_write, int, 0644);
MODULE_PARM_DESC(debug, "dummy write");
module_param_named(slice, pwi2s_slice, int, 0644);
MODULE_PARM_DESC(slice, "pwi2s slice");

#define dprintk(level, fmt, ...) do { \
	if (pwi2s_debug >= level) \
		printk(KERN_DEBUG "PWI2S %*s %s:%d: "fmt, level, "", __func__, __LINE__, ## __VA_ARGS__); \
	} while (0)
#else
#define dprintk(...) do { } while (0)
#endif

//software work around for pop noise
//will be remove after hardware fix it
//this GPIO will reset the op-AMP


#define TOPAZEH_SYS_BASE 0xffff0000

#if 0
static unsigned int *pGPIO;
#endif

#if 0
// ARM clock
#define ARM_CLK 330000000
// SYS registers for I2S clock divisor
#define SYS_CLKCTRL2 0x0c
#define SYS_CLKCTRL5 0x18
#define SYS_MISC2 0x54
#define SYS_RESERVED_0 0x7c

#define SYS_CLKCTRL2_XCLK2I2S (1 << 8)

#define SYS_CLKCTRL5_CLK_DIV_MASK 0x7fc00000
#define SYS_CLKCTRL5_CLK_DIV_SHIFT 22

#define SYS_MISC2_EXT_AUD_CLK_EN (1 << 12)

#define SYS_RESERVED_0_I2SCLK_SEL2_I2SCLKIN1 0
#define SYS_RESERVED_0_I2SCLK_SEL2_I2SCLKIN2 1
#define SYS_RESERVED_0_I2SCLK_SEL2_HDMISCK0 2
#define SYS_RESERVED_0_I2SCLK_SEL2_HDMISCK1 3
#define SYS_RESERVED_0_I2SCLK_SEL2_ARMDIVI2SCLK 4
#define SYS_RESERVED_0_I2SCLK_SEL2_XCLK_CKG 5
#else
#define ARM_CLK 333000000
#define SYS_IPCLKCTRL 0x00
#define SYS_CLKOFF 0x2c
#define SYS_CLKDIV 0x38
#define SYS_CLKDIV_EN 0x3c
#define SYS_PADMUX_REG1 0x68
#define SYS_PADMUX_REG2 0x6c
#define SYS_IPCLKCTRL_I2SPOLOUT (1 << 1)
#define SYS_IPCLKCTRL_I2OSRCSEL (1 << 3)
#define SYS_CLKOFF_I2SOFF (1 << 23)
#define SYS_CLKDIV_I2SMCLK_MASK (0x3e00)
#define SYS_CLKDIV_I2SMCLK_SHIFT 9
#define SYS_CLKDIV_ARM_I2SMCLK_MASK (0x1ff)
#define SYS_CLKDIV_ARM_I2SMCLK_SHIFT 0
#define SYS_CLKDIV_I2SMCLK_EN (1 << 1)
#define SYS_CLKDIV_ARM_I2SMCLK_EN (1)
#define SYS_PADMUX_REG1_AUDIOMCLK_EN (1 << 26)
#define SYS_PADMUX_REG2_I2SOD_OUTPUT_ENABLE (1 << 2)
#define SYS_PADMUX_REG2_I2SOWS_OUTPUT_ENABLE (1 << 3)
#endif


// register bank 1
#define I2SREG_0 0xfc00
#define I2SREG_1 0xfc04
#define I2SREG_2 0xfc08
#define I2SREG_3 0xfc0c
#define I2SREG_4 0xfc10
#define I2SREG_5 0xfc14
#define I2SREG_6 0xfc18
#define I2SREG_7 0xfc1c
#define I2SREG_8 0xfc20
#define I2SREG_9 0xfc24

#define I2SREG_0_MASK_BYPASS         0x0002
#define I2SREG_0_SHIFT_BYPASS        1
#define I2SREG_0_MASK_MUTE           0x0004
#define I2SREG_0_SHIFT_MUTE          2
#define I2SREG_0_MASK_INPOL          0x0008
#define I2SREG_0_SHIFT_INPOL         3
#define I2SREG_0_MASK_OUTPOL         0x0010
#define I2SREG_0_SHIFT_OUTPOL        4
#define I2SREG_0_MASK_CLKOFF         0x0020
#define I2SREG_0_SHIFT_CLKOFF        5
#define I2SREG_0_MASK_EN             0x0040
#define I2SREG_0_SHIFT_EN            6
#define I2SREG_0_MASK_RDENABLE       0x0080
#define I2SREG_0_SHIFT_RDENABLE      7
#define I2SREG_0_MASK_PREFETCHLINES  0xff00
#define I2SREG_0_SHIFT_PREFETCHLINES 8
#define I2SREG_0_MASK_LINESIZE       0x000f0000
#define I2SREG_0_SHIFT_LINESIZE      16
#define I2SREG_0_MASK_WSBITWIDTH     0xff000000
#define I2SREG_0_SHIFT_WSBITWIDTH    24

#define I2SREG_1_MASK_MEMSIZE        0x3fff
#define I2SREG_1_SHIFT_MEMSIZE       0
#define I2SREG_1_MASK_DELAYSLICE     0x0fff0000
#define I2SREG_1_SHIFT_DELAYSLICE    16

#define I2SREG_2_MASK_DELAYNUMBER    0x0fff
#define I2SREG_2_SHIFT_DELAYNUMBER   0
#define I2SREG_2_MASK_WRVARLENGTH    0x00010000
#define I2SREG_2_SHIFT_WRVARLENGTH   16
#define I2SREG_2_MASK_WRLINEINCBY2   0x00020000
#define I2SREG_2_SHIFT_WRLINEINCBY2  17
#define I2SREG_2_MASK_AUTOFIELDEN    0x00040000
#define I2SREG_2_SHIFT_AUTOFIELDEN   18
#define I2SREG_2_MASK_WRITESTALL     0x00080000
#define I2SREG_2_SHIFT_WRITESTALL    19
#define I2SREG_2_MASK_WRFUD          0x00700000
#define I2SREG_2_SHIFT_WRFUD         20
#define I2SREG_2_MASK_WRBURSTLENGTH  0x1f000000
#define I2SREG_2_SHIFT_WRBURSTLENGTH 24

#define I2SREG_3_MASK_WRSTARTPIXEL   0x07ff
#define I2SREG_3_SHIFT_WRSTARTPIXEL  0
#define I2SREG_3_MASK_WRSTARTLINEA   0x3fff0000
#define I2SREG_3_SHIFT_WRSTARTLINEA  16

#define I2SREG_4_MASK_WRSTARTLINEB   0x3fff
#define I2SREG_4_SHIFT_WRSTARTLINEB  0
#define I2SREG_4_MASK_READEN         0x00010000
#define I2SREG_4_SHIFT_READEN        16
#define I2SREG_4_MASK_RDLINEINCBY2   0x00020000
#define I2SREG_4_SHIFT_RDLINEINCBY2  17
#define I2SREG_4_MASK_RDVARLENGTH    0x00040000
#define I2SREG_4_SHIFT_RDVARLENGTH   18
#define I2SREG_4_MASK_RDBURSTLENGTH  0x00f00000
#define I2SREG_4_SHIFT_RDBURSTLENGTH 20
#define I2SREG_4_MASK_RDFUD          0x07000000
#define I2SREG_4_SHIFT_RDFUD         24

#define I2SREG_5_MASK_MAXRDDEPTH     0x000f
#define I2SREG_5_SHIFT_MAXRDDEPTH    0
#define I2SREG_5_MASK_SRCSEL         0x0700
#define I2SREG_5_SHIFT_SRCSEL        8
#define I2SREG_5_MASK_RDSTARTPIXEL   0x07ff0000
#define I2SREG_5_SHIFT_RDSTARTPIXEL  16

#define I2SREG_6_MASK_RDSTARTLINEA   0x3fff
#define I2SREG_6_SHIFT_RDSTARTLINEA  0
#define I2SREG_6_MASK_RDSTARTLINEB   0x3fff0000
#define I2SREG_6_SHIFT_RDSTARTLINEB  16

#define I2SREG_7_MASK_WORDDROPPED    0x0001
#define I2SREG_7_SHIFT_WORDDROPPED   0
#define I2SREG_7_MASK_UNDERFLOW      0x0002
#define I2SREG_7_SHIFT_UNDERFLOW     1
#define I2SREG_7_MASK_CLKDETEN       0x0004
#define I2SREG_7_SHIFT_CLKDETEN      2
#define I2SREG_7_MASK_NOCLK          0x0008
#define I2SREG_7_SHIFT_NOCLK         3
#define I2SREG_7_MASK_MODE           0x0010
#define I2SREG_7_SHIFT_MODE          4
#define I2SREG_7_MASK_WSMODE         0x0020
#define I2SREG_7_SHIFT_WSMODE        5
#define I2SREG_7_MASK_POLSTOUT       0x0040
#define I2SREG_7_SHIFT_POLSTOUT      6
#define I2SREG_7_MASK_CLKOUTSEL      0x0080
#define I2SREG_7_SHIFT_CLKOUTSEL     7
#define I2SREG_7_MASK_MINCLKS        0xff00
#define I2SREG_7_SHIFT_MINCLKS       8
#define I2SREG_7_MASK_CLKDETWIN      0x07ff0000
#define I2SREG_7_SHIFT_CLKDETWIN     16
#define I2SREG_7_MASK_I2OCLKOFF      0x80000000
#define I2SREG_7_SHIFT_I2OCLKOFF     31

#define I2SREG_8_MASK_AUDBASEADDR    0xffffffff
#define I2SREG_8_SHIFT_AUDBASEADDR   0

#define I2SREG_9_MASK_AUDLINESIZE    0xff
#define I2SREG_9_SHIFT_AUDLINESIZE   0

// register bank 2, +0x1000
#define I2S2REG_0 0xff00
#define I2S2REG_1 0xff04
#define I2S2REG_2 0xff08
#define I2S2REG_3 0xff0c
#define I2S2REG_4 0xff10
#define I2S2REG_5 0xff14
#define I2S2REG_6 0xff18
#define I2S2REG_7 0xff1c

#define I2S2REG_18 0xff50


#define I2S2REG_0_MASK_GOTOSTOP           0x00010000
#define I2S2REG_0_SHIFT_GOTOSTOP          16

#define I2S2REG_1_MASK_STOPATMEM          0x0001
#define I2S2REG_1_SHIFT_STOPATMEM         0
#define I2S2REG_1_MASK_STOPATLINE         0x1fff0000
#define I2S2REG_1_SHIFT_STOPATLINE        16

#define I2S2REG_2_MASK_WROVERINTSTAT      0x0001
#define I2S2REG_2_SHIFT_WROVERINTSTAT     0
#define I2S2REG_2_MASK_WROVERINTSTATRAW   0x00010000
#define I2S2REG_2_SHIFT_WROVERINTSTATRAW  16

#define I2S2REG_3_MASK_WROVERINTEN        0x0001
#define I2S2REG_3_SHIFT_WROVERINTEN       0
#define I2S2REG_3_MASK_WROVERINTSET       0x00010000
#define I2S2REG_3_SHIFT_WROVERINTSET      16

#define I2S2REG_4_MASK_WROVERINTCLR       0x0001
#define I2S2REG_4_SHIFT_WROVERINTCLR      0
#define I2S2REG_4_MASK_WROVERINTPRI       0x00010000
#define I2S2REG_4_SHIFT_WROVERINTPRI      16

#define I2S2REG_5_MASK_STOPINTSTAT        0x0001
#define I2S2REG_5_SHIFT_STOPINTSTAT       0
#define I2S2REG_5_MASK_STOPINTSTATRAW     0x00010000
#define I2S2REG_5_SHIFT_STOPINTSTATRAW    16

#define I2S2REG_6_MASK_STOPINTEN          0x0001
#define I2S2REG_6_SHIFT_STOPINTEN         0
#define I2S2REG_6_MASK_STOPINTSET         0x00010000
#define I2S2REG_6_SHIFT_STOPINTSET        16

#define I2S2REG_7_MASK_STOPINTCLR         0x0001
#define I2S2REG_7_SHIFT_STOPINTCLR        0
#define I2S2REG_7_MASK_STOPINTPRI         0x00010000
#define I2S2REG_7_SHIFT_STOPINTPRI        16


#define I2S2REG_18_MASK_I2S_STANDARD_EN         0x0001
#define I2S2REG_18_SHIFT_I2S_STANDARD_EN        0
#define I2S2REG_18_MASK_PCMFORMAT_SEL           0x0002
#define I2S2REG_18_SHIFT_PCMFORMAT_SEL          1
#define I2S2REG_18_MASK_I2S_LEFT_FIRST           0x000C
#define I2S2REG_18_SHIFT_I2S_LEFT_FIRST          2
#define I2S2REG_18_MASK_I2S_MSB_SWITCH           0x0010
#define I2S2REG_18_SHIFT_I2S_MSB_SWITCH          0x4
#define I2S2REG_18_MASK_CONVI2S2PCM_EN           0x0020
#define I2S2REG_18_SHIFT_CONVI2S2PCM_EN          0x5



//topazeh audio PLL
#define HDMI_RX_REG_0 0x00
#define HDMI_RX_REG_1 0x04
#define HDMI_RX_REG_0_MASK_AUD_PLL_REF_SEL   0x01
#define HDMI_RX_REG_0_SHIFT_AUD_PLL_REF_SEL  0 //Audio PLL Reference Clock Selection 0 : hdmi 1: xternal
#define HDMI_RX_REG_0_MASK_AUD_PLL_BYPASS     0x80
#define HDMI_RX_REG_0_SHIFT_AUD_PLL_BYPASS    7
#define HDMI_RX_REG_0_MASK_AUD_PLL_REFDIV   0x3F00
#define HDMI_RX_REG_0_SHIFT_AUD_PLL_REFDIV  8
#define HDMI_RX_REG_0_MASK_AUD_PLL_FBDIV  0xFFF0000
#define HDMI_RX_REG_0_SHIFT_AUD_PLL_FBDIV  16
#define HDMI_RX_REG_1_MASK_AUD_PLL_FRAC  0xFFFFFF
#define HDMI_RX_REG_1_SHIFT_AUD_PLL_FRAC  0
#define HDMI_RX_REG_1_MASK_AUD_PLL_POSTDIV1  0x7000000
#define HDMI_RX_REG_1_SHIFT_AUD_PLL_POSTDIV1  24
#define HDMI_RX_REG_1_MASK_AUD_PLL_POSTDIV2   0x38000000
#define HDMI_RX_REG_1_SHIFT_AUD_PLL_POSTDIV2  27

//vicotria sys v20
#define SYS_V20_CLKSRCCTL_REG 0x00000010
#define SYS_V20_CLKSRCCTL_REG_MASK_I2SCLK_SRCSEL  0xC000
#define SYS_V20_CLKSRCCTL_REG_SHIFT_I2SCLK_SRCSEL  14
#define SYS_V20_CLKSRCCTL_REG_MASK_AUDIOMCLK_SRCSEL  0x2000
#define SYS_V20_CLKSRCCTL_REG_SHIFT_AUDIOMCLK_SRCSEL  13
#define SYS_V20_CLKSRCCTL_REG_MASK_I2SOCLKSRCSEL  0x20000000
#define SYS_V20_CLKSRCCTL_REG_SHIFT_I2SOCLKSRCSEL  29

#define SYS_V20_CLKDIVEN_REG  0x00000014
#define SYS_V20_CLKDIVEN_REG_MASK_I2SCLKDIV_EN  0x400
#define SYS_V20_CLKDIVEN_REG_SHIFT_I2SCLKDIV_EN  10
#define SYS_V20_CLKDIVEN_REG_MASK_AUDIOMCLKDIV_EN  0x100000
#define SYS_V20_CLKDIVEN_REG_SHIFT_AUDIOMCLKDIV_EN  20



struct pwi2s {
	struct platform_device	*pdev;
	struct resource		*res1;
	//struct resource		*res2;
	int			 irq_wrover;
	int			 irq_stop;
	void __iomem		*regs1;
	//void __iomem		*regs2;
	void __iomem		*sys_regs;

	struct snd_card		*card;
	struct snd_pcm		*pcm;
	struct snd_hwdep	*hw;

	void			*sample_buf;
	dma_addr_t		 sample_buf_phys;
	size_t			 sample_buf_size;

	int			 enable_delay;
	u32			 audio_input_src;
	u32			 time_slice;
	u32			 delay;

	int			 playback;

	struct snd_pcm_substream *substream;
	int			 buffer_b;
	int			 set_pointer_to_end;
};


static struct snd_pcm_hardware pwi2s_playback_hw = {
	.info =		    SNDRV_PCM_INFO_INTERLEAVED
				| SNDRV_PCM_INFO_BATCH
				| SNDRV_PCM_INFO_BLOCK_TRANSFER,
	.formats =          SNDRV_PCM_FMTBIT_S16_LE,
	.rates =            SNDRV_PCM_RATE_32000 | SNDRV_PCM_RATE_44100 | SNDRV_PCM_RATE_48000
				| SNDRV_PCM_RATE_96000 | SNDRV_PCM_RATE_192000,
	.rate_min =         32000,
	.rate_max =         192000,
	.channels_min =     2,
	.channels_max =     2,
	.buffer_bytes_max = 0x2000,
	.period_bytes_min = 0x1000,
	.period_bytes_max = 0x1000,
	.periods_min =      1,
	.periods_max =      2,
};

//----------------------------------------------------------------------

static inline u32 reg_read(struct pwi2s *pwi2s, unsigned int reg)
{
	void __iomem *addr = pwi2s->regs1 + reg;
	u32 val;

	//if (reg >= 0x1000)
		//addr = pwi2s->regs2 + reg - 0x1000;
	val = readl(addr);
	dprintk(10, "%04x = %08x\n", reg, val);
	return val;
}

static void reg_write(struct pwi2s *pwi2s, unsigned int reg, u32 val)
{
	void __iomem *addr = pwi2s->regs1 + reg;

	//if (reg >= 0x1000)
	//	addr = pwi2s->regs2 + reg - 0x1000;

	dprintk(10, "%04x = %08x\n", reg, val);
	writel(val, addr);
}

#define get_field(pwi2s, reg, fld) \
	((reg_read(pwi2s, reg) & reg ## _MASK_ ## fld) >> reg ## _SHIFT_ ## fld)

#define set_field(pwi2s, reg, fld, val) \
	do { \
		reg_write(pwi2s, reg, (reg_read(pwi2s, reg) & ~reg ## _MASK_ ## fld) \
			  | (val << reg ## _SHIFT_ ## fld)); \
	} while (0)

//----------------------------------------------------------------------
static unsigned int g_nSampleRate = 0;
#if 0
static int pwi2s_set_sample_rate(struct pwi2s *pwi2s, unsigned int rate)
{
	u32 val, clk_div;

	dprintk(2, "%u\n", rate);

	if (rate < 8000)
		return -EINVAL;

	g_nSampleRate = rate;

	// use ARM clock + divisor to generate I2S clock
	val = readl(pwi2s->sys_regs + SYS_CLKCTRL2);
	val &= ~SYS_CLKCTRL2_XCLK2I2S;
	writel(val, pwi2s->sys_regs + SYS_CLKCTRL2);
	val = readl(pwi2s->sys_regs + SYS_MISC2);
	val &= ~SYS_MISC2_EXT_AUD_CLK_EN;
	writel(val, pwi2s->sys_regs + SYS_MISC2);
	val = readl(pwi2s->sys_regs + SYS_RESERVED_0);
	val &= ~7;
	val |= SYS_RESERVED_0_I2SCLK_SEL2_ARMDIVI2SCLK;
	writel(val, pwi2s->sys_regs + SYS_RESERVED_0);

	rate *= 2 * 16; // 2 channels 16bit
	// use correct rounding
	// ARM_CLK_DIV register value = clk_div + 1
	clk_div = (ARM_CLK - rate / 2) / rate;

	// this read-modify-write is potentially unsafe, but
	// no one else writes that register except during early boot init
	val = readl(pwi2s->sys_regs + SYS_CLKCTRL5);
	val &= ~SYS_CLKCTRL5_CLK_DIV_MASK;
	val |= clk_div << SYS_CLKCTRL5_CLK_DIV_SHIFT;
	writel(val, pwi2s->sys_regs + SYS_CLKCTRL5);
	return 0;
}

static int pwi2s_enable_adc_mclck(struct pwi2s *pwi2s, int source, unsigned int rate)
{
	u32 val;

	dprintk(2, "\n");

	// audio ADC provides 24bit samples packaged as 32bit -> double the rate
	// additionally it is configured for quad mode -> rate * 4
	pwi2s_set_sample_rate(pwi2s, rate * 2 * 4);

	// enable MCLK output to ADC
	val = readl(pwi2s->sys_regs + SYS_MISC2);
	val |= SYS_MISC2_EXT_AUD_CLK_EN;
	writel(val, pwi2s->sys_regs + SYS_MISC2);

	if (source > 3)
		source = 3;
	val = readl(pwi2s->sys_regs + SYS_RESERVED_0);
	val &= ~7;
	val |= source;
	writel(val, pwi2s->sys_regs + SYS_RESERVED_0);
	return 0;
}
#else
static int pwi2s_set_sample_rate(struct pwi2s *pwi2s, unsigned int rate)
{
	u32 mclk_div, mclk, sclk_div;

	dprintk(2, "%u\n", rate);

	printk("pwi2s_set_sample_rate \n");


	if (rate < 8000)
		return -EINVAL;

	g_nSampleRate = rate;

	if (pwi2s->enable_delay) {
		mclk = rate * 2 * 32 * 4;
		sclk_div = 3;
	}
	else
	{
		/*
			Denny YU:
				For 48k 16bits	Divider Value:216.7968750 -> 217 (31*7) -> 218 (109 * 2)  -> 216 (27 * 8)
				sclk_div = 6 -> 1
				mclk_div = 30 -> 108

				For 48k 16bits	Divider Value: 218  more slower
				sclk_div = 3
				mclk_div = 58

				For 48k 16bits	Divider Value: 216  more faster
				sclk_div = 7
				mclk_div = 26
		*/

		if(rate == 48000)
		{
#if 1
			// more faster
			mclk_div = 26;
			sclk_div = 7;
			goto I2S_Setting;
#else
			// more slower
			mclk_div = 108;
			sclk_div = 1;
			goto I2S_Setting;
#endif
		}

#if 1
		/*
			For 44.1k those value are more better for audio quility.
		*/
		if(rate == 44100)
		{
			mclk_div = 58;
			sclk_div = 3;
			goto I2S_Setting;
		}
#endif
		// note: 128x for 192KHz, others use 256x
		if (rate > 96000)
			sclk_div = 3;
		else
			sclk_div = 7;
		mclk = rate * 2 * 16 * (sclk_div + 1);
	}
	mclk_div = (ARM_CLK - mclk / 2) / mclk;   //26


I2S_Setting:

//	printk(KERN_ALERT"[QQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQQ111]\n");
//	printk(KERN_ALERT"rate: %u, mclk_div: %i, sclk_div: %i\n", rate, mclk_div, sclk_div);
	dprintk(2, "rate: %u, mclk_div: %i, sclk_div: %i\n", rate, mclk_div, sclk_div);



#if 0
	// this read-modify-write is potentially unsafe, but
	// no one else writes that register except during early boot init
	val = readl(pwi2s->sys_regs + SYS_CLKDIV);
	val &= ~(SYS_CLKDIV_I2SMCLK_MASK | SYS_CLKDIV_ARM_I2SMCLK_MASK);
	val |= (sclk_div << SYS_CLKDIV_I2SMCLK_SHIFT)
		| (mclk_div << SYS_CLKDIV_ARM_I2SMCLK_SHIFT);
	writel(val, pwi2s->sys_regs + SYS_CLKDIV);
	val = readl(pwi2s->sys_regs + SYS_CLKDIV_EN);
	val |= SYS_CLKDIV_I2SMCLK_EN | SYS_CLKDIV_ARM_I2SMCLK_EN;
	writel(val, pwi2s->sys_regs + SYS_CLKDIV_EN);
#endif

	return 0;
}

static int pwi2s_enable_adc_mclck(struct pwi2s *pwi2s, int source, unsigned int rate)
{
//	u32 val;

	dprintk(2, "\n");

	printk("pwi2s_enable_adc_mclck \n");

#if 0
	pwi2s_set_sample_rate(pwi2s, rate);


	val = readl(pwi2s->sys_regs + SYS_PADMUX_REG1);
	if (source >= 2 && source <= 5)
		val &= ~SYS_PADMUX_REG1_AUDIOMCLK_EN;
	else
		val |= SYS_PADMUX_REG1_AUDIOMCLK_EN;
	writel(val, pwi2s->sys_regs + SYS_PADMUX_REG1);

#endif

	return 0;
}
#endif

static int pwi2s_start(struct pwi2s *pwi2s, int delay_mode)
{
#if 0
	u32 val;
#endif

	printk("    pwi2s_start \n");

	dprintk(2, "%s\n", delay_mode ? "delay" : "stream");
	pwi2s->buffer_b = 0;
	if (delay_mode) {
		set_field(pwi2s, I2SREG_1, DELAYSLICE, pwi2s->time_slice);
		set_field(pwi2s, I2SREG_2, DELAYNUMBER, pwi2s->delay);
		set_field(pwi2s, I2SREG_5, SRCSEL, pwi2s->audio_input_src);
		pwi2s_enable_adc_mclck(pwi2s, pwi2s->audio_input_src, 44100);
	} else
	{
		set_field(pwi2s, I2SREG_5, SRCSEL, 7);
		set_field(pwi2s, I2SREG_2, DELAYNUMBER, 1);
		set_field(pwi2s, I2SREG_1, DELAYSLICE, pwi2s_slice);
	}
	set_field(pwi2s, I2S2REG_4, WROVERINTCLR, 1);
	set_field(pwi2s, I2S2REG_4, WROVERINTCLR, 0);
	set_field(pwi2s, I2S2REG_7, STOPINTCLR, 1);
	set_field(pwi2s, I2S2REG_7, STOPINTCLR, 0);
	set_field(pwi2s, I2S2REG_3, WROVERINTEN, 1);
	set_field(pwi2s, I2S2REG_6, STOPINTEN, 1);
	set_field(pwi2s, I2SREG_0, BYPASS, 0);
	set_field(pwi2s, I2SREG_0, MUTE, 0);


#if 0

	val = readl(pwi2s->sys_regs + SYS_CLKOFF);
	val &= ~SYS_CLKOFF_I2SOFF;
	writel(val, pwi2s->sys_regs + SYS_CLKOFF);

	val = readl(pwi2s->sys_regs + SYS_IPCLKCTRL);
	val |= SYS_IPCLKCTRL_I2SPOLOUT;
	writel(val, pwi2s->sys_regs + SYS_IPCLKCTRL);
#endif
	set_field(pwi2s, I2SREG_0, RDENABLE, 1);
	set_field(pwi2s, I2SREG_4, READEN, 1);
	set_field(pwi2s, I2SREG_0, EN, 1);
#if defined(CONFIG_RUBY_CHIP_C0)
	*pGPIO |= 0x2;
#endif
	return 0;
}

static int pwi2s_stop(struct pwi2s *pwi2s)
{
#if 0
	u32 val;
#endif
	printk("    pwi2s_stop \n");
	dprintk(2, "\n");
#if 0
	*pGPIO &= 0xFFFFFFFD;
#endif
	set_field(pwi2s, I2SREG_0, MUTE, 1);
	set_field(pwi2s, I2S2REG_3, WROVERINTEN, 0);
	set_field(pwi2s, I2S2REG_6, STOPINTEN, 0);
	set_field(pwi2s, I2SREG_0, RDENABLE, 0);
	set_field(pwi2s, I2SREG_4, READEN, 0);
	set_field(pwi2s, I2SREG_0, EN, 0);
	set_field(pwi2s, I2SREG_0, BYPASS, 1);
#if 0
	val = readl(pwi2s->sys_regs + SYS_CLKOFF);
	val |= SYS_CLKOFF_I2SOFF;
	writel(val, pwi2s->sys_regs + SYS_CLKOFF);

	val = readl(pwi2s->sys_regs + SYS_IPCLKCTRL);
	val &= ~SYS_IPCLKCTRL_I2SPOLOUT;
	writel(val, pwi2s->sys_regs + SYS_IPCLKCTRL);
#endif
	return 0;
}

static int pwi2s_set_buffer_config(struct pwi2s *pwi2s, struct pwi2s_buffer_config *bufcfg)
{
	u32 mem_lines;
#if 1

	printk("pwi2s_set_buffer_config \n");

	dprintk(2, "ls %u, buf %08x, sz %u\n",
		bufcfg->ap_line_size, bufcfg->buf_addr, bufcfg->buf_size);
	if (pwi2s->substream)
		snd_pcm_stop(pwi2s->substream, SNDRV_PCM_STATE_DISCONNECTED);
	pwi2s_stop(pwi2s);
	pwi2s->enable_delay = 0;
	iounmap(pwi2s->sample_buf);
	pwi2s->sample_buf = NULL;
	pwi2s->sample_buf_phys = 0;
	pwi2s->sample_buf_size = 0;

	if (bufcfg->buf_size == 0)
		return 0;

	if (bufcfg->ap_line_size > 15 || bufcfg->ap_line_size == 0
	    || bufcfg->buf_addr > 0x10000000 || bufcfg->buf_size > 0x10000000)
		return -EINVAL;

	pwi2s->sample_buf_phys = bufcfg->buf_addr;
	pwi2s->sample_buf_size = bufcfg->buf_size;
	pwi2s->sample_buf = ioremap(bufcfg->buf_addr, bufcfg->buf_size);
	if (!pwi2s->sample_buf) {
		pwi2s->sample_buf_phys = 0;
		pwi2s->sample_buf_size = 0;
		return -ENOMEM;
	}
	dprintk(3, "vbuf %p\n", pwi2s->sample_buf);
	// half of the buffer is wasted by channel indicator bits
	pwi2s_playback_hw.buffer_bytes_max = pwi2s->sample_buf_size / 2;
	pwi2s_playback_hw.period_bytes_min = pwi2s->sample_buf_size / 4;
	pwi2s_playback_hw.period_bytes_max = pwi2s->sample_buf_size / 4;

	mem_lines = bufcfg->buf_size / (bufcfg->ap_line_size * 32);

#if 1
	set_field(pwi2s, I2SREG_8, AUDBASEADDR, bufcfg->buf_addr);
	set_field(pwi2s, I2SREG_9, AUDLINESIZE, bufcfg->ap_line_size);
#endif
	set_field(pwi2s, I2SREG_0, LINESIZE, bufcfg->ap_line_size);
	// MEMSIZE is just one buffer
	set_field(pwi2s, I2SREG_1, MEMSIZE, mem_lines / 2);

	set_field(pwi2s, I2SREG_1, DELAYSLICE, pwi2s_slice);
	set_field(pwi2s, I2SREG_2, DELAYNUMBER, 1);
	set_field(pwi2s, I2SREG_5, SRCSEL, 7);

	if (pwi2s->substream)
		snd_pcm_hw_constraint_minmax(pwi2s->substream->runtime,
					     SNDRV_PCM_HW_PARAM_BUFFER_BYTES,
					     pwi2s_playback_hw.buffer_bytes_max,
					     pwi2s_playback_hw.buffer_bytes_max);
#endif
	return 0;
}

static int pwi2s_set_delay_mode(struct pwi2s *pwi2s, struct pwi2s_delay_config *delcfg)
{

#if 1
	dprintk(2, "\n");

	printk("pwi2s_set_delay_mode \n");


	if (delcfg->audio_input_src > 5 || delcfg->time_slice > 0xfff
	    || delcfg->delay > 0xfff)
		return -EINVAL;
	if (!pwi2s->sample_buf)
		return -EPROTO;
	pwi2s_stop(pwi2s);

	if (delcfg->enable_delay) {
		pwi2s->enable_delay = 1;
		pwi2s->audio_input_src = delcfg->audio_input_src;
		pwi2s->time_slice = delcfg->time_slice;
		pwi2s->delay = delcfg->delay;
	} else {
		pwi2s->enable_delay = 0;
		pwi2s->audio_input_src = 7;
		pwi2s->time_slice = 10;
		pwi2s->delay = 1;
	}
	set_field(pwi2s, I2SREG_1, DELAYSLICE, pwi2s->time_slice);
	set_field(pwi2s, I2SREG_2, DELAYNUMBER, pwi2s->delay);
	set_field(pwi2s, I2SREG_5, SRCSEL, pwi2s->audio_input_src);

	if (delcfg->enable_delay)
		pwi2s_start(pwi2s, 1);
#endif
	return 0;
}

static int pwi2s_set_pointer_mode(struct pwi2s *pwi2s, struct pwi2s_pointer_config *pntcfg)
{
	dprintk(2, "old set_pointer_to_end: %d\n", pwi2s->set_pointer_to_end);
	pwi2s->set_pointer_to_end = pntcfg->set_pointer_to_end;
	dprintk(2, "new set_pointer_to_end: %d\n", pwi2s->set_pointer_to_end);

	return 0;
}

#if (!defined(CONFIG_TOPAZ_CHIP))
static int pwi2s_set_gpio_mode(struct pwi2s *pwi2s, struct pwi2s_gpio_config *gpiocfg)
{
#if 0
	if(gpiocfg->enable_gpio)
	{
		*pGPIO |= 0x2;
	}
	else
	{
		*pGPIO &= 0xFFFFFFFD;
	}
#endif

	return 0;
}
#endif
//----------------------------------------------------------------------

static int pwi2s_hwdep_open(struct snd_hwdep *hw, struct file *file)
{
	dprintk(2, "\n");



	return 0;
}

static int pwi2s_hwdep_release(struct snd_hwdep *hw, struct file *file)
{
	struct pwi2s *pwi2s = hw->private_data;

	dprintk(2, "\n");
	if (pwi2s->substream)
		snd_pcm_stop(pwi2s->substream, SNDRV_PCM_STATE_DISCONNECTED);
	pwi2s_stop(pwi2s);
	pwi2s->enable_delay = 0;
	pwi2s->audio_input_src = 7;
	pwi2s->time_slice = 10;
	pwi2s->delay = 1;
	iounmap(pwi2s->sample_buf);
	pwi2s->sample_buf = NULL;
	pwi2s->sample_buf_phys = 0;
	pwi2s->sample_buf_size = 0;
	return 0;
}

static int pwi2s_hwdep_ioctl(struct snd_hwdep *hw, struct file *file,
			     unsigned int cmd, unsigned long arg)
{
#if 0
	u32 val;
#endif
	struct pwi2s *pwi2s = hw->private_data;
	struct pwi2s_buffer_config bufcfg;
	struct pwi2s_delay_config delcfg;
	struct pwi2s_pointer_config pntcfg;
	struct pwi2s_gpio_config gpiocfg;

	dprintk(2, "\n");
	switch (cmd) {
	case PWI2S_SET_BUFFER_CONFIG:
		if (copy_from_user(&bufcfg, (void __user *)arg, sizeof(bufcfg)))
			return -EFAULT;
		return pwi2s_set_buffer_config(pwi2s, &bufcfg);

	case PWI2S_SET_DELAY_MODE:
		if (copy_from_user(&delcfg, (void __user *)arg, sizeof(delcfg)))
			return -EFAULT;
		return pwi2s_set_delay_mode(pwi2s, &delcfg);

	case PWI2S_SET_POINTER_MODE:
		if (copy_from_user(&pntcfg, (void __user *)arg, sizeof(pntcfg)))
			return -EFAULT;
		return pwi2s_set_pointer_mode(pwi2s, &pntcfg);

	case PWI2S_SET_GPIO_MODE:
		if (copy_from_user(&gpiocfg, (void __user *)arg, sizeof(gpiocfg)))
			return -EFAULT;
#if 0
		return pwi2s_set_gpio_mode(pwi2s, &gpiocfg);
#else
		return 0;
#endif
	case PWI2S_GET_BYPASS_MODE:
		return get_field(pwi2s, I2SREG_0, BYPASS);

	case PWI2S_SET_BYPASS_MODE:
		set_field(pwi2s, I2SREG_0, BYPASS, (arg != 0));
		return 0;

	case PWI2S_GET_POLOUT_MODE:
#if 0

		val = readl(pwi2s->sys_regs + SYS_IPCLKCTRL);
		if (val & SYS_IPCLKCTRL_I2SPOLOUT)
			return 1;
		else
			return 0;
#endif
		break;
	case PWI2S_SET_POLOUT_MODE:
#if 0

		val = readl(pwi2s->sys_regs + SYS_CLKOFF);
		if (arg != 0)
			val &= ~SYS_CLKOFF_I2SOFF;
		else
			val |= SYS_CLKOFF_I2SOFF;
		writel(val, pwi2s->sys_regs + SYS_CLKOFF);

		val = readl(pwi2s->sys_regs + SYS_IPCLKCTRL);
		if (arg != 0)
			val |= SYS_IPCLKCTRL_I2SPOLOUT;
		else
			val &= ~SYS_IPCLKCTRL_I2SPOLOUT;
		writel(val, pwi2s->sys_regs + SYS_IPCLKCTRL);
		return 0;
#endif
		break;
	}
	return -ENOSYS;
}

//----------------------------------------------------------------------

static int pwi2s_playback_open(struct snd_pcm_substream *substream)
{
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime *runtime = substream->runtime;

	dprintk(2, "\n");
	runtime->hw = pwi2s_playback_hw;
	pwi2s->substream = substream;
	snd_pcm_hw_constraint_minmax(runtime, SNDRV_PCM_HW_PARAM_BUFFER_BYTES,
				     pwi2s_playback_hw.buffer_bytes_max,
				     pwi2s_playback_hw.buffer_bytes_max);
	return 0;
}

static int pwi2s_playback_close(struct snd_pcm_substream *substream)
{
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);

	dprintk(2, "\n");
	pwi2s->substream = NULL;
	return 0;
}

static int pwi2s_pcm_hw_params(struct snd_pcm_substream *substream,
				    struct snd_pcm_hw_params *hw_params)
{
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);

	dprintk(2, "\n");
	if (!pwi2s->sample_buf)
		return -EPROTO;
	if (!pwi2s->playback) {
		if (pwi2s->enable_delay)
			// preempty delay mode for playback
			pwi2s_stop(pwi2s);
		pwi2s->playback = 1;
	}
	return 0;
}

static int pwi2s_pcm_hw_free(struct snd_pcm_substream *substream)
{
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);

	dprintk(2, "\n");
	if (pwi2s->playback) {
		if (pwi2s->enable_delay)
			pwi2s_start(pwi2s, 1);
		pwi2s->playback = 0;
	}
	return 0;
}

static int pwi2s_pcm_prepare(struct snd_pcm_substream *substream)
{
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);
	struct snd_pcm_runtime *runtime = substream->runtime;
	int rc;

	dprintk(2, "\n");
	if (!pwi2s->playback)
		return -EPROTO;

	if (runtime->format != SNDRV_PCM_FORMAT_S16_LE)
		return -EINVAL;
	if (runtime->channels != 2)
		return -EINVAL;
	if (runtime->rate > 192000)
		return -EINVAL;
	dprintk(2, "format %i, channels %i, rate %i\n", runtime->format, runtime->channels, runtime->rate);
#if 1
	// in topaz MCLK is output by SOC
	rc = pwi2s_enable_adc_mclck(pwi2s, pwi2s->audio_input_src, runtime->rate);
#else
	rc = pwi2s_set_sample_rate(pwi2s, runtime->rate);
#endif
	return rc;
}

static int pwi2s_pcm_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);
	int rc = 0;

	dprintk(2, "%d\n", cmd);
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		/* do something to start the PCM engine */
		if (!pwi2s->playback)
			return -EPROTO;
		pwi2s_start(pwi2s, 0);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		/* do something to stop the PCM engine */
		if (!pwi2s->playback)
			return 0;
		pwi2s_stop(pwi2s);
		break;
	default:
		rc = -EINVAL;
	}
	return rc;
}

static unsigned int g_nStartTime = 0;
static snd_pcm_uframes_t pwi2s_pcm_pointer(struct snd_pcm_substream *substream)
{
#if 1
	snd_pcm_uframes_t pointer;
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);

	// ALSA expects the pointer to be somewhere within the
	// buffer currently played back; we don't know what position
	// the hardware is currently reading, so we'll have to report
	// the start of the buffer
	if (pwi2s->buffer_b)
		pointer = pwi2s_playback_hw.period_bytes_min / 4; // 2ch, 16bit
	else
		pointer = 0;
	dprintk(2, "%d -> %lu\n", pwi2s->buffer_b, pointer);
	return pointer;
#else
	snd_pcm_uframes_t pointer, pointer_max;
	unsigned int nNowTime = jiffies_to_msecs(jiffies);
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);

	// ALSA expects the pointer to be somewhere within the
	// buffer currently played back; we don't know what position
	// the hardware is currently reading, so we'll have to report
	// the start of the buffer

	if(pwi2s->set_pointer_to_end == 0)
	{
		if (pwi2s->buffer_b)
			pointer = pwi2s_playback_hw.period_bytes_min / 4; // 2ch, 16bit
		else
			pointer = 0;
	}
	else
	{
		pointer = g_nSampleRate * (nNowTime - g_nStartTime) / 4000;
		if (pwi2s->buffer_b)
		{
			pointer += pwi2s_playback_hw.period_bytes_min / 4;
			pointer_max = pwi2s_playback_hw.period_bytes_min / 2 - 1; // 2ch, 16bit
		}
		else
		{
			pointer_max = pwi2s_playback_hw.period_bytes_min / 4 - 1; // 2ch, 16bit
		}
		pointer = pointer > pointer_max ? pointer_max : pointer;

		/*
		   if (pwi2s->buffer_b)
		   pointer = pwi2s_playback_hw.period_bytes_min / 2 - 1; // 2ch, 16bit
		   else
		   pointer = pwi2s_playback_hw.period_bytes_min / 4 - 1; // 2ch, 16bit
		 */
	}

	dprintk(2, "%d -> %lu\n", pwi2s->buffer_b, pointer);
	return pointer;
#endif
}

static void pwi2s_sample_convert(u32 src, u32 *dst)
{
	// its for left justified data format
	//u32 left = 0x00000000, right = 0xaaaaaaaa;
	// note: i2s data format requires LRCK to toggle one cycle in advance
	u32 left = 0x80000000, right = 0x2aaaaaaa;

	int i;

	//if (unlikely(pwi2s_dummy_write))
	//	return;
	// note: this reverses the order of the bits
	for (i = 15; i >= 0; i--) {
		if (src & 1)
			left |= 1 << (i * 2);
		src >>= 1;
	}
	for (i = 15; i >= 0; i--) {
		if (src & 1)
			right |= 1 << (i * 2);
		src >>= 1;
	}

	// and the hw also has different endianness
	*dst++ = swab32(left);
	*dst = swab32(right);
}

static int pwi2s_pcm_copy(struct snd_pcm_substream *substream, int channel,
			  snd_pcm_uframes_t pos, void __user *buf, snd_pcm_uframes_t count)
{
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);
	u32 __user *src; // 2chan 16bit
	u32 *dst; // 2chan 16bit + channel indicators
	u32 tmp;

	dprintk(2, "ch %d, pos %lu, buf %p, count %lu\n",
		channel, pos, buf, count);
	if (!pwi2s->playback || !pwi2s->sample_buf)
		return -EPROTO;
	src = buf;
	dst = pwi2s->sample_buf;
	dst += pos * 2;  // half the buffer is channel indicator bits
	dprintk(3, "%p -> %p (%lu u32)\n", src, dst, count);
	while (count--) {
		if (get_user(tmp, src++)) {
			dprintk(1, "EFAULT accessing %p\n", --src);
			return -EFAULT;
		}
		pwi2s_sample_convert(tmp, dst);
		dst += 2;
	}
	dprintk(3, "next dst = %p\n", dst);
	return 0;
}

static int pwi2s_pcm_silence(struct snd_pcm_substream *substream, int channel,
			     snd_pcm_uframes_t pos, snd_pcm_uframes_t count)
{
// suppose it is not necessary
#if 0
	struct pwi2s *pwi2s = snd_pcm_substream_chip(substream);
	u32 *dst; // 2chan 16bit

	dprintk(2, "ch %d, pos %lu, count %lu\n", channel, pos, count);
	if (!pwi2s->playback || !pwi2s->sample_buf)
		return -EPROTO;
	BUG_ON(!pwi2s->sample_buf);
	dst = pwi2s->sample_buf;
	dst += pos * 2;  // half the buffer is channel indicator bits
	while (count--) {
		// write zeros for left + right channel
		// note: i2s data format requires LRCK to toggle one cycle in advance
		*dst++ = 0x00000080;
		*dst++ = 0xaaaaaa2a;
	}
#endif
	return 0;
}

static struct snd_pcm_ops pwi2s_playback_ops = {
	.open      = pwi2s_playback_open,
	.close     = pwi2s_playback_close,
	.ioctl     = snd_pcm_lib_ioctl,
	.hw_params = pwi2s_pcm_hw_params,
	.hw_free   = pwi2s_pcm_hw_free,
	.prepare   = pwi2s_pcm_prepare,
	.trigger   = pwi2s_pcm_trigger,
	.pointer   = pwi2s_pcm_pointer,
	.copy	   = pwi2s_pcm_copy,
	.silence   = pwi2s_pcm_silence,
};


static irqreturn_t pwi2s_irq(int irq, void *dev)
{
	struct pwi2s *pwi2s = dev;
	u32 status;

	printk("pwi2s_irq \n");

#if 1

	dprintk(6, "%d\n", irq);
	if (irq == pwi2s->irq_wrover) {
		status = get_field(pwi2s, I2S2REG_2, WROVERINTSTAT);
		set_field(pwi2s, I2S2REG_4, WROVERINTCLR, 1);
		set_field(pwi2s, I2S2REG_4, WROVERINTCLR, 0);
		// on each WROVER irq, switch between A and B buffer half
		pwi2s->buffer_b = !pwi2s->buffer_b;
		g_nStartTime = jiffies_to_msecs(jiffies);
		dprintk(5, "WROVER IRQ %u: new buffer is %d\n", status, pwi2s->buffer_b);
		snd_pcm_period_elapsed(pwi2s->substream);
	} else if (irq == pwi2s->irq_stop) {
		status = get_field(pwi2s, I2S2REG_5, STOPINTSTAT);
		set_field(pwi2s, I2S2REG_7, STOPINTCLR, 1);
		set_field(pwi2s, I2S2REG_7, STOPINTCLR, 0);
		dprintk(5, "STOP IRQ %u\n", status);
	}
	return IRQ_HANDLED;
#else
	return IRQ_HANDLED;

#endif

}

static void pwi2s_init_hw(struct pwi2s *pwi2s)
{
#if 0
	u32 val;
#endif

#if 1

	dprintk(2, "\n");

	// configure audio processing write FIFO
	set_field(pwi2s, I2SREG_2, WRVARLENGTH, 1);
	set_field(pwi2s, I2SREG_2, WRLINEINCBY2, 0);
	set_field(pwi2s, I2SREG_2, AUTOFIELDEN, 0);
	set_field(pwi2s, I2SREG_2, WRITESTALL, 0);
	set_field(pwi2s, I2SREG_2, WRFUD, 0);
	set_field(pwi2s, I2SREG_2, WRBURSTLENGTH, 8);
	set_field(pwi2s, I2SREG_3, WRSTARTPIXEL, 0);
	set_field(pwi2s, I2SREG_3, WRSTARTLINEA, 0);
	set_field(pwi2s, I2SREG_4, WRSTARTLINEB, 0);

	// configure audio processing read FIFO
	set_field(pwi2s, I2SREG_4, RDVARLENGTH, 1);
	set_field(pwi2s, I2SREG_4, RDLINEINCBY2, 0);
	set_field(pwi2s, I2SREG_4, RDBURSTLENGTH, 8);
	set_field(pwi2s, I2SREG_4, RDFUD, 0);
	set_field(pwi2s, I2SREG_5, RDSTARTPIXEL, 0);
	set_field(pwi2s, I2SREG_6, RDSTARTLINEA, 0);
	set_field(pwi2s, I2SREG_6, RDSTARTLINEB, 0);

	// configure so data transitions on rising edge of clock
	set_field(pwi2s, I2SREG_0, INPOL, 0);
	// I2S word size
	set_field(pwi2s, I2SREG_0, WSBITWIDTH, 32);

	// initialize to disabled state (no clock, no memory access)
	set_field(pwi2s, I2SREG_0, BYPASS, 1);
	set_field(pwi2s, I2SREG_0, MUTE, 1);
#if  0
	val = readl(pwi2s->sys_regs + SYS_CLKOFF);
	val |= SYS_CLKOFF_I2SOFF;
	writel(val, pwi2s->sys_regs + SYS_CLKOFF);

	// I2SPOLOUT set 1 when start play, now it make I2OSCLK low
	val = readl(pwi2s->sys_regs + SYS_IPCLKCTRL);
	val &= ~SYS_IPCLKCTRL_I2SPOLOUT;
	writel(val, pwi2s->sys_regs + SYS_IPCLKCTRL);

	val = readl(pwi2s->sys_regs + SYS_IPCLKCTRL);
	val &= ~SYS_IPCLKCTRL_I2OSRCSEL;
	writel(val, pwi2s->sys_regs + SYS_IPCLKCTRL);

	val = readl(pwi2s->sys_regs + SYS_PADMUX_REG2);
	val |= SYS_PADMUX_REG2_I2SOD_OUTPUT_ENABLE
		| SYS_PADMUX_REG2_I2SOWS_OUTPUT_ENABLE;
	writel(val, pwi2s->sys_regs + SYS_PADMUX_REG2);
#endif
	set_field(pwi2s, I2SREG_0, EN, 0);
	set_field(pwi2s, I2SREG_0, RDENABLE, 0);
	set_field(pwi2s, I2SREG_4, READEN, 0);
	set_field(pwi2s, I2SREG_7, MINCLKS, 3);

	// disable and clear irqs
	set_field(pwi2s, I2S2REG_3, WROVERINTEN, 0);
	set_field(pwi2s, I2S2REG_4, WROVERINTPRI, 0);
	set_field(pwi2s, I2S2REG_4, WROVERINTCLR, 1);
	set_field(pwi2s, I2S2REG_4, WROVERINTCLR, 0);
	set_field(pwi2s, I2S2REG_6, STOPINTEN, 0);
	set_field(pwi2s, I2S2REG_7, STOPINTPRI, 0);
	set_field(pwi2s, I2S2REG_7, STOPINTCLR, 1);
	set_field(pwi2s, I2S2REG_7, STOPINTCLR, 0);

#endif

}

static int pwi2s_probe(struct platform_device *pdev)
{

#if 1

	struct snd_card *card;
	struct snd_pcm *pcm;
	struct snd_hwdep *hw;
	struct pwi2s *pwi2s;
	//struct resource *res1, *res2;
	struct resource *res1;
	int rc, irq_wrover, irq_stop;


	printk("pwi2s_probe 3aaaaa \n");
	//printk("i2s  = 0x%x\n",res1->start);
	//printk("i2s 2 = 0x%x\n",res2->start);
	//printk("sys     = 0x%x\n",TOPAZEH_SYS_BASE);


	dprintk(2, "\n");
	irq_wrover = platform_get_irq_byname(pdev, "wrover");
	if (irq_wrover < 0) {
		dprintk(1, "Cannot get wrover IORESOURCE_IRQ\n");
		return -ENODEV;
	}
	irq_stop = platform_get_irq_byname(pdev, "stop");
	if (irq_stop < 0) {
		dprintk(1, "Cannot get stop IORESOURCE_IRQ\n");
		return -ENODEV;
	}

	res1 = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res1) {
		dprintk(1, "Cannot get IORESOURCE_MEM 1\n");
		return -ENODEV;
	}

#if 0
	res2 = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!res2) {
		dprintk(1, "Cannot get IORESOURCE_MEM 2\n");
		return -ENODEV;
	}
#endif

	res1 = request_mem_region(res1->start, resource_size(res1), pdev->name);
	if (!res1) {
		dev_err(&pdev->dev, "could not claim io area 1\n");
		return -EBUSY;
	}


#if 0
	res2 = request_mem_region(res2->start, resource_size(res2), pdev->name);
	if (!res2) {
		dev_err(&pdev->dev, "could not claim io area 2\n");
		return -EBUSY;
	}
#endif

	rc = snd_card_new(&pdev->dev, 0, "pwi2s", THIS_MODULE, sizeof(*pwi2s), &card);
	if (rc) {
		dprintk(1, "snd_card_create failed: %d\n", rc);
		//release_resource(res2);
		release_resource(res1);
		return rc;
	}
	dprintk(3, "card is at %p, private_data is %p\n", card, card->private_data);
	pwi2s = card->private_data;
	pwi2s->irq_wrover = -1;
	pwi2s->irq_stop = -1;
	pwi2s->card = card;
	pwi2s->pdev = pdev;
	pwi2s->res1 = res1;
	//pwi2s->res2 = res2;
	pwi2s->set_pointer_to_end = 0;
	platform_set_drvdata(pdev, pwi2s);

	pwi2s->regs1 = ioremap(res1->start, resource_size(res1));
	if (!pwi2s->regs1) {
		dev_err(&pdev->dev, "could not map io area 1\n");
		rc = -ENOMEM;
		goto out;
	}

	//pwi2s->regs2 = pwi2s->regs1;


#if 0
	pwi2s->regs2 = ioremap(res2->start, resource_size(res2));
	if (!pwi2s->regs2) {
		dev_err(&pdev->dev, "could not map io area 2\n");
		rc = -ENOMEM;
		goto out;
	}
#endif

	pwi2s->sys_regs = ioremap(TOPAZEH_SYS_BASE, 0x100);
	if (!pwi2s->sys_regs) {
		dev_err(&pdev->dev, "could not map sys io area\n");
		rc = -ENOMEM;
		goto out;
	}

	//dprintk(3, "regs1 %p, regs2 %p, sys_regs %p\n",
	//	pwi2s->regs1, pwi2s->regs2, pwi2s->sys_regs);

	pwi2s_init_hw(pwi2s);
	dprintk(3, "hardware intialized\n");

	rc = request_irq(irq_wrover, pwi2s_irq, 0, pdev->name, pwi2s);
	if (rc) {
		dev_err(&pdev->dev, "could not claim wrover irq\n");
		goto out;
	}
	pwi2s->irq_wrover = irq_wrover;
	rc = request_irq(irq_stop, pwi2s_irq, 0, pdev->name, pwi2s);
	if (rc) {
		dev_err(&pdev->dev, "could not claim stop irq\n");
		goto out;
	}
	pwi2s->irq_stop = irq_stop;
	dprintk(3, "registered irqs\n");
#if 1
	rc = snd_pcm_new(card, "pwi2s", 0, 1, 0, &pcm);
	if (rc) {
		dprintk(1, "snd_pcm_new failed: %d\n", rc);
		goto out;
	}
	dprintk(3, "added pcm device at %p\n", pcm);
	pwi2s->pcm = pcm;
	pcm->private_data = pwi2s;
	strcpy(pcm->name, "pwi2s");
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK, &pwi2s_playback_ops);



	rc = snd_hwdep_new(card, "buf conf", 0, &hw);
	if (rc) {
		dprintk(1, "snd_hwdep_new failed: %d\n", rc);
		goto out;
	}
	dprintk(3, "added hwdep device at %p\n", hw);
	pwi2s->hw = hw;
	hw->private_data = pwi2s;
	hw->exclusive = 1;
	strcpy(hw->name, hw->id);
	hw->ops.open = pwi2s_hwdep_open;
	hw->ops.release = pwi2s_hwdep_release;
	hw->ops.ioctl = pwi2s_hwdep_ioctl;

	snd_card_set_dev(card, &pdev->dev);
	strcpy(card->driver, "pwi2s");
	strcpy(card->shortname, "pwi2s");
	strcpy(card->longname, "Pixelworks I2S");
	dprintk(3, "register card\n");
	rc = snd_card_register(card);
	if (rc)
		dprintk(1, "snd_card_register failed: %d\n", rc);
	else
		dprintk(2, "card registered\n");

	printk("......card registered");

#endif


out:
	if (rc)
		snd_card_free(card);



	return rc;
#else

	struct resource *res1, *res2;
	res1 = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res1) {
		dprintk(1, "Cannot get IORESOURCE_MEM 1\n");
		return -ENODEV;
	}
	res2 = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!res2) {
		dprintk(1, "Cannot get IORESOURCE_MEM 2\n");
		return -ENODEV;
	}
	printk("    pwi2s_probe \n");
	printk("i2s  = 0x%x\n",res1->start);
	printk("i2s 2 = 0x%x\n",res2->start);
	printk("sys     = 0x%x\n",TOPAZEH_SYS_BASE);


	return -ENOMEM;


#endif

}

static int pwi2s_remove(struct platform_device *pdev)
{
#if 0
	u32 val;
#endif
	struct pwi2s *pwi2s = platform_get_drvdata(pdev);

	dprintk(2, "\n");
	if (pwi2s->irq_wrover != -1)
		free_irq(pwi2s->irq_wrover, pwi2s);
	if (pwi2s->irq_stop != -1)
		free_irq(pwi2s->irq_stop, pwi2s);



#if 0
	val = readl(pwi2s->sys_regs + SYS_IPCLKCTRL);
	val &= ~SYS_IPCLKCTRL_I2SPOLOUT;
	writel(val, pwi2s->sys_regs + SYS_IPCLKCTRL);
#endif

	iounmap(pwi2s->regs1);
	//iounmap(pwi2s->regs2);
	iounmap(pwi2s->sys_regs);
	release_resource(pwi2s->res1);
	//release_resource(pwi2s->res2);
	platform_set_drvdata(pdev, NULL);
	snd_card_free(pwi2s->card);

	return 0;
}

#if 0


static struct platform_driver pwi2sdrv = {
	.probe          = pwi2s_probe,
	.remove         = pwi2s_remove,
	.driver         = {
		.name   = "pwi2s",
		.owner  = THIS_MODULE,
	},
};
#else

static struct of_device_id pwi2s_hw_of_match[]={
	{ .compatible = "pixelworks,i2s_delay_v2" },
	{ },
};

static struct platform_driver pwi2sdrv = {
	.probe      = pwi2s_probe,
	.remove     = pwi2s_remove,
	//.suspend    = pw2wire_hw_suspend,
	//.resume     = pw2wire_hw_resume,
	.driver     = {
		.name       = "pwi2s",
		.owner = THIS_MODULE,
		.of_match_table = pwi2s_hw_of_match,
	}
};



#endif


static int __init pwi2s_init_module(void)
{
	dprintk(2, "\n");
	printk("pwi2s_init_module Jan 21 10:15\n");


#if 0
	pGPIO = ioremap(0xffca03fc,0x4);
	*pGPIO &= 0xFFFFFFFD;
#endif
	return platform_driver_register(&pwi2sdrv);
}

static void __exit pwi2s_exit_module(void)
{

	printk("pwi2s_exit_module 2222\n");

	dprintk(2, "\n");
	platform_driver_unregister(&pwi2sdrv);
	dprintk(2, "driver unloaded\n");
}

module_init(pwi2s_init_module);
module_exit(pwi2s_exit_module);
