/*
 * pwgpio.c -- Pixelworks gpio API
 *
 * Copyright (C) 2007-2012 Pixelworks Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/io.h>
#include <linux/list.h>
#include <linux/spinlock.h>
#include <linux/of_irq.h>
#include <linux/of.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/device.h>
#include <asm/uaccess.h>

#include <mach/platform.h>

#include <mach/gpio.h>
//#include <linux/gpio.h>

#include "types.h"
#include "ral.h"
#include "gpio_v2_ral.h"
#include <pxlw/pwgpio.h>
#include <mach/pwgpio.h>
#include <mach/hardware.h>

MODULE_AUTHOR ("Pixelworks Inc.");
MODULE_LICENSE("GPL");


#ifdef CONFIG_ARCH_PIXELWORKS_TOPAZ
#define SYS_PORT_CTRL		0xFFFA0064
#elif defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
#define SYS_PORT_CTRL		0xfd000000
#else
#error chip not supported
#endif

#ifdef CONFIG_ARCH_PIXELWORKS_TOPAZEH
#define REG_ADDR_SYS_PADALTFUNCCTRL1 0x00000500
#define REG_ADDR_SYS_PADALTFUNCCTRL2 0x00000504
#define REG_ADDR_SYS_PADALTFUNCCTRL3 0x00000508
#define REG_ADDR_SYS_PADPULLCTRL1 0x00000514
#endif
// hardware ID values
#define GPIO_V2_PARTNO   0x061
#define GPIO_V2_DESIGNER 0x41
#define GPIO_V2_REVISION 0x00
#define GPIO_V2_NUM_PINS 8

static int gpio_mask[16];
static int pin_mask[32];
module_param_array(pin_mask, int, NULL, 0);


#define DEBUG
#ifdef DEBUG
static int pwgpio_debug = 1;
module_param_named(debug, pwgpio_debug, int, 0644);
MODULE_PARM_DESC(debug, "debug level (0 off, 1 errors, 2 trace, 3 full)");

#define dprintk(level, fmt, ...) \
	do { \
		if (pwgpio_debug >= level) \
			printk(KERN_DEBUG "PWGPIO: %s:%d: " fmt, \
			       __func__, __LINE__, ## __VA_ARGS__); \
	} while (0)
#define ddprintk(level, pwgpio, fmt, ...) \
	do { \
		if (pwgpio_debug >= level) \
			printk(KERN_DEBUG "%s: %s:%d: " fmt, \
			       dev_name(&pwgpio->pwgpio_data->pdev->dev), \
			       __func__, __LINE__, ## __VA_ARGS__); \
	} while (0)
#else
#define dprintk(...) do { } while (0)
#define ddprintk(...) do { } while (0)
#endif

static dev_t pwgpio_dev;
struct pwgpio_events_buffer {
	u32 write_pos;
	u32 read_pos;
	u32 freq_bkup;
	u32 time_stamp;
	struct pwgpio_events_record buf[PWGPIO_EVENT_RECORD_MAX];
};

struct pwgpio_instance {
	struct list_head node;		// node on struct pwgpio_data.bufs
	struct pwgpio_data *pwgpio_data;// back pointer
	u32 used_pins;

	// irq trigger events
	// note: these are not buffered since multiple readers can
	// selectively read a few bits using a mask; thus we just
	// or event bits and on read() reset them all
	u32 events;

	wait_queue_head_t wq;	// for waking up process on GPIO input event
};

struct pwgpio_data {
        unsigned long phys_base_addr;
        unsigned long reg_size;
        void *base_addr;
        int irq;

        // device numbers (allocated dynamically)
        int major;
        int minor;


	// ringbuffer for input data, one per reader
	struct list_head bufs;
	// lock to protect the bufs list, and also
	// all pwgpio_instance.events and used_pins
	spinlock_t lock;

        struct cdev cdev;       // char dev structure
        struct device *dev;
        struct platform_device *pdev;   // keep for reference

        BlockInfo blkinfo;

	struct pwgpio_events_buffer events_buf;
};

static int pwgpio_major;    //Dynamic allocation
static struct class *pwgpio_class;

static int pwgpio_open(struct inode *inode, struct file *filp)
{

        struct pwgpio_data *pwgpio_data = container_of(inode->i_cdev, struct pwgpio_data, cdev);
	struct pwgpio_instance *pwgpio;
	unsigned long flags;

	pwgpio = kzalloc(sizeof(*pwgpio), GFP_KERNEL);
	if (!pwgpio) {
		dev_err(&pwgpio_data->pdev->dev, "out of memory\n");
		return -ENOMEM;
	}

	init_waitqueue_head(&pwgpio->wq);
	pwgpio->pwgpio_data = pwgpio_data;
	pwgpio->used_pins = pin_mask[(pwgpio_data->minor * 2) + 1];	
	ddprintk(1, pwgpio, "usedpins: 0x%x\n", pwgpio->used_pins);

	spin_lock_irqsave(&pwgpio_data->lock, flags);
	list_add(&pwgpio->node, &pwgpio_data->bufs);
	spin_unlock_irqrestore(&pwgpio_data->lock, flags);

	filp->private_data = pwgpio; // remember for later use
	return 0;
}

static int pwgpio_release(struct inode *inode, struct file *filp)
{
	struct pwgpio_instance *pwgpio = filp->private_data;
	struct pwgpio_data *pwgpio_data = pwgpio->pwgpio_data;
	unsigned long flags;
	u32 ie;

        ddprintk(2, pwgpio, "\n");
        spin_lock_irqsave(&pwgpio_data->lock, flags);
        // disable irqs
        GPIO_V2_IE_GET(&pwgpio_data->blkinfo, IE(ie));
        ie &= ~pwgpio->used_pins;
        GPIO_V2_IE_SET(&pwgpio_data->blkinfo, IE(ie));

        list_del(&pwgpio->node);
        spin_unlock_irqrestore(&pwgpio_data->lock, flags);

	kfree(pwgpio);
	
	return 0;
}

static unsigned int pwgpio_poll(struct file *filp, poll_table *wait)
{
	struct pwgpio_instance *pwgpio = filp->private_data;
	int mask = 0;

	ddprintk(3, pwgpio, "\n");
	poll_wait(filp, &pwgpio->wq, wait);

	if (pwgpio->events != 0)
		mask = POLLIN | POLLRDNORM;

	return mask;
}

static u32 event_timestamp[PWGPIO_EVENT_RECORD_MAX];
static u32 event_interval[PWGPIO_EVENT_RECORD_MAX];

static void ascend_sort( u32 *a, u32 length )
{
        u32 i;
        u32 j;
        u32 tmp;

        for( i = 0; i < length; i++ )
        {
                for( j = i; j < length; j++ )
                {
                        if( a[i] > a[j] )
                        {
                                tmp  = a[i];
                                a[i] = a[j];
                                a[j] = tmp;
                        }
                }
        }
}
static void prepare_samples( u32 cnt )
{
        u32 i;

        if( cnt < 2 )
        {
                return;
        }

        //Get interval between samples
        for( i = 0; i < cnt - 1; i++ )
        {
                event_interval[i] = event_timestamp[i + 1] - event_timestamp[i];
        }

        //Sort interval values in ascend order
        ascend_sort( event_interval, cnt - 1 );
}


/* return 32 bit timestamp with ~50nsec resolution
 * FIXME: it would be better to change struct pwgpio_events_record
 * to use 64 nsec timestamps
 */
static u32 get_timestamp(void)
{
	unsigned long long nsec = sched_clock();
	return (u32)(nsec >> 6);
}

static irqreturn_t pwgpio_interrupt(int irq, void *dev_id)
{
	struct pwgpio_instance *pwgpio ;
	struct pwgpio_data *pwgpio_data = dev_id;
	//struct pwgpio_events_buffer *buffer;
	//struct pwgpio_events_record *record;
	u32 events, level_trig, mask;

        //dprintk(1,"pwgpio_interrupt, irq: %d, id: %d, blkinfo 0x%08x, addr 0x%08x\n",irq, pwgpio_data->minor, &pwgpio_data->blkinfo, pwgpio_data->base_addr);

	 //get triggered irqs
        GPIO_V2_MIS_GET(&pwgpio_data->blkinfo, MIS(events));
        if (!events)
        {
            dprintk(3, "pwgpio_interrupt: 0x%02x\n", events);
            return IRQ_NONE; // irq is from other GPIO block
        }

         dprintk(3, "pwgpio_interrupt: 0x%02x\n", events);	

         // if this is a level triggered irq, disable it
        // (userspace has to reenable it after it has
        // taken care of the irq source)
        GPIO_V2_IS_GET(&pwgpio_data->blkinfo, IS(level_trig));
        level_trig &= events;
        if (level_trig) {
                GPIO_V2_IE_GET(&pwgpio_data->blkinfo, IE(mask));
                dprintk(3, "pwgpio_interrupt: disable level triggered irqs %02x "
                        "(mask old %02xm new %02x)\n",
                        level_trig, mask, mask & ~level_trig);
                mask &= ~level_trig;
                GPIO_V2_IE_SET(&pwgpio_data->blkinfo, IE(mask));
        }


      
	spin_lock(&pwgpio_data->lock);
        list_for_each_entry(pwgpio, &pwgpio_data->bufs, node){
        dprintk(3, "pwgpio_interrupt: 0x%02x, used_pins: 0x%x\n", events, pwgpio->used_pins);
        if (!(events & pwgpio->used_pins))
                continue;

               pwgpio->events |= events;
                wake_up(&pwgpio->wq);
                {
                        //Record the event
                        struct pwgpio_events_buffer *buffer;
                        struct pwgpio_events_record *record;
                        buffer = &pwgpio_data->events_buf;                        
                        dprintk(3, "read_pos: %d, write_pos:%d, MAX: %d\n", buffer->read_pos,  buffer->write_pos, PWGPIO_EVENT_RECORD_MAX);
                        //Adjust read/write position
                        if( buffer->read_pos  >= PWGPIO_EVENT_RECORD_MAX &&
                            buffer->write_pos >= PWGPIO_EVENT_RECORD_MAX )
                        {
                                buffer->read_pos  -= PWGPIO_EVENT_RECORD_MAX;
                                buffer->write_pos -= PWGPIO_EVENT_RECORD_MAX;
                        }
                        //Save the events
                        record = &buffer->buf[buffer->write_pos % PWGPIO_EVENT_RECORD_MAX];
                         record->events = events | (events<<16); //High word:History, Low word:Event
                        record->tickcount = get_timestamp();

                        //Adjust read/write position
                        buffer->write_pos++;
                        if( buffer->write_pos - buffer->read_pos >= PWGPIO_EVENT_RECORD_MAX )
                        {
                                buffer->read_pos = buffer->write_pos - PWGPIO_EVENT_RECORD_MAX;
                                //Adjust read/write position
                                if( buffer->read_pos >= PWGPIO_EVENT_RECORD_MAX )
                                {
                                        buffer->read_pos  -= PWGPIO_EVENT_RECORD_MAX;
                                        buffer->write_pos -= PWGPIO_EVENT_RECORD_MAX;
                                }
                        }
                }
       } 
        spin_unlock(&pwgpio_data->lock);

        // clear interrupt
        GPIO_V2_IC_SET(&pwgpio_data->blkinfo, IC(events));

	return IRQ_HANDLED;
}

static u32 convert_to_frequency(u32 cnt)
{
	unsigned long long ulval;
	u32 freq;
	u32 v;

	//Get the frequency
	freq = 0;

	v = event_interval[cnt - 1]; //Last data contains longest period
	if (v) {
		ulval = 1000000000ull >> 6; // see get_timestamp()
		do_div(ulval, v);
		freq = (u32)ulval;
	}

	return freq;
}
/*
static int pwgpio_configure(struct pwgpio_instance *pwgpio,
			    int i, int pin_used, int out, int irq_mode)
{
	struct pwgpio_data *pwgpio_data = pwgpio->pwgpio_data;
	int gpio, irq, irqf, rc, reenable;

	gpio = pwgpio_data->gpio_base + i;

	if (!pin_used) {
		if (pwgpio->gpio[i] == gpio) {
			ddprintk(3, pwgpio, "release gpio %d/irq %d\n",
				 gpio, pwgpio->irq[i]);
			if (pwgpio->irq[i])
				free_irq(pwgpio->irq[i], pwgpio);
			gpio_free(gpio);
			pwgpio->gpio[i] = -1;
			pwgpio->irq[i] = 0;
			pwgpio->irqf[i] = 0;
		}
		return 0;
	}

	if (pwgpio->gpio[i] == -1) {
		ddprintk(3, pwgpio, "request gpio %d (%s)\n", gpio,
			 out ? "out" : "in");
		rc = gpio_request_one(gpio,
				      out ? GPIOF_DIR_OUT : GPIOF_DIR_IN,
				      "pwgpio");
		if (rc < 0) {
			dev_err(&pwgpio_data->pdev->dev,
				"request_gpio error %d\n", rc);
			return rc;
		}
		pwgpio->gpio[i] = gpio;
	} else {
		ddprintk(3, pwgpio, "update gpio %d (%s)\n", gpio,
			 out ? "out" : "in");
		if (out)
			gpio_direction_output(gpio, gpio_get_value(gpio));
		else
			gpio_direction_input(gpio);
	}

	switch (irq_mode) {
	case PWGPIO_IRQ_MODE_NONE:
	default:
		irqf = 0;
		break;
	case PWGPIO_IRQ_MODE_LEVEL_LOW:
		irqf = IRQF_TRIGGER_LOW;
		break;
	case PWGPIO_IRQ_MODE_LEVEL_HIGH:
		irqf = IRQF_TRIGGER_HIGH;
		break;
	case PWGPIO_IRQ_MODE_EDGE_RISING:
		irqf = IRQF_TRIGGER_RISING;
		break;
	case PWGPIO_IRQ_MODE_EDGE_FALLING:
		irqf = IRQF_TRIGGER_FALLING;
		break;
	case PWGPIO_IRQ_MODE_EDGE_BOTH:
		irqf = IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING;
		break;
	}
	reenable = pwgpio->irqf[i] & IRQF_NEED_REENABLE;
	pwgpio->irqf[i] &= ~IRQF_NEED_REENABLE;
	if (pwgpio->irqf[i] != irqf && pwgpio->irq[i]) {
		ddprintk(3, pwgpio, "gpio %d: release irq %d\n", gpio, pwgpio->irq[i]);
		free_irq(pwgpio->irq[i], pwgpio);
		pwgpio->irq[i] = 0;
		pwgpio->irqf[i] = 0;
	}
	if (pwgpio->irqf[i] != irqf) {
		irq = gpio_to_irq(gpio);
		ddprintk(3, pwgpio, "gpio %d: request irq %d (irqf %#x)\n",
			 gpio, irq, irqf);
		rc = request_irq(irq,
				 pwgpio_interrupt,
				 irqf, "pwgpio", pwgpio);
		if (rc < 0) {
			dev_err(&pwgpio_data->pdev->dev,
				"request_irq error %d\n", rc);
			return rc;
		}
		pwgpio->irq[i] = irq;
		pwgpio->irqf[i] = irqf;
	}
	else if (irqf && reenable)
		enable_irq(pwgpio->irq[i]);
	return 0;
}
*/
static long pwgpio_ioctl(struct file *filp, unsigned cmd, unsigned long arg)
{
	struct pwgpio_instance *pwgpio = filp->private_data, *orb;
	struct pwgpio_data *pwgpio_data = pwgpio->pwgpio_data;
	unsigned long flags;
	int result = 0;

        ddprintk(2, pwgpio, "Port %d cmd = %08x, arg %08lx\n",pwgpio_data->minor, cmd, arg);

	switch (cmd) {
	case PWGPIO_CAPABILITIES:
	{
		struct pwgpio_capabilitites caps = {
                        .num_pins = GPIO_V2_NUM_PINS
		};
		ddprintk(2, pwgpio, "PWGPIO_CAPABILITIES\n");
		if (copy_to_user((void __user *) arg, &caps, sizeof(caps)))
			return -EFAULT;
		break;
	}
	case PWGPIO_CONFIGURE:
	{
		struct pwgpio_config conf;
                u32 ie, dir, level, bothedges, positive;
                int i;

                if (copy_from_user(&conf, (void __user *) arg, sizeof(conf))) {
                        result = -EFAULT;
                        break;
                }
                ddprintk(2, pwgpio, "PWGPIO_CONFIGURE pins %02x, outputs %02x\n",
                         conf.pin_mask, conf.is_output);
                if (conf.pin_mask >= (1 << GPIO_V2_NUM_PINS))
                        return -EINVAL;
                // check if pin is reserved
                //printk("XXXX[0x%x], YYYY[0x%x]\n",(u32)pwgpio_data->pdev->dev.platform_data,conf.pin_mask);
                if (conf.pin_mask & (u32)pwgpio_data->pdev->dev.platform_data)
                        return -EBUSY;

		spin_lock_irqsave(&pwgpio_data->lock, flags);

		// check if one of the pins is in use by someone else
		list_for_each_entry(orb, &pwgpio_data->bufs, node) {
			if ((orb != pwgpio) && (conf.pin_mask & orb->used_pins)) {
				spin_unlock_irqrestore(&pwgpio_data->lock, flags);
				return -EBUSY;
			}
		}

                // disable irqs on our pins
                GPIO_V2_IE_GET(&pwgpio_data->blkinfo, IE(ie));
                ie &= ~pwgpio->used_pins;
                GPIO_V2_IE_SET(&pwgpio_data->blkinfo, IE(ie));
                // set pin direction
                GPIO_V2_DIR_GET(&pwgpio_data->blkinfo, DIR(dir));
// The Logic is wrong, FIXED by Antson Hu
//              dir &= conf.pin_mask;
//              dir &= conf.pin_mask & conf.is_output;
                dir = (dir & ~conf.pin_mask) | (conf.pin_mask & conf.is_output);
                GPIO_V2_DIR_SET(&pwgpio_data->blkinfo, DIR(dir));

                GPIO_V2_IS_GET(&pwgpio_data->blkinfo,IS(level));
                level &= conf.pin_mask;
                GPIO_V2_IBE_GET(&pwgpio_data->blkinfo,IBE(bothedges));
                bothedges &= conf.pin_mask;
                GPIO_V2_IEV_GET(&pwgpio_data->blkinfo,IEV(positive));
                positive &= conf.pin_mask;
                for (i = 0; i < GPIO_V2_NUM_PINS; i++) {
                        if (conf.irq_mode[i] != PWGPIO_IRQ_MODE_NONE)
                                ie |= (1 << i);
                        switch (conf.irq_mode[i])
                        {
                        case PWGPIO_IRQ_MODE_LEVEL_LOW:
                                level |= (1 << i);
                                break;
                        case PWGPIO_IRQ_MODE_LEVEL_HIGH:
                                level |= (1 << i);
                                positive |= (1 << i);
                                break;
                        case PWGPIO_IRQ_MODE_EDGE_RISING:
                                positive |= (1 << i);
                                break;
                        case PWGPIO_IRQ_MODE_EDGE_FALLING:
                                break;
                        case PWGPIO_IRQ_MODE_EDGE_BOTH:
                                bothedges |= (1 << i);
                                break;
                        default:
                                ;
                        }
                }
                GPIO_V2_IS_SET(&pwgpio_data->blkinfo,IS(level));
                GPIO_V2_IBE_SET(&pwgpio_data->blkinfo,IBE(bothedges));
                GPIO_V2_IEV_SET(&pwgpio_data->blkinfo,IEV(positive));

                // enable irqs on our pins
                pwgpio->used_pins = conf.pin_mask;
                GPIO_V2_IE_SET(&pwgpio_data->blkinfo, IE(ie));
		spin_unlock_irqrestore(&pwgpio_data->lock, flags);
		break;
	}
	case PWGPIO_SET:
	{
		struct pwgpio_pin_value val;
                u32 addr;

		if (copy_from_user(&val, (void __user *) arg, sizeof(val))) {
			result = -EFAULT;
			break;
		}
		ddprintk(2, pwgpio, "PWGPIO_SET val %02x mask %02x used %02x\n",
			 val.pin_value, val.pin_mask, pwgpio->used_pins);
                if (val.pin_mask >= (1 << GPIO_V2_NUM_PINS))
                        return -EINVAL;

                spin_lock_irqsave(&pwgpio_data->lock, flags);
                if (val.pin_mask & ~pwgpio->used_pins) {
                        // pin might be claimed by someone else
                        spin_unlock_irqrestore(&pwgpio_data->lock, flags);
                        return -EPERM;
                }
                // this address manipulation is for using the hw mask feature
                addr = pwgpio_data->blkinfo.BaseAddress + (val.pin_mask << 2);
                dprintk(2,"blkinfo 0x%08x, addr 0x%08x, val 0x%08x \n",(unsigned int)&pwgpio_data->blkinfo,addr, val.pin_value);
		  
                pwgpio_data->blkinfo.RegWrite8(&pwgpio_data->blkinfo,
                                               addr, 0, val.pin_value);
                
		spin_unlock_irqrestore(&pwgpio_data->lock, flags);
		break;
	}
	case PWGPIO_GET:
	{
		struct pwgpio_pin_value val;
                u32 addr;
                RALuint8 data;

                if (copy_from_user(&val, (void __user *) arg, sizeof(val))) {
                        result = -EFAULT;
                        break;
                }
                if (val.pin_mask >= (1 << GPIO_V2_NUM_PINS))
                        return -EINVAL;

                spin_lock_irqsave(&pwgpio_data->lock, flags);
                if (val.pin_mask & ~pwgpio->used_pins) {
                        // pin might be claimed by someone else
                        spin_unlock_irqrestore(&pwgpio_data->lock, flags);
                        return -EPERM;
                }
                // this address manipulation is for using the hw mask feature
                addr = pwgpio_data->blkinfo.BaseAddress + (val.pin_mask << 2);
                pwgpio_data->blkinfo.RegRead8(&pwgpio_data->blkinfo, addr, &data);
                spin_unlock_irqrestore(&pwgpio_data->lock, flags);

                val.pin_value = data;
		ddprintk(2, pwgpio, "PWGPIO_GET mask %02x = val %02x\n",
			 val.pin_mask, val.pin_value);
		if (copy_to_user((void __user *) arg, &val, sizeof(val))) {
			result = -EFAULT;
			break;
		}
		break;
	}
	case PWGPIO_GETEVENT:
	{
		struct pwgpio_pin_value val;

		if (copy_from_user(&val, (void __user *) arg, sizeof(val))) {
			result = -EFAULT;
			break;
		}
                if (val.pin_mask >= (1 << GPIO_V2_NUM_PINS))
			return -EINVAL;

again:
		spin_lock_irqsave(&pwgpio_data->lock, flags);
		if (val.pin_mask & ~pwgpio->used_pins) {
			// pin might be claimed by someone else
			spin_unlock_irqrestore(&pwgpio_data->lock, flags);
			return -EPERM;
		}
		val.pin_value = pwgpio->events & val.pin_mask;
		pwgpio->events ^= val.pin_value;
		if (!val.pin_value) {
			spin_unlock_irqrestore(&pwgpio_data->lock, flags);
			if (filp->f_flags & O_NONBLOCK)
				return -EWOULDBLOCK;
			result = wait_event_interruptible(pwgpio->wq,
							  pwgpio->events != 0);
			if (result < 0)
				return result;
			goto again;
		}
		spin_unlock_irqrestore(&pwgpio_data->lock, flags);

		ddprintk(2, pwgpio, "PWGPIO_GETEVENT mask %02x = val %02x\n",
			 val.pin_mask, val.pin_value);
		if (copy_to_user((void __user *) arg, &val, sizeof(val))) {
			result = -EFAULT;
			break;
		}
		break;
	}
	case PWGPIO_GETFREQUENCY:
	{
		//Assuming user assigned only 1 pin. If 2 or more pins are
		//assigned, the return value is not meaningful.
		struct pwgpio_pin_value val;
		struct pwgpio_events_buffer *buffer;
		u32 index_cnt;
		u32 read_pos;
		u32 current_time;

		struct pwgpio_events_record *record;

		current_time = get_timestamp();

		if (copy_from_user(&val, (void __user *) arg, sizeof(val))) {
			result = -EFAULT;
			break;
		}
                if (val.pin_mask >= (1 << GPIO_V2_NUM_PINS))
			return -EINVAL;

		if (val.pin_mask & ~pwgpio->used_pins)
			// pin might be claimed by someone else
			return -EPERM;

		index_cnt   = 0;

		//Search in the history for the event
		buffer = &pwgpio_data->events_buf;

		val.pin_value = 0;  //Initialize

		//Do measure based on input signal
		read_pos = buffer->read_pos;
		while (read_pos < buffer->write_pos &&
		       index_cnt < PWGPIO_EVENT_RECORD_MAX) {
			record = &buffer->buf[read_pos % PWGPIO_EVENT_RECORD_MAX];

			//Check if this event is in history
			if (record->events & val.pin_mask) {
				//Save the timestamp
				event_timestamp[index_cnt] = record->tickcount;

				//Count up
				index_cnt++;
				//Reset the event
				record->events &= ~val.pin_mask;
			}
			read_pos++;
		}

		//Calculate frequency
		//Only valid when at least 2 edges detected
		if (index_cnt > 1) {
			//Prepare the samples
			prepare_samples(index_cnt);

			//Convert to frequency
			val.pin_value = convert_to_frequency(index_cnt - 1);
		} else if (index_cnt == 1)
			//Frequency is forced to be unchanged
			val.pin_value = buffer->freq_bkup;
		else
			; //No samples. Freq is 0.

		buffer->time_stamp = current_time;
		buffer->freq_bkup = val.pin_value;

		//Reset the event history
		//Do lock the data here
		spin_lock_irqsave(&pwgpio_data->lock, flags);
		buffer->read_pos = buffer->write_pos;
		buffer->buf[buffer->write_pos % PWGPIO_EVENT_RECORD_MAX].events = 0;
		buffer->buf[buffer->write_pos % PWGPIO_EVENT_RECORD_MAX].tickcount = buffer->time_stamp;
		spin_unlock_irqrestore(&pwgpio_data->lock, flags);

		ddprintk(2, pwgpio, "PWGPIO_GETFREQUNCY mask %02x = val %08x\n",
			 val.pin_mask, val.pin_value);
		if (copy_to_user((void __user *) arg, &val, sizeof(val))) {
			result = -EFAULT;
			break;
		}
		break;
	}
	case PWGPIO_GETEVENTHISTORY:
	{
		struct pwgpio_events_buffer *buffer;
		struct pwgpio_events_history *history;
		u32 index_dest;

		history = kzalloc(sizeof(*history), GFP_KERNEL);
		buffer = &pwgpio_data->events_buf;

		index_dest = 0;
		while (buffer->read_pos < buffer->write_pos &&
		       index_dest < PWGPIO_EVENT_RECORD_MAX) {
			history->record[index_dest].events =
				buffer->buf[buffer->read_pos % PWGPIO_EVENT_RECORD_MAX].events>>16;
			history->record[index_dest].tickcount =
				buffer->buf[buffer->read_pos % PWGPIO_EVENT_RECORD_MAX].tickcount;
			spin_lock_irqsave(&pwgpio_data->lock, flags);
			buffer->read_pos++;
			spin_unlock_irqrestore(&pwgpio_data->lock, flags);
			index_dest++;
		}
		history->events_num = index_dest;
		ddprintk(2, pwgpio, "PWGPIO_GETEVENTHISTORY %d\n", history->events_num);
		if (copy_to_user((void __user *) arg, history, sizeof(*history)))
			result = -EFAULT;
		kfree(history);
		break;
	}
	case PWGPIO_STARTMEASURING:
	{
		struct pwgpio_events_buffer *buffer;

		buffer = &pwgpio_data->events_buf;
		spin_lock_irqsave(&pwgpio_data->lock, flags);
		buffer->time_stamp = get_timestamp();
		buffer->read_pos = buffer->write_pos;
		buffer->buf[buffer->write_pos % PWGPIO_EVENT_RECORD_MAX].events = 0;
		buffer->buf[buffer->write_pos % PWGPIO_EVENT_RECORD_MAX].tickcount = buffer->time_stamp;
		buffer->freq_bkup = 0;

		spin_unlock_irqrestore(&pwgpio_data->lock, flags);

		ddprintk(2, pwgpio, "PWGPIO_STARTMEASURING\n");
		break;
	}
	default:
		result = -ENOTTY;
	}
	return result;
}

static const struct file_operations pwgpio_fops = {
	.owner		= THIS_MODULE,
	.open		= pwgpio_open,
	.release	= pwgpio_release,
	.poll		= pwgpio_poll,
	.unlocked_ioctl = pwgpio_ioctl
};

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
typedef struct _alt_func_ctrl
{
    int id;                 // Port ID
    unsigned int offset;    // Offset
    unsigned int shift;     // Shift
} alt_func_ctrl;
typedef struct _pad_pull_ctrl
{
    int id;                 // Port ID
    unsigned int mask;      // Mask to match
    unsigned int offset;    // Offset
    unsigned int shift;     // Shift
} pad_pull_ctrl;
#endif

static int pwgpio_port_ctrl(struct device *dev, int id)
{
#ifdef CONFIG_ARCH_PIXELWORKS_TOPAZ
	void __iomem *port_ctrl;
	unsigned long port_val;
	unsigned int  shift;
    
	if ((id < 0) || (id > 9)) {
		dev_err(dev, "Invalid GPIO port id %d.\n", id);
		return -ENXIO;
	}

	if (id >= 2 && id <= 7) {
		if (id <= 5) {
			port_ctrl = ioremap(SYS_PORT_CTRL, 0x4);
			shift = (id - 2) * 8;
		} else {
			port_ctrl = ioremap(SYS_PORT_CTRL + 0x04, 0x4);
			shift = (id - 6) * 8;
		}

		if (!port_ctrl) {
			dev_err(dev, "ioremap for gpio_base (0x%08x) failed\n", SYS_PORT_CTRL);
			return -ENOMEM;
		}

		if (gpio_mask[id]) {
			port_val = readl(port_ctrl);
			port_val &= ~(0xffU << shift);
			port_val |= (gpio_mask[id] << shift);
			writel(port_val, port_ctrl);
		}
		iounmap(port_ctrl);
	}
#elif defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
    void __iomem *port_ctrl;
	unsigned long port_val;
	unsigned int  shift;
    unsigned int  addr;
	int  i;
    static const alt_func_ctrl port_alt_config[] =
    {
        {  5, 0,  0 },  //PORT_EXF
        { 12, 0,  8 },  //PORT_EXM
        { 10, 0, 16 },  //PORT_EXK
        {  9, 0, 24 },  //PORT_EXJ
        {  4, 4,  0 },  //PORT_EXE
        { 14, 4,  8 },  //PORT_EXO
        { 15, 4, 16 },  //PORT_EXP
        { 13, 4, 24 },  //PORT_EXN
        {  6, 8,  0 },  //PORT_EXG
        {  7, 8,  8 },  //PORT_EXH
        {  8, 8, 16 },  //PORT_EXI
        { 11, 8, 24 },  //PORT_EXL
    };
#define NUM_OF_ALT_PORTS    (sizeof(port_alt_config)/sizeof(alt_func_ctrl))
    
    static const pad_pull_ctrl pad_pull_config[] =
    {
        {  4, 0x01, 0, 3 }, //PORT_EXE pin0
        {  4, 0x02, 0, 3 }, //PORT_EXE pin1
        {  4, 0x04, 0, 4 }, //PORT_EXE pin2
        {  4, 0x08, 0, 4 }, //PORT_EXE pin3
        { 13, 0x40, 0, 2 }, //PORT_EXN pin6
        { 13, 0x80, 0, 2 }, //PORT_EXN pin7
    };
#define NUM_OF_PULL_CONFIG    (sizeof(pad_pull_config)/sizeof(pad_pull_ctrl))
    
	if ((id < 0) || (id > 15)) {
		dev_err(dev, "Invalid GPIO port id %d.\n", id);
		return -ENXIO;
	}

    for( i = 0; i < NUM_OF_ALT_PORTS; i++ )
    {
        if( id == port_alt_config[i].id && gpio_mask[id] )
        {
            addr = SYS_PORT_CTRL + REG_ADDR_SYS_PADALTFUNCCTRL1 + port_alt_config[i].offset;
            shift = port_alt_config[i].shift;
            port_ctrl = ioremap(addr, 0x4);
            if ( !port_ctrl ) {
                dev_err(dev, "ioremap for gpio alt config (0x%08x) failed\n", addr);
                return -ENOMEM;
            }

            port_val = readl(port_ctrl);
            dprintk(2,"id: %d, mask: 0x%02x, shift=%d, old port_val: 0x%08lx\n", id, gpio_mask[id], shift, port_val);
            port_val &= ~(0xffU << shift);
            port_val |= ((unsigned long)gpio_mask[id] << shift);

            writel(port_val, port_ctrl);
            dprintk(2,"new port_val: 0x%08lx\n", port_val);
            
            iounmap(port_ctrl);
        }
    }
    
    for( i = 0; i < NUM_OF_PULL_CONFIG; i++ )
    {
        //If PORT_E pin3,2,1,0 or PORT_N pin6,7 are used as GPIO, then the pull-up/pull-down
        //control of TW0, TW1, TW2 should be disabled.
        if( id == pad_pull_config[i].id &&
            ((unsigned int)gpio_mask[id] & pad_pull_config[i].mask) == pad_pull_config[i].mask )
        {
            addr = SYS_PORT_CTRL + REG_ADDR_SYS_PADPULLCTRL1 + pad_pull_config[i].offset;
            shift = pad_pull_config[i].shift;
            port_ctrl = ioremap(addr, 0x4);
            if ( !port_ctrl ) {
                dev_err(dev, "ioremap for gpio pad pull config (0x%08x) failed\n", addr);
                return -ENOMEM;
            }
            
            port_val = readl(port_ctrl);
            dprintk(2,"id: %d, gpio pull up/down control. old: 0x%08lx\n", id, port_val);
            port_val &= ~(1 << shift);
            writel(port_val, port_ctrl);
            dprintk(2,"new: 0x%08lx\n", port_val);
            
            iounmap(port_ctrl);
        }
    }
    
#endif
	return 0;
}

static int pwgpio_probe(struct platform_device *pdev)
{
//        struct resource *res, *ires;
        struct resource *res;
        struct pwgpio_data *pwgpio_data;
        void *vaddr = NULL;
        u32 tmp1, tmp2, partno, designer, revision;
        int result = 0;
//        struct device_node *np = pdev->dev.of_node;
        u32 pins;

        dev_info(&pdev->dev, "pwgpio_probe, reserved pins: 0x%02lx\n",
                (unsigned long) pdev->dev.platform_data);
        pins =  (u32)pdev->dev.platform_data;
        res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
        if (!res) {
                dev_err(&pdev->dev, "failed to get IORESOURCE_MEM\n");
                return -ENXIO;
        }
    /*    ires = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
        if (!ires) {
                dev_err(&pdev->dev, "failed to get IORESOURCE_IRQ\n");
                return -ENXIO;
        }
*/
        if (!request_mem_region(res->start, res->end - res->start + 1, dev_name(&pdev->dev))) {
                dev_err(&pdev->dev, "IO mem at %08x is in use\n", res->start);
                return -EBUSY;
        }

	pwgpio_data = kzalloc(sizeof(*pwgpio_data), GFP_KERNEL);
        if (!pwgpio_data) {
                dev_err(&pdev->dev, "out of memory\n");
                result = -ENOMEM;
                goto err;
        }
        INIT_LIST_HEAD(&pwgpio_data->bufs);
        spin_lock_init(&pwgpio_data->lock);
        pwgpio_data->pdev = pdev;
        pwgpio_data->phys_base_addr = res->start;
        pwgpio_data->reg_size = res->end - res->start + 1;
        pwgpio_data->irq = -1;

        dprintk(2,"pdev 0x%08x, id %d, phy_add 0x%08x, reg_size 0x%08lx\n",(unsigned int)pwgpio_data->pdev,pdev->id,res->start, pwgpio_data->reg_size);
		
        vaddr = ioremap(pwgpio_data->phys_base_addr, pwgpio_data->reg_size);
        if (IS_ERR(vaddr)) {
                dev_err(&pdev->dev, "ioremap failed\n");
                result = PTR_ERR(vaddr);
                goto err;
        }
        pwgpio_data->base_addr = vaddr;
        dev_info(&pdev->dev, "phys 0x%08x virt %p\n", res->start, vaddr);

        RAL_INIT(&pwgpio_data->blkinfo, (uint32) pwgpio_data->base_addr, NULL);

        GPIO_V2_PERIPH_ID0_GET(&pwgpio_data->blkinfo, PARTNUMBER0(tmp1));
        partno = tmp1;
        GPIO_V2_PERIPH_ID1_GET(&pwgpio_data->blkinfo, PARTNUMBER1(tmp1), DESIGNER0(tmp2));
        partno |= (tmp1 << 8);
        designer = tmp2;
        GPIO_V2_PERIPH_ID2_GET(&pwgpio_data->blkinfo, DESIGNER1(tmp1), REVISION(tmp2));
        designer |= (tmp1 << 4);
        revision = tmp2;
        if (partno != GPIO_V2_PARTNO) {
                dev_err(&pdev->dev, "partno mismatch 0x%03x != 0x%03x\n",
                        partno, GPIO_V2_PARTNO);
                result = -ENXIO;
                goto err;
        }
        if (designer != GPIO_V2_DESIGNER) {
                dev_err(&pdev->dev, "designer mismatch 0x%02x != 0x%02x\n",
                        designer, GPIO_V2_DESIGNER);
                result = -ENXIO;
                goto err;
        }
        if (revision != GPIO_V2_REVISION) {
                dev_err(&pdev->dev, "revision mismatch 0x%02x != 0x%02x\n",
                        revision, GPIO_V2_REVISION);
                result = -ENXIO;
                goto err;
        }
        
        gpio_mask[pdev->id] = pin_mask[pdev->id * 2];         
        if(pins!=0)
           gpio_mask[pdev->id] = pins; 
          
        pwgpio_port_ctrl(&pdev->dev, pdev->id);

        // Use dynamic major
        pwgpio_data->major = pwgpio_major;  //PWGPIO_MAJOR;
        pwgpio_data->minor = pdev->id; // hack       

        pwgpio_data->irq =  platform_get_irq(pdev, 0);//irq_of_parse_and_map(np, 0);
        if (! pwgpio_data->irq) {
                dev_err(&pdev->dev, "IRQ missing or invalid\n");
                return -EINVAL;
        }
        result = devm_request_irq(&pdev->dev, pwgpio_data->irq+1, pwgpio_interrupt, IRQF_SHARED,
                                     dev_name(&pdev->dev), pwgpio_data);
        if (result) {
                dev_err(&pdev->dev, "Failed to request irq %i\n", pwgpio_data->irq);
                goto err;
        }

/*
       if (-1 != ires->start)
        {
                result = request_irq(ires->start, pwgpio_interrupt, IRQF_SHARED,
                                     dev_name(&pdev->dev), pwgpio_data);
                if (result) {
                        dev_err(&pdev->dev, "failed to get irq %d\n", ires->start);
                        goto err;
                }
                pwgpio_data->irq = ires->start;
        }
 */
        dprintk(2, "irq: %d, name: %s\n", pwgpio_data->irq, dev_name(&pdev->dev));
        cdev_init(&pwgpio_data->cdev, &pwgpio_fops);
        pwgpio_data->cdev.owner = THIS_MODULE;
        pwgpio_data->cdev.kobj.parent = &pdev->dev.kobj;        

        result = cdev_add(&pwgpio_data->cdev, MKDEV(pwgpio_data->major, pwgpio_data->minor), 1);
        if (result < 0) {
                dev_err(&pdev->dev, "can't register cdev %d/%d\n",
                        pwgpio_data->major, pwgpio_data->minor);
                goto err;
        }
        platform_set_drvdata(pdev, pwgpio_data);
	pwgpio_data->dev = device_create(pwgpio_class, &pdev->dev,
					 MKDEV(pwgpio_data->major, pwgpio_data->minor),
					 pwgpio_data, "pwgpio%d", pdev->id);
	if (IS_ERR(pwgpio_data->dev)) {
		result = PTR_ERR(pwgpio_data->dev);
		cdev_del(&pwgpio_data->cdev);
		goto err;
	}
        return 0;

err:
        if (pwgpio_data) {
                if (-1 != pwgpio_data->irq)
                    devm_free_irq(&pdev->dev, pwgpio_data->irq+1, pwgpio_data);
                if (pwgpio_data->base_addr)
                        iounmap(pwgpio_data->base_addr);
                kfree(pwgpio_data);
        }
        release_mem_region(res->start, res->end - res->start + 1);
	return result;
}

static int pwgpio_remove(struct platform_device *pdev)
{
	struct pwgpio_data *pwgpio_data;

        dev_dbg(&pdev->dev, "pwgpio_remove\n");
        pwgpio_data = platform_get_drvdata(pdev);
        platform_set_drvdata(pdev, NULL);

        device_unregister(pwgpio_data->dev);
        cdev_del(&pwgpio_data->cdev);
//        if (-1 != pwgpio_data->irq)
//It's already freed            devm_free_irq(&pdev->dev, pwgpio_data->irq+1, pwgpio_data);
        if (pwgpio_data->base_addr)
                iounmap(pwgpio_data->base_addr);
        release_mem_region(pwgpio_data->phys_base_addr, pwgpio_data->reg_size);
        kfree(pwgpio_data);
        return 0;
}

#define pwgpio_suspend NULL
#define pwgpio_resume NULL

static struct of_device_id pwgpio_of_match[]={
        { .compatible = "pixelworks, pwgpio_v2" },
        { },
};

static struct platform_driver pwgpio_driver = {
        .probe      = pwgpio_probe,
        .remove     = pwgpio_remove,
        .suspend    = pwgpio_suspend,
        .resume     = pwgpio_resume,
	.driver	    = {
		.name	    = "pwgpio",
		.owner	    = THIS_MODULE,
                .of_match_table = pwgpio_of_match,
	}
};

#define MAXNUM_DATA 16
static int __init pwgpio_init(void)
{
        int result;        
        dprintk(2, "driver loaded\n");
	pwgpio_class = class_create(THIS_MODULE, "pwgpio");
	if (IS_ERR(pwgpio_class)) {
		printk(KERN_ERR "ERROR: pwgpio_init: class_create failed\n");
		return PTR_ERR(pwgpio_class);
	}

         result = alloc_chrdev_region(&pwgpio_dev,
                                 0, MAXNUM_DATA, "pwgpio");
        if (result) {
                printk(KERN_ERR "ERROR: pwgpio_init: alloc_chrdev_region failed\n");
                class_destroy(pwgpio_class);
                return result;
        }
        pwgpio_major = MAJOR(pwgpio_dev);
        result = platform_driver_register(&pwgpio_driver);
	if (result)
        {
                printk(KERN_ERR "ERROR: pwgpio_init: platform_driver_register failed\n");
                unregister_chrdev_region(MKDEV(MAJOR(pwgpio_dev), 0), MAXNUM_DATA);

		class_destroy(pwgpio_class);
        }
        return result;
}

static __exit void pwgpio_cleanup(void)
{
        dprintk(2, "driver removed\n");
        platform_driver_unregister(&pwgpio_driver);
         unregister_chrdev_region(MKDEV(MAJOR(pwgpio_dev), 0), MAXNUM_DATA);
	class_destroy(pwgpio_class);
}

module_init(pwgpio_init);
module_exit(pwgpio_cleanup);
