/* -------------------------------------------------------------------------- *\
   pw2wireslave.c -- driver for Ruby 2wire slave

   Copyright (c) 2007-2013 Pixelworks Inc.

   --------------------------------------------------------------------------
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\* -------------------------------------------------------------------------- */
//
//History
//Version  0.1 : 2009/09/10 Pixelworks Y.Wang
//         0.2 : 2010/06/09 Pixelworks Japan HZhou
//         0.3 : 2010/06/24 Minor change to add release bus at all address mode
//         0.4 : 2010/09/23 Add support for Topaz
//         0.5 : 2010/10/28 Add support for polling
//         2018/07/14 : Bug fix. Wrong use of spinlock caused sleep when preempt disabled.
//
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/of_platform.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/hardirq.h>
#include <linux/ioport.h>
#include <linux/io.h>
#include <linux/list.h>
#include <linux/jiffies.h>

#include <pxlw/pw2wireslave.h>

#include "types.h"
#include "ral.h"
#include "2wire_v1_ral.h"


MODULE_AUTHOR ("Pixelworks Inc.");
MODULE_LICENSE("GPL");


#define DEBUG
//#define DEBUG_BUF   //Define this to debug when working as a module only

#ifdef DEBUG
static int pw2wireslave_debug = 1;
module_param_named(debug, pw2wireslave_debug, int, 0644);
MODULE_PARM_DESC(debug, "debug level (0 off, 1 errors, 2 trace, 3 full)");

#ifdef DEBUG_BUF
#define DEBUG_LOG   //For debugging as module to dump log when exiting
#define DEBUG_BUF_SIZE (64*1024)
static char debug_buffer[DEBUG_BUF_SIZE + 1024]; //With extra 1KBytes to guard memory
static char *debug_buf_p = &debug_buffer[0];
static int debug_len;
static int overwrapped = 0;
static unsigned int time_stamp;

#define dprintk(level, fmt, ...)                                        \
do {                                                                    \
        if (pw2wireslave_debug >= level)                                \
        {                                                               \
            time_stamp  = jiffies;                                      \
            debug_len = snprintf(debug_buf_p, 256, "%08X %s:%d: " fmt, time_stamp, __FUNCTION__, __LINE__, ## __VA_ARGS__); \
            debug_buf_p += debug_len;                                   \
            if( debug_buf_p > debug_buffer + DEBUG_BUF_SIZE )           \
            {                                                           \
                debug_buf_p = &debug_buffer[0];                         \
                overwrapped = 1;                                        \
            }                                                           \
        }                                                               \
    } while (0)
#else
#define dprintk(level, fmt, ...)                                        \
do {                                                                    \
        if (pw2wireslave_debug >= level)                                \
            printk("%s:%d: " fmt, __FUNCTION__, __LINE__, ## __VA_ARGS__); \
    } while (0)
#endif
#else
#define dprintk(...) do { } while (0)
#endif


#define FALSE (0)
#define TRUE  (1)
#define MAXNUM_SLAVES   3
#define MAX_DATA_STORAGE  256

//Allow single open only
static atomic_t pw2wireslave_dev_available = ATOMIC_INIT(1);

//Slave working mode
typedef enum
{
	smINVALID = 0,
	smSINGLE_MODE,
	smALL_MODE,
} eSLAVE_MODE;
typedef enum
{
	srsINVALID = 0,
	srsIDLE,
	srsADDR,
	srsCMD,
	srsRX,
	srsTX,
	srsSTOP,
	srsHOLD,
	srsDELIMITER = 0xffff,
} eSLAVE_RUN_STATE;

#define INVALID_ADDRESS (0xffffffff)

struct pw2wireslave_data {
	unsigned long phys_base_addr;
	unsigned long reg_size;
	void *base_addr;
	int irq;

	struct cdev cdev;
	struct device *dev;
	struct platform_device *pdev;

	wait_queue_head_t wq;
	spinlock_t lock;

	BlockInfo blkinfo;

	unsigned long interrupt_status; //Current interrupt status
	unsigned long event;            //Current processing event

	struct pw2wireslave_config conf;
	unsigned char addr_bits;

	unsigned int  addr_configured;

	eSLAVE_MODE mode;           //Slave working mode
	eSLAVE_RUN_STATE state;     //Slave state

	unsigned int  my_address[MAXNUM_SLAVES];    //Slave address array
	unsigned int  my_address_num;               //Number of slave addresses

	unsigned int  saved_address;        //The address of working device
	unsigned int  saved_address_index;  //Depends on saved_address
	unsigned int  backup_saved_address; //Backup saved address for user to get

	unsigned char tx_data_length;   //Actual data length in tx buffer
	unsigned int  tx_data_pos;      //Current tx data position
	unsigned int  tx_data_ready[MAXNUM_SLAVES]; //Slave data ready flag

	unsigned int  rx_data_length;   //Actual data length in rx buffer
	unsigned int  rx_data_pos;      //Current rx data position

	unsigned int  rx_data_ready;    //Flag between user and kernel space. Indicating
	//that data arrived from master

	unsigned char rx_buffer[MAX_DATA_STORAGE];      //RX buffer used by interrupt handler
	unsigned char rx_temp_buffer[MAX_DATA_STORAGE]; //RX buffer for user space
    unsigned char user_temp_buf[MAX_DATA_STORAGE];//Buffer before copying to user 2018/07/14
	unsigned char tx_buffer[MAXNUM_SLAVES][MAX_DATA_STORAGE];
};

typedef struct {
	unsigned long   event;
	void    (*func)(struct pw2wireslave_data *);
} pw2wireslave_action_t;

typedef struct {
	eSLAVE_RUN_STATE    state;
	pw2wireslave_action_t *action_table;
} pw2wireslave_statemachine_t;
static void pw2wireslave_statemachine(struct pw2wireslave_data *);

#define INVALID_EVENT   (-1)
static const unsigned long pw2wireslave_event_mask[] =
{
	//In order of priority
	TW_V1_INTSTAT_m_START(),
	TW_V1_INTSTAT_m_SBYRCVD(),
	TW_V1_INTSTAT_m_SHOLD(),
	TW_V1_INTSTAT_m_SWRREQ(),
	TW_V1_INTSTAT_m_SRDREQ(),
	TW_V1_INTSTAT_m_SWFLIM(),
	TW_V1_INTSTAT_m_SRFLIM(),
	TW_V1_INTSTAT_m_SWFLOCKED(),
	TW_V1_INTSTAT_m_SNACK(),
	TW_V1_INTSTAT_m_STOP(),
	INVALID_EVENT,
};

static void pw2wireslave_invalid_start( struct pw2wireslave_data *);
static void pw2wireslave_invalid_sbyrcvd( struct pw2wireslave_data *);
static void pw2wireslave_invalid_swrreq( struct pw2wireslave_data *);
static void pw2wireslave_invalid_srdreq( struct pw2wireslave_data *);
static void pw2wireslave_invalid_swflim( struct pw2wireslave_data *);
static void pw2wireslave_invalid_srflim( struct pw2wireslave_data *);
static void pw2wireslave_invalid_swflocked( struct pw2wireslave_data *);
static void pw2wireslave_invalid_snack( struct pw2wireslave_data *);
static void pw2wireslave_invalid_stop( struct pw2wireslave_data *);
static void pw2wireslave_invalid_shold( struct pw2wireslave_data *);

static const pw2wireslave_action_t pw2wireslave_state_invalid_actions[] =
{
	//In order of priority
	{ TW_V1_INTSTAT_m_START(),      pw2wireslave_invalid_start      },
	{ TW_V1_INTSTAT_m_SBYRCVD(),    pw2wireslave_invalid_sbyrcvd    },
	{ TW_V1_INTSTAT_m_SWRREQ(),     pw2wireslave_invalid_swrreq     },
	{ TW_V1_INTSTAT_m_SRDREQ(),     pw2wireslave_invalid_srdreq     },
	{ TW_V1_INTSTAT_m_SHOLD(),      pw2wireslave_invalid_shold      },
	{ TW_V1_INTSTAT_m_SWFLIM(),     pw2wireslave_invalid_swflim     },
	{ TW_V1_INTSTAT_m_SRFLIM(),     pw2wireslave_invalid_srflim     },
	{ TW_V1_INTSTAT_m_SWFLOCKED(),  pw2wireslave_invalid_swflocked  },
	{ TW_V1_INTSTAT_m_SNACK(),      pw2wireslave_invalid_snack      },
	{ TW_V1_INTSTAT_m_STOP(),       pw2wireslave_invalid_stop       },
	{ 0, NULL },
};

static void pw2wireslave_idle_start( struct pw2wireslave_data *);
static void pw2wireslave_idle_sbyrcvd( struct pw2wireslave_data *);
static void pw2wireslave_idle_swrreq( struct pw2wireslave_data *);
static void pw2wireslave_idle_srdreq( struct pw2wireslave_data *);
static void pw2wireslave_idle_swflim( struct pw2wireslave_data *);
static void pw2wireslave_idle_srflim( struct pw2wireslave_data *);
static void pw2wireslave_idle_swflocked( struct pw2wireslave_data *);
static void pw2wireslave_idle_snack( struct pw2wireslave_data *);
static void pw2wireslave_idle_stop( struct pw2wireslave_data *);
static void pw2wireslave_idle_shold( struct pw2wireslave_data *);

static const pw2wireslave_action_t pw2wireslave_state_idle_actions[] =
{
	//In order of priority
	{ TW_V1_INTSTAT_m_START(),      pw2wireslave_idle_start     },
	{ TW_V1_INTSTAT_m_SBYRCVD(),    pw2wireslave_idle_sbyrcvd   },
	{ TW_V1_INTSTAT_m_SWRREQ(),     pw2wireslave_idle_swrreq    },
	{ TW_V1_INTSTAT_m_SRDREQ(),     pw2wireslave_idle_srdreq    },
	{ TW_V1_INTSTAT_m_SWFLIM(),     pw2wireslave_idle_swflim    },
	{ TW_V1_INTSTAT_m_SRFLIM(),     pw2wireslave_idle_srflim    },
	{ TW_V1_INTSTAT_m_SWFLOCKED(),  pw2wireslave_idle_swflocked },
	{ TW_V1_INTSTAT_m_SNACK(),      pw2wireslave_idle_snack     },
	{ TW_V1_INTSTAT_m_STOP(),       pw2wireslave_idle_stop      },
	{ TW_V1_INTSTAT_m_SHOLD(),      pw2wireslave_idle_shold     },
	{ 0, NULL },
};

static void pw2wireslave_addr_start( struct pw2wireslave_data *);
static void pw2wireslave_addr_sbyrcvd( struct pw2wireslave_data *);
static void pw2wireslave_addr_swrreq( struct pw2wireslave_data *);
static void pw2wireslave_addr_srdreq( struct pw2wireslave_data *);
static void pw2wireslave_addr_swflim( struct pw2wireslave_data *);
static void pw2wireslave_addr_srflim( struct pw2wireslave_data *);
static void pw2wireslave_addr_swflocked( struct pw2wireslave_data *);
static void pw2wireslave_addr_snack( struct pw2wireslave_data *);
static void pw2wireslave_addr_stop( struct pw2wireslave_data *);
static void pw2wireslave_addr_shold( struct pw2wireslave_data *);

static const pw2wireslave_action_t pw2wireslave_state_addr_actions[] =
{
	//In order of priority
	{ TW_V1_INTSTAT_m_START(),      pw2wireslave_addr_start     },
	{ TW_V1_INTSTAT_m_SBYRCVD(),    pw2wireslave_addr_sbyrcvd   },
	{ TW_V1_INTSTAT_m_SWRREQ(),     pw2wireslave_addr_swrreq    },
	{ TW_V1_INTSTAT_m_SRDREQ(),     pw2wireslave_addr_srdreq    },
	{ TW_V1_INTSTAT_m_SWFLIM(),     pw2wireslave_addr_swflim    },
	{ TW_V1_INTSTAT_m_SRFLIM(),     pw2wireslave_addr_srflim    },
	{ TW_V1_INTSTAT_m_SWFLOCKED(),  pw2wireslave_addr_swflocked },
	{ TW_V1_INTSTAT_m_SNACK(),      pw2wireslave_addr_snack     },
	{ TW_V1_INTSTAT_m_STOP(),       pw2wireslave_addr_stop      },
	{ TW_V1_INTSTAT_m_SHOLD(),      pw2wireslave_addr_shold     },
	{ 0, NULL },
};

static void pw2wireslave_cmd_start( struct pw2wireslave_data *);
static void pw2wireslave_cmd_sbyrcvd( struct pw2wireslave_data *);
static void pw2wireslave_cmd_swrreq( struct pw2wireslave_data *);
static void pw2wireslave_cmd_srdreq( struct pw2wireslave_data *);
static void pw2wireslave_cmd_swflim( struct pw2wireslave_data *);
static void pw2wireslave_cmd_srflim( struct pw2wireslave_data *);
static void pw2wireslave_cmd_swflocked( struct pw2wireslave_data *);
static void pw2wireslave_cmd_snack( struct pw2wireslave_data *);
static void pw2wireslave_cmd_stop( struct pw2wireslave_data *);
static void pw2wireslave_cmd_shold( struct pw2wireslave_data *);

static const pw2wireslave_action_t pw2wireslave_state_cmd_actions[] =
{
	//In order of priority
	{ TW_V1_INTSTAT_m_START(),      pw2wireslave_cmd_start      },
	{ TW_V1_INTSTAT_m_SBYRCVD(),    pw2wireslave_cmd_sbyrcvd    },
	{ TW_V1_INTSTAT_m_SWRREQ(),     pw2wireslave_cmd_swrreq     },
	{ TW_V1_INTSTAT_m_SRDREQ(),     pw2wireslave_cmd_srdreq     },
	{ TW_V1_INTSTAT_m_SWFLIM(),     pw2wireslave_cmd_swflim     },
	{ TW_V1_INTSTAT_m_SRFLIM(),     pw2wireslave_cmd_srflim     },
	{ TW_V1_INTSTAT_m_SWFLOCKED(),  pw2wireslave_cmd_swflocked  },
	{ TW_V1_INTSTAT_m_SNACK(),      pw2wireslave_cmd_snack      },
	{ TW_V1_INTSTAT_m_STOP(),       pw2wireslave_cmd_stop       },
	{ TW_V1_INTSTAT_m_SHOLD(),      pw2wireslave_cmd_shold      },
	{ 0, NULL },
};

static void pw2wireslave_rx_start( struct pw2wireslave_data *);
static void pw2wireslave_rx_sbyrcvd( struct pw2wireslave_data *);
static void pw2wireslave_rx_swrreq( struct pw2wireslave_data *);
static void pw2wireslave_rx_srdreq( struct pw2wireslave_data *);
static void pw2wireslave_rx_swflim( struct pw2wireslave_data *);
static void pw2wireslave_rx_srflim( struct pw2wireslave_data *);
static void pw2wireslave_rx_swflocked( struct pw2wireslave_data *);
static void pw2wireslave_rx_snack( struct pw2wireslave_data *);
static void pw2wireslave_rx_stop( struct pw2wireslave_data *);
static void pw2wireslave_rx_shold( struct pw2wireslave_data *);

static const pw2wireslave_action_t pw2wireslave_state_rx_actions[] =
{
	//In order of priority
	{ TW_V1_INTSTAT_m_START(),      pw2wireslave_rx_start       },
	{ TW_V1_INTSTAT_m_SBYRCVD(),    pw2wireslave_rx_sbyrcvd     },
	{ TW_V1_INTSTAT_m_SWRREQ(),     pw2wireslave_rx_swrreq      },
	{ TW_V1_INTSTAT_m_SRDREQ(),     pw2wireslave_rx_srdreq      },
	{ TW_V1_INTSTAT_m_SWFLIM(),     pw2wireslave_rx_swflim      },
	{ TW_V1_INTSTAT_m_SRFLIM(),     pw2wireslave_rx_srflim      },
	{ TW_V1_INTSTAT_m_SWFLOCKED(),  pw2wireslave_rx_swflocked   },
	{ TW_V1_INTSTAT_m_SNACK(),      pw2wireslave_rx_snack       },
	{ TW_V1_INTSTAT_m_STOP(),       pw2wireslave_rx_stop        },
	{ TW_V1_INTSTAT_m_SHOLD(),      pw2wireslave_rx_shold       },
	{ 0, NULL },
};

static void pw2wireslave_tx_start( struct pw2wireslave_data *);
static void pw2wireslave_tx_sbyrcvd( struct pw2wireslave_data *);
static void pw2wireslave_tx_swrreq( struct pw2wireslave_data *);
static void pw2wireslave_tx_srdreq( struct pw2wireslave_data *);
static void pw2wireslave_tx_swflim( struct pw2wireslave_data *);
static void pw2wireslave_tx_srflim( struct pw2wireslave_data *);
static void pw2wireslave_tx_swflocked( struct pw2wireslave_data *);
static void pw2wireslave_tx_snack( struct pw2wireslave_data *);
static void pw2wireslave_tx_stop( struct pw2wireslave_data *);
static void pw2wireslave_tx_shold( struct pw2wireslave_data *);

static const pw2wireslave_action_t pw2wireslave_state_tx_actions[] =
{
	//In order of priority
	{ TW_V1_INTSTAT_m_START(),      pw2wireslave_tx_start       },
	{ TW_V1_INTSTAT_m_SBYRCVD(),    pw2wireslave_tx_sbyrcvd     },
	{ TW_V1_INTSTAT_m_SWRREQ(),     pw2wireslave_tx_swrreq      },
	{ TW_V1_INTSTAT_m_SRDREQ(),     pw2wireslave_tx_srdreq      },
	{ TW_V1_INTSTAT_m_SWFLIM(),     pw2wireslave_tx_swflim      },
	{ TW_V1_INTSTAT_m_SRFLIM(),     pw2wireslave_tx_srflim      },
	{ TW_V1_INTSTAT_m_SWFLOCKED(),  pw2wireslave_tx_swflocked   },
	{ TW_V1_INTSTAT_m_SNACK(),      pw2wireslave_tx_snack       },
	{ TW_V1_INTSTAT_m_STOP(),       pw2wireslave_tx_stop        },
	{ TW_V1_INTSTAT_m_SHOLD(),      pw2wireslave_tx_shold       },
	{ 0, NULL },
};

static void pw2wireslave_stop_start( struct pw2wireslave_data *);
static void pw2wireslave_stop_sbyrcvd( struct pw2wireslave_data *);
static void pw2wireslave_stop_swrreq( struct pw2wireslave_data *);
static void pw2wireslave_stop_srdreq( struct pw2wireslave_data *);
static void pw2wireslave_stop_swflim( struct pw2wireslave_data *);
static void pw2wireslave_stop_srflim( struct pw2wireslave_data *);
static void pw2wireslave_stop_swflocked( struct pw2wireslave_data *);
static void pw2wireslave_stop_snack( struct pw2wireslave_data *);
static void pw2wireslave_stop_stop( struct pw2wireslave_data *);
static void pw2wireslave_stop_shold( struct pw2wireslave_data *);

static const pw2wireslave_action_t pw2wireslave_state_stop_actions[] =
{
	//In order of priority
	{ TW_V1_INTSTAT_m_START(),      pw2wireslave_stop_start     },
	{ TW_V1_INTSTAT_m_SBYRCVD(),    pw2wireslave_stop_sbyrcvd   },
	{ TW_V1_INTSTAT_m_SWRREQ(),     pw2wireslave_stop_swrreq    },
	{ TW_V1_INTSTAT_m_SRDREQ(),     pw2wireslave_stop_srdreq    },
	{ TW_V1_INTSTAT_m_SWFLIM(),     pw2wireslave_stop_swflim    },
	{ TW_V1_INTSTAT_m_SRFLIM(),     pw2wireslave_stop_srflim    },
	{ TW_V1_INTSTAT_m_SWFLOCKED(),  pw2wireslave_stop_swflocked },
	{ TW_V1_INTSTAT_m_SNACK(),      pw2wireslave_stop_snack     },
	{ TW_V1_INTSTAT_m_STOP(),       pw2wireslave_stop_stop      },
	{ TW_V1_INTSTAT_m_SHOLD(),      pw2wireslave_stop_shold     },
	{ 0, NULL },
};

static void pw2wireslave_hold_start( struct pw2wireslave_data *);
static void pw2wireslave_hold_sbyrcvd( struct pw2wireslave_data *);
static void pw2wireslave_hold_swrreq( struct pw2wireslave_data *);
static void pw2wireslave_hold_srdreq( struct pw2wireslave_data *);
static void pw2wireslave_hold_swflim( struct pw2wireslave_data *);
static void pw2wireslave_hold_srflim( struct pw2wireslave_data *);
static void pw2wireslave_hold_swflocked( struct pw2wireslave_data *);
static void pw2wireslave_hold_snack( struct pw2wireslave_data *);
static void pw2wireslave_hold_stop( struct pw2wireslave_data *);
static void pw2wireslave_hold_shold( struct pw2wireslave_data *);

static const pw2wireslave_action_t pw2wireslave_state_hold_actions[] =
{
	//In order of priority
	{ TW_V1_INTSTAT_m_START(),      pw2wireslave_hold_start     },
	{ TW_V1_INTSTAT_m_SBYRCVD(),    pw2wireslave_hold_sbyrcvd   },
	{ TW_V1_INTSTAT_m_SWRREQ(),     pw2wireslave_hold_swrreq    },
	{ TW_V1_INTSTAT_m_SRDREQ(),     pw2wireslave_hold_srdreq    },
	{ TW_V1_INTSTAT_m_SWFLIM(),     pw2wireslave_hold_swflim    },
	{ TW_V1_INTSTAT_m_SRFLIM(),     pw2wireslave_hold_srflim    },
	{ TW_V1_INTSTAT_m_SWFLOCKED(),  pw2wireslave_hold_swflocked },
	{ TW_V1_INTSTAT_m_SNACK(),      pw2wireslave_hold_snack     },
	{ TW_V1_INTSTAT_m_STOP(),       pw2wireslave_hold_stop      },
	{ TW_V1_INTSTAT_m_SHOLD(),      pw2wireslave_hold_shold     },
	{ 0, NULL },
};

static const pw2wireslave_statemachine_t slave_statemachine_func_table[] =
{
	{ srsINVALID,   (pw2wireslave_action_t *)pw2wireslave_state_invalid_actions  },
	{ srsIDLE,      (pw2wireslave_action_t *)pw2wireslave_state_idle_actions     },
	{ srsADDR,      (pw2wireslave_action_t *)pw2wireslave_state_addr_actions     },
	{ srsCMD,       (pw2wireslave_action_t *)pw2wireslave_state_cmd_actions      },
	{ srsRX,        (pw2wireslave_action_t *)pw2wireslave_state_rx_actions       },
	{ srsTX,        (pw2wireslave_action_t *)pw2wireslave_state_tx_actions       },
	{ srsSTOP,      (pw2wireslave_action_t *)pw2wireslave_state_stop_actions     },
	{ srsHOLD,      (pw2wireslave_action_t *)pw2wireslave_state_hold_actions     },
	{ srsDELIMITER, NULL },
};

////////////////////////////////////////////////////////////////////////////////
//
// Functions
//
////////////////////////////////////////////////////////////////////////////////
static inline void pw2wireslave_unlock( struct pw2wireslave_data *slave_data )
{
	unsigned char fifolock;

	TW_V1_SCTL_GET1(&slave_data->blkinfo, SWFLOCK(fifolock));
	if(fifolock)
	{
		TW_V1_SCTL_SET1(&slave_data->blkinfo, UNLOCK(0x1));
		dprintk(3, "UNLOCK\n");
	}
}

static inline void pw2wireslave_reply_posack( struct pw2wireslave_data *slave_data )
{
	unsigned char ackmode;

	//Send POSACK
	TW_V1_SCTL_GET(&slave_data->blkinfo, AUTOACK(ackmode));
	if(!ackmode)
	{
		TW_V1_SCTL_SET(&slave_data->blkinfo, ACKCMD(0x1));   //POSACK
		dprintk(2, "POSACK\n");
	}
}

static inline void pw2wireslave_reply_negack( struct pw2wireslave_data *slave_data )
{
	TW_V1_SCTL_SET(&slave_data->blkinfo, ACKCMD(0x3));   //NEGACK
	dprintk(2, "NEGACK\n");
}

static inline void pw2wireslave_release_bus( struct pw2wireslave_data *slave_data )
{
	TW_V1_SCTL_SET(&slave_data->blkinfo, ACKCMD(0x2));
	dprintk(2, "RELEASE BUS\n");
}

static inline void pw2wireslave_send_one( struct pw2wireslave_data *slave_data )
{
	unsigned char txdata;
	int i;

	i = slave_data->saved_address_index;

	txdata = slave_data->tx_buffer[i][slave_data->tx_data_pos];
	TW_V1_SWFIFO_SET1(&slave_data->blkinfo, WDATA(txdata));
	dprintk(3, "set tx data[%d] %02X\n", slave_data->tx_data_pos, txdata);
	slave_data->tx_data_pos++;
}

static inline void pw2wireslave_send( struct pw2wireslave_data *slave_data )
{
	unsigned char fifoentries;
	unsigned char fifosize;
	unsigned char tx_max_num;
	int i;

	i = slave_data->saved_address_index;

	if( slave_data->tx_data_ready[i] == FALSE )
	{
		dprintk(3, "Data not ready!\n");
		return;
	}

	//Get number of FIFO entries and FIFO size
	TW_V1_SFIFOSTAT_GET1(&slave_data->blkinfo, WFNE(fifoentries));
	TW_V1_SFIFOSTAT_GET1(&slave_data->blkinfo, WFSZ(fifosize));
	tx_max_num = fifosize - fifoentries;

	if( slave_data->tx_data_pos == slave_data->tx_data_length )
	{
		dprintk(3, "send %d dummy bytes\n", tx_max_num);
		while( tx_max_num-- )
		{
			TW_V1_SWFIFO_SET1(&slave_data->blkinfo, WDATA(0x0));
			pw2wireslave_unlock( slave_data );
		}

		return;
	}

	while( slave_data->tx_data_pos < slave_data->tx_data_length && tx_max_num-- )
	{
		pw2wireslave_send_one( slave_data );
		pw2wireslave_unlock( slave_data );
	}
}

static inline void pw2wireslave_reset_readfifo( struct pw2wireslave_data *slave_data )
{
	//Reset read FIFO
	TW_V1_SCTL_SET1(&slave_data->blkinfo, SRFRST(0x1));
}

static inline void pw2wireslave_reset_writefifo( struct pw2wireslave_data *slave_data )
{
	//Reset read/write FIFO
	TW_V1_SCTL_SET(&slave_data->blkinfo, SWFRST(0x1), UNLOCK(0x1));
}

static inline void pw2wireslave_reset_allfifos( struct pw2wireslave_data *slave_data )
{
	//Reset read/write FIFO
	TW_V1_SCTL_SET(&slave_data->blkinfo, SRFRST(0x1), SWFRST(0x1), UNLOCK(0x1));
}

static inline void pw2wireslave_recv_one( struct pw2wireslave_data *slave_data )
{
	unsigned long rxdata;

	TW_V1_SRFIFO_GET(&slave_data->blkinfo, ALL(rxdata));
	dprintk(3, "rx data %02X\n", (unsigned char)rxdata);

	slave_data->rx_buffer[slave_data->rx_data_pos] = (unsigned char)rxdata;
	slave_data->rx_data_pos++;

	pw2wireslave_reply_posack(slave_data);
}

static inline void pw2wireslave_recv_all( struct pw2wireslave_data *slave_data )
{
	unsigned char unread;

	TW_V1_SFIFOSTAT_GET1(&slave_data->blkinfo, RFNE(unread));

	while( unread )
	{
		pw2wireslave_recv_one( slave_data );
		TW_V1_SFIFOSTAT_GET1(&slave_data->blkinfo, RFNE(unread));
	}
}

//Enable 2wire slave
//Input : device
//Return: -
//
static void pw2wireslave_enable(struct pw2wireslave_data *slave_data)
{
	//Enable operation
	TW_V1_SCTL_SET1(&slave_data->blkinfo,SLAVEEN(0x1));
}
//
//Disable hw master
//Input : device
//Return: -
//
static void pw2wireslave_disable(struct pw2wireslave_data *slave_data)
{
	//Disable operation
	TW_V1_SCTL_SET1(&slave_data->blkinfo,SLAVEEN(0x0));
}
//
//Disable hw interrupt
//Input : device
//Return: -
//
static void pw2wireslave_disable_interrupt(struct pw2wireslave_data *slave_data)
{
	//Disable operation
	TW_V1_INTEN_SET(&slave_data->blkinfo,
			SNACK(0x0),
			SBYRCVD(0x0),
			SWRREQ(0x0),
			SRDREQ(0x0),
			SRFLIM(0x0),
			SWFLIM(0x0),
			SWFLOCKED(0x0),
			SGENCALL(0x0),
			SHOLD(0x0),
			START(0x0),
			STOP(0x0));
}
//Enable hw interrupt
//Input : device
//Return: -
//
static void pw2wireslave_enable_interrupt(struct pw2wireslave_data *slave_data)
{
	//Enable operation
	TW_V1_INTEN_SET(&slave_data->blkinfo,
			SNACK(0x1),
			SBYRCVD(0x1),
			SWRREQ(0x1),
			SRDREQ(0x1),
			SRFLIM(0x1),
			SWFLIM(0x1),
			SWFLOCKED(0x1),
			SHOLD(0x1),
			STOP(0x1),
			START(0x1));
}

//Clear hw interrupt event
//Input : interrup mask
//Return: -
//
static inline void pw2wireslave_clear_event(struct pw2wireslave_data *slave_data, unsigned long event_mask)
{
	switch( event_mask )
	{
	case TW_V1_INTSTAT_m_START():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, START(0x1));
		break;
	case TW_V1_INTSTAT_m_SBYRCVD():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, SBYRCVD(0x1));
		break;
	case TW_V1_INTSTAT_m_SHOLD():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, SHOLD(0x1));
		break;
	case TW_V1_INTSTAT_m_SWRREQ():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, SWRREQ(0x1));
		break;
	case TW_V1_INTSTAT_m_SRDREQ():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, SRDREQ(0x1));
		break;
	case TW_V1_INTSTAT_m_SWFLIM():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, SWFLIM(0x1));
		break;
	case TW_V1_INTSTAT_m_SRFLIM():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, SRFLIM(0x1));
		break;
	case TW_V1_INTSTAT_m_SWFLOCKED():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, SWFLOCKED(0x1));
		break;
	case TW_V1_INTSTAT_m_SNACK():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, SNACK(0x1));
		break;
	case TW_V1_INTSTAT_m_STOP():
		TW_V1_INTCLR_SET(&slave_data->blkinfo, STOP(0x1));
		break;
	}
}

//
//Reset hw slave
//Input : device
//Return: -
//
static void pw2wireslave_reset(struct pw2wireslave_data *slave_data)
{
	//Reset control logic
	TW_V1_SCTL_SET1(&slave_data->blkinfo,SLAVERST(0x1));

	//Reset read and write FIFO
	pw2wireslave_reset_allfifos( slave_data );

	//Set write fifo threshold
	TW_V1_SCTL_SET1(&slave_data->blkinfo, SWFTHR(0x00));

	//Set read fifo threshold
	TW_V1_SCTL_SET1(&slave_data->blkinfo, SRFTHR(0x00));

#if defined(CONFIG_RUBY_CHIP_B0) || defined(CONFIG_RUBY_CHIP_C0)
	//Preset ACKCMD
	TW_V1_SCTL_SET(&slave_data->blkinfo, ACKCMD(0x1));
#endif

	//Clear all interrupt status
	TW_V1_INTCLR_SET(&slave_data->blkinfo,
			 SNACK(0x1),
			 SBYRCVD(0x1),
			 SWRREQ(0x1),
			 SRDREQ(0x1),
			 SRFLIM(0x1),
			 SWFLIM(0x1),
			 SWFLOCKED(0x1),
			 SGENCALL(0x1),
			 SHOLD(0x1),
			 START(0x1),
			 STOP(0x1),
			 MNACK(0x1),
			 MBYRCVD(0x1),
			 MARBLOST(0x1),
			 MBYCDONE(0x1),
			 MRFLIM(0x1),
			 MWFLIM(0x1),
			 MILLCMD(0x1),
			 MHOLD(0x1)
			);

	TW_V1_SCTL_SET1(&slave_data->blkinfo, GENCALL_EN(slave_data->conf.gencall));
	TW_V1_SCTL_SET1(&slave_data->blkinfo, ALLADDREN(slave_data->conf.alladdren));
	TW_V1_SCTL_SET1(&slave_data->blkinfo, AUTOACK(slave_data->conf.autoack));

	if(slave_data->conf.gencall)
	{
		TW_V1_INTEN_SET1(&slave_data->blkinfo, SGENCALL(slave_data->conf.gencall));
	}

	if( slave_data->addr_configured == TRUE )
	{
		if( slave_data->conf.alladdren == 0 )
		{
			//Single address mode
			TW_V1_SCTL_SET1(&slave_data->blkinfo, ADDRMODE(slave_data->addr_bits));
			TW_V1_SCTL_SET1(&slave_data->blkinfo, SLVADDR(slave_data->my_address[0]));
		}
		else
		{
			//All address mode already done
		}
	}
}

static int pw2wireslave_open(struct inode *inode, struct file *filp)
{
	struct pw2wireslave_data *slave_data = container_of(inode->i_cdev, struct pw2wireslave_data, cdev);
	int i;

	//Allow only one user to open this device once
	if( !atomic_dec_and_test(&pw2wireslave_dev_available) )
	{
		atomic_inc(&pw2wireslave_dev_available);
		return -EBUSY;
	}

	//Initialize wait queue
	init_waitqueue_head(&slave_data->wq);

	slave_data->mode = smINVALID;
	slave_data->state = srsINVALID;
	for( i=0; i < MAXNUM_SLAVES; i++ )
	{
		slave_data->tx_data_ready[i] = FALSE;
		slave_data->my_address[i] = INVALID_ADDRESS;
	}
	slave_data->my_address_num = 0;

	//Initial address index with invalid value
	slave_data->saved_address_index = MAXNUM_SLAVES;

	filp->private_data = slave_data;

	dprintk(2, "pw2wireslave_open\n");

	return 0;
}

static int pw2wireslave_release(struct inode *inode, struct file *filp)
{
	atomic_inc(&pw2wireslave_dev_available);

	dprintk(2, "pw2wireslave_release\n");
	return 0;
}

static unsigned int pw2wireslave_poll(struct file *filp, poll_table *wait)
{
	struct pw2wireslave_data *slave_data = filp->private_data;
	unsigned int mask = 0;

	poll_wait(filp, &slave_data->wq, wait);

	if( slave_data->rx_data_ready != 0 )
	{
		mask = POLLIN | POLLRDNORM;
		dprintk(2, "PW2WIRESLAVE poll ready\n");
	}

	return mask;
}

static long pw2wireslave_ioctl(struct file *filp, unsigned cmd, unsigned long arg)
{
	struct pw2wireslave_data *slave_data = filp->private_data;
	int result = 0;
	unsigned long flags;

	switch(cmd)
	{
	case PW2WIRESLAVE_ADDRSETTING: {
		struct pw2wireslave_addressinfo addressinfo;
		unsigned int iAddress;

		memset(&addressinfo,0,sizeof(struct pw2wireslave_addressinfo));
		if(copy_from_user(&addressinfo, (struct pw2wireslave_addressinfo __user*)arg, sizeof(addressinfo)))
		{
			result = -EFAULT;
			break;
		}

		spin_lock_irqsave(&slave_data->lock, flags);

		dprintk(2, "PW2WIRESLAVE_ADDRSETTING\n");

		iAddress = addressinfo.slavaddr;
		slave_data->saved_address = addressinfo.slavaddr;
		slave_data->saved_address_index = 0;

		if(!addressinfo.addrmode)   //7 bit mode
		{
			iAddress >>= 1;
		}

		slave_data->conf.alladdren = 0;
		slave_data->conf.autoack = 1;
		slave_data->addr_bits = addressinfo.addrmode;
		slave_data->my_address[0] = iAddress;
		slave_data->my_address_num = 1;
		slave_data->addr_configured = TRUE;
		slave_data->mode = smSINGLE_MODE;
		slave_data->state= srsIDLE;

		pw2wireslave_reset(slave_data);
		pw2wireslave_enable_interrupt(slave_data);

		spin_unlock_irqrestore(&slave_data->lock, flags);
		break;
	}
	case PW2WIRESLAVE_CONFIGURE: {
		struct pw2wireslave_config conf;

		memset(&conf,0,sizeof(struct pw2wireslave_config));

		if (copy_from_user(&conf, (void __user *) arg, sizeof(struct pw2wireslave_config)))
		{
			result = -EFAULT;
			break;
		}

		spin_lock_irqsave(&slave_data->lock, flags);

		memcpy( &slave_data->conf, &conf, sizeof(struct pw2wireslave_config) );

		dprintk(2,"Alladdren %d, Genecall: %d, Autoack %d\n",
			slave_data->conf.alladdren,
			slave_data->conf.gencall,
			slave_data->conf.autoack);

		TW_V1_SCTL_SET1(&slave_data->blkinfo, GENCALL_EN(slave_data->conf.gencall));
		TW_V1_SCTL_SET1(&slave_data->blkinfo, ALLADDREN(slave_data->conf.alladdren));
		TW_V1_SCTL_SET1(&slave_data->blkinfo, AUTOACK(slave_data->conf.autoack));

		if(slave_data->conf.gencall)
		{
			TW_V1_INTEN_SET1(&slave_data->blkinfo, SGENCALL(slave_data->conf.gencall));
		}

		spin_unlock_irqrestore(&slave_data->lock, flags);
		break;
	}

	case PW2WIRESLAVE_XDATASET: {
		struct pw2wireslave_rdwr_data   rdwr_pa;
		u8 __user *data_ptrs;
		u8  *buf;
		int i, j;
		int pos;

		if (copy_from_user(&rdwr_pa, (struct pw2wireslave_rdwr_data __user *) arg, sizeof(rdwr_pa)))
		{
			result = -EFAULT;
			break;
		}
		data_ptrs = (u8 __user *)rdwr_pa.buf;

		rdwr_pa.buf = kmalloc(rdwr_pa.nLen, GFP_KERNEL);

		if(rdwr_pa.buf == NULL)
		{
			result = -ENOMEM;
			break;
		}
		if(copy_from_user(rdwr_pa.buf, data_ptrs, rdwr_pa.nLen))
		{
			result = -EFAULT;
			break;
		}

		spin_lock_irqsave(&slave_data->lock, flags);

		dprintk(2, "PW2WIRESLAVE_XDATASET\n");

		buf = rdwr_pa.buf;
		switch( slave_data->mode )
		{
		case    smSINGLE_MODE:
			dprintk(3, "address=%08X\n", slave_data->my_address[0]);
			for(j = 0; j < rdwr_pa.nLen; j++)
			{
				pos = slave_data->tx_data_length + j;

				slave_data->tx_buffer[0][pos] = *buf;
				buf++;
				dprintk(3, "[%d] data=%02X\n", j, slave_data->tx_buffer[0][pos]);
			}
			slave_data->tx_data_length += rdwr_pa.nLen;
			slave_data->tx_data_ready[0] = TRUE;
			if( slave_data->state == srsHOLD )
			{
				//Send POSACK
				pw2wireslave_reply_posack(slave_data);
				slave_data->state = srsTX;
				if( slave_data->tx_data_pos == 0 )
				{
					pw2wireslave_reset_writefifo( slave_data );
				}
				pw2wireslave_send( slave_data );
			}
			else if( slave_data->state == srsTX )
			{
				if( slave_data->tx_data_pos == 0 )
				{
					pw2wireslave_reset_writefifo( slave_data );
				}
				pw2wireslave_send( slave_data );
			}
			break;
		case    smALL_MODE:
			for(i = 0; i < slave_data->my_address_num; i++)
			{
				if(rdwr_pa.addr == slave_data->my_address[i])
				{
					dprintk(3, "address=%08X\n", slave_data->my_address[i]);

					for(j = 0; j < rdwr_pa.nLen; j++)
					{
						pos = slave_data->tx_data_length + j;

						slave_data->tx_buffer[i][pos] = *buf;
						buf++;
						dprintk(3, "[%d] data=%02X\n", j, slave_data->tx_buffer[i][pos]);
					}
					slave_data->tx_data_length += rdwr_pa.nLen;
					slave_data->tx_data_ready[i] = TRUE;
					if( slave_data->saved_address_index == i )
					{
						if( slave_data->state == srsHOLD )
						{
							//Send POSACK
							pw2wireslave_reply_posack(slave_data);
							slave_data->state = srsTX;
							if( slave_data->tx_data_pos == 0 )
							{
								pw2wireslave_reset_writefifo( slave_data );
							}
							pw2wireslave_send( slave_data );
						}
						else if( slave_data->state == srsTX )
						{
							if( slave_data->tx_data_pos == 0 )
							{
								pw2wireslave_reset_writefifo( slave_data );
							}
							pw2wireslave_send( slave_data );
						}
					}
					break;
				}
			}

			break;
		default:
			break;
		}

		kfree(rdwr_pa.buf);

		spin_unlock_irqrestore(&slave_data->lock, flags);
		break;
	}
	case PW2WIRESLAVE_DATAGET: {
#if 1 //2018/07/14
        unsigned int i, len;
        spin_lock_irqsave(&slave_data->lock, flags);
        len = slave_data->rx_data_length;
        memcpy(&slave_data->user_temp_buf[0], &slave_data->rx_temp_buffer[0], slave_data->rx_data_length);
        memset(&slave_data->rx_temp_buffer[0], 0, MAX_DATA_STORAGE);
        slave_data->rx_data_length = 0;
        spin_unlock_irqrestore(&slave_data->lock, flags);
        
        dprintk(2, "PW2WIRESLAVE_DATAGET\n");
        for( i = 0; i < len; i++ )
		{
            dprintk(3, "0x%02X\n", slave_data->user_temp_buf[i]);
        }
        if(copy_to_user((u8 __user *) arg, &slave_data->user_temp_buf[0], len))
        {
            result = - EFAULT;
        }
#else
		unsigned int i;
		spin_lock_irqsave(&slave_data->lock, flags);

		dprintk(2, "PW2WIRESLAVE_DATAGET\n");
		for( i = 0; i < slave_data->rx_data_length; i++ )
		{
			dprintk(3, "0x%02X\n", slave_data->rx_temp_buffer[i]);
		}
		if(copy_to_user((u8 __user *) arg, &slave_data->rx_temp_buffer[0], slave_data->rx_data_length))
		{
			result = - EFAULT;
		}
		memset( &slave_data->rx_temp_buffer[0], 0, MAX_DATA_STORAGE );
		slave_data->rx_data_length = 0;
		spin_unlock_irqrestore(&slave_data->lock, flags);
#endif
		break;
	}
	case PW2WIRESLAVE_ALLSLAVEADDR: {
		struct pw2wireslave_allslaveaddr allslaveaddr;
		int i=0;
		u16 __user *slaveaddr_ptrs;
		u16  *buf;
		int tmp;

//2018/07/14		spin_lock_irqsave(&slave_data->lock, flags);

		dprintk(2, "PW2WIRESLAVE_ALLSLAVEADDR\n");
		memset(&allslaveaddr,0,sizeof(struct pw2wireslave_allslaveaddr));

		if(copy_from_user(&allslaveaddr, (struct pw2wireslave_allslaveaddr __user *) arg, sizeof(allslaveaddr)))
		{
			result = -EFAULT;
//2018/07/14			spin_unlock_irqrestore(&slave_data->lock, flags);
			break;
		}

		slaveaddr_ptrs = (u16 __user *)allslaveaddr.addr;

		tmp = (int)allslaveaddr.nslave * (int)sizeof(u16 __user *);
		allslaveaddr.addr = kmalloc(tmp, GFP_KERNEL);

		if(allslaveaddr.addr == NULL)
		{
			result = -ENOMEM;
//2018/07/14			spin_unlock_irqrestore(&slave_data->lock, flags);
			break;
		}

		if(copy_from_user(allslaveaddr.addr, slaveaddr_ptrs, tmp))
		{
			kfree( allslaveaddr.addr );
			result = -EFAULT;
//2018/07/14			spin_unlock_irqrestore(&slave_data->lock, flags);
			break;
		}

		buf = allslaveaddr.addr;
		slave_data->my_address_num = allslaveaddr.nslave;
		if( slave_data->my_address_num > MAXNUM_SLAVES )
		{
			slave_data->my_address_num = MAXNUM_SLAVES;
		}

		for(i = 0; i < slave_data->my_address_num; i++)
		{
			slave_data->my_address[i] = *buf;
			buf++;
		}

		kfree( allslaveaddr.addr );

		slave_data->conf.alladdren = 1;
		slave_data->conf.autoack = 0;
		slave_data->mode = smALL_MODE;
		slave_data->state= srsIDLE;
		slave_data->addr_configured = TRUE;

		pw2wireslave_reset(slave_data);

		pw2wireslave_enable_interrupt(slave_data);

//2018/07/14		spin_unlock_irqrestore(&slave_data->lock, flags);
		break;
	}
	case PW2WIRESLAVE_MASTEWRITE_DATAREADY: {
		if (wait_event_interruptible(slave_data->wq, slave_data->rx_data_ready))
		{
			dprintk(1, "PW2WIRESLAVE_MASTEWRITE_DATAREADY interrupted\n");
		}

//2018/07/14		spin_lock_irqsave(&slave_data->lock, flags);

		dprintk(2, "PW2WIRESLAVE_MASTEWRITE_DATAREADY\n");
		if (copy_to_user((void *)arg, &slave_data->rx_data_ready, sizeof(unsigned int)))
		{
			result = -EFAULT;
//2018/07/14			spin_unlock_irqrestore(&slave_data->lock, flags);
			break;
		}

//2018/07/14		spin_unlock_irqrestore(&slave_data->lock, flags);
		break;
	}
	case PW2WIRESLAVE_XDATAINFO: {
		struct pw2wireslave_Getdatainfo datainfo;

		spin_lock_irqsave(&slave_data->lock, flags);
		dprintk(2, "PW2WIRESLAVE_XDATAINFO\n");
		datainfo.cFlag = slave_data->rx_data_ready;
		datainfo.cAddr = slave_data->backup_saved_address;
		datainfo.uReceivedCount = slave_data->rx_data_length;
		dprintk(3, "%d, addr=0x%02X, len=%d\n", datainfo.cFlag, datainfo.cAddr, datainfo.uReceivedCount);
		if (slave_data->rx_data_ready == 1)
		{
			slave_data->rx_data_ready = 0;
			slave_data->backup_saved_address = INVALID_ADDRESS;
		}
		spin_unlock_irqrestore(&slave_data->lock, flags);   //2018/07/14
		if (copy_to_user((struct pw2wireslave_Getdatainfo __user*)arg, &datainfo, sizeof(datainfo)))
		{
			result = -EFAULT;
//2018/07/14			spin_unlock_irqrestore(&slave_data->lock, flags);
			break;
		}
//2018/07/14		spin_unlock_irqrestore(&slave_data->lock, flags);
		break;
	}
	default:
		result = -ENOTTY;
		break;
	}
	return result;
}

static int pw2wireslave_major;
static struct class *pw2wireslave_class;
static struct file_operations pw2wireslave_fops = {
	.owner          = THIS_MODULE,
	.open           = pw2wireslave_open,
	.release        = pw2wireslave_release,
	.unlocked_ioctl = pw2wireslave_ioctl,
	.poll           = pw2wireslave_poll
};

// Check if the address is for this device
// Return 1 - Address match
//        0 - Address unmatch
//Note : Only for all address mode
static inline int pw2wireslave_validateaddress(struct pw2wireslave_data *slave_data)
{
	int i;
	int ret = 0;    //Address unmatch
	unsigned int address;

	TW_V1_SRFIFO_GET(&slave_data->blkinfo, ALL(address));
	dprintk(2, "SBYRCVD, Address=%02X\n", address);

	for(i = 0; i < slave_data->my_address_num; i++)
	{
		if( (address & 0xFE) == slave_data->my_address[i] )
		{
			slave_data->saved_address = address;
			slave_data->saved_address_index = i;
			ret = 1;
			break;
		}
	}

	return ret;
}

//
//Acions in INVALID state
//
static void pw2wireslave_invalid_start( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	//Release bus
	pw2wireslave_release_bus(slave_data);
}

static void pw2wireslave_invalid_sbyrcvd( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_invalid_swrreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_invalid_srdreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_invalid_swflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_invalid_srflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_invalid_swflocked( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_invalid_snack( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_invalid_stop( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_invalid_shold( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

//
//Actions in IDLE state
//
static void pw2wireslave_idle_start( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "mode=%d\n", slave_data->mode);

	switch( slave_data->mode )
	{
	case smSINGLE_MODE:
		slave_data->state = srsCMD;
		break;
	case smALL_MODE:
		slave_data->state = srsADDR;
		slave_data->saved_address = INVALID_ADDRESS;
		break;
	default:
		break;
	}
}

static void pw2wireslave_idle_sbyrcvd( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_idle_swrreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_idle_srdreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_idle_swflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");
	//Reset fifos
	pw2wireslave_reset_allfifos( slave_data );
}

static void pw2wireslave_idle_srflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");
	//Reset fifos
	pw2wireslave_reset_allfifos( slave_data );
}

static void pw2wireslave_idle_swflocked( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");
	//Reset fifos
	pw2wireslave_reset_allfifos( slave_data );
}

static void pw2wireslave_idle_snack( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_idle_stop( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_reset_allfifos( slave_data );
}

static void pw2wireslave_idle_shold( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_reset_allfifos( slave_data );
}

//
//Actions in ADDR state
//
static void pw2wireslave_addr_start( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");
	//NOP
}

static void pw2wireslave_addr_sbyrcvd( struct pw2wireslave_data *slave_data )
{
	if( pw2wireslave_validateaddress( slave_data ) )
	{
		if( slave_data->saved_address & 0x01 )
		{
			if( slave_data->tx_data_ready[slave_data->saved_address_index] == TRUE )
			{
				//Send POSACK in regardless of ack mode
				pw2wireslave_reply_posack(slave_data);
				slave_data->state = srsTX;
			}
			else
			{
#if defined(CONFIG_RUBY_CHIP_B0) || defined(CONFIG_RUBY_CHIP_C0)
				//Send NEGACK in regardless of ack mode
				pw2wireslave_reply_negack(slave_data);
				dprintk(2, "NEGACK\n");
				pw2wireslave_reset_allfifos( slave_data );
				slave_data->state = srsSTOP;
#elif defined(CONFIG_TOPAZ_CHIP) || defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
				dprintk(2, "Hold the SCL\n");
				slave_data->state = srsHOLD;
#else
#error  Please define chip name
#endif
			}
		}
		else
		{
			//Send POSACK in regardless of ack mode
			pw2wireslave_reply_posack(slave_data);

			//Reset AUTOACK mode as configured
			TW_V1_SCTL_SET1(&slave_data->blkinfo, AUTOACK(slave_data->conf.autoack));

			slave_data->state = srsRX;
			slave_data->rx_data_pos = 0;
		}
	}
	else
	{
		slave_data->state = srsIDLE;

		//Release bus
		pw2wireslave_release_bus(slave_data);
	}
}

static void pw2wireslave_addr_swrreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_addr_srdreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_addr_swflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_addr_srflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_addr_swflocked( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_addr_snack( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_addr_stop( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	slave_data->state = srsIDLE;
	//Release bus
	pw2wireslave_release_bus(slave_data);
}

static void pw2wireslave_addr_shold( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

//
//Actions in CMD state
//
static void pw2wireslave_cmd_start( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_cmd_sbyrcvd( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	//Receive data
	pw2wireslave_recv_all( slave_data );
	slave_data->state = srsRX;
}

static void pw2wireslave_cmd_swrreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	slave_data->state = srsRX;
	slave_data->rx_data_pos = 0;
}

static void pw2wireslave_cmd_srdreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	//Master read
	if( slave_data->tx_data_ready[slave_data->saved_address_index] == TRUE )
	{
		slave_data->state = srsTX;
		if( slave_data->tx_data_pos == 0 )
		{
			pw2wireslave_reset_writefifo( slave_data );
		}
		pw2wireslave_send( slave_data );
	}
	else
	{
		//Send NEGACK in regardless of ack mode
#if defined(CONFIG_RUBY_CHIP_B0) || defined(CONFIG_RUBY_CHIP_C0)
		//Send NEGACK in regardless of ack mode
		pw2wireslave_reply_negack(slave_data);   //NEGACK
		pw2wireslave_reset_allfifos( slave_data );
		slave_data->state = srsSTOP;
#elif defined(CONFIG_TOPAZ_CHIP) || defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
		dprintk(2, "Hold the SCL\n");
		slave_data->state = srsHOLD;
#else
#error  Please define chip name
#endif
	}
}

static void pw2wireslave_cmd_swflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_cmd_srflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_cmd_swflocked( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_cmd_snack( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_cmd_stop( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_cmd_shold( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	//Master read
	if( slave_data->tx_data_ready[slave_data->saved_address_index] == TRUE )
	{
		if( slave_data->tx_data_pos == 0 )
		{
			pw2wireslave_reset_writefifo( slave_data );
		}
		pw2wireslave_send( slave_data );
	}
}

//
//Actions in RX state
//
static void pw2wireslave_rx_start( struct pw2wireslave_data *slave_data )
{
	dprintk(2, "RE-START\n");

	//Re-start
	switch( slave_data->mode )
	{
	case smSINGLE_MODE:
		slave_data->state = srsCMD;
		break;
	case smALL_MODE:
		slave_data->state = srsADDR;
		slave_data->saved_address = INVALID_ADDRESS;
		break;
	default:
		break;
	}
	slave_data->rx_data_pos = 0;
}

static void pw2wireslave_rx_sbyrcvd( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_recv_all( slave_data );
}

static void pw2wireslave_rx_swrreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");
	//NOP
}

static void pw2wireslave_rx_srdreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "Enter TX state\n");
	//Enter TX state
	slave_data->state = srsTX;
	slave_data->rx_data_pos = 0;
}

static void pw2wireslave_rx_swflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_rx_srflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_recv_all( slave_data );
}

static void pw2wireslave_rx_swflocked( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_rx_snack( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_rx_stop( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	if( slave_data->rx_data_pos > 0 )
	{
		slave_data->rx_data_length = slave_data->rx_data_pos;
		memcpy(slave_data->rx_temp_buffer, slave_data->rx_buffer, MAX_DATA_STORAGE);
		slave_data->rx_data_ready = 1;
		slave_data->backup_saved_address = slave_data->saved_address;

		memset(slave_data->rx_buffer, 0, MAX_DATA_STORAGE);
		slave_data->rx_data_pos = 0;

		wake_up_interruptible( &slave_data->wq );
		dprintk(2, "wake up user addr=%02X\n", slave_data->backup_saved_address);
	}
	//Reset all FIFO especially empty write FIFO
	pw2wireslave_reset_allfifos( slave_data );
	slave_data->tx_data_length = 0;
	slave_data->tx_data_pos    = 0;

	slave_data->state = srsIDLE;
}

static void pw2wireslave_rx_shold( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_recv_all( slave_data );
}

//
//Actions in TX state
//
static void pw2wireslave_tx_start( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_tx_sbyrcvd( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "IGNORE\n");
	//NOP
}

static void pw2wireslave_tx_swrreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_tx_srdreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_tx_swflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_send( slave_data );
}

static void pw2wireslave_tx_srflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_tx_swflocked( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_send( slave_data );
}

static void pw2wireslave_tx_snack( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	slave_data->state = srsSTOP;
}

static void pw2wireslave_tx_stop( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_reset_allfifos( slave_data );

	slave_data->state = srsIDLE;
	slave_data->tx_data_ready[slave_data->saved_address_index] = FALSE;
	slave_data->tx_data_length = 0;
	slave_data->tx_data_pos    = 0;
}

static void pw2wireslave_tx_shold( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_send( slave_data );
}

//
//Actions in STOP state
//
static void pw2wireslave_stop_start( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_stop_sbyrcvd( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_stop_swrreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_stop_srdreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_stop_swflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_stop_srflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_stop_swflocked( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");

	pw2wireslave_unlock( slave_data );
}

static void pw2wireslave_stop_snack( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_stop_stop( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "\n");

	pw2wireslave_reset_allfifos( slave_data );

	slave_data->state = srsIDLE;
	slave_data->tx_data_ready[slave_data->saved_address_index] = FALSE;
	slave_data->tx_data_length = 0;
	slave_data->tx_data_pos    = 0;
}

static void pw2wireslave_stop_shold( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

//
//Actions in HOLD state
//
static void pw2wireslave_hold_start( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_sbyrcvd( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_swrreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_srdreq( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_swflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_srflim( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_swflocked( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_snack( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_stop( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "ERROR\n");
	//NOP
}

static void pw2wireslave_hold_shold( struct pw2wireslave_data *slave_data )
{
	dprintk(3, "HOLDING BUS\n");
	//NOP
}

static void pw2wireslave_statemachine(struct pw2wireslave_data *slave_data )
{
	unsigned long interrupt_status;
	const unsigned long *event_mask;
	const pw2wireslave_statemachine_t *statemachine;
	const pw2wireslave_action_t   *action;

	while(1)
	{
		TW_V1_INTSTAT_GET(&slave_data->blkinfo, ALL(interrupt_status));
		if( interrupt_status == 0 )
		{
			break;
		}

		slave_data->interrupt_status = interrupt_status;

		dprintk(2, "INTSTAT=0x%08X\n", (unsigned int)interrupt_status);

		event_mask = (unsigned long *)pw2wireslave_event_mask;

		while( interrupt_status )
		{
			if( interrupt_status & (*event_mask) )
			{
				slave_data->event = *event_mask;

				statemachine = (pw2wireslave_statemachine_t *)&slave_statemachine_func_table[0];
				while( statemachine->action_table != NULL )
				{
					if( statemachine->state == slave_data->state )
					{
						action = (pw2wireslave_action_t *)statemachine->action_table;
						while( action->func != NULL )
						{
							if( slave_data->event == action->event )
							{
								action->func(slave_data);
							}
							action++;
						}

						interrupt_status &= ~(*event_mask);
						pw2wireslave_clear_event(slave_data, *event_mask);
						break;
					}
					statemachine++;
				}
			}
			event_mask++;
		}
	}
}

static irqreturn_t pw2wireslave_interrupt_handler(int irq, void *dev_id)
{
	struct pw2wireslave_data *slave_data = dev_id;

//	local_irq_disable();        //Fix:this is unnecessary

	spin_lock(&slave_data->lock);

	pw2wireslave_statemachine( slave_data );

	spin_unlock( &slave_data->lock );

//	local_irq_enable(); //Fix: this is unnecessary
	return IRQ_HANDLED;
}


static int pw2wireslave_probe(struct platform_device *pdev)
{
	struct resource *res;
	struct pw2wireslave_data *slave_data;
	void *vaddr = NULL;
	int result = 0;
	u32 i2c_id;

	if (of_property_read_u32(pdev->dev.of_node, "i2c-id", &i2c_id)) {
		dev_err(&pdev->dev, "i2c-id is not set\n");
		return -EINVAL;
	}
	if (i2c_id > MAXNUM_SLAVES) {
		dev_err(&pdev->dev, "i2c-id is out of range (%u, max %d)\n",
			i2c_id, MAXNUM_SLAVES);
		return -EINVAL;
	}
	dprintk(2, "probe %s%d\n", pdev->name, i2c_id);

#ifdef DEBUG_LOG
	memset( debug_buffer, 0, sizeof(debug_buffer) );
	debug_buf_p = &debug_buffer[0];
#endif

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "failed to get IORESOURCE_MEM\n");
		return -ENXIO;
	}

	if (!request_mem_region(res->start, res->end - res->start + 1, dev_name(&pdev->dev))) {
		dev_err(&pdev->dev, "IO mem at %08x is in use\n", res->start);
		return -EBUSY;
	}

	slave_data = kzalloc(sizeof(struct pw2wireslave_data), GFP_KERNEL);
	if (!slave_data) {
		dev_err(&pdev->dev, "out of memory\n");
		result = -ENOMEM;
		goto err;
	}

	slave_data->pdev = pdev;
	slave_data->phys_base_addr = res->start;
	slave_data->reg_size = res->end - res->start + 1;
	slave_data->irq = -1;

	spin_lock_init(&slave_data->lock);

	vaddr = ioremap(slave_data->phys_base_addr, slave_data->reg_size);
	if (IS_ERR(vaddr)) {
		dev_err(&pdev->dev, "ioremap failed\n");
		result = PTR_ERR(vaddr);
		goto err;
	}
	slave_data->base_addr = vaddr;
	dprintk(2, "phys 0x%08x virt %p\n", res->start, vaddr);

	RAL_INIT(&slave_data->blkinfo, (uint32) slave_data->base_addr, NULL);

    //Get IRQ# and check error 
    slave_data->irq = platform_get_irq(pdev, 0);
    if (slave_data->irq < 0) {
        dev_err(&pdev->dev, "IRQ missing or invalid\n");
        goto err;
    }

    //Request IRQ with devm call
    result = devm_request_irq(&pdev->dev, slave_data->irq, pw2wireslave_interrupt_handler, IRQF_SHARED,
                              dev_name(&pdev->dev), slave_data);
    if (result) {
        slave_data->irq = result;
        dev_err(&pdev->dev, "failed to get irq %d\n", slave_data->irq);
        goto err;
    }

	cdev_init(&slave_data->cdev, &pw2wireslave_fops);
	slave_data->cdev.owner = THIS_MODULE;
	result = cdev_add(&slave_data->cdev, MKDEV(pw2wireslave_major, i2c_id), 1);
	if (result < 0) {
		dev_err(&pdev->dev, "can't register cdev %d/%d\n",
                pw2wireslave_major, i2c_id);
		goto err;
	}
	platform_set_drvdata(pdev, slave_data);
	pw2wireslave_enable(slave_data);
	pw2wireslave_reset(slave_data);
	pw2wireslave_disable_interrupt(slave_data);

	slave_data->dev = device_create(pw2wireslave_class, &pdev->dev,
                                    MKDEV(pw2wireslave_major, i2c_id),
                                    slave_data, "i2c_slave-%d", i2c_id);
	if (IS_ERR(slave_data->dev)) {
		result = PTR_ERR(slave_data->dev);
		cdev_del(&slave_data->cdev);
		goto err;
	}
	return 0;

err:
	pw2wireslave_reset(slave_data);
	pw2wireslave_disable(slave_data);

	if (slave_data) {
		if (0 < slave_data->irq)    //Free IRQ if it is already requested successfully
		{
			devm_free_irq(&pdev->dev, slave_data->irq, slave_data);
		}
		if (slave_data->base_addr)
		{
			iounmap(slave_data->base_addr);
		}
		kfree(slave_data);
	}
	release_mem_region(res->start, res->end - res->start + 1);
	return result;
}

static int pw2wireslave_remove(struct platform_device *pdev)
{
	struct pw2wireslave_data *slave_data;

#ifdef DEBUG_LOG
	{
		char *start_p;
		char *end_p;
		char *tmp_buf;
		size_t tmp_siz;

		dprintk(2, "END OF TRACE\n");

		tmp_buf = kzalloc( DEBUG_BUF_SIZE + 1024, GFP_KERNEL );

		tmp_siz = debug_buf_p - &debug_buffer[0];
		if( overwrapped )
		{
			tmp_siz = DEBUG_BUF_SIZE;
		}

		memcpy( tmp_buf, &debug_buffer[0], tmp_siz );

		start_p = tmp_buf;
		end_p   = tmp_buf;

		while( *end_p )
		{
			if( *end_p != '\n' )
			{
				end_p++;
			}
			else
			{
				*end_p++ = 0;
				printk( "%s\n", start_p );
				start_p = end_p;
			}
		}
		kfree( tmp_buf );
	}
#endif
	dev_dbg(&pdev->dev, "pw2wireslave_remove\n");
	slave_data = platform_get_drvdata(pdev);
	pw2wireslave_disable(slave_data);
	platform_set_drvdata(pdev, NULL);

	device_unregister(slave_data->dev);
	cdev_del(&slave_data->cdev);
//	if (-1 != slave_data->irq)      //this is unnessary when using devm_request_irq
//	{
//		free_irq(slave_data->irq, slave_data);
//	}
	if (slave_data->base_addr)
	{
		iounmap(slave_data->base_addr);
	}
	release_mem_region(slave_data->phys_base_addr, slave_data->reg_size);
	kfree(slave_data);
	return 0;
}

#define pw2wireslave_suspend NULL
#define pw2wireslave_resume NULL

static struct of_device_id pw2wireslave_of_match[] = {
       { .compatible = "pixelworks,two_wire_slave_v1" },
       { },
};

static struct platform_driver pw2wireslave_driver = {
	.probe      = pw2wireslave_probe,
	.remove     = pw2wireslave_remove,
	.suspend    = pw2wireslave_suspend,
	.resume     = pw2wireslave_resume,
	.driver     = {
		.name       = "pwhw2wire_slave",
		.owner      = THIS_MODULE,
		.of_match_table = pw2wireslave_of_match,
	}
};

static int __init pw2wireslave_init(void)
{
	int rc;
	dev_t pw2wireslave_dev;

	dprintk(1, "\n");
	pw2wireslave_class = class_create(THIS_MODULE, "two_wire_slave");
	if (IS_ERR(pw2wireslave_class)) {
		printk(KERN_ERR "ERROR: pw2wireslave_init: class_create failed\n");
		return PTR_ERR(pw2wireslave_class);
	}
	rc = alloc_chrdev_region(&pw2wireslave_dev,
				 0, MAXNUM_SLAVES, "two_wire_slave");
	if (rc) {
		printk(KERN_ERR "ERROR: pw2wireslave_init: alloc_chrdev_region failed\n");
		class_destroy(pw2wireslave_class);
		return rc;
	}
	pw2wireslave_major = MAJOR(pw2wireslave_dev);
	rc = platform_driver_register(&pw2wireslave_driver);
	if (rc) {
		printk(KERN_ERR "ERROR: pw2wireslave_init: platform_driver_register failed\n");
		unregister_chrdev_region(MKDEV(pw2wireslave_major, 0), MAXNUM_SLAVES);
		class_destroy(pw2wireslave_class);
		return rc;
	}
	return 0;
}

static __exit void pw2wireslave_exit(void)
{
	dprintk(1, "\n");
	platform_driver_unregister(&pw2wireslave_driver);
	unregister_chrdev_region(MKDEV(pw2wireslave_major, 0), MAXNUM_SLAVES);  //Fix error when reloading
	class_destroy(pw2wireslave_class);
}

module_init(pw2wireslave_init);
module_exit(pw2wireslave_exit);
