/** @file		showimage.c
 *	@author		Allen.Kuo
 *	@version	1.0
 *	@date		2008.12.26
 *
 * <pre>
 * --------------------------------------------------------------------------
 * Demo / Test Load a Image
 * --------------------------------------------------------------------------
 * </pre>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "utilDef.h"
#include "utilPoolMgr.h"

#include "SDL.h"
#include "SDL_image.h"
#include "PW_SDL_image.h"
#include "PW_IMAGE_filters.h"

/* Import Jpeg Files */
#include "jpg_res_sample.h"

#define USING_DOUBLE_BUFFER		0
#define USING_YUV444_IMAGEL		0
#define WXGA_W					(1280)
#define WXGA_H					(768)
#define JPEG_POOL_SIZE			(250 * 1024)

/* Debug Tools */
#ifdef PW_SDL_EMU
#define L_PRINT      printf
#if defined(_DEBUG) || defined(DEBUG)
#define L_TRACE      printf
#else
#define L_TRACE      __noop
#endif
#else
#define L_PRINT(...)	fprintf(stderr, __VA_ARGS__)
#if defined(_DEBUG) || defined(DEBUG)
#define L_TRACE(...)	fprintf(stderr, __VA_ARGS__)
#else
#define L_TRACE(...)
#endif
#endif

// debug tools
#define TEST_LOG_START(msg)		(test_log(msg, 0, 0))
#define TEST_LOG_END(c)			(test_log(NULL, 1, (c)))
#define TEST_LOG_END1(c)		(test_log(NULL, 1, (c)), waitkey())
#define TEST_LOG_END2(c, t)		(test_log(NULL, 1, (c)), waitms((t)))
#define TEST_LOG_END3(c, t)		(test_log(NULL, 1, (c)), waitevent0((t)))
#ifdef PW_SDL_EMU
#define TEST_LOG_END4(c)		TEST_LOG_END1(c)
#else
#define TEST_LOG_END4(c)		(test_log(NULL, 1, (c)), waitms((5000)))
#endif

static void test_log(char *msg, int done, int counter)
{
	static Uint32 s_nTicks;
	static char * s_pcMsg;
	if (done)
	{
		Uint32 nTicksPassed = SDL_GetTicks() - s_nTicks;
		if (counter == 0)
		{
			L_PRINT("%s - done, time elapse %u ms\n", 
				s_pcMsg, (unsigned int)nTicksPassed);
		}
		else
		{
			L_PRINT("%s - done, time elapse %u ms, fps %.2f\n", 
				s_pcMsg, (unsigned int)nTicksPassed, (1000 * (float)counter / nTicksPassed));
		}
		L_PRINT("=============================================================\n");
	}
	else
	{
		s_pcMsg = msg;
		s_nTicks = SDL_GetTicks();
		L_PRINT("\n\n");
		L_PRINT("=============================================================\n");
		L_PRINT("%s - start\n", s_pcMsg);
	}
}

static int waitkey()
{
	SDL_Event event;
	L_PRINT("Press a key!\n");
	while(1)
	{
		while (SDL_PollEvent(&event))
		{
			switch (event.type)
			{
			case SDL_KEYDOWN:
				return event.key.keysym.unicode;
			case SDL_QUIT:
				return -1;
			}
		}
	}
	return 0;
}

static int waitevent0(Uint32 ms)
{
	int cc = 0;
	SDL_Event event;
	Uint32 ticks;
	L_PRINT("Wait %d ms!\n", ms);
	ticks = SDL_GetTicks();
	while(SDL_GetTicks() - ticks < 2000)
	{
		SDL_PollEvent(&event);
		cc++;
	}
	return cc;
}

static int waitms(Uint32 ms)
{
	L_PRINT("Delay %d ms!\n", ms);
	SDL_Delay(ms);
	return 0;
}

static int get_simple_rnd()
{
	return rand();
}

/* Call this instead of exit(), so we can clean up SDL: atexit() is evil. */
static void quit(int rc)
{
	SDL_Quit();
	exit(rc);
}

int main(int argc, char *argv[])
{
	SDL_Surface *screen;
	SDL_Surface *bitmap;
	SDL_Surface *hwsurface;
	Uint8  video_bpp;
	Uint32 videoflags;
	Uint32 color;
	SDL_Rect rcClip;
	SDL_RWops *rwops;
	PW_IMAGE_FILTERS filters[2];
	PW_IMAGE_FILTER *filterResample;
	PW_IMAGE_FILTER *filterColorConvert;
	void * jpeg_buf;
	Uint32 jpeg_buf_size;

	/* Initialize SDL */
	if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) {
		fprintf(stderr, "Couldn't initialize SDL: %s\n",SDL_GetError());
		return(1);
	}

	video_bpp = 24;
	videoflags = SDL_HWSURFACE;

	/* Set 1280x768 video mode */
	if ( (screen = SDL_SetVideoMode(WXGA_W,WXGA_H,video_bpp,videoflags)) == NULL ) {
		fprintf(stderr, "Couldn't set 1280x768x%d video mode: %s\n",
						video_bpp, SDL_GetError());
		quit(2);
	}

	TEST_LOG_START("SW: Load JPG ......");
	bitmap = IMG_Load("samples/sample.jpg");
	if (bitmap == NULL)
	{
		fprintf(stderr, "Cannot load bitmap\n");
		quit(1);
	}
	SDL_BlitSurface(bitmap, NULL, screen, NULL);
	SDL_Flip(screen);
	SDL_FreeSurface(bitmap);
	TEST_LOG_END4(0);

	TEST_LOG_START("SW: Load PNG ......");
	bitmap = IMG_Load("samples/sample.png");
	if (bitmap == NULL)
	{
		fprintf(stderr, "Cannot load bitmap\n");
		quit(1);
	}
	SDL_BlitSurface(bitmap, NULL, screen, NULL);
	SDL_Flip(screen);
	SDL_FreeSurface(bitmap);
	TEST_LOG_END4(0);

	/* Clear Screen */
	color = SDL_MapRGB(screen->format, 0, 0, 0);
	SDL_FillRect(screen, NULL, color);
	SDL_Flip(screen);

	/* Load JPEG file to screen directly */
	TEST_LOG_START("SW: Load JPG Directly to Screen");
	PW_IMG_Load("samples/sample.jpg", screen, NULL, NULL);
	SDL_Flip(screen);
	TEST_LOG_END4(0);

	/* Clear Screen */
	color = SDL_MapRGB(screen->format, 0, 0, 0);
	SDL_FillRect(screen, NULL, color);
	SDL_Flip(screen);

	/* Load JPEG file to screen directly */
	TEST_LOG_START("SW: Load JPG Directly to Screen with Clip");
	rcClip.x = 201;
	rcClip.y = 199;
	rcClip.w = 507;
	rcClip.h = 497;
	PW_IMG_Load("samples/sample.jpg", screen, &rcClip, NULL);
	SDL_Flip(screen);
	TEST_LOG_END4(0);

	/* Create Color Convert Filter */
	filterColorConvert = PW_IMAGE_FILTER_colorconvert_create(0);
	if (filterColorConvert == NULL)
	{
		fprintf(stderr, "Cannot create color convert filter\n");
		quit(1);
	}

	/* Create Resample Filter */
	filterResample = PW_IMAGE_FILTER_resample_create(0);
	if (filterResample == NULL)
	{
		fprintf(stderr, "Cannot create resample filter\n");
		quit(1);
	}

	/* Setup Filters */
	filters[0].filter = filterColorConvert;
	filters[1].filter = filterResample;
	filters[0].next = &filters[1];
	filters[1].next = NULL;

	/* Clear Screen */
	color = SDL_MapRGB(screen->format, 0, 0, 0);
	SDL_FillRect(screen, NULL, color);
	SDL_Flip(screen);

	TEST_LOG_START("SW: Load Large JPG to Screen with Resample Filter");
	PW_IMG_Load("samples/large.jpg", screen, NULL, filters);
	SDL_Flip(screen);
	TEST_LOG_END4(0);

	/* Clear Screen */
	color = SDL_MapRGB(screen->format, 0, 0, 0);
	SDL_FillRect(screen, NULL, color);
	SDL_Flip(screen);

	TEST_LOG_START("SW: Load Small JPG to Screen with Resample Filter");
	rcClip.x = 100;
	rcClip.y = 100;
	rcClip.w = screen->w - 200;
	rcClip.h = screen->h - 200;
	PW_IMG_Load("samples/small.jpg", screen, &rcClip, filters);
	SDL_Flip(screen);
	TEST_LOG_END4(0);

	/* Trying hardware jpeg decoding */

	/* Clear Screen */
	color = SDL_MapRGB(screen->format, 0, 0, 0);
	SDL_FillRect(screen, NULL, color);
	SDL_Flip(screen);

	rwops = SDL_RWFromMem((void *)m_acJpe_res_sample_data, sizeof(m_acJpe_res_sample_data));

	TEST_LOG_START("SW: Load jpeg in RAM in Software Codec");
	PW_IMG_LoadTyped_RW(rwops, 0, "JPG", screen, NULL, NULL);
	SDL_Flip(screen);
	TEST_LOG_END4(0);

	/* Clear Screen */
	color = SDL_MapRGB(screen->format, 0, 0, 0);
	SDL_FillRect(screen, NULL, color);
	SDL_Flip(screen);

	SDL_RWseek(rwops, 0, RW_SEEK_SET);

	TEST_LOG_START("SW: Load jpeg in RAM in Software Codec with Clip and Filters");
	rcClip.x = 10;
	rcClip.y = 10;
	rcClip.w = screen->w - 600;
	rcClip.h = screen->h - 400;
	PW_IMG_LoadTyped_RW(rwops, 0, "JPG", screen, &rcClip, filters);
	SDL_Flip(screen);
	TEST_LOG_END4(0);

	SDL_RWclose(rwops);

	/* Clear Screen */
	color = SDL_MapRGB(screen->format, 0, 0, 0);
	SDL_FillRect(screen, NULL, color);
	SDL_Flip(screen);

	/* Init Jpeg Pool */
	utilPoolMgrJpegPoolInit();

	TEST_LOG_START("HW: Load jpeg and generate HW Surface in Hardware Codec");
	rwops = SDL_RWFromFile("samples/small.jpg", "rb");
	rwops->hidden.mem.base = NULL;	/* hack it to make sure hw codec not using */
	hwsurface = PW_IMG_LoadTyped_RW(rwops, 0, "PHJ", NULL, NULL, NULL);
	if ((hwsurface->flags & SDL_HWSURFACE) == SDL_SWSURFACE)
		L_PRINT("Cannot create HW surface, using SW sufrace to replace it\n");
	SDL_BlitSurface(hwsurface, NULL, screen, NULL);
	SDL_Flip(screen);
	TEST_LOG_END4(0);


	SDL_FreeSurface(hwsurface);
	SDL_RWclose(rwops);


	/* Load Demo Jpeg to Buffer */
	jpeg_buf = malloc(JPEG_POOL_SIZE);
	if (jpeg_buf == NULL)
	{
		L_PRINT("Load Jpeg File: Out of memory \n");
		quit(1);
	}


	/* Load Demo Jpeg to Buffer */
	rwops = SDL_RWFromFile("samples/tiny.jpg", "rb");
	jpeg_buf_size = SDL_RWread(rwops, jpeg_buf, 1, JPEG_POOL_SIZE);
	if (jpeg_buf_size == 0 || jpeg_buf_size < (Uint32)SDL_RWseek(rwops, 0, RW_SEEK_END))
	{
		L_PRINT("Load Jpeg File: Read JPEG file with Errors\n");
		quit(1);
	}
	SDL_RWclose(rwops);
	rwops = SDL_RWFromMem(jpeg_buf, jpeg_buf_size);

	TEST_LOG_START("HW: Load jpeg to Screen directly in Hardware Codec");
	hwsurface = PW_IMG_LoadTyped_RW(rwops, 0, "PHJ", NULL, NULL, NULL);
	if ((hwsurface->flags & SDL_HWSURFACE) == SDL_SWSURFACE)
		L_PRINT("Cannot create HW surface, using SW sufrace to replace it\n");
	SDL_BlitSurface(hwsurface, NULL, screen, NULL);
	SDL_Flip(screen);
	TEST_LOG_END4(0);

	SDL_FreeSurface(hwsurface);
	SDL_RWclose(rwops);
	
	TEST_LOG_START("SW: Load sample jpeg from file storage to RAM");
	rwops = SDL_RWFromFile("samples/420.jpg", "rb");
	jpeg_buf_size = SDL_RWread(rwops, jpeg_buf, 1, JPEG_POOL_SIZE);
	if (jpeg_buf_size == 0 || jpeg_buf_size < (Uint32)SDL_RWseek(rwops, 0, RW_SEEK_END))
	{
		L_PRINT("Load Jpeg File: Read JPEG file with Errors\n");
		quit(1);
	}
	SDL_RWclose(rwops);
	rwops = SDL_RWFromMem(jpeg_buf, jpeg_buf_size);	
	TEST_LOG_END4(0);


	/* Clear Screen */
	color = SDL_MapRGB(screen->format, 0, 0, 0);
	SDL_FillRect(screen, NULL, color);
	SDL_Flip(screen);

	TEST_LOG_START("HW: Load jpeg to Screen directly in Hardware Codec with Clip");
	rcClip.x = 50;
	rcClip.y = 50;
	rcClip.w = screen->w - 100;
	rcClip.h = screen->h - 100;
	PW_IMG_LoadTyped_RW(rwops, 0, "PHJ", screen, NULL, NULL);
	SDL_Flip(screen);
	TEST_LOG_END4(0);

	
	
	utilPoolMgrSystemPoolSave(spiFB0,"/mnt/spiflashrw/spiFB0");
	utilPoolMgrSystemPoolSave(spiFB1,"/mnt/spiflashrw/spiFB1");
	utilPoolMgrSystemPoolSave(spiFB2,"/mnt/spiflashrw/spiFB2");
	utilPoolMgrSystemPoolSave(spiY,"/mnt/spiflashrw/spiY");
	utilPoolMgrSystemPoolSave(spiCB,"/mnt/spiflashrw/spiCB");
	utilPoolMgrSystemPoolSave(spiCR,"/mnt/spiflashrw/spiCR");




	SDL_RWclose(rwops);

	free(jpeg_buf);

	/* Finalize Jpeg Pool */
	utilPoolMgrJpegPoolFinal();

	/* Clean up filters */
	PW_IMAGE_FILTER_colorconvert_free(filterColorConvert);
	PW_IMAGE_FILTER_resample_free(filterResample);

#if 0
	/* Hardware surface testing */
	TEST_LOG_START("Hardware surfaces testing");
	{
		int i;
		SDL_Surface * asfcList[1000];
		SDL_Surface * sfcSwSurface;
		int nSfcSize = sizeof(asfcList) / sizeof(asfcList[0]);

		sfcSwSurface = SDL_CreateRGBSurface(SDL_HWSURFACE, 8, 8, screen->format->BitsPerPixel,
			screen->format->Rmask, screen->format->Gmask, screen->format->Bmask, screen->format->Amask);
		if (sfcSwSurface == NULL)
		{
			L_PRINT("Create Software Surface Failed\n", i);
			quit(1);
		}

		for (i = 0; i < nSfcSize; i++)
		{
			asfcList[i] = SDL_CreateRGBSurface(SDL_HWSURFACE, 8, 8, screen->format->BitsPerPixel,
				screen->format->Rmask, screen->format->Gmask, screen->format->Bmask, screen->format->Amask);
			if (asfcList[i] == NULL || (asfcList[i]->flags & SDL_HWSURFACE) == SDL_SWSURFACE)
			{
				L_PRINT("Create Hardware Surface Failed: index %d\n", i);
			}
		}
		for (i = 0; i < nSfcSize; i++)
		{
			if (asfcList[i] != NULL)
			{
				if (SDL_LockSurface(asfcList[i]) != 0)
				{
					L_PRINT("Create Hardware Surface Lock Failed: index %d\n", i);
				}
			}
		}
		for (i = 0; i < nSfcSize; i++)
		{
			if (asfcList[i] != NULL)
			{
				SDL_UnlockSurface(asfcList[i]);
			}
		}
		for (i = 0; i < nSfcSize; i++)
		{
			if (asfcList[i] != NULL)
			{
				SDL_BlitSurface(sfcSwSurface, NULL, asfcList[i], NULL);
			}
		}
		for (i = 0; i < nSfcSize; i++)
		{
			if (asfcList[i] != NULL)
			{
				SDL_FreeSurface(asfcList[i]);
				asfcList[i] = NULL;
			}
		}
		if (sfcSwSurface != NULL)
		{
			SDL_FreeSurface(sfcSwSurface);
			sfcSwSurface = NULL;
		}
	}
	TEST_LOG_END4(0);
#endif
	
	SDL_Quit();
	return(0);
}
