/*
    SDL_image:  An example image loading library for use with SDL
    Copyright (C) 1997-2009 Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Sam Lantinga
    slouken@libsdl.org
*/

/* This is a TIFF image file loading framework */

#include <stdio.h>

#include "SDL_image.h"
#include "utilPoolMgr.h"

#ifdef LOAD_TIF

#include <tiffio.h>

#include "PW_SDL_image.h"


static struct {
	int loaded;
	void *handle;
	TIFF* (*TIFFClientOpen)(const char*, const char*, thandle_t, TIFFReadWriteProc, TIFFReadWriteProc, TIFFSeekProc, TIFFCloseProc, TIFFSizeProc, TIFFMapFileProc, TIFFUnmapFileProc);
	void (*TIFFClose)(TIFF*);
	int (*TIFFGetField)(TIFF*, ttag_t, ...);
	int (*TIFFReadRGBAImage)(TIFF*, uint32, uint32, uint32*, int);
	TIFFErrorHandler (*TIFFSetErrorHandler)(TIFFErrorHandler);
} lib;

#ifdef LOAD_TIF_DYNAMIC
int IMG_InitTIF()
{
	if ( lib.loaded == 0 ) {
		lib.handle = SDL_LoadObject(LOAD_TIF_DYNAMIC);
		if ( lib.handle == NULL ) {
			return -1;
		}
		lib.TIFFClientOpen =
			(TIFF* (*)(const char*, const char*, thandle_t, TIFFReadWriteProc, TIFFReadWriteProc, TIFFSeekProc, TIFFCloseProc, TIFFSizeProc, TIFFMapFileProc, TIFFUnmapFileProc))
			SDL_LoadFunction(lib.handle, "TIFFClientOpen");
		if ( lib.TIFFClientOpen == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.TIFFClose =
			(void (*)(TIFF*))
			SDL_LoadFunction(lib.handle, "TIFFClose");
		if ( lib.TIFFClose == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.TIFFGetField =
			(int (*)(TIFF*, ttag_t, ...))
			SDL_LoadFunction(lib.handle, "TIFFGetField");
		if ( lib.TIFFGetField == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.TIFFReadRGBAImage =
			(int (*)(TIFF*, uint32, uint32, uint32*, int))
			SDL_LoadFunction(lib.handle, "TIFFReadRGBAImage");
		if ( lib.TIFFReadRGBAImage == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.TIFFSetErrorHandler =
			(TIFFErrorHandler (*)(TIFFErrorHandler))
			SDL_LoadFunction(lib.handle, "TIFFSetErrorHandler");
		if ( lib.TIFFSetErrorHandler == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
	}
	++lib.loaded;

	return 0;
}
void IMG_QuitTIF()
{
	if ( lib.loaded == 0 ) {
		return;
	}
	if ( lib.loaded == 1 ) {
		SDL_UnloadObject(lib.handle);
	}
	--lib.loaded;
}
#else
int IMG_InitTIF()
{
	if ( lib.loaded == 0 ) {
		lib.TIFFClientOpen = TIFFClientOpen;
		lib.TIFFClose = TIFFClose;
		lib.TIFFGetField = TIFFGetField;
		lib.TIFFReadRGBAImage = TIFFReadRGBAImage;
		lib.TIFFSetErrorHandler = TIFFSetErrorHandler;
	}
	++lib.loaded;

	return 0;
}
void IMG_QuitTIF()
{
	if ( lib.loaded == 0 ) {
		return;
	}
	if ( lib.loaded == 1 ) {
	}
	--lib.loaded;
}
#endif /* LOAD_TIF_DYNAMIC */

/*
 * These are the thunking routine to use the SDL_RWops* routines from
 * libtiff's internals.
*/

static tsize_t tiff_read(thandle_t fd, tdata_t buf, tsize_t size)
{
	return SDL_RWread((SDL_RWops*)fd, buf, 1, size);
}

static toff_t tiff_seek(thandle_t fd, toff_t offset, int origin)
{
	return SDL_RWseek((SDL_RWops*)fd, offset, origin);
}

static tsize_t tiff_write(thandle_t fd, tdata_t buf, tsize_t size)
{
	return SDL_RWwrite((SDL_RWops*)fd, buf, 1, size);
}

static int tiff_close(thandle_t fd)
{
	/*
	 * We don't want libtiff closing our SDL_RWops*, but if it's not given
         * a routine to try, and if the image isn't a TIFF, it'll segfault.
	 */
	return 0;
}

static int tiff_map(thandle_t fd, tdata_t* pbase, toff_t* psize)
{
	return (0);
}

static void tiff_unmap(thandle_t fd, tdata_t base, toff_t size)
{
	return;
}

static toff_t tiff_size(thandle_t fd)
{
	Uint32 save_pos;
	toff_t size;

	save_pos = SDL_RWtell((SDL_RWops*)fd);
	SDL_RWseek((SDL_RWops*)fd, 0, SEEK_END);
        size = SDL_RWtell((SDL_RWops*)fd);
	SDL_RWseek((SDL_RWops*)fd, save_pos, SEEK_SET);
	return size;
}

int IMG_isTIF(SDL_RWops* src)
{
	int start;
	int is_TIF;
	Uint8 magic[4];

	if ( !src )
		return 0;
	start = SDL_RWtell(src);
	is_TIF = 0;
	if ( SDL_RWread(src, magic, 1, sizeof(magic)) == sizeof(magic) ) {
		if ( (magic[0] == 'I' &&
                      magic[1] == 'I' &&
		      magic[2] == 0x2a &&
                      magic[3] == 0x00) ||
		     (magic[0] == 'M' &&
                      magic[1] == 'M' &&
		      magic[2] == 0x00 &&
                      magic[3] == 0x2a) ) {
			is_TIF = 1;
		}
	}
	SDL_RWseek(src, start, SEEK_SET);
	return(is_TIF);
}

#if defined(IMG_PARTIAL_DECODE)
SDL_Surface *PW_IMG_LoadTIF_RW(SDL_RWops *src, SDL_Surface *target, SDL_Rect *rect, PW_IMAGE_FILTERS *filters, int type);
typedef struct _PNG_RGB_24Pixel{	
	unsigned char nRed;	
	unsigned char nGreen;	
	unsigned char nBlue;
	} PW_RGB_24Pixel;
typedef struct _PNG_RGB_32Pixel{	
	unsigned char nRed;	
	unsigned char nGreen;	
	unsigned char nBlue;	
	unsigned char nAlpha;
} PW_RGB_32Pixel;

void PW_TIFF_LIB_ConvertRGBA2RGB(unsigned *src_buf, unsigned int src_width ,unsigned int src_pitch,
										unsigned int src_bit_depth, unsigned int src_color_channels, PwImageColorFormat src_color_format,
										unsigned *dst_buf, unsigned int dst_width, unsigned int dst_pitch,
										unsigned int dst_bit_depth, unsigned int dst_color_channels,PwImageColorFormat dst_color_format)

{
	int i,j;
	PW_RGB_32Pixel *src_pixel_buf = src_buf;
	PW_RGB_24Pixel *dst_pixel_buf = dst_buf;

	//for(j=0; j < src_height; j ++)
	{
		for(i=0; i < src_width ; i ++)
		{
			dst_pixel_buf->nBlue = src_pixel_buf->nBlue;
			dst_pixel_buf->nGreen = src_pixel_buf->nGreen;		
			dst_pixel_buf->nRed = src_pixel_buf->nRed;		
				
			src_pixel_buf++;
			dst_pixel_buf++;
		}
	}


}

static int PW_TIFF_Read_Strip_Image(TIFF *in, SDL_Surface *target, SDL_Rect *rect, PW_IMAGE_FILTERS *filters)
{

    uint32  width, height;		/* image width & height */
	uint32 rowsperstrip;
    uint32* raster;			/* retrieve RGBA image */
    uint32  *wrk_line;
    int	    ok = 1;
    uint32  row;
	int x, y, clip_w, clip_h, offset_x;
	int		line_buf_size;
	int		stripe_buf_size;
	
	int i,j;
	Uint8 * dst_line_buf;
	PW_IMAGE_FILTER_CONTEXT filter_context;	

	/* Retrieve the dimensions of the image from the TIFF tags */
    TIFFGetField(in, TIFFTAG_IMAGEWIDTH, &width);
    TIFFGetField(in, TIFFTAG_IMAGELENGTH, &height);
	TIFFGetField(in, TIFFTAG_ROWSPERSTRIP, &rowsperstrip);
	
    if( !TIFFGetField(in, TIFFTAG_ROWSPERSTRIP, &rowsperstrip) ) {
        IMG_TRACE("[PW_TIFF_Read_Strip_Image Error] Source image not in strips \n");
        return (0);
    }

	stripe_buf_size = width * rowsperstrip * sizeof (uint32);
	IMG_TRACE("[PW_TIFF_Read_Strip_Image] required stripe buffer buffer = %d \n",stripe_buf_size);	
	line_buf_size = width * sizeof (uint32);
	dst_line_buf = (Uint8 *) malloc(line_buf_size);
	IMG_TRACE("[PW_TIFF_Read_Strip_Image] required color conversion buffer = %d \n",line_buf_size);	
	
	if(dst_line_buf == 0)
	{
        IMG_TRACE("[PW_TIFF_Read_Strip_Image Error] Not enough memory for conversion buffer \n");
        return (0);
	}

    /*
     * Allocate strip buffer
     */
    raster = (uint32*)malloc(stripe_buf_size);
    if (raster == 0) 
	{
		free(dst_line_buf);
        IMG_TRACE("[PW_TIFF_Read_Strip_Image Error] No space for raster (stripe) buffer");
        return (0);
    }

    /*
     * Allocate a scanline buffer for swapping during the vertical
     * mirroring pass.
     */
    wrk_line = (uint32*)malloc(line_buf_size);
    if (!wrk_line) {
		free(dst_line_buf);
		free(raster);
        IMG_TRACE("[PW_TIFF_Read_Strip_Image Error]  No space for line swap buffer");
        return (0);

    }

	if ( SDL_MUSTLOCK(target) ) {
		if ( SDL_LockSurface(target) < 0 ) {
			IMG_TRACE("Cannot Lock Surface");
			free(raster);
			free(wrk_line);
			free(dst_line_buf);
			return (0);			
		}
	}



	/* PW_SDL Spec. */
	if (rect == NULL) {
		x = 0;
		y = 0;
		clip_w = 0;
		clip_h = 0;
	} else {
		x = rect->x;
		y = rect->y;
		clip_w = rect->w;
		clip_h = rect->h;
		if (clip_w <= 0 || clip_h <= 0)
		{
			free(raster);
			free(wrk_line);
			free(dst_line_buf);		
			return (0);
		}
	}

	if (clip_w == 0) clip_w = target->w;		
	if (clip_h == 0) clip_h = target->h;

	filter_context.filters = filters;		
	filter_context.image = target;		
	filter_context.clip.x = x;		
	filter_context.clip.y = y;		
	filter_context.clip.w = clip_w;		
	filter_context.clip.h = clip_h;		
	PW_FILTER_TOOLS_init(&filter_context);		
	PW_FILTER_TOOLS_config(&filter_context, (int)width, (int)height, picfRGB);


    /*
     * Loop over the strips.
     */
    for( row = 0; ok && row < height; row += rowsperstrip )
    {
        int	rows_to_write, i_row;

        /* Read the strip into an RGBA array */
        if (!TIFFReadRGBAStrip(in, row, raster)) {
            ok = 0;
            break;
        }

      /*
         * Figure out the number of scanlines actually in this strip.
         */
        if( row + rowsperstrip > height )
            rows_to_write = height - row;
        else
            rows_to_write = rowsperstrip;

        /*
         * For some reason the TIFFReadRGBAStrip() function chooses the
         * lower left corner as the origin.  Vertically mirror scanlines.
         */
        for( i_row = 0; i_row < rows_to_write / 2; i_row++ )
        {
            uint32	*top_line, *bottom_line;

            top_line = raster + width * i_row;
            bottom_line = raster + width * (rows_to_write-i_row-1);

            memcpy(wrk_line, top_line, 4*width);
            memcpy(top_line, bottom_line, 4*width);
            memcpy(bottom_line, wrk_line, 4*width);
        }
        for( i_row = 0; ok && i_row < rows_to_write ; i_row++ )
        {
			PW_TIFF_LIB_ConvertRGBA2RGB(raster + (width * i_row), width,line_buf_size,
				8,4, picfGRAYSCALE,
				dst_line_buf,width, line_buf_size,
				8,3,picfRGB);

			if (PW_FILTER_TOOLS_sendPixels(&filter_context, 					
				(int)row + i_row, (Uint8 *)dst_line_buf, target->pitch, picfRGB) != 0)				
				{
					ok = 0;
					break;	
		}
    }
    }



	PW_FILTER_TOOLS_finalize(&filter_context);	
	if ( SDL_MUSTLOCK(target) ) {
		SDL_UnlockSurface(target);
	}	

	free(raster);
	free(wrk_line);
	free(dst_line_buf);
	return 1;

	
}


SDL_Surface *PW_IMG_LoadTIF_RW(SDL_RWops *src, SDL_Surface *target, SDL_Rect *rect, PW_IMAGE_FILTERS *filters, int type)
{
	int start;
	TIFF* tiff;
	SDL_Surface* surface = NULL;
	Uint32 img_width, img_height, tile_width, tile_height;
	Uint32 Rmask, Gmask, Bmask, Amask;
	Uint32 x, y;
	Uint32 half;
	Uint32 video_flags;

	surface = NULL;
	tiff = NULL;
	
	if ( !src ) {
		/* The error message has been set in SDL_RWFromFile */
		return NULL;
	}
	start = SDL_RWtell(src);
	if ( IMG_InitTIF() < 0 ) {
		return NULL;
	}
	/* turn off memory mapped access with the m flag */
	tiff = TIFFClientOpen("SDL_image", "rm", (thandle_t)src, 
		tiff_read, tiff_write, tiff_seek, tiff_close, tiff_size, tiff_map, tiff_unmap);
	if(!tiff)
		goto error;

	/* Retrieve the dimensions of the image from the TIFF tags */
	TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, &img_width);
	TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, &img_height);	
	Rmask = 0x000000FF;
	Gmask = 0x0000FF00;
	Bmask = 0x00FF0000;
	Amask = 0xFF000000;
	IMG_TRACE("TIFF : img_width = %d,img_height = %d\n",img_width,img_height);
	if(!target)
	{
		if ((img_width >  IMG_TIF_MAX_WIDTH ||
			img_height > IMG_TIF_MAX_HEIGHT) ||
			(img_width < IMG_TIF_MIN_WIDTH &&
			img_height < IMG_TIF_MIN_HEIGHT))
		{
			IMG_TRACE("TIFF : not support size. supported size IMG_TIF_MIN_WIDTH = %d,IMG_TIF_MIN_HEIGHT = %d,IMG_TIF_MAX_WIDTH = %d,IMG_TIF_MAX_HEIGHT = %d\n",IMG_TIF_MIN_WIDTH,IMG_TIF_MIN_HEIGHT,IMG_TIF_MAX_WIDTH,IMG_TIF_MAX_HEIGHT);
			goto error;
		}
	
		if (type == SDL_IMAGE_LOAD_PWSURFACE)
			video_flags = SDL_SWSURFACE|SDL_PWSWSURFACE;
		else
			video_flags = SDL_SWSURFACE;
		surface = SDL_AllocSurface(video_flags, img_width, img_height, 32,
			Rmask, Gmask, Bmask, Amask);
		if(!surface)
			goto error;
	}
	else
	{
		surface = target;
	}

	if(target != surface)
	{
		//decode entire image
		if(!lib.TIFFReadRGBAImage(tiff, img_width, img_height, surface->pixels, 0))
			goto error;
		
		/* libtiff loads the image upside-down, flip it back */
		half = img_height / 2;
		for(y = 0; y < half; y++)
		{
				Uint32 *top = (Uint32 *)surface->pixels + y * surface->pitch/4;
				Uint32 *bot = (Uint32 *)surface->pixels
						  + (img_height - y - 1) * surface->pitch/4;
			for(x = 0; x < img_width; x++)
			{
					Uint32 tmp = top[x];
				top[x] = bot[x];
				bot[x] = tmp;
			}
		}
	}
	else
	{
		//decode strip!! and scale to target size
		if(!PW_TIFF_Read_Strip_Image(tiff, surface, rect, filters))
		{
			goto error;
		}
	}


	
	TIFFClose(tiff);
	IMG_QuitTIF();
	return surface;
	
error:
	
	if (tiff)
	{
		TIFFClose(tiff);
	}
	SDL_RWseek(src, start, SEEK_SET);
	if ( surface && surface != target) {
		SDL_FreeSurface(surface);
	}
	IMG_QuitTIF();
	return NULL;



}

SDL_Surface* IMG_LoadTIF_RW(SDL_RWops* src,int type)
{
	//SDL_Surface *PW_IMG_LoadTIF_RW(SDL_RWops *src, SDL_Surface *target, SDL_Rect *rect, PW_IMAGE_FILTERS *filters, int type)

	return PW_IMG_LoadTIF_RW(src, NULL, NULL, NULL,type);

}
#else

SDL_Surface* IMG_LoadTIF_RW(SDL_RWops* src,int type)
{
	int start;
	TIFF* tiff;
	SDL_Surface* surface = NULL;
	Uint32 img_width, img_height;
	Uint32 Rmask, Gmask, Bmask, Amask;
	Uint32 x, y;
	Uint32 half;
	Uint32 video_flags;

	if ( !src ) {
		/* The error message has been set in SDL_RWFromFile */
		return NULL;
	}
	start = SDL_RWtell(src);

	if ( IMG_InitTIF() < 0 ) {
		return NULL;
	}

	/* turn off memory mapped access with the m flag */
	tiff = lib.TIFFClientOpen("SDL_image", "rm", (thandle_t)src, 
		tiff_read, tiff_write, tiff_seek, tiff_close, tiff_size, tiff_map, tiff_unmap);
	if(!tiff)
		goto error;

	/* Retrieve the dimensions of the image from the TIFF tags */
	lib.TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, &img_width);
	lib.TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, &img_height);

	Rmask = 0x000000FF;
	Gmask = 0x0000FF00;
	Bmask = 0x00FF0000;
	Amask = 0xFF000000;


	IMG_TRACE("TIFF : img_width = %d,img_height = %d\n",img_width,img_height);

	if ((img_width >  IMG_TIF_MAX_WIDTH ||
		img_height > IMG_TIF_MAX_HEIGHT) ||
		(img_width < IMG_TIF_MIN_WIDTH &&
		img_height < IMG_TIF_MIN_HEIGHT))
	{
		IMG_TRACE("TIFF : not support size. supported size IMG_TIF_MIN_WIDTH = %d,IMG_TIF_MIN_HEIGHT = %d,IMG_TIF_MAX_WIDTH = %d,IMG_TIF_MAX_HEIGHT = %d\n",IMG_TIF_MIN_WIDTH,IMG_TIF_MIN_HEIGHT,IMG_TIF_MAX_WIDTH,IMG_TIF_MAX_HEIGHT);
		goto error;
	}

	if (type == SDL_IMAGE_LOAD_PWSURFACE)
	{
		video_flags = SDL_SWSURFACE|SDL_PWSWSURFACE;
	}
	else
	{
		video_flags = SDL_SWSURFACE;
	}

	surface = SDL_AllocSurface(video_flags, img_width, img_height, 32,
		Rmask, Gmask, Bmask, Amask);
	if(!surface)
		goto error;
	
	if(!lib.TIFFReadRGBAImage(tiff, img_width, img_height, surface->pixels, 0))
		goto error;

	/* libtiff loads the image upside-down, flip it back */
	half = img_height / 2;
	for(y = 0; y < half; y++)
	{
	        Uint32 *top = (Uint32 *)surface->pixels + y * surface->pitch/4;
	        Uint32 *bot = (Uint32 *)surface->pixels
		              + (img_height - y - 1) * surface->pitch/4;
		for(x = 0; x < img_width; x++)
		{
		        Uint32 tmp = top[x];
			top[x] = bot[x];
			bot[x] = tmp;
		}
	}
	lib.TIFFClose(tiff);
	IMG_QuitTIF();
	
	return surface;

error:
	if (tiff)
	{
		lib.TIFFClose(tiff);
	}
	SDL_RWseek(src, start, SEEK_SET);
	if ( surface ) {
		SDL_FreeSurface(surface);
	}
	IMG_QuitTIF();
	return NULL;
}

#endif

#else

/* See if an image is contained in a data source */
int IMG_isTIF(SDL_RWops *src)
{
	return(0);
}

/* Load a TIFF type image from an SDL datasource */
SDL_Surface *IMG_LoadTIF_RW(SDL_RWops *src,int type)
{
	return(NULL);
}

#endif /* LOAD_TIF */
