/*
    SDL_image:  An example image loading library for use with SDL
    Copyright (C) 1997-2009 Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Sam Lantinga
    slouken@libsdl.org
*/

/* This is a PNG image file loading framework */

#include <stdlib.h>
#include <stdio.h>

#include "SDL_image.h"
#include "PW_SDL_image.h"
#include "utilPoolMgr.h"

#ifdef LOAD_PNG

/*=============================================================================
        File: SDL_png.c
     Purpose: A PNG loader and saver for the SDL library      
    Revision: 
  Created by: Philippe Lavoie          (2 November 1998)
              lavoie@zeus.genie.uottawa.ca
 Modified by: 

 Copyright notice:
          Copyright (C) 1998 Philippe Lavoie
 
          This library is free software; you can redistribute it and/or
          modify it under the terms of the GNU Library General Public
          License as published by the Free Software Foundation; either
          version 2 of the License, or (at your option) any later version.
 
          This library is distributed in the hope that it will be useful,
          but WITHOUT ANY WARRANTY; without even the implied warranty of
          MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
          Library General Public License for more details.
 
          You should have received a copy of the GNU Library General Public
          License along with this library; if not, write to the Free
          Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Comments: The load and save routine are basically the ones you can find
             in the example.c file from the libpng distribution.

  Changes:
    5/17/99 - Modified to use the new SDL data sources - Sam Lantinga

=============================================================================*/

#include "SDL_endian.h"

#ifdef macintosh
#define MACOS
#endif
#include <png.h>


static struct {
	int loaded;
	void *handle;
	png_infop (*png_create_info_struct) (png_structp png_ptr);
	png_structp (*png_create_read_struct) (png_const_charp user_png_ver, png_voidp error_ptr, png_error_ptr error_fn, png_error_ptr warn_fn);
	void (*png_destroy_read_struct) (png_structpp png_ptr_ptr, png_infopp info_ptr_ptr, png_infopp end_info_ptr_ptr);
	png_uint_32 (*png_get_IHDR) (png_structp png_ptr, png_infop info_ptr, png_uint_32 *width, png_uint_32 *height, int *bit_depth, int *color_type, int *interlace_method, int *compression_method, int *filter_method);
	png_voidp (*png_get_io_ptr) (png_structp png_ptr);
	png_uint_32 (*png_get_tRNS) (png_structp png_ptr, png_infop info_ptr, png_bytep *trans, int *num_trans, png_color_16p *trans_values);
	png_uint_32 (*png_get_valid) (png_structp png_ptr, png_infop info_ptr, png_uint_32 flag);
	void (*png_read_image) (png_structp png_ptr, png_bytepp image);
	void (*png_read_info) (png_structp png_ptr, png_infop info_ptr);
	void (*png_read_update_info) (png_structp png_ptr, png_infop info_ptr);
	void (*png_set_expand) (png_structp png_ptr);
	void (*png_set_gray_to_rgb) (png_structp png_ptr);
	void (*png_set_packing) (png_structp png_ptr);
	void (*png_set_read_fn) (png_structp png_ptr, png_voidp io_ptr, png_rw_ptr read_data_fn);
	void (*png_set_strip_16) (png_structp png_ptr);
	int (*png_sig_cmp) (png_bytep sig, png_size_t start, png_size_t num_to_check);
} lib;

#ifdef LOAD_PNG_DYNAMIC
int IMG_InitPNG()
{
	if ( lib.loaded == 0 ) {
		lib.handle = SDL_LoadObject(LOAD_PNG_DYNAMIC);
		if ( lib.handle == NULL ) {
			return -1;
		}
		lib.png_create_info_struct =
			(png_infop (*) (png_structp))
			SDL_LoadFunction(lib.handle, "png_create_info_struct");
		if ( lib.png_create_info_struct == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_create_read_struct =
			(png_structp (*) (png_const_charp, png_voidp, png_error_ptr, png_error_ptr))
			SDL_LoadFunction(lib.handle, "png_create_read_struct");
		if ( lib.png_create_read_struct == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_destroy_read_struct =
			(void (*) (png_structpp, png_infopp, png_infopp))
			SDL_LoadFunction(lib.handle, "png_destroy_read_struct");
		if ( lib.png_destroy_read_struct == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_get_IHDR =
			(png_uint_32 (*) (png_structp, png_infop, png_uint_32 *, png_uint_32 *, int *, int *, int *, int *, int *))
			SDL_LoadFunction(lib.handle, "png_get_IHDR");
		if ( lib.png_get_IHDR == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_get_io_ptr =
			(png_voidp (*) (png_structp))
			SDL_LoadFunction(lib.handle, "png_get_io_ptr");
		if ( lib.png_get_io_ptr == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_get_tRNS =
			(png_uint_32 (*) (png_structp, png_infop, png_bytep *, int *, png_color_16p *))
			SDL_LoadFunction(lib.handle, "png_get_tRNS");
		if ( lib.png_get_tRNS == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_get_valid =
			(png_uint_32 (*) (png_structp, png_infop, png_uint_32))
			SDL_LoadFunction(lib.handle, "png_get_valid");
		if ( lib.png_get_valid == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_read_image =
			(void (*) (png_structp, png_bytepp))
			SDL_LoadFunction(lib.handle, "png_read_image");
		if ( lib.png_read_image == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_read_info =
			(void (*) (png_structp, png_infop))
			SDL_LoadFunction(lib.handle, "png_read_info");
		if ( lib.png_read_info == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_read_update_info =
			(void (*) (png_structp, png_infop))
			SDL_LoadFunction(lib.handle, "png_read_update_info");
		if ( lib.png_read_update_info == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_set_expand =
			(void (*) (png_structp))
			SDL_LoadFunction(lib.handle, "png_set_expand");
		if ( lib.png_set_expand == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_set_gray_to_rgb =
			(void (*) (png_structp))
			SDL_LoadFunction(lib.handle, "png_set_gray_to_rgb");
		if ( lib.png_set_gray_to_rgb == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_set_packing =
			(void (*) (png_structp))
			SDL_LoadFunction(lib.handle, "png_set_packing");
		if ( lib.png_set_packing == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_set_read_fn =
			(void (*) (png_structp, png_voidp, png_rw_ptr))
			SDL_LoadFunction(lib.handle, "png_set_read_fn");
		if ( lib.png_set_read_fn == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_set_strip_16 =
			(void (*) (png_structp))
			SDL_LoadFunction(lib.handle, "png_set_strip_16");
		if ( lib.png_set_strip_16 == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
		lib.png_sig_cmp =
			(int (*) (png_bytep, png_size_t, png_size_t))
			SDL_LoadFunction(lib.handle, "png_sig_cmp");
		if ( lib.png_sig_cmp == NULL ) {
			SDL_UnloadObject(lib.handle);
			return -1;
		}
	}
	++lib.loaded;

	return 0;
}
void IMG_QuitPNG()
{
	if ( lib.loaded == 0 ) {
		return;
	}
	if ( lib.loaded == 1 ) {
		SDL_UnloadObject(lib.handle);
	}
	--lib.loaded;
}
#else
int IMG_InitPNG()
{
	if ( lib.loaded == 0 ) {
		lib.png_create_info_struct = png_create_info_struct;
		lib.png_create_read_struct = png_create_read_struct;
		lib.png_destroy_read_struct = png_destroy_read_struct;
		lib.png_get_IHDR = png_get_IHDR;
		lib.png_get_io_ptr = png_get_io_ptr;
		lib.png_get_tRNS = png_get_tRNS;
		lib.png_get_valid = png_get_valid;
		lib.png_read_image = png_read_image;
		lib.png_read_info = png_read_info;
		lib.png_read_update_info = png_read_update_info;
		lib.png_set_expand = png_set_expand;
		lib.png_set_gray_to_rgb = png_set_gray_to_rgb;
		lib.png_set_packing = png_set_packing;
		lib.png_set_read_fn = png_set_read_fn;
		lib.png_set_strip_16 = png_set_strip_16;
		lib.png_sig_cmp = png_sig_cmp;
	}
	++lib.loaded;

	return 0;
}
void IMG_QuitPNG()
{
	if ( lib.loaded == 0 ) {
		return;
	}
	if ( lib.loaded == 1 ) {
	}
	--lib.loaded;
}
#endif /* LOAD_PNG_DYNAMIC */

/* See if an image is contained in a data source */
int IMG_isPNG(SDL_RWops *src)
{
	int start;
	int is_PNG;
	Uint8 magic[4];

	if ( !src )
		return 0;
	start = SDL_RWtell(src);
	is_PNG = 0;
	if ( SDL_RWread(src, magic, 1, sizeof(magic)) == sizeof(magic) ) {
                if ( magic[0] == 0x89 &&
                     magic[1] == 'P' &&
                     magic[2] == 'N' &&
                     magic[3] == 'G' ) {
			is_PNG = 1;
		}
	}
	SDL_RWseek(src, start, SEEK_SET);
	return(is_PNG);
}

/* Load a PNG type image from an SDL datasource */
static void png_read_data(png_structp ctx, png_bytep area, png_size_t size)
{
	SDL_RWops *src;

	src = (SDL_RWops *)lib.png_get_io_ptr(ctx);
	SDL_RWread(src, area, size, 1);
}

SDL_Surface *IMG_LoadPNG_RW(SDL_RWops *src,int type)
{
	return PW_IMG_LoadPNG_RW(src, NULL, NULL, NULL,type);
}

/* PW_SDL Spec. PNG doesn't support filter */
#if !defined(IMG_PARTIAL_DECODE)

SDL_Surface *PW_IMG_LoadPNG_RW(SDL_RWops *src, SDL_Surface *target, SDL_Rect *rect, PW_IMAGE_FILTERS *filters,int type)
{
	int start;
	const char *error;
	SDL_Surface *volatile surface;
	png_structp png_ptr;
	png_infop info_ptr;
	png_uint_32 width, height;
	int bit_depth, color_type, interlace_type;
	Uint32 Rmask;
	Uint32 Gmask;
	Uint32 Bmask;
	Uint32 Amask;
	SDL_Palette *palette;
	png_bytep *volatile row_pointers;
	int row, i;
	volatile int ckey = -1;
	png_color_16 *transv;
	/* PW_SDL Spec. */
	int x, y, clip_w, clip_h, offset_x;

	if ( filters )
	{
		IMG_SetError("PNG, doesn't support filter, yet!");
		return NULL;
	}

	if ( !src ) {
		/* The error message has been set in SDL_RWFromFile */
		return NULL;
	}
	start = SDL_RWtell(src);

	if ( IMG_InitPNG() < 0 ) {
		return NULL;
	}

	/* PW_SDL Spec. */
	if (rect == NULL) {
		x = 0;
		y = 0;
		clip_w = 0;
		clip_h = 0;
	} else {
		x = rect->x;
		y = rect->y;
		clip_w = rect->w;
		clip_h = rect->h;
		if (clip_w <= 0 || clip_h <= 0)
			return NULL;
	}
	
	/* Initialize the data we will clean up when we're done */
	error = NULL;
	png_ptr = NULL; info_ptr = NULL; row_pointers = NULL; surface = NULL;

	/* Create the PNG loading context structure */
	png_ptr = lib.png_create_read_struct(PNG_LIBPNG_VER_STRING,
					  NULL,NULL,NULL);
	if (png_ptr == NULL) {
		error = "Couldn't allocate memory for PNG file or incompatible PNG dll";
		goto done;
	}

	 /* Allocate/initialize the memory for image information.  REQUIRED. */
	info_ptr = lib.png_create_info_struct(png_ptr);
	if (info_ptr == NULL) {
		error = "Couldn't create image information for PNG file";
		goto done;
	}

	/* Set error handling if you are using setjmp/longjmp method (this is
	 * the normal method of doing things with libpng).  REQUIRED unless you
	 * set up your own error handlers in png_create_read_struct() earlier.
	 */
	if ( setjmp(png_ptr->jmpbuf) ) {
		error = "Error reading the PNG file.";
		goto done;
	}

	/* Set up the input control */
	lib.png_set_read_fn(png_ptr, src, png_read_data);

	/* Read PNG header info */
	lib.png_read_info(png_ptr, info_ptr);
	lib.png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
			&color_type, &interlace_type, NULL, NULL);

	/* PW_SDL Spec. */
	if ( target != NULL  && ( target->format->BitsPerPixel < 24 || 
		target->format->BitsPerPixel != bit_depth*info_ptr->channels) ) {
		error = "Target format is not matched";
		goto done;
	}
	if ( target != NULL && (target->w < x + (int)width || target->h < y + (int)height) ) {
		error = "Target destination is not matched";
		goto done;
	}

	/* tell libpng to strip 16 bit/color files down to 8 bits/color */
	lib.png_set_strip_16(png_ptr) ;

	/* Extract multiple pixels with bit depths of 1, 2, and 4 from a single
	 * byte into separate bytes (useful for paletted and grayscale images).
	 */
	lib.png_set_packing(png_ptr);

	/* scale greyscale values to the range 0..255 */
	if(color_type == PNG_COLOR_TYPE_GRAY)
		lib.png_set_expand(png_ptr);

	/* For images with a single "transparent colour", set colour key;
	   if more than one index has transparency, or if partially transparent
	   entries exist, use full alpha channel */
	if (lib.png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS)) {
	        int num_trans;
		Uint8 *trans;
		lib.png_get_tRNS(png_ptr, info_ptr, &trans, &num_trans,
			     &transv);
		if(color_type == PNG_COLOR_TYPE_PALETTE) {
		    /* Check if all tRNS entries are opaque except one */
		    int i, t = -1;
		    for(i = 0; i < num_trans; i++)
			if(trans[i] == 0) {
			    if(t >= 0)
				break;
			    t = i;
			} else if(trans[i] != 255)
			    break;
		    if(i == num_trans) {
			/* exactly one transparent index */
			ckey = t;
		    } else {
			/* more than one transparent index, or translucency */
			lib.png_set_expand(png_ptr);
		    }
		} else
		    ckey = 0; /* actual value will be set later */
	}

	if ( color_type == PNG_COLOR_TYPE_GRAY_ALPHA )
		lib.png_set_gray_to_rgb(png_ptr);

	lib.png_read_update_info(png_ptr, info_ptr);

	lib.png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
			&color_type, &interlace_type, NULL, NULL);

	/* Allocate the SDL surface to hold the image */
	Rmask = Gmask = Bmask = Amask = 0 ; 
	if ( color_type != PNG_COLOR_TYPE_PALETTE ) {
		if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) {
			Rmask = 0x000000FF;
			Gmask = 0x0000FF00;
			Bmask = 0x00FF0000;
			Amask = (info_ptr->channels == 4) ? 0xFF000000 : 0;
		} else {
		        int s = (info_ptr->channels == 4) ? 0 : 8;
			Rmask = 0xFF000000 >> s;
			Gmask = 0x00FF0000 >> s;
			Bmask = 0x0000FF00 >> s;
			Amask = 0x000000FF >> s;
		}
	}

	/* PW_SDL Spec. */
	if ( target == NULL ) {
		Uint32 video_flags;


		IMG_TRACE("PNG : width = %d, height = %d\n",width,height);

		if ((width > IMG_PNG_MAX_WIDTH || 
			height > IMG_PNG_MAX_HEIGHT) ||
			(width < IMG_PNG_MIN_WIDTH &&
			height < IMG_PNG_MIN_HEIGHT))
		{
            IMG_TRACE("PNG :  not support size. supported size IMG_PNG_MIN_WIDTH = %d,IMG_PNG_MIN_HEIGHT = %d,IMG_PNG_MAX_WIDTH = %d,IMG_PNG_MAX_HEIGHT = %d\n",IMG_PNG_MIN_WIDTH,IMG_PNG_MIN_HEIGHT,IMG_PNG_MAX_WIDTH,IMG_PNG_MAX_HEIGHT);			
			goto done;
		}

		if (type == SDL_IMAGE_LOAD_PWSURFACE)
		{
			video_flags = SDL_SWSURFACE|SDL_PWSWSURFACE;
		}
		else
		{
			video_flags = SDL_SWSURFACE;
		}
		surface = SDL_AllocSurface(video_flags, width, height,
				bit_depth*info_ptr->channels, Rmask,Gmask,Bmask,Amask);
	} else {
		surface = target;
	}
	if ( surface == NULL ) {
		error = "Out of memory";
		goto done;
	}

	if ( SDL_MUSTLOCK(surface) ) {
		if ( SDL_LockSurface(surface) < 0 ) {
			error = "Cannot Lock Surface";
			goto done;
		}
	}

	if(ckey != -1) {
	        if(color_type != PNG_COLOR_TYPE_PALETTE)
			/* FIXME: Should these be truncated or shifted down? */
		        ckey = SDL_MapRGB(surface->format,
			                 (Uint8)transv->red,
			                 (Uint8)transv->green,
			                 (Uint8)transv->blue);
	        SDL_SetColorKey(surface, SDL_SRCCOLORKEY, ckey);
	}

	/* Create the array of pointers to image data */
	row_pointers = (png_bytep*) malloc(sizeof(png_bytep)*height);
	if ( (row_pointers == NULL) ) {
		if ( SDL_MUSTLOCK(surface) ) {
			SDL_UnlockSurface(surface);
		}
		error = "Out of memory";
		goto done;
	}

	/* PW_SDL Spec. */
	offset_x = x*surface->format->BytesPerPixel;
	for (row = 0; row < (int)height; row++) {
		row_pointers[row] = (png_bytep)
				(Uint8 *)surface->pixels + (row + y)*surface->pitch + offset_x;
	}

	/* Read the entire image in one go */
	lib.png_read_image(png_ptr, row_pointers);

	/* and we're done!  (png_read_end() can be omitted if no processing of
	 * post-IDAT text/time/etc. is desired)
	 * In some cases it can't read PNG's created by some popular programs (ACDSEE),
	 * we do not want to process comments, so we omit png_read_end

	lib.png_read_end(png_ptr, info_ptr);
	*/

	/* Load the palette, if any */
	palette = surface->format->palette;
	if ( palette ) {
	    if(color_type == PNG_COLOR_TYPE_GRAY) {
		palette->ncolors = 256;
		for(i = 0; i < 256; i++) {
/* #ifdef PW_FB_UYV444B */
#ifdef PW_SDL_PALETTE_DIRECT
			{
				int r, g, b, y, u, v;
				r = i;
				g = i;
				b = i;
				y = ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
				u = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
				v = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;
				palette->colors[i].r = (Uint8)u;
				palette->colors[i].g = (Uint8)y;
				palette->colors[i].b = (Uint8)v;
			}
#else
			palette->colors[i].r = i;
			palette->colors[i].g = i;
			palette->colors[i].b = i;
#endif
		}
	    } else if (info_ptr->num_palette > 0 ) {
		palette->ncolors = info_ptr->num_palette; 
		for( i=0; i<info_ptr->num_palette; ++i ) {
/* #ifdef PW_FB_UYV444B */
#ifdef PW_SDL_PALETTE_DIRECT
			{
				int r, g, b, y, u, v;
				r = info_ptr->palette[i].red;
				g = info_ptr->palette[i].green;
				b = info_ptr->palette[i].blue;
				y = ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
				u = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
				v = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;
				palette->colors[i].r = (Uint8)u;
				palette->colors[i].g = (Uint8)y;
				palette->colors[i].b = (Uint8)v;
			}
#else
		    palette->colors[i].b = info_ptr->palette[i].blue;
		    palette->colors[i].g = info_ptr->palette[i].green;
		    palette->colors[i].r = info_ptr->palette[i].red;
#endif
		}
	    }
	}

	if ( SDL_MUSTLOCK(surface) ) {
		SDL_UnlockSurface(surface);
	}

done:	/* Clean up and return */
	if ( png_ptr ) {
		lib.png_destroy_read_struct(&png_ptr,
		                        info_ptr ? &info_ptr : (png_infopp)0,
								(png_infopp)0);
	}
	if ( row_pointers ) {
		free(row_pointers);
	}
	if ( error ) {
		SDL_RWseek(src, start, SEEK_SET);
		if ( surface != NULL && surface != target ) {
			SDL_FreeSurface(surface);
			surface = NULL;
		}
		IMG_QuitPNG();
		IMG_SetError(error);
	} else {
		IMG_QuitPNG();
	}
	return(surface); 
}
#else
//Support filter ( requires at least one line buf mem)

typedef struct _PNG_RGB_24Pixel{	
	unsigned char nRed;	
	unsigned char nGreen;	
	unsigned char nBlue;
	} PW_RGB_24Pixel;
typedef struct _PNG_RGB_32Pixel{	
	unsigned char nRed;	
	unsigned char nGreen;	
	unsigned char nBlue;	
	unsigned char nAlpha;
} PW_RGB_32Pixel;




void PW_PNG_LIB_ConvertRGB2RGB(unsigned *src_buf, unsigned int src_width ,unsigned int src_pitch,
										unsigned int src_bit_depth, unsigned int src_color_channels, PwImageColorFormat src_color_format,
										unsigned *dst_buf, unsigned int dst_width, unsigned int dst_pitch,
										unsigned int dst_bit_depth, unsigned int dst_color_channels,PwImageColorFormat dst_color_format)

{
	int i;
	PW_RGB_24Pixel *src_pixel_buf = src_buf;
	PW_RGB_24Pixel *dst_pixel_buf = dst_buf;

	for(i=0; i < src_width ; i ++)
	{
		dst_pixel_buf->nBlue = src_pixel_buf->nBlue;
		dst_pixel_buf->nGreen = src_pixel_buf->nGreen;		
		dst_pixel_buf->nRed = src_pixel_buf->nRed;		
			
		src_pixel_buf++;
		dst_pixel_buf++;
	}


}



void PW_PNG_LIB_ConvertRGBA2RGB(unsigned *src_buf, unsigned int src_width ,unsigned int src_pitch,
										unsigned int src_bit_depth, unsigned int src_color_channels, PwImageColorFormat src_color_format,
										unsigned *dst_buf, unsigned int dst_width, unsigned int dst_pitch,
										unsigned int dst_bit_depth, unsigned int dst_color_channels,PwImageColorFormat dst_color_format)

{
	int i;
	PW_RGB_32Pixel *src_pixel_buf = src_buf;
	PW_RGB_24Pixel *dst_pixel_buf = dst_buf;

	for(i=0; i < src_width ; i ++)
	{
		dst_pixel_buf->nBlue = src_pixel_buf->nBlue;
		dst_pixel_buf->nGreen = src_pixel_buf->nGreen;		
		dst_pixel_buf->nRed = src_pixel_buf->nRed;		
			
		src_pixel_buf++;
		dst_pixel_buf++;
	}


}



void PW_PNG_LIB_ConvertGRAYSCALE2RGB(unsigned *src_buf, unsigned int src_width ,unsigned int src_pitch,
										unsigned int src_bit_depth, unsigned int src_color_channels, PwImageColorFormat src_color_format,
										unsigned *dst_buf, unsigned int dst_width, unsigned int dst_pitch,
										unsigned int dst_bit_depth, unsigned int dst_color_channels,PwImageColorFormat dst_color_format)

{
	int i;
	unsigned char *src_pixel_buf = src_buf;
	PW_RGB_24Pixel *dst_pixel_buf = dst_buf;

	for(i=0; i < src_width ; i ++)
	{
		dst_pixel_buf->nBlue = *src_pixel_buf;
		dst_pixel_buf->nGreen = *src_pixel_buf;		
		dst_pixel_buf->nRed = *src_pixel_buf;		
			
		src_pixel_buf++;
		dst_pixel_buf++;
	}


}




void PW_PNG_LIB_ColorCovertion(unsigned *src_buf, unsigned int src_width ,unsigned int src_pitch,
									unsigned int src_bit_depth, unsigned int src_color_channels, PwImageColorFormat src_color_format,
									unsigned *dst_buf, unsigned int dst_width, unsigned int dst_pitch,
									unsigned int dst_bit_depth, unsigned int dst_color_channels,PwImageColorFormat dst_color_format)
{

	switch(dst_color_format)
	{
		case picfRGB:
		{
			switch(src_color_format)
			{
				case picfGRAYSCALE:
				PW_PNG_LIB_ConvertGRAYSCALE2RGB(src_buf,src_width,src_pitch,src_bit_depth,src_color_channels,src_color_format,
					dst_buf, dst_width, dst_pitch,dst_bit_depth, dst_color_channels, dst_color_format);
				break;
				case picfPALETTE:
				PW_PNG_LIB_ConvertRGB2RGB(src_buf,src_width,src_pitch,src_bit_depth,src_color_channels,src_color_format,
					dst_buf, dst_width, dst_pitch,dst_bit_depth, dst_color_channels, dst_color_format);
				break;
			
				case picfRGB:
				PW_PNG_LIB_ConvertRGB2RGB(src_buf,src_width,src_pitch,src_bit_depth,src_color_channels,src_color_format,
					dst_buf, dst_width, dst_pitch,dst_bit_depth, dst_color_channels, dst_color_format);
				break;
				case picfRGBA:
				PW_PNG_LIB_ConvertRGBA2RGB(src_buf,src_width,src_pitch,src_bit_depth,src_color_channels,src_color_format,
					dst_buf, dst_width, dst_pitch,dst_bit_depth, dst_color_channels, dst_color_format);
				break;
			}
			break;
		}
		break;

	}
}
                               



SDL_Surface *PW_IMG_LoadPNG_RW(SDL_RWops *src, SDL_Surface *target, SDL_Rect *rect, PW_IMAGE_FILTERS *filters,int type)
{
	int start;
	const char *error;
	SDL_Surface *volatile surface;
	png_structp png_ptr;
	png_infop info_ptr;
	png_uint_32 width, height;
	int bit_depth, color_type, interlace_type;
	Uint32 Rmask;
	Uint32 Gmask;
	Uint32 Bmask;
	Uint32 Amask;
	SDL_Palette *palette;
	png_bytep *volatile row_pointers;
	int row, i;
	volatile int ckey = -1;
	png_color_16 *transv;
	/* PW_SDL Spec. */
	int x, y, clip_w, clip_h, offset_x;

	/*
	if ( filters )
	{
		IMG_SetError("PNG, doesn't support filter, yet!");
		return NULL;
	}
	*/

	if ( !src ) {
		/* The error message has been set in SDL_RWFromFile */
		return NULL;
	}
	start = SDL_RWtell(src);

	if ( IMG_InitPNG() < 0 ) {
		return NULL;
	}

	/* PW_SDL Spec. */
	if (rect == NULL) {
		x = 0;
		y = 0;
		clip_w = 0;
		clip_h = 0;
	} else {
		x = rect->x;
		y = rect->y;
		clip_w = rect->w;
		clip_h = rect->h;
		if (clip_w <= 0 || clip_h <= 0)
			return NULL;
	}
	
	/* Initialize the data we will clean up when we're done */
	error = NULL;
	png_ptr = NULL; info_ptr = NULL; row_pointers = NULL; surface = NULL;

	/* Create the PNG loading context structure */
	png_ptr = lib.png_create_read_struct(PNG_LIBPNG_VER_STRING,
					  NULL,NULL,NULL);
	if (png_ptr == NULL) {
		error = "Couldn't allocate memory for PNG file or incompatible PNG dll";
		goto done;
	}

	 /* Allocate/initialize the memory for image information.  REQUIRED. */
	info_ptr = lib.png_create_info_struct(png_ptr);
	if (info_ptr == NULL) {
		error = "Couldn't create image information for PNG file";
		goto done;
	}

	/* Set error handling if you are using setjmp/longjmp method (this is
	 * the normal method of doing things with libpng).  REQUIRED unless you
	 * set up your own error handlers in png_create_read_struct() earlier.
	 */
	if ( setjmp(png_ptr->jmpbuf) ) {
		error = "Error reading the PNG file.";
		goto done;
	}

	/* Set up the input control */
	lib.png_set_read_fn(png_ptr, src, png_read_data);

	/* Read PNG header info */
	lib.png_read_info(png_ptr, info_ptr);
	lib.png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
			&color_type, &interlace_type, NULL, NULL);

	/* PW_SDL Spec. */
	/*
	if ( target != NULL  && ( target->format->BitsPerPixel < 24 || 
		target->format->BitsPerPixel != bit_depth*info_ptr->channels) ) {
		error = "Target format is not matched";
		goto done;
	}
	if ( target != NULL && (target->w < x + (int)width || target->h < y + (int)height) ) {
		error = "Target destination is not matched";
		goto done;
	}
	*/
	

	/* tell libpng to strip 16 bit/color files down to 8 bits/color */
	lib.png_set_strip_16(png_ptr) ;

	/* Extract multiple pixels with bit depths of 1, 2, and 4 from a single
	 * byte into separate bytes (useful for paletted and grayscale images).
	 */
	lib.png_set_packing(png_ptr);

	/* scale greyscale values to the range 0..255 */
	if(color_type == PNG_COLOR_TYPE_GRAY)
		lib.png_set_expand(png_ptr);

	/* For images with a single "transparent colour", set colour key;
	   if more than one index has transparency, or if partially transparent
	   entries exist, use full alpha channel */
	if (lib.png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS)) {
	        int num_trans;
		Uint8 *trans;
		lib.png_get_tRNS(png_ptr, info_ptr, &trans, &num_trans,
			     &transv);
		if(color_type == PNG_COLOR_TYPE_PALETTE) {
		    /* Check if all tRNS entries are opaque except one */
		    int i, t = -1;
		    for(i = 0; i < num_trans; i++)
			if(trans[i] == 0) {
			    if(t >= 0)
				break;
			    t = i;
			} else if(trans[i] != 255)
			    break;
		    if(i == num_trans) {
			/* exactly one transparent index */
			ckey = t;
		    } else {
			/* more than one transparent index, or translucency */
			lib.png_set_expand(png_ptr);
		    }
		} else
		    ckey = 0; /* actual value will be set later */
	}

	if ( color_type == PNG_COLOR_TYPE_GRAY_ALPHA )
		lib.png_set_gray_to_rgb(png_ptr);

	lib.png_read_update_info(png_ptr, info_ptr);

	lib.png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
			&color_type, &interlace_type, NULL, NULL);

	/* Allocate the SDL surface to hold the image */
	Rmask = Gmask = Bmask = Amask = 0 ; 
	if ( color_type != PNG_COLOR_TYPE_PALETTE ) {
		if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) {
			Rmask = 0x000000FF;
			Gmask = 0x0000FF00;
			Bmask = 0x00FF0000;
			Amask = (info_ptr->channels == 4) ? 0xFF000000 : 0;
		} else {
		        int s = (info_ptr->channels == 4) ? 0 : 8;
			Rmask = 0xFF000000 >> s;
			Gmask = 0x00FF0000 >> s;
			Bmask = 0x0000FF00 >> s;
			Amask = 0x000000FF >> s;
		}
	}

	/* PW_SDL Spec. */
	if ( target == NULL ) {
		Uint32 video_flags;


		IMG_TRACE("PNG : width = %d, height = %d\n",width,height);

		if ((width > IMG_PNG_MAX_WIDTH || 
			height > IMG_PNG_MAX_HEIGHT) ||
			(width < IMG_PNG_MIN_WIDTH &&
			height < IMG_PNG_MIN_HEIGHT))
		{
            IMG_TRACE("PNG :  not support size. supported size IMG_PNG_MIN_WIDTH = %d,IMG_PNG_MIN_HEIGHT = %d,IMG_PNG_MAX_WIDTH = %d,IMG_PNG_MAX_HEIGHT = %d\n",IMG_PNG_MIN_WIDTH,IMG_PNG_MIN_HEIGHT,IMG_PNG_MAX_WIDTH,IMG_PNG_MAX_HEIGHT);			
			goto done;
		}

		if (type == SDL_IMAGE_LOAD_PWSURFACE)
		{
			video_flags = SDL_SWSURFACE|SDL_PWSWSURFACE;
		}
		else
		{
			video_flags = SDL_SWSURFACE;
		}
		surface = SDL_AllocSurface(video_flags, width, height,
				bit_depth*info_ptr->channels, Rmask,Gmask,Bmask,Amask);
		
	} else {
		surface = target;
	}
	if ( surface == NULL ) {
		error = "Out of memory";
		goto done;
	}

	if ( SDL_MUSTLOCK(surface) ) {
		if ( SDL_LockSurface(surface) < 0 ) {
			error = "Cannot Lock Surface";
			goto done;
		}
	}

	if(ckey != -1) {
	        if(color_type != PNG_COLOR_TYPE_PALETTE)
			/* FIXME: Should these be truncated or shifted down? */
		        ckey = SDL_MapRGB(surface->format,
			                 (Uint8)transv->red,
			                 (Uint8)transv->green,
			                 (Uint8)transv->blue);
	        SDL_SetColorKey(surface, SDL_SRCCOLORKEY, ckey);
	}

	/* Create the array of pointers to image data */
	row_pointers = (png_bytep*) malloc(sizeof(png_bytep)*height);
	if ( (row_pointers == NULL) ) {
		if ( SDL_MUSTLOCK(surface) ) {
			SDL_UnlockSurface(surface);
		}
		error = "Out of memory";
		goto done;
	}

	/* PW_SDL Spec. */

	if(target == NULL)
	{
		offset_x = x*surface->format->BytesPerPixel;
		for (row = 0; row < (int)height; row++) {
			row_pointers[row] = (png_bytep)
					(Uint8 *)surface->pixels + (row + y)*surface->pitch + offset_x;
		}
		/* Read the entire image in one go */
		lib.png_read_image(png_ptr, row_pointers);
	}
	else
	{

		int 	line_pitch = surface->pitch;	
		int		line_buf_size = png_get_rowbytes(png_ptr,info_ptr);
		Uint8 * src_line_buf = (Uint8 *) png_malloc(png_ptr, line_buf_size * 4);
		Uint8 * dst_line_buf = (Uint8 *) png_malloc(png_ptr, line_buf_size * 4);  //max pixel size is 4 bytes

		int number_passes;
		int pass;
		/* Turn on interlace handling.	REQUIRED if you are not using
		 * png_read_image().  To see how to handle interlacing passes,
		 * see the png_read_row() method below:
		 */
		number_passes = png_set_interlace_handling(png_ptr); //lnterlace requires whole image buffer
		
		if(!src_line_buf || !dst_line_buf || number_passes > 1)
		{
			if(src_line_buf)
				png_free(png_ptr,src_line_buf);				
			if(dst_line_buf)
				png_free(png_ptr,dst_line_buf);

			if ( SDL_MUSTLOCK(surface) ) {
				SDL_UnlockSurface(surface);
			}			
			surface = NULL;
			error = "Line buffer out of memory";		
			goto done;
		}
		for (row = 0; row < (int)height; row++) 
		{			
			row_pointers[row] = (png_bytep) src_line_buf;		
		}

		png_set_strip_alpha(png_ptr);
		png_set_packing(png_ptr);
		png_set_packswap(png_ptr);


		/* Expand grayscale images to the full 8 bits from 1, 2, or 4 bits/pixel */
		if (color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8)
		  png_set_expand_gray_1_2_4_to_8(png_ptr);

		/* Expand paletted colors into true RGB triplets */
		if (color_type == PNG_COLOR_TYPE_PALETTE)
		   png_set_palette_to_rgb(png_ptr);

		/* Expand paletted or RGB images with transparency to full alpha channels
		* so the data will be available as RGBA quartets.
		*/
		if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
			png_set_tRNS_to_alpha(png_ptr);		

		if (color_type & PNG_COLOR_MASK_COLOR)
		{
			int num_palette;
			png_colorp palette;

			/* This reduces the image to the application supplied palette */
			if (0)
			{
				/* An array of colors to which the image should be dithered */
				png_color std_color_cube[256];
				png_set_dither(png_ptr, std_color_cube, 256,
				256, png_uint_16p_NULL, 0);
			}
			/* This reduces the image to the palette supplied in the file */
			else if (png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette))
			{
				png_uint_16p histogram = NULL;
				png_get_hIST(png_ptr, info_ptr, &histogram);
				png_set_dither(png_ptr, palette, num_palette,
			            256, histogram, 0);
			}
		}	

		png_set_filter(png_ptr, 0,PNG_FILTER_NONE );
		
		if (clip_w == 0) clip_w = surface->w;		
		if (clip_h == 0) clip_h = surface->h;
		PW_IMAGE_FILTER_CONTEXT filter_context;		
		filter_context.filters = filters;		
		filter_context.image = surface;		
		filter_context.clip.x = x;		
		filter_context.clip.y = y;		
		filter_context.clip.w = clip_w;		
		filter_context.clip.h = clip_h;		
		PW_FILTER_TOOLS_init(&filter_context);		
		PW_FILTER_TOOLS_config(&filter_context, (int)width, (int)height, picfRGB);

		printf("line_buf_size = %d \n",line_buf_size);
		printf("number_passes = %d \n",number_passes);
		printf("src_width = %d src_height %d \n",width,height);
		printf("dst_width = %d dst_height %d \n",surface->w,surface->h);		
		printf("bit_depth = %d \n",bit_depth);
		printf("info_ptr->channels = %d \n",info_ptr->channels);
		printf("color_type = %d \n",color_type);
		printf("Clip Rect = (%d, %d, %d, %d) \n",filter_context.clip.x,filter_context.clip.y,filter_context.clip.w,filter_context.clip.h);
		printf("PNG_COLOR_TYPE_GRAY = %d \n",PNG_COLOR_TYPE_GRAY);
		printf("PNG_COLOR_TYPE_PALETTE = %d \n",PNG_COLOR_TYPE_PALETTE);
		printf("PNG_COLOR_TYPE_RGB = %d \n",PNG_COLOR_TYPE_RGB);
		printf("PNG_COLOR_TYPE_RGB_ALPHA = %d \n",PNG_COLOR_TYPE_RGB_ALPHA);
		printf("PNG_COLOR_TYPE_GRAY_ALPHA = %d \n",PNG_COLOR_TYPE_GRAY_ALPHA);
		printf("PNG_COLOR_TYPE_RGBA = %d \n",PNG_COLOR_TYPE_RGBA);			
		printf("PNG_COLOR_TYPE_GA = %d \n",PNG_COLOR_TYPE_GA);
		
		for (pass = 0; pass < number_passes; pass++)	
		{
			for ( row = 0 ; row < height; row++) 			
			{				
				png_read_rows(png_ptr, &row_pointers[row], png_bytepp_NULL, 1);

				if(color_type == PNG_COLOR_TYPE_RGB_ALPHA && bit_depth ==8)
				{
					//do nothing 
					if (PW_FILTER_TOOLS_sendPixels(&filter_context, 					
						(int)row, (Uint8 *)row_pointers[row], line_pitch, picfRGB) != 0)				
							break;		
					
				}
				else if(color_type == PNG_COLOR_TYPE_RGB && bit_depth ==8)
				{
					//do nothing 
					if (PW_FILTER_TOOLS_sendPixels(&filter_context, 					
						(int)row, (Uint8 *)row_pointers[row], line_pitch, picfRGB) != 0)				
							break;							

				}
				else if(color_type == PNG_COLOR_TYPE_PALETTE && bit_depth ==8)
				{
					//do nothing 
					if (PW_FILTER_TOOLS_sendPixels(&filter_context, 					
						(int)row, (Uint8 *)row_pointers[row], line_pitch, picfRGB) != 0)				
							break;							
				}
				else if(color_type == PNG_COLOR_TYPE_GRAY && bit_depth ==8 && info_ptr->channels == 1)
				{
					PW_PNG_LIB_ColorCovertion(src_line_buf, width,line_buf_size,
						bit_depth,info_ptr->channels, picfGRAYSCALE,
						dst_line_buf,width, line_buf_size,
						8,3,picfRGB);
					
					if (PW_FILTER_TOOLS_sendPixels(&filter_context, 					
						(int)row, (Uint8 *)dst_line_buf, line_pitch, picfRGB) != 0)				
							break;							
					
				}				
				else
				{
					error = "Invalid Color Format";	
					PW_FILTER_TOOLS_finalize(&filter_context);	
					png_free(png_ptr,src_line_buf);
					png_free(png_ptr,dst_line_buf);
					if ( SDL_MUSTLOCK(surface) ) {
						SDL_UnlockSurface(surface);
					}			
					surface = NULL;				
					goto done;
				}

			}		
		}
		PW_FILTER_TOOLS_finalize(&filter_context);	
		png_free(png_ptr,src_line_buf);
		png_free(png_ptr,dst_line_buf);		
	}
	

	/* and we're done!  (png_read_end() can be omitted if no processing of
	 * post-IDAT text/time/etc. is desired)
	 * In some cases it can't read PNG's created by some popular programs (ACDSEE),
	 * we do not want to process comments, so we omit png_read_end

	lib.png_read_end(png_ptr, info_ptr);
	*/

	
	/* Load the palette, if any */
	palette = surface->format->palette;
	if ( palette ) {
	    if(color_type == PNG_COLOR_TYPE_GRAY) {
		palette->ncolors = 256;
		for(i = 0; i < 256; i++) {
/* #ifdef PW_FB_UYV444B */
#ifdef PW_SDL_PALETTE_DIRECT
			{
				int r, g, b, y, u, v;
				r = i;
				g = i;
				b = i;
				y = ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
				u = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
				v = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;
				palette->colors[i].r = (Uint8)u;
				palette->colors[i].g = (Uint8)y;
				palette->colors[i].b = (Uint8)v;
			}
#else
			palette->colors[i].r = i;
			palette->colors[i].g = i;
			palette->colors[i].b = i;
#endif
		}
	    } else if (info_ptr->num_palette > 0 ) {
		palette->ncolors = info_ptr->num_palette; 
		for( i=0; i<info_ptr->num_palette; ++i ) {
/* #ifdef PW_FB_UYV444B */
#ifdef PW_SDL_PALETTE_DIRECT
			{
				int r, g, b, y, u, v;
				r = info_ptr->palette[i].red;
				g = info_ptr->palette[i].green;
				b = info_ptr->palette[i].blue;
				y = ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
				u = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
				v = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;
				palette->colors[i].r = (Uint8)u;
				palette->colors[i].g = (Uint8)y;
				palette->colors[i].b = (Uint8)v;
			}
#else
		    palette->colors[i].b = info_ptr->palette[i].blue;
		    palette->colors[i].g = info_ptr->palette[i].green;
		    palette->colors[i].r = info_ptr->palette[i].red;
#endif
		}
	    }
	}

	if ( SDL_MUSTLOCK(surface) ) {
		SDL_UnlockSurface(surface);
	}

done:	/* Clean up and return */
	if ( png_ptr ) {
		lib.png_destroy_read_struct(&png_ptr,
		                        info_ptr ? &info_ptr : (png_infopp)0,
								(png_infopp)0);
	}
	if ( row_pointers ) {
		free(row_pointers);
	}
	if ( error ) {
		printf("PNG ERR = %s \n",error);	
		SDL_RWseek(src, start, SEEK_SET);
		if ( surface != NULL && surface != target ) {
			SDL_FreeSurface(surface);
			surface = NULL;
		}
		IMG_QuitPNG();
		IMG_SetError(error);
	} else {
		IMG_QuitPNG();
	}
	return(surface); 
}



#endif




#else

/* See if an image is contained in a data source */
int IMG_isPNG(SDL_RWops *src)
{
	return(0);
}

/* Load a PNG type image from an SDL datasource */
SDL_Surface *IMG_LoadPNG_RW(SDL_RWops *src,int type)
{
	return(NULL);
}

SDL_Surface *PW_IMG_LoadPNG_RW(SDL_RWops *src, SDL_Surface *target, SDL_Rect *rect, PW_IMAGE_FILTERS *filters,int type)
{
	return(NULL);
}

#endif /* LOAD_PNG */
