/*
    SDL_image:  An example image loading library for use with SDL
    Copyright (C) 1997-2009 Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Sam Lantinga
    slouken@libsdl.org
*/

/* This is a BMP image file loading framework */

#include <stdio.h>
#include <string.h>

#include "SDL_image.h"
#include "PW_SDL_image.h"
#include "utilPoolMgr.h"

#ifdef LOAD_BMP

/* See if an image is contained in a data source */
int IMG_isBMP(SDL_RWops *src)
{
	int start;
	int is_BMP;
	char magic[2];

	if ( !src )
		return 0;
	start = SDL_RWtell(src);
	is_BMP = 0;
	if ( SDL_RWread(src, magic, sizeof(magic), 1) ) {
		if ( strncmp(magic, "BM", 2) == 0 ) {
			is_BMP = 1;
		}
	}
	SDL_RWseek(src, start, SEEK_SET);
	return(is_BMP);
}

#include "SDL_error.h"
#include "SDL_video.h"
#include "SDL_endian.h"

/* Compression encodings for BMP files */
#ifndef BI_RGB
#define BI_RGB		0
#define BI_RLE8		1
#define BI_RLE4		2
#define BI_BITFIELDS	3
#endif

static int readRlePixels(SDL_Surface * surface, SDL_RWops * src, int isRle8)
{
	/*
	| Sets the surface pixels from src.  A bmp image is upside down.
	*/
	int pitch = surface->pitch;
	int height = surface->h;
	Uint8 *start = (Uint8 *)surface->pixels;
	Uint8 *end = start + (height*pitch);
	Uint8 *bits = end-pitch, *spot;
	int ofs = 0;
	Uint8 ch;
	Uint8 needsPad;

#define COPY_PIXEL(x)	spot = &bits[ofs++]; if(spot >= start && spot < end) *spot = (x)

	for (;;) {
		if ( !SDL_RWread(src, &ch, 1, 1) ) return 1;
		/*
		| encoded mode starts with a run length, and then a byte
		| with two colour indexes to alternate between for the run
		*/
		if ( ch ) {
			Uint8 pixel;
			if ( !SDL_RWread(src, &pixel, 1, 1) ) return 1;
			if ( isRle8 ) {                 /* 256-color bitmap, compressed */
				do {
					COPY_PIXEL(pixel);
				} while (--ch);
			} else {                         /* 16-color bitmap, compressed */
				Uint8 pixel0 = pixel >> 4;
				Uint8 pixel1 = pixel & 0x0F;
				for (;;) {
					COPY_PIXEL(pixel0);	/* even count, high nibble */
					if (!--ch) break;
					COPY_PIXEL(pixel1);	/* odd count, low nibble */
					if (!--ch) break;
				}
			}
		} else {
			/*
			| A leading zero is an escape; it may signal the end of the bitmap,
			| a cursor move, or some absolute data.
			| zero tag may be absolute mode or an escape
			*/
			if ( !SDL_RWread(src, &ch, 1, 1) ) return 1;
			switch (ch) {
			case 0:                         /* end of line */
				ofs = 0;
				bits -= pitch;               /* go to previous */
				break;
			case 1:                         /* end of bitmap */
				return 0;                    /* success! */
			case 2:                         /* delta */
				if ( !SDL_RWread(src, &ch, 1, 1) ) return 1;
				ofs += ch;
				if ( !SDL_RWread(src, &ch, 1, 1) ) return 1;
				bits -= (ch * pitch);
				break;
			default:                        /* no compression */
				if (isRle8) {
					needsPad = ( ch & 1 );
					do {
						Uint8 pixel;
						if ( !SDL_RWread(src, &pixel, 1, 1) ) return 1;
						COPY_PIXEL(pixel);
					} while (--ch);
				} else {
					needsPad = ( ((ch+1)>>1) & 1 ); /* (ch+1)>>1: bytes size */
					for (;;) {
						Uint8 pixel;
						if ( !SDL_RWread(src, &pixel, 1, 1) ) return 1;
						COPY_PIXEL(pixel >> 4);
						if (!--ch) break;
						COPY_PIXEL(pixel & 0x0F);
						if (!--ch) break;
					}
				}
				/* pad at even boundary */
				if ( needsPad && !SDL_RWread(src, &ch, 1, 1) ) return 1;
				break;
			}
		}
	}
}

#if !defined(IMG_PARTIAL_DECODE)
static SDL_Surface *LoadBMP_RW (SDL_RWops *src, int freesrc,int type)
{
	int was_error;
	long fp_offset;
	int bmpPitch;
	int i, pad;
	SDL_Surface *surface;
	Uint32 Rmask;
	Uint32 Gmask;
	Uint32 Bmask;
	Uint32 Amask;
	SDL_Palette *palette;
	Uint8 *bits;
	int ExpandBMP;
	Uint32 video_flags;

	/* The Win32 BMP file header (14 bytes) */
	char   magic[2];
	Uint32 bfSize;
	Uint16 bfReserved1;
	Uint16 bfReserved2;
	Uint32 bfOffBits;

	/* The Win32 BITMAPINFOHEADER struct (40 bytes) */
	Uint32 biSize;
	Sint32 biWidth;
	Sint32 biHeight;
	Uint16 biPlanes;
	Uint16 biBitCount;
	Uint32 biCompression;
	Uint32 biSizeImage;
	Sint32 biXPelsPerMeter;
	Sint32 biYPelsPerMeter;
	Uint32 biClrUsed;
	Uint32 biClrImportant;


	/* Make sure we are passed a valid data source */
	surface = NULL;
	was_error = 0;
	if ( src == NULL ) {
		was_error = 1;
		goto done;
	}

	/* Read in the BMP file header */
	fp_offset = SDL_RWtell(src);
	SDL_ClearError();
	if ( SDL_RWread(src, magic, 1, 2) != 2 ) {
		SDL_Error(SDL_EFREAD);
		was_error = 1;
		goto done;
	}
	if ( strncmp(magic, "BM", 2) != 0 ) {
		SDL_SetError("File is not a Windows BMP file");
		was_error = 1;
		goto done;
	}
	bfSize		= SDL_ReadLE32(src);
	bfReserved1	= SDL_ReadLE16(src);
	bfReserved2	= SDL_ReadLE16(src);
	bfOffBits	= SDL_ReadLE32(src);

	/* Read the Win32 BITMAPINFOHEADER */
	biSize		= SDL_ReadLE32(src);
	if ( biSize == 12 ) {
		biWidth		= (Uint32)SDL_ReadLE16(src);
		biHeight	= (Uint32)SDL_ReadLE16(src);
		biPlanes	= SDL_ReadLE16(src);
		biBitCount	= SDL_ReadLE16(src);
		biCompression	= BI_RGB;
		biSizeImage	= 0;
		biXPelsPerMeter	= 0;
		biYPelsPerMeter	= 0;
		biClrUsed	= 0;
		biClrImportant	= 0;
	} else {
		biWidth		= SDL_ReadLE32(src);
		biHeight	= SDL_ReadLE32(src);
		biPlanes	= SDL_ReadLE16(src);
		biBitCount	= SDL_ReadLE16(src);
		biCompression	= SDL_ReadLE32(src);
		biSizeImage	= SDL_ReadLE32(src);
		biXPelsPerMeter	= SDL_ReadLE32(src);
		biYPelsPerMeter	= SDL_ReadLE32(src);
		biClrUsed	= SDL_ReadLE32(src);
		biClrImportant	= SDL_ReadLE32(src);
	}

	/* Check for read error */
	if ( strcmp(SDL_GetError(), "") != 0 ) {
		was_error = 1;
		goto done;
	}

	/* Expand 1 and 4 bit bitmaps to 8 bits per pixel */
	switch (biBitCount) {
		case 1:
		case 4:
			ExpandBMP = biBitCount;
			biBitCount = 8;
			break;
		default:
			ExpandBMP = 0;
			break;
	}

	/* RLE4 and RLE8 BMP compression is supported */
	Rmask = Gmask = Bmask = Amask = 0;
	switch (biCompression) {
		case BI_RGB:
			/* If there are no masks, use the defaults */
			if ( bfOffBits == (14+biSize) ) {
				/* Default values for the BMP format */
				switch (biBitCount) {
					case 15:
					case 16:
						Rmask = 0x7C00;
						Gmask = 0x03E0;
						Bmask = 0x001F;
						break;
					case 24:
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
					        Rmask = 0x000000FF;
					        Gmask = 0x0000FF00;
					        Bmask = 0x00FF0000;
#else
						Rmask = 0x00FF0000;
						Gmask = 0x0000FF00;
						Bmask = 0x000000FF;
#endif
						break;
					case 32:
						Amask = 0xFF000000;
						Rmask = 0x00FF0000;
						Gmask = 0x0000FF00;
						Bmask = 0x000000FF;
						break;
					default:
						break;
				}
				break;
			}
			/* Fall through -- read the RGB masks */

		default:
			switch (biBitCount) {
				case 15:
				case 16:
				case 32:
					Rmask = SDL_ReadLE32(src);
					Gmask = SDL_ReadLE32(src);
					Bmask = SDL_ReadLE32(src);
					Amask = SDL_ReadLE32(src);
					break;
				default:
					break;
			}
			break;
	}

	IMG_TRACE("BMP : biWidth = %d,biHeight = %d\n",biWidth,biHeight);

	if ((biWidth > IMG_BMP_MAX_WIDTH) ||
		(biHeight > IMG_BMP_MAX_HEIGHT) ||
		(biWidth < IMG_BMP_MIN_WIDTH) ||
		(biHeight < IMG_BMP_MIN_HEIGHT))
	{
		IMG_TRACE("BMP : not support size. supported size IMG_BMP_MIN_WIDTH = %d,IMG_BMP_MIN_WIDTH = %d,IMG_BMP_MAX_WIDTH = %d,IMG_BMP_MAX_HEIGHT = %d\n",IMG_BMP_MIN_WIDTH,IMG_BMP_MIN_WIDTH,IMG_BMP_MAX_WIDTH,IMG_BMP_MAX_HEIGHT);
		was_error = 1;
		goto done;
	}

	if (type == SDL_IMAGE_LOAD_PWSURFACE)
	{
		video_flags = SDL_SWSURFACE|SDL_PWSWSURFACE;
	}
	else
	{
		video_flags = SDL_SWSURFACE;
	}

	/* Create a compatible surface, note that the colors are RGB ordered */
	surface = SDL_CreateRGBSurface(video_flags,
			biWidth, biHeight, biBitCount, Rmask, Gmask, Bmask, Amask);
	if ( surface == NULL ) {
		was_error = 1;
		goto done;
	}

	/* Load the palette, if any */
	palette = (surface->format)->palette;
	if ( palette ) {
		if ( SDL_RWseek(src, fp_offset+14+biSize, SEEK_SET) < 0 ) {
			SDL_Error(SDL_EFSEEK);
			was_error = 1;
			goto done;
		}

		/*
		| guich: always use 1<<bpp b/c some bitmaps can bring wrong information
		| for colorsUsed
		*/
		/* if ( biClrUsed == 0 ) {  */
		biClrUsed = 1 << biBitCount;
		/* } */
		if ( biSize == 12 ) {
			for ( i = 0; i < (int)biClrUsed; ++i ) {
				SDL_RWread(src, &palette->colors[i].b, 1, 1);
				SDL_RWread(src, &palette->colors[i].g, 1, 1);
				SDL_RWread(src, &palette->colors[i].r, 1, 1);
				palette->colors[i].unused = 0;
			}	
		} else {
			for ( i = 0; i < (int)biClrUsed; ++i ) {
				SDL_RWread(src, &palette->colors[i].b, 1, 1);
				SDL_RWread(src, &palette->colors[i].g, 1, 1);
				SDL_RWread(src, &palette->colors[i].r, 1, 1);
				SDL_RWread(src, &palette->colors[i].unused, 1, 1);
			}	
		}
		palette->ncolors = biClrUsed;
	}

	/* Read the surface pixels.  Note that the bmp image is upside down */
	if ( SDL_RWseek(src, fp_offset+bfOffBits, SEEK_SET) < 0 ) {
		SDL_Error(SDL_EFSEEK);
		was_error = 1;
		goto done;
	}
	if ((biCompression == BI_RLE4) || (biCompression == BI_RLE8)) {
		was_error = readRlePixels(surface, src, biCompression == BI_RLE8);
		if (was_error) SDL_SetError("Error reading from BMP");
		goto done;
	}
	bits = (Uint8 *)surface->pixels+(surface->h*surface->pitch);
	switch (ExpandBMP) {
		case 1:
			bmpPitch = (biWidth + 7) >> 3;
			pad  = (((bmpPitch)%4) ? (4-((bmpPitch)%4)) : 0);
			break;
		case 4:
			bmpPitch = (biWidth + 1) >> 1;
			pad  = (((bmpPitch)%4) ? (4-((bmpPitch)%4)) : 0);
			break;
		default:
			pad  = ((surface->pitch%4) ?
					(4-(surface->pitch%4)) : 0);
			break;
	}
	while ( bits > (Uint8 *)surface->pixels ) {
		bits -= surface->pitch;
		switch (ExpandBMP) {
			case 1:
			case 4: {
			Uint8 pixel = 0;
			int   shift = (8-ExpandBMP);
			for ( i=0; i<surface->w; ++i ) {
				if ( i%(8/ExpandBMP) == 0 ) {
					if ( !SDL_RWread(src, &pixel, 1, 1) ) {
						SDL_SetError(
					"Error reading from BMP");
						was_error = 1;
						goto done;
					}
				}
				*(bits+i) = (pixel>>shift);
				pixel <<= ExpandBMP;
			} }
			break;

			default:
			if ( SDL_RWread(src, bits, 1, surface->pitch)
							 != surface->pitch ) {
				SDL_Error(SDL_EFREAD);
				was_error = 1;
				goto done;
			}
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
			/* Byte-swap the pixels if needed. Note that the 24bpp
			   case has already been taken care of above. */
			switch(biBitCount) {
				case 15:
				case 16: {
				        Uint16 *pix = (Uint16 *)bits;
					for(i = 0; i < surface->w; i++)
					        pix[i] = SDL_Swap16(pix[i]);
					break;
				}

				case 32: {
				        Uint32 *pix = (Uint32 *)bits;
					for(i = 0; i < surface->w; i++)
					        pix[i] = SDL_Swap32(pix[i]);
					break;
				}
			}
#endif
			break;
		}
		/* Skip padding bytes, ugh */
		if ( pad ) {
			Uint8 padbyte;
			for ( i=0; i<pad; ++i ) {
				SDL_RWread(src, &padbyte, 1, 1);
			}
		}
	}
done:
	if ( was_error ) {
		if ( src ) {
			SDL_RWseek(src, fp_offset, SEEK_SET);
		}
		if ( surface ) {
			SDL_FreeSurface(surface);
		}
		surface = NULL;
	}
	if ( freesrc && src ) {
		SDL_RWclose(src);
	}
	return(surface);
}

/* Load a BMP type image from an SDL datasource */
SDL_Surface *IMG_LoadBMP_RW(SDL_RWops *src,int type)
{
	return(LoadBMP_RW(src, 0, type));
}
#else

typedef struct _PNG_RGB_24Pixel{	
	unsigned char nRed;	
	unsigned char nGreen;	
	unsigned char nBlue;
} PW_RGB_24Pixel;


static Uint32 PW_BMP_SDL_Get32BytesPixel(Uint32 pixelAddr, int x)
{
	Uint32 pixel;
	Uint32 *pixelBuf = (Uint32 *)pixelAddr;
	pixelBuf = pixelBuf + x;
	pixel = *pixelBuf;
	return pixel;
}

static Uint32 PW_BMP_SDL_Get24BytesPixel(Uint32 pixelAddr, int x)
{
	Uint32 pixel;
	PW_RGB_24Pixel *pixelBuf = (PW_RGB_24Pixel *)pixelAddr;
	pixelBuf = pixelBuf + x;
	pixel = ((Uint32)pixelBuf->nBlue) << 16 | ((Uint32)pixelBuf->nGreen) << 8 | ((Uint32)pixelBuf->nRed);
	return pixel;
}

static Uint32 PW_BMP_SDL_Get16BytesPixel(Uint32 pixelAddr, int x)
{
	Uint32 pixel;
	Uint16 *pixelBuf = (Uint16 *)pixelAddr;
	pixelBuf = pixelBuf + x;
	pixel = *pixelBuf;
	return pixel;
}


static Uint32 PW_BMP_SDL_Get8BytesPixel(Uint32 pixelAddr, int x)
{
	Uint32 pixel;
	Uint8 *pixelBuf = (Uint8 *)pixelAddr;
	pixelBuf = pixelBuf + x;
	pixel = *pixelBuf;
	return pixel;
}



static void PW_BMP_Read_Line_Buf(SDL_Surface *line_buf_surface, unsigned char *rgb_buffer)
{
		int i;
		int Bpp;
		Uint8 Red, Green,Blue;
		Uint32 pixelAddr;
		PW_RGB_24Pixel *dst_pixel_buf;
		Bpp = line_buf_surface->format->BytesPerPixel;
		dst_pixel_buf = (PW_RGB_24Pixel*)rgb_buffer;
		pixelAddr = ((Uint32)(line_buf_surface->pixels));
		for(i =0; i < line_buf_surface->w ; i++)
		{
			Uint32 pixel;
			
			switch(Bpp)
			{
			case 4:
				pixel = PW_BMP_SDL_Get32BytesPixel(pixelAddr,i);
				break;
			case 3:
				pixel = PW_BMP_SDL_Get24BytesPixel(pixelAddr,i);
				break;
			case 2:
				pixel = PW_BMP_SDL_Get16BytesPixel(pixelAddr,i);
				break;
			case 1:
				pixel = PW_BMP_SDL_Get8BytesPixel(pixelAddr,i);
				break;
			default:
				printf("Invalid Format \n");
				return;
			}

			SDL_GetRGB(pixel, line_buf_surface->format, &Red, &Green, &Blue);
			dst_pixel_buf->nRed = Red;
			dst_pixel_buf->nGreen = Green;
			dst_pixel_buf->nBlue = Blue;
			dst_pixel_buf++;
			
	
		}
}

SDL_Surface *PW_IMG_LoadBMP_RW(SDL_RWops *src, SDL_Surface *target, SDL_Rect *rect, PW_IMAGE_FILTERS *filters, int type)
{
	int was_error;
	long fp_offset;
	int bmpPitch;
	int i, pad;
	SDL_Surface *surface;
	Uint32 Rmask;
	Uint32 Gmask;
	Uint32 Bmask;
	Uint32 Amask;
	SDL_Palette *palette;
	Uint8 *bits;
	int ExpandBMP;
	Uint32 video_flags;

	/* The Win32 BMP file header (14 bytes) */
	char   magic[2];
	Uint32 bfSize;
	Uint16 bfReserved1;
	Uint16 bfReserved2;
	Uint32 bfOffBits;

	/* The Win32 BITMAPINFOHEADER struct (40 bytes) */
	Uint32 biSize;
	Sint32 biWidth;
	Sint32 biHeight;
	Uint16 biPlanes;
	Uint16 biBitCount;
	Uint32 biCompression;
	Uint32 biSizeImage;
	Sint32 biXPelsPerMeter;
	Sint32 biYPelsPerMeter;
	Uint32 biClrUsed;
	Uint32 biClrImportant;

	int currentY;
	int x, y, clip_w, clip_h, offset_x;
	PW_IMAGE_FILTER_CONTEXT filter_context;
	unsigned char *dst_line_buf = NULL;
	SDL_Surface *line_buf_surface = NULL;	

	/* Make sure we are passed a valid data source */
	surface = NULL;
	was_error = 0;
	if ( src == NULL ) {
		was_error = 1;
		goto done;
	}

	/* Read in the BMP file header */
	fp_offset = SDL_RWtell(src);
	SDL_ClearError();
	if ( SDL_RWread(src, magic, 1, 2) != 2 ) {
		SDL_Error(SDL_EFREAD);
		was_error = 1;
		goto done;
	}
	if ( strncmp(magic, "BM", 2) != 0 ) {
		SDL_SetError("File is not a Windows BMP file");
		was_error = 1;
		goto done;
	}
	bfSize		= SDL_ReadLE32(src);
	bfReserved1	= SDL_ReadLE16(src);
	bfReserved2	= SDL_ReadLE16(src);
	bfOffBits	= SDL_ReadLE32(src);

	/* Read the Win32 BITMAPINFOHEADER */
	biSize		= SDL_ReadLE32(src);
	if ( biSize == 12 ) {
		biWidth		= (Uint32)SDL_ReadLE16(src);
		biHeight	= (Uint32)SDL_ReadLE16(src);
		biPlanes	= SDL_ReadLE16(src);
		biBitCount	= SDL_ReadLE16(src);
		biCompression	= BI_RGB;
		biSizeImage	= 0;
		biXPelsPerMeter	= 0;
		biYPelsPerMeter	= 0;
		biClrUsed	= 0;
		biClrImportant	= 0;
	} else {
		biWidth		= SDL_ReadLE32(src);
		biHeight	= SDL_ReadLE32(src);
		biPlanes	= SDL_ReadLE16(src);
		biBitCount	= SDL_ReadLE16(src);
		biCompression	= SDL_ReadLE32(src);
		biSizeImage	= SDL_ReadLE32(src);
		biXPelsPerMeter	= SDL_ReadLE32(src);
		biYPelsPerMeter	= SDL_ReadLE32(src);
		biClrUsed	= SDL_ReadLE32(src);
		biClrImportant	= SDL_ReadLE32(src);
	}

	/* Check for read error */
	if ( strcmp(SDL_GetError(), "") != 0 ) {
		was_error = 1;
		goto done;
	}

	/* Expand 1 and 4 bit bitmaps to 8 bits per pixel */
	switch (biBitCount) {
		case 1:
		case 4:
			ExpandBMP = biBitCount;
			biBitCount = 8;
			break;
		default:
			ExpandBMP = 0;
			break;
	}

	/* RLE4 and RLE8 BMP compression is supported */
	Rmask = Gmask = Bmask = Amask = 0;
	switch (biCompression) {
		case BI_RGB:
			/* If there are no masks, use the defaults */
			if ( bfOffBits == (14+biSize) ) {
				/* Default values for the BMP format */
				switch (biBitCount) {
					case 15:
					case 16:
						Rmask = 0x7C00;
						Gmask = 0x03E0;
						Bmask = 0x001F;
						break;
					case 24:
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
					        Rmask = 0x000000FF;
					        Gmask = 0x0000FF00;
					        Bmask = 0x00FF0000;
#else
						Rmask = 0x00FF0000;
						Gmask = 0x0000FF00;
						Bmask = 0x000000FF;
#endif
						break;
					case 32:
						Amask = 0xFF000000;
						Rmask = 0x00FF0000;
						Gmask = 0x0000FF00;
						Bmask = 0x000000FF;
						break;
					default:
						break;
				}
				break;
			}
			/* Fall through -- read the RGB masks */

		default:
			switch (biBitCount) {
				case 15:
				case 16:
				case 32:
					Rmask = SDL_ReadLE32(src);
					Gmask = SDL_ReadLE32(src);
					Bmask = SDL_ReadLE32(src);
					Amask = SDL_ReadLE32(src);
					break;
				default:
					break;
			}
			break;
	}

	IMG_TRACE("BMP : biWidth = %d,biHeight = %d\n",biWidth,biHeight);

	if (type == SDL_IMAGE_LOAD_PWSURFACE)
	{
		video_flags = SDL_SWSURFACE|SDL_PWSWSURFACE;
	}
	else
	{
		video_flags = SDL_SWSURFACE;
	}

	/* Create a compatible surface, note that the colors are RGB ordered */
	/* PW_SDL Spec. */
	if (rect == NULL) {
		x = 0;
		y = 0;
		clip_w = 0;
		clip_h = 0;
	} else {
		x = rect->x;
		y = rect->y;
		clip_w = rect->w;
		clip_h = rect->h;
		if (clip_w <= 0 || clip_h <= 0)
		{
			goto done;
		}
	}


	if(!target)
	{
		if ((biWidth > IMG_BMP_MAX_WIDTH) ||
			(biHeight > IMG_BMP_MAX_HEIGHT) ||
			(biWidth < IMG_BMP_MIN_WIDTH) ||
			(biHeight < IMG_BMP_MIN_HEIGHT))
		{
			IMG_TRACE("BMP : not support size. supported size IMG_BMP_MIN_WIDTH = %d,IMG_BMP_MIN_WIDTH = %d,IMG_BMP_MAX_WIDTH = %d,IMG_BMP_MAX_HEIGHT = %d\n",IMG_BMP_MIN_WIDTH,IMG_BMP_MIN_WIDTH,IMG_BMP_MAX_WIDTH,IMG_BMP_MAX_HEIGHT);
			was_error = 1;
			goto done;
		}
		surface = SDL_CreateRGBSurface(video_flags,
			biWidth, biHeight, biBitCount, Rmask, Gmask, Bmask, Amask);
		if(!surface)
		{
			IMG_TRACE("BMP : SDL_CreateRGBSurface (Out of Memory)\n");
			was_error = 1;
			goto done;
		}
	}
	else
	{
		//target buffer has allocated
		//requires entire frame buffer for decoding
		if (((biCompression == BI_RLE4) || (biCompression == BI_RLE8)) &&
			biWidth != target->w && biHeight != target->h) 
		{
			printf("RLE encoding require entire image \n");
			IMG_TRACE("BMP : RLE requires entire image frame\n");
			was_error = 1;
			goto done;

		}
		surface = target;
	}
//read entire image to surface
	if(target != surface)
	{
		/* Load the palette, if any */
		palette = (surface->format)->palette;
		if ( palette ) {
			if ( SDL_RWseek(src, fp_offset+14+biSize, SEEK_SET) < 0 ) {
				SDL_Error(SDL_EFSEEK);
				was_error = 1;
				goto done;
			}

			/*
			| guich: always use 1<<bpp b/c some bitmaps can bring wrong information
			| for colorsUsed
			*/
			/* if ( biClrUsed == 0 ) {  */
			biClrUsed = 1 << biBitCount;
			/* } */
			if ( biSize == 12 ) {
				for ( i = 0; i < (int)biClrUsed; ++i ) {
					SDL_RWread(src, &palette->colors[i].b, 1, 1);
					SDL_RWread(src, &palette->colors[i].g, 1, 1);
					SDL_RWread(src, &palette->colors[i].r, 1, 1);
					palette->colors[i].unused = 0;
				}	
			} else {
				for ( i = 0; i < (int)biClrUsed; ++i ) {
					SDL_RWread(src, &palette->colors[i].b, 1, 1);
					SDL_RWread(src, &palette->colors[i].g, 1, 1);
					SDL_RWread(src, &palette->colors[i].r, 1, 1);
					SDL_RWread(src, &palette->colors[i].unused, 1, 1);
				}	
			}
			palette->ncolors = biClrUsed;
		}

		/* Read the surface pixels.  Note that the bmp image is upside down */
		if ( SDL_RWseek(src, fp_offset+bfOffBits, SEEK_SET) < 0 ) {
			SDL_Error(SDL_EFSEEK);
			was_error = 1;
			goto done;
		}
		
		if ((biCompression == BI_RLE4) || (biCompression == BI_RLE8)) {
			was_error = readRlePixels(surface, src, biCompression == BI_RLE8);
			if (was_error) SDL_SetError("Error reading from BMP");
			goto done;
		}

		bits = (Uint8 *)surface->pixels+(surface->h*surface->pitch);
		switch (ExpandBMP) {
			case 1:
				bmpPitch = (biWidth + 7) >> 3;
				pad  = (((bmpPitch)%4) ? (4-((bmpPitch)%4)) : 0);
				break;
			case 4:
				bmpPitch = (biWidth + 1) >> 1;
				pad  = (((bmpPitch)%4) ? (4-((bmpPitch)%4)) : 0);
				break;
			default:
				pad  = ((surface->pitch%4) ?
						(4-(surface->pitch%4)) : 0);
				break;
		}

		while ( bits > (Uint8 *)surface->pixels ) {
			bits -= surface->pitch;
			switch (ExpandBMP) {
				case 1:
				case 4: {
				Uint8 pixel = 0;
				int   shift = (8-ExpandBMP);
				for ( i=0; i<surface->w; ++i ) {
					if ( i%(8/ExpandBMP) == 0 ) {
						if ( !SDL_RWread(src, &pixel, 1, 1) ) {
							SDL_SetError(
						"Error reading from BMP");
							was_error = 1;
							goto done;
						}
					}
					*(bits+i) = (pixel>>shift);
					pixel <<= ExpandBMP;
				} }
				break;

				default:
				if ( SDL_RWread(src, bits, 1, surface->pitch)
								 != surface->pitch ) {
					SDL_Error(SDL_EFREAD);
					was_error = 1;
					goto done;
				}
	#if SDL_BYTEORDER == SDL_BIG_ENDIAN
				/* Byte-swap the pixels if needed. Note that the 24bpp
				   case has already been taken care of above. */
				switch(biBitCount) {
					case 15:
					case 16: {
							Uint16 *pix = (Uint16 *)bits;
						for(i = 0; i < surface->w; i++)
								pix[i] = SDL_Swap16(pix[i]);
						break;
					}

					case 32: {
							Uint32 *pix = (Uint32 *)bits;
						for(i = 0; i < surface->w; i++)
								pix[i] = SDL_Swap32(pix[i]);
						break;
					}
				}
	#endif
				break;
			}
			/* Skip padding bytes, ugh */
			if ( pad ) {
				Uint8 padbyte;
				for ( i=0; i<pad; ++i ) {
					SDL_RWread(src, &padbyte, 1, 1);
				}
			}
		}
	}

	//resample to target surface instead of creating new surface (target == surface)
	else
	{
		//target == surface
		//create line buffer
		printf("Scan Line Read \n");
		line_buf_surface = SDL_CreateRGBSurface(video_flags,
				biWidth, 1, biBitCount, Rmask, Gmask, Bmask, Amask);
		if(!line_buf_surface)
		{
			IMG_TRACE("Can't Create SDL Line Buffer (Out of Memory)");
			goto done;
				
		}
		dst_line_buf = malloc(biWidth * 4);
		if(!dst_line_buf)
		{
			IMG_TRACE("Can't Create Color Conversion Line Buffer (Out of Memory)");
			was_error = 1;			
			goto done;
		}

		
		if ( SDL_MUSTLOCK(line_buf_surface) ) {
			if ( SDL_LockSurface(line_buf_surface) < 0 ) {
				IMG_TRACE("Cannot Lock Surface");
				was_error = 1;
				goto done;				
			}
		}
		
		/* Load the palette, if any */
		palette = (line_buf_surface->format)->palette;
		if ( palette ) {
			if ( SDL_RWseek(src, fp_offset+14+biSize, SEEK_SET) < 0 ) {
				SDL_Error(SDL_EFSEEK);
				was_error = 1;
				goto done;
			}

			/*
			| guich: always use 1<<bpp b/c some bitmaps can bring wrong information
			| for colorsUsed
			*/
			/* if ( biClrUsed == 0 ) {  */
			biClrUsed = 1 << biBitCount;
			/* } */
			if ( biSize == 12 ) {
				for ( i = 0; i < (int)biClrUsed; ++i ) {
					SDL_RWread(src, &palette->colors[i].b, 1, 1);
					SDL_RWread(src, &palette->colors[i].g, 1, 1);
					SDL_RWread(src, &palette->colors[i].r, 1, 1);
					palette->colors[i].unused = 0;
				}	
			} else {
				for ( i = 0; i < (int)biClrUsed; ++i ) {
					SDL_RWread(src, &palette->colors[i].b, 1, 1);
					SDL_RWread(src, &palette->colors[i].g, 1, 1);
					SDL_RWread(src, &palette->colors[i].r, 1, 1);
					SDL_RWread(src, &palette->colors[i].unused, 1, 1);
				}	
			}
			palette->ncolors = biClrUsed;
		}

		/* Read the surface pixels.  Note that the bmp image is upside down */
		if ( SDL_RWseek(src, fp_offset+bfOffBits, SEEK_SET) < 0 ) {
			SDL_Error(SDL_EFSEEK);
			was_error = 1;
			goto done;
		}

		bits = (Uint8 *)line_buf_surface->pixels;
		
		switch (ExpandBMP) {
			case 1:
				bmpPitch = (biWidth + 7) >> 3;
				pad  = (((bmpPitch)%4) ? (4-((bmpPitch)%4)) : 0);
				break;
			case 4:
				bmpPitch = (biWidth + 1) >> 1;
				pad  = (((bmpPitch)%4) ? (4-((bmpPitch)%4)) : 0);
				break;
			default:
				pad  = ((line_buf_surface->pitch%4) ?
						(4-(line_buf_surface->pitch%4)) : 0);
				break;
		}

		if ( SDL_MUSTLOCK(target) ) {
			if ( SDL_LockSurface(target) < 0 ) {
				IMG_TRACE("Cannot Lock Surface");
				was_error = 1;
				goto done;	
			}
		}

		if (clip_w == 0) clip_w = target->w;		
		if (clip_h == 0) clip_h = target->h;

		filter_context.filters = filters;		
		filter_context.image = target;		
		filter_context.clip.x = x;		
		filter_context.clip.y = y;		
		filter_context.clip.w = clip_w;		
		filter_context.clip.h = clip_h;		
		PW_FILTER_TOOLS_init(&filter_context);		
		PW_FILTER_TOOLS_config(&filter_context, (int)biWidth, (int)biHeight, picfRGB);

		//read line
		currentY = 0;
		while ( currentY < biHeight ) {
			bits = (Uint8 *)line_buf_surface->pixels;
			switch (ExpandBMP) {
				case 1:
				case 4: {
				Uint8 pixel = 0;
				int   shift = (8-ExpandBMP);
				for ( i=0; i<line_buf_surface->w; ++i ) {
					if ( i%(8/ExpandBMP) == 0 ) {
						if ( !SDL_RWread(src, &pixel, 1, 1) ) {
							SDL_SetError(
						"Error reading from BMP");
							was_error = 1;
							goto done;
						}
					}
					*(bits+i) = (pixel>>shift);
					pixel <<= ExpandBMP;
				} }
				break;

				default:
				
				if ( SDL_RWread(src, bits, 1, line_buf_surface->pitch)
								 != line_buf_surface->pitch ) {
					SDL_Error(SDL_EFREAD);
					was_error = 1;
					goto done;
				}

	#if SDL_BYTEORDER == SDL_BIG_ENDIAN
				/* Byte-swap the pixels if needed. Note that the 24bpp
				   case has already been taken care of above. */
				switch(biBitCount) {
					case 15:
					case 16: {
							Uint16 *pix = (Uint16 *)bits;
						for(i = 0; i < surface->w; i++)
								pix[i] = SDL_Swap16(pix[i]);
						break;
					}

					case 32: {
							Uint32 *pix = (Uint32 *)bits;
						for(i = 0; i < surface->w; i++)
								pix[i] = SDL_Swap32(pix[i]);
						break;
					}
				}
	#endif
				break;
			}
			/* Skip padding bytes, ugh */
			
			if ( pad ) {
				Uint8 padbyte;
				for ( i=0; i<pad; ++i ) {
					SDL_RWread(src, &padbyte, 1, 1);
				}
			}
			//convert line buffer
			PW_BMP_Read_Line_Buf(line_buf_surface, dst_line_buf);

			if (PW_FILTER_TOOLS_sendPixels(&filter_context, 					
				(int)biHeight - currentY -1, (Uint8 *)dst_line_buf, target->pitch, picfRGB) != 0)				
					break;
			currentY++;
		}

		PW_FILTER_TOOLS_finalize(&filter_context);	
		if ( SDL_MUSTLOCK(target) ) {
			SDL_UnlockSurface(target);
		}	
		
		if ( SDL_MUSTLOCK(line_buf_surface) ) {
			SDL_UnlockSurface(line_buf_surface);
		}
		
	}


done:
	if ( was_error ) {
		if ( src ) {
			SDL_RWseek(src, fp_offset, SEEK_SET);
		}
		if ( surface && target != surface ) {
			SDL_FreeSurface(surface);
		}
		surface = NULL;
	}

	//printf(" free dst_line_buf \n");
	if(dst_line_buf)
		free(dst_line_buf);
	if(line_buf_surface)
		SDL_FreeSurface(line_buf_surface);

	return(surface);
}



/* Load a BMP type image from an SDL datasource */
SDL_Surface *IMG_LoadBMP_RW(SDL_RWops *src,int type)
{
	return(PW_IMG_LoadBMP_RW(src, NULL, NULL, NULL, type));
}



#endif






#else

/* See if an image is contained in a data source */
int IMG_isBMP(SDL_RWops *src)
{
	return(0);
}

/* Load a BMP type image from an SDL datasource */
SDL_Surface *IMG_LoadBMP_RW(SDL_RWops *src,int type)
{
	return(NULL);
}

#endif /* LOAD_BMP */
