/*
    SDL_image:  An example image loading library for use with SDL
    Copyright (C) 1997-2009 Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Sam Lantinga
    slouken@libsdl.org
*/

/* A simple library to load images of various formats as SDL surfaces */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "SDL_image.h"

#define ARRAYSIZE(a) (sizeof(a) / sizeof((a)[0]))

/* Table of image detection and loading functions */
static struct {
	char *type;
	int (SDLCALL *is)(SDL_RWops *src);
	SDL_Surface *(SDLCALL *load)(SDL_RWops *src,int type);
} supported[] = {
	/* keep magicless formats first */
	{ "PHJ", IMG_isPHJ, IMG_LoadPHJ_RW },
	{ "JPG", IMG_isJPG, IMG_LoadJPG_RW },
	{ "TGA", NULL,      IMG_LoadTGA_RW },
	{ "BMP", IMG_isBMP, IMG_LoadBMP_RW },
	{ "GIF", IMG_isGIF, IMG_LoadGIF_RW },
	{ "LBM", IMG_isLBM, IMG_LoadLBM_RW },
	{ "PCX", IMG_isPCX, IMG_LoadPCX_RW },
	{ "PNG", IMG_isPNG, IMG_LoadPNG_RW },
	{ "PNM", IMG_isPNM, IMG_LoadPNM_RW }, /* P[BGP]M share code */
	{ "TIF", IMG_isTIF, IMG_LoadTIF_RW },
	{ "XCF", IMG_isXCF, IMG_LoadXCF_RW },
	{ "XPM", IMG_isXPM, IMG_LoadXPM_RW },
	{ "XV",  IMG_isXV,  IMG_LoadXV_RW  }
};

const SDL_version *IMG_Linked_Version(void)
{
	static SDL_version linked_version;
	SDL_IMAGE_VERSION(&linked_version);
	return(&linked_version);
}

/* Load an image from a file */
SDL_Surface *IMG_Load(const char *file)
{
    SDL_RWops *src = SDL_RWFromFile(file, "rb");
    char *ext = strrchr(file, '.');
    if(ext) {
        ext++;
    }
    if(!src) {
        /* The error message has been set in SDL_RWFromFile */
        return NULL;
    }
    return IMG_LoadTyped_RW(src, 1, ext, SDL_IMAGE_LOAD_NEW_SURFACE);
}

/* Load an image from an SDL datasource (for compatibility) */
SDL_Surface *IMG_Load_RW(SDL_RWops *src, int freesrc)
{
    return IMG_LoadTyped_RW(src, freesrc, NULL, SDL_IMAGE_LOAD_NEW_SURFACE);
}

/* Load an image from a file */
SDL_Surface *IMG_Load_Type(const char *file,int load_type)
{
    SDL_RWops *src = SDL_RWFromFile(file, "rb");
    char *ext = strrchr(file, '.');
    if(ext) {
        ext++;
    }
    if(!src) {
        /* The error message has been set in SDL_RWFromFile */
        return NULL;
    }
    return IMG_LoadTyped_RW(src, 1, ext, load_type);
}

/* Portable case-insensitive string compare function */
static int IMG_string_equals(const char *str1, const char *str2)
{
	while ( *str1 && *str2 ) {
		if ( toupper((Uint8)*str1) !=
		     toupper((Uint8)*str2) )
			break;
		++str1;
		++str2;
	}
	return (!*str1 && !*str2);
}

#define RGB16BITS_484_REPRESENTATION 0
#define RGB16BITS_565_REPRESENTATION 1
#define RGB565_PIXEL(r,g,b)  ((Uint16)( (((Uint16)((Uint8)(r))<<8)&0xF800) | (((Uint16)((Uint8)(g))<<3)&0x7E0) | (((Uint8)(b)>>3)&0x01F) ))
#define RGB555_PIXEL(r,g,b)  ((Uint16)( (((Uint16)((Uint8)(r))<<7)&0x7C00) | (((Uint16)((Uint8)(g))<<2)&0x3E0) | (((Uint8)(b)>>3)&0x01F) ))
#define RGB484_PIXEL(r,g,b)  ((Uint16)( (((Uint16)((Uint8)(r))<<8)&0xF000) | (((Uint16)((Uint8)(g))<<4)&0x0FF0) | (((Uint8)(b)>>4)&0x0F) ))

#define RGB_FROM_RGB565(p, r, g, b)									\
{																	\
	(r) = ((((p)&0xF800)>>11)<<3);		 							\
	(g) = ((((p)&0x07E0)>>5)<<2); 									\
	(b) = (((p)&0x001F)<<3); 										\
}
#define RGB_FROM_RGB555(p, r, g, b)									\
{																	\
	r = ((((p)&0x7C00)>>10)<<3);		 							\
	g = ((((p)&0x03E0)>>5)<<3); 									\
	b = (((p)&0x001F)<<3); 											\
}

#define RGB888_YUV888_PLANES(r, g, b, y, u, v)						\
{																	\
	(y) = (( 66 * (r) + 129 * (g) +  25 * (b) + 128) >> 8) + 16;	\
	(u) = ((-38 * (r) -  74 * (g) + 112 * (b) + 128) >> 8) + 128;	\
	(v) = ((112 * (r) -  94 * (g) -  18 * (b) + 128) >> 8) + 128;	\
}

/* Allen K, due to SDL_ReallocFormat is not a public function, we redesign a remap function for that */
static void IMG_RemapFormat(SDL_Surface *surface,
							Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask)
{
	/* rip from SDL_pixels.c and we just support the mask remap */
	if (surface && surface->format)	{
		SDL_PixelFormat *format = surface->format;
		Uint32 mask;

		if ( Rmask || Bmask || Gmask ) { /* Packed pixels with custom mask */
			format->palette = NULL;
			format->Rshift = 0;
			format->Rloss = 8;
			if ( Rmask ) {
				for ( mask = Rmask; !(mask&0x01); mask >>= 1 )
					++format->Rshift;
				for ( ; (mask&0x01); mask >>= 1 )
					--format->Rloss;
			}
			format->Gshift = 0;
			format->Gloss = 8;
			if ( Gmask ) {
				for ( mask = Gmask; !(mask&0x01); mask >>= 1 )
					++format->Gshift;
				for ( ; (mask&0x01); mask >>= 1 )
					--format->Gloss;
			}
			format->Bshift = 0;
			format->Bloss = 8;
			if ( Bmask ) {
				for ( mask = Bmask; !(mask&0x01); mask >>= 1 )
					++format->Bshift;
				for ( ; (mask&0x01); mask >>= 1 )
					--format->Bloss;
			}
			format->Ashift = 0;
			format->Aloss = 8;
			if ( Amask ) {
				for ( mask = Amask; !(mask&0x01); mask >>= 1 )
					++format->Ashift;
				for ( ; (mask&0x01); mask >>= 1 )
					--format->Aloss;
			}
			format->Rmask = Rmask;
			format->Gmask = Gmask;
			format->Bmask = Bmask;
			format->Amask = Amask;
		}
	}
}
/* Load an image from an SDL datasource, optionally specifying the type */
SDL_Surface *IMG_LoadTyped_RW(SDL_RWops *src, int freesrc, char *type, int load_mode)
{
	int i;
	SDL_Surface *image;

	/* Make sure there is something to do.. */
	if ( src == NULL ) {
		IMG_SetError("Passed a NULL data source");
		return(NULL);
	}

	/* See whether or not this data source can handle seeking */
	if ( SDL_RWseek(src, 0, SEEK_CUR) < 0 ) {
		IMG_SetError("Can't seek in this data source");
		if(freesrc)
			SDL_RWclose(src);
		return(NULL);
	}

	/* Detect the type of image being loaded */
	image = NULL;
	for ( i=0; i < ARRAYSIZE(supported); ++i ) {
		if(supported[i].is) {
			if(!supported[i].is(src))
				continue;
		} else {
			/* magicless format */
			if(!type
			   || !IMG_string_equals(type, supported[i].type))
				continue;
		}
#ifdef DEBUG_IMGLIB
		fprintf(stderr, "IMGLIB: Loading image as %s\n",
			supported[i].type);
#endif
		image = supported[i].load(src,load_mode);

		if(freesrc)
			SDL_RWclose(src);

#ifdef PW_FB_UYV444B
		if (image)
		{
			Uint8 * pixels;
			int l, j, k, r, g, b, y, u ,v;

			if (image->format->BitsPerPixel <= 8 && image->format->palette != NULL)
			{
				SDL_Palette *palette;
				int i;

				palette = image->format->palette;
				/* Allen K, put the PALETTE-YUV convert code here. Slow... but simple */
				for(i=0; i<palette->ncolors; ++i)
				{
					r = palette->colors[i].r;
					g = palette->colors[i].g;
					b = palette->colors[i].b;
					RGB888_YUV888_PLANES(r, g, b, y, u, v);
					palette->colors[i].r = (Uint8)u;
					palette->colors[i].g = (Uint8)y;
					palette->colors[i].b = (Uint8)v;
				}
			}
			/* removed, Allen K, we don't support 16 bits here. It's hard to us to do UYV 16 bits compress           */
			/* Allen K, use the compress YUV555/565??? to represent RGB555/565.                                      */
			/* Due to 16 bits 555 or 565 losing lots of primary Y color, I represent it to 484 another weird color:) */
			else if (image->format->BitsPerPixel == 16)
			{
				if (SDL_LockSurface(image) == 0)
				{
					Uint16 * pixels;
					pixels = (Uint16 *)image->pixels;

					/* RGB555 */
					if (image->format->Gmask == 0x03E0)
					{
						for (l = 0; l < image->h; ++l)
						{					
							for (k=0; k < image->w; ++k)
							{
								RGB_FROM_RGB555(pixels[k], r, g, b);
								RGB888_YUV888_PLANES(r, g, b, y, u, v);
#if RGB16BITS_484_REPRESENTATION
								pixels[k] = RGB484_PIXEL(u, y, v);
#elif RGB16BITS_565_REPRESENTATION
								pixels[k] = RGB565_PIXEL(u, y, v);
#else
								pixels[k] = RGB555_PIXEL(u, y, v);
#endif
							}
							pixels += image->pitch / 2;
						}
					}
					/* RGB565 */
					else
					{
						for (l = 0; l < image->h; ++l)
						{					
							for (k=0; k < image->w; ++k)
							{
								RGB_FROM_RGB565(pixels[k], r, g, b);
								RGB888_YUV888_PLANES(r, g, b, y, u, v);
#if RGB16BITS_484_REPRESENTATION
								pixels[k] = RGB484_PIXEL(u, y, v);
#elif RGB16BITS_565_REPRESENTATION
								pixels[k] = RGB565_PIXEL(u, y, v);
#else
								pixels[k] = RGB565_PIXEL(u, y, v);
#endif
							}
							pixels += image->pitch / 2;
						}
					}
					SDL_UnlockSurface(image);
#if RGB16BITS_484_REPRESENTATION
					IMG_RemapFormat(image, 0xF000, 0x0FF0, 0x000F, 0);
#elif RGB16BITS_565_REPRESENTATION
					IMG_RemapFormat(image, 0xF800, 0x07E0, 0x001F, 0);
#endif
				}
				else
				{
					SDL_FreeSurface(image);
					image = NULL;
					IMG_SetError("Couldn't support RGB-YUV 16 bits convert");
				}
			}
			/* do color convert for 24 bits */
			else if (image->format->BitsPerPixel == 24)
			{
				if (SDL_LockSurface(image) == 0)
				{
					pixels = (Uint8 *)image->pixels;

					/* Allen K. Pass through HW SURFACE, it must comes from HW decoder */
					/* The compile flag PHJ_MUST_HWSF must not be defined              */
					if ((image->flags & SDL_HWSURFACE) == SDL_HWSURFACE)
					{
						SDL_UnlockSurface(image);
						return image;
					}
					/* just need do byte swap for JPEG YUV format YUV => UYV */
					else if (supported[i].load == IMG_LoadPHJ_RW || supported[i].load == IMG_LoadJPG_RW)
					{
						for (l = 0; l < image->h; ++l)
						{					
							for (k=0, j=0; k < image->w; ++k, j+=3)
							{
								r = pixels[j + 0];
								pixels[j + 0] = pixels[j + 1];
								pixels[j + 1] = r;
							}
							pixels += image->pitch;
						}
					}
					/* non PHJ, JPG */
					else
					{
						if (image->format->Rmask == 0x00FF0000)
						{
							for (l = 0; l < image->h; ++l)
							{					
								for (k=0, j=0; k < image->w; ++k, j+=3)
								{
									b = pixels[j + 0];
									g = pixels[j + 1];
									r = pixels[j + 2];
									RGB888_YUV888_PLANES(r, g, b, y, u, v);
									pixels[j + 0] = (Uint8)u;
									pixels[j + 1] = (Uint8)y;
									pixels[j + 2] = (Uint8)v;
								}
								pixels += image->pitch;
							}
						}
						else
						{
							for (l = 0; l < image->h; ++l)
							{					
								for (k=0, j=0; k < image->w; ++k, j+=3)
								{
									r = pixels[j + 0];
									g = pixels[j + 1];
									b = pixels[j + 2];
									RGB888_YUV888_PLANES(r, g, b, y, u, v);
									pixels[j + 0] = (Uint8)u;
									pixels[j + 1] = (Uint8)y;
									pixels[j + 2] = (Uint8)v;
								}
								pixels += image->pitch;
							}
						}
					}
					SDL_UnlockSurface(image);
				}
				else
				{
					SDL_FreeSurface(image);
					image = NULL;
					IMG_SetError("Couldn't support RGB-YUV 24 bits convert");
				}
			} /* BitsPerPixel == 24 */
			/* do color convert for 32 bits. leave alpha channel to the same place. */
			else if (image->format->BitsPerPixel == 32)
			{
				if (SDL_LockSurface(image) == 0)
				{
					pixels = (Uint8 *)image->pixels;

					/* PNG::RGBA */
					if (image->format->Rmask == 0xFF000000)
					{
						for (l = 0; l < image->h; ++l)
						{					
							for (k=0, j=0; k < image->w; ++k, j+=4)
							{
								b = pixels[j + 1];
								g = pixels[j + 2];
								r = pixels[j + 3];
								RGB888_YUV888_PLANES(r, g, b, y, u, v);
								pixels[j + 1] = (Uint8)v;
								pixels[j + 2] = (Uint8)y;
								pixels[j + 3] = (Uint8)u;
							}
							pixels += image->pitch;
						}
					}
					/* BMP::ARGB */
					else if (image->format->Rmask == 0x00FF0000)
					{
						for (l = 0; l < image->h; ++l)
						{					
							for (k=0, j=0; k < image->w; ++k, j+=4)
							{
								b = pixels[j + 0];
								g = pixels[j + 1];
								r = pixels[j + 2];
								RGB888_YUV888_PLANES(r, g, b, y, u, v);
								pixels[j + 2] = (Uint8)u;
								pixels[j + 1] = (Uint8)y;
								pixels[j + 0] = (Uint8)v;
							}
							pixels += image->pitch;
						}
					}
					/* Actually, it implies the format in image->format->Rmask == 0x000000FF */
					/* PNG::ABGR */
					else
					{
						for (l = 0; l < image->h; ++l)
						{					
							for (k=0, j=0; k < image->w; ++k, j+=4)
							{
								r = pixels[j + 0];
								g = pixels[j + 1];
								b = pixels[j + 2];
								RGB888_YUV888_PLANES(r, g, b, y, u, v);
								pixels[j + 0] = (Uint8)u;
								pixels[j + 1] = (Uint8)y;
								pixels[j + 2] = (Uint8)v;
							}
							pixels += image->pitch;
						}
					}
					SDL_UnlockSurface(image);
				}
				else
				{
					SDL_FreeSurface(image);
					image = NULL;
					IMG_SetError("Couldn't support RGB-YUV 32 bits convert");
				}
			} /* BitsPerPixel ==32 */
		}
#else
        // RGB frame buffer
	if (image)
    {
        if (image->format->BitsPerPixel == 24)
        {
			Uint8 * pixels;
		    Uint8 r, g, b;
			int l, j, k, y, u ,v;

    	    IMG_TRACE("IMG : image->format->BitsPerPixel == 24\n");

				if (SDL_LockSurface(image) == 0)
				{
					pixels = (Uint8 *)image->pixels;

					/* Allen K. Pass through HW SURFACE, it must comes from HW decoder */
					/* The compile flag PHJ_MUST_HWSF must not be defined              */
					if ((image->flags & SDL_HWSURFACE) == SDL_HWSURFACE)
					{
						SDL_UnlockSurface(image);
						return image;
					}
					/* just need do byte swap for JPEG YUV format YUV => UYV */
					else if (supported[i].load == IMG_LoadPHJ_RW || supported[i].load == IMG_LoadJPG_RW)
					{
					}
					/* non PHJ, JPG */
					else
					{
#if 0                	        
						if (image->format->Rmask == 0x00FF0000)
						{
                	        IMG_TRACE("IMG : switch color\n");
							for (l = 0; l < image->h; ++l)
							{					
								for (k=0, j=0; k < image->w; ++k, j+=3)
								{
									b = pixels[j + 0];
									g = pixels[j + 1];
									r = pixels[j + 2];
									pixels[j + 0] = r;
									pixels[j + 1] = g;
									pixels[j + 2] = b;
								}
								pixels += image->pitch;
							}
						}
						else
						{
						}
#endif							
					}
					SDL_UnlockSurface(image);
				}
				else
				{
					SDL_FreeSurface(image);
					image = NULL;
					IMG_SetError("Couldn't support RGB-YUV 24 bits convert");
				}

        }
    }
#endif
		return image;
	}

	if ( freesrc ) {
		SDL_RWclose(src);
	}
	IMG_SetError("Unsupported image format");
	return NULL;
}

/* Invert the alpha of a surface for use with OpenGL
   This function is a no-op and only kept for backwards compatibility.
 */
int IMG_InvertAlpha(int on)
{
    return 1;
}
