/*
 * based on altera_spi.c
 * (C) Copyright 2011 - Franck JULLIEN <elec4fun@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

//#define DEBUG

#include <common.h>
#include <errno.h>
#include <init.h>
#include <driver.h>
#include <spi/spi.h>
#include <io.h>
#include <clock.h>
#include <linux/stringify.h>

#define AUG06X_CLKDIV 2		/* HPER */
static unsigned long aug060x_clkdiv;
static unsigned long aug060x_clk_ci;

#define AUG060X_CR	0x00	/* SPI Control Register */
#define AUG060X_TC	0x04	/* SPI Transmit Control Register */
#define AUG060X_RC	0x08	/* SPI Receive Control Register */
#define AUG060X_IEN	0x0c	/* SPI Interrupt Enable Register */
#define AUG060X_ST	0x10	/* SPI Status Register */
#define AUG060X_ICLR	0x14	/* SPI Interrupt Clear Register */
#define AUG060X_SS	0x18	/* SPI Slave Select Register */
#define AUG060X_CT	0x1c	/* SPI Clock Time Register */
#define AUG060X_TCD	0x20	/* SPI Transmit Clock Delay Register */
#define AUG060X_RCD	0x24	/* SPI Receive Clock Delay */
#define AUG060X_SSD	0x28	/* SPI Slave Select Delay */
#define AUG060X_TDC	0x30	/* SPI Transmit Data Count Register */
#define AUG060X_RDC	0x34	/* SPI Receive Data Count Register */
#define AUG060X_TDF	0x38	/* SPI Transmit Data FIFO Register */
#define AUG060X_RDF	0x3c	/* SPI Receive Data FIFO Register */


struct aug060x_spi {
	struct spi_master	master;
	void __iomem		*regs;
};


static int aug060x_spi_setup(struct spi_device *spi)
{
	dev_dbg(spi->master->dev, "mode 0x%08x, chip select %d, bits_per_word: %d, speed: %d\n",
		spi->mode, spi->chip_select, spi->bits_per_word, spi->max_speed_hz);
	return 0;
}

static unsigned aug060x_spi_do_xfer(struct spi_device *spi,
				    struct spi_transfer *t, u32 cr)
{
	struct aug060x_spi *aug060x_spi = container_of(spi->master, struct aug060x_spi, master);
	unsigned int tdc, rdc, rt, rc, tc;
	const u8 *txbuf = t->tx_buf;
	u8 *rxbuf = t->rx_buf;
	u32 *rxbuf32 = NULL;
	int ri = 0, ti = 0;
	u32 st, val;

	dev_dbg(spi->master->dev, "tx %p rx %p len %u%s\n", txbuf, rxbuf, t->len,
		t->rx_nbits == SPI_NBITS_QUAD ? ", 4x" :
		t->rx_nbits == SPI_NBITS_DUAL ? ", 2x" : "");

	if (((t->len & 3) == 0) && (((u32)t->rx_buf & 3) == 0))
		rxbuf32 = t->rx_buf;

	/* it is necessary to set CR here with EN=0 before setting TDC and RDC,
	 * and later set EN=1 otherwise the 2x and 4x reads don't work
	 */
	if (t->rx_nbits == SPI_NBITS_QUAD)
		cr |= 4 << 26;
	else if (t->rx_nbits == SPI_NBITS_DUAL)
		cr |= 2 << 26;
	writel(cr, aug060x_spi->regs + AUG060X_CR);

	// SPI word length 8, MSB first, AMBA size 8 or 32 bits, irq threshold = 31
	rc = (31 << 20) | (1 << 12) | ((rxbuf32 ? 0 : 2) << 8) | 8;
	// SPI word length 8, MSB first, AMBA size 32 bits, irq threshold = 31
	tc = (31 << 20) | (1 << 12) | (2 << 8) | 8;

	rdc = rxbuf ? t->len : 0;
	tdc = txbuf ? t->len : 0;

	rt = tdc ? 3 : 2;
	writel((rt << 26) | rc, aug060x_spi->regs + AUG060X_RC);
	rc |= 2 << 26;
	writel(tc, aug060x_spi->regs + AUG060X_TC);
	dev_dbg(spi->master->dev, "TC %08x\n", readl(aug060x_spi->regs + AUG060X_TC));
	dev_dbg(spi->master->dev, "RC %08x\n", readl(aug060x_spi->regs + AUG060X_RC));

	writel(tdc, aug060x_spi->regs + AUG060X_TDC);
	writel(rdc, aug060x_spi->regs + AUG060X_RDC);
	dev_dbg(spi->master->dev, "TDC %08x\n", readl(aug060x_spi->regs + AUG060X_TDC));
	dev_dbg(spi->master->dev, "RDC %08x\n", readl(aug060x_spi->regs + AUG060X_RDC));

	/* enable */
	dev_dbg(spi->master->dev, "CR %08x\n", cr);
	writel(cr | (1 << 31), aug060x_spi->regs + AUG060X_CR);

	/* prime TX FIFO */
	while (ti < tdc && readl(aug060x_spi->regs + AUG060X_ST) & (1 << 6)) {
		writel((1 << 6), aug060x_spi->regs + AUG060X_ICLR);
		dev_dbg(spi->master->dev, "TX %02x\n", txbuf[ti]);
		writel(txbuf[ti++], aug060x_spi->regs + AUG060X_TDF);
	}
	if (ti == tdc)
		writel(rc, aug060x_spi->regs + AUG060X_RC);

	while (ri < rdc || ti < tdc) {
		st = readl(aug060x_spi->regs + AUG060X_ST);
		if ((ri < rdc) && (st & (1 << 5))) {
			writel((1 << 5), aug060x_spi->regs + AUG060X_ICLR);
			val = readl(aug060x_spi->regs + AUG060X_RDF);
			dev_dbg(spi->master->dev, "RX %08x\n", val);
			if (rxbuf32) {
				*rxbuf32++ = val;
				ri += 4;
			} else
				rxbuf[ri++] = val;
		}
		if (ti < tdc && (st & (1 << 6))) {
			writel((1 << 6), aug060x_spi->regs + AUG060X_ICLR);
			dev_dbg(spi->master->dev, "TX %02x\n", txbuf[ti]);
			writel(txbuf[ti++], aug060x_spi->regs + AUG060X_TDF);
			if (ti == tdc)
				writel(rc, aug060x_spi->regs + AUG060X_RC);
		}
	}
	if (tdc > rdc) {
		// wait for TX FIFO to drain
		while (!(readl(aug060x_spi->regs + AUG060X_ST) & (1 << 10)))
			;
		writel((1 << 10), aug060x_spi->regs + AUG060X_ICLR);
	}

	/* disable */
	writel(cr, aug060x_spi->regs + AUG060X_CR);
	return tdc + rdc;
}

static int aug060x_spi_transfer(struct spi_device *spi, struct spi_message *mesg)
{
	struct aug060x_spi *aug060x_spi = container_of(spi->master, struct aug060x_spi, master);
	struct spi_transfer *t;
	u8 polarity, phase;
	u32 cr;

	dev_dbg(spi->master->dev, "%s start\n", __func__);

	writel(0, aug060x_spi->regs + AUG060X_TCD);
	writel(0, aug060x_spi->regs + AUG060X_RCD);
	writel(0, aug060x_spi->regs + AUG060X_SSD);
	writel(0, aug060x_spi->regs + AUG060X_IEN);
	writel(~0, aug060x_spi->regs + AUG060X_ICLR);

	writel(aug060x_clkdiv, aug060x_spi->regs + AUG060X_CT);

	polarity = !!(spi->mode & SPI_CPOL);
	phase = !!(spi->mode & SPI_CPHA);
	/* use sw slave select, push-pull outputs, master mode */
	cr = (1 << 6) | (phase << 20) | (polarity << 21) | (1 << 24) |
		(aug060x_clk_ci ? (1<<7) : 0);

	mesg->actual_length = 0;
	writel(1 << spi->chip_select, aug060x_spi->regs + AUG060X_SS);
	list_for_each_entry(t, &mesg->transfers, transfer_list)
		mesg->actual_length += aug060x_spi_do_xfer(spi, t, cr);
	writel(0, aug060x_spi->regs + AUG060X_SS);

	dev_dbg(spi->master->dev, "%s done\n", __func__);
	return 0;
}

static int aug060x_set_hper(struct device_d *dev,
			    struct param_d *param,
			    const char *val)
{
	unsigned long hper = simple_strtoul(val, NULL, 0);
	if (hper < 2)
		return -EINVAL;
	aug060x_clkdiv = hper;

	return dev_param_set_generic(dev, param, val);
}

static int aug060x_set_ci(struct device_d *dev,
			    struct param_d *param,
			    const char *val)
{
	unsigned long ci = simple_strtoul(val, NULL, 0);
	if (ci > 2)
		return -EINVAL;
	aug060x_clk_ci = ci;

	return dev_param_set_generic(dev, param, val);
}

static int aug060x_spi_probe(struct device_d *dev)
{
	struct spi_master *master;
	struct aug060x_spi *aug060x_spi;

	dev_dbg(dev, "%s\n", __func__);
	aug060x_spi = xzalloc(sizeof(*aug060x_spi));
	aug060x_spi->regs = dev_request_mem_region(dev, 0);

	master = &aug060x_spi->master;
	master->dev = dev;
	master->setup = aug060x_spi_setup;
	master->transfer = aug060x_spi_transfer;
	master->num_chipselect = 1;
	master->bus_num = 0;

	dev_add_param(dev, "hper", aug060x_set_hper, NULL, 0);
	dev_set_param(dev, "hper", __stringify(AUG06X_CLKDIV));
	dev_add_param(dev, "ci", aug060x_set_ci, NULL, 0);
	dev_set_param(dev, "ci", "0");
	spi_register_master(master);

	return 0;
}

static struct driver_d aug060x_spi_driver = {
	.name  = "aug0603_spi",
	.probe = aug060x_spi_probe,
};
device_platform_driver(aug060x_spi_driver);
