/*
   -------------------------------------------------------------------------
   pw2wire-hw-algo.c two wire driver algorithms for Pixelworks Ruby adapters

   Antson Hu, Pixelworks, Inc.
   HZhou, Pixelworks Japan, Inc.

   Copyright 2007-2017 Pixelworks Inc.

   ---------------------------------------------------------------------------
   This file was highly leveraged from pw2wire-hw-algo.c, which was created
   by Antson Hu:


     Copyright (C) 1995-1997 Simon G. Vogl
                   1998-2000 Hans Berglund

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/* ------------------------------------------------------------------------- */

/* With some changes from Kysti Mlkki <kmalkki@cc.hut.fi> and
   Frodo Looijaard <frodol@dds.nl> ,and also from Martin Bailey
   <mbailey@littlefeet-inc.com> */

//#define VERBOSE_DEBUG

#include <common.h>
#include <clock.h>
#include <i2c/i2c.h>
#include <malloc.h>
#include <io.h>
#include <driver.h>
#include <init.h>
#include <printk.h>

#include "i2c-pw2wire.h"

struct BlockInfo {
	u32 BaseAddress;
	int errorState;
	int (*RegRead8) (void* blkinfo, u32 addr, u8*  pValue);
	int (*RegRead16)(void* blkinfo, u32 addr, u16* pValue);
	int (*RegRead24)(void* blkinfo, u32 addr, u32*  pValue);
	int (*RegRead32)(void* blkinfo, u32 addr, u32*  pValue);

	int (*RegWrite8) (void* blkinfo, u32 addr, u8 mask, u8 value);
	int (*RegWrite16)(void* blkinfo, u32 addr, u16 mask, u16 value);
	int (*RegWrite24)(void* blkinfo, u32 addr, u32 mask, u32 value);
	int (*RegWrite32)(void* blkinfo, u32 addr, u32 mask, u32 value);
};

static int RegRead8(void * blkinfo, u32  dwAddr, u8* pdwValue)
{
	*pdwValue = *((volatile u32* )dwAddr);
	return 0;
}

static int RegRead16(void * blkinfo, u32  dwAddr, u16* pdwValue)
{
	*pdwValue = *((volatile u32* )dwAddr);
	return 0;
}

static int RegRead24(void * blkinfo, u32  dwAddr, u32* pdwValue)
{
	u32 temp=0;
	temp  =  *((volatile u8* ) dwAddr) & 0xff;
	temp += (*((volatile u8* ) dwAddr+1) & 0xff)<<8;
	temp += (*((volatile u8* ) dwAddr+2)   & 0xff)<<16;
	*pdwValue = temp;

	return 0;
}

static int RegRead32(void * blkinfo, u32  dwAddr, u32* pdwValue)
{
	*pdwValue = *((volatile u32* )dwAddr);
	return 0;
}

static int RegWrite8(void * blkinfo, u32 dwAddr, u8 dwMask, u8 dwValue)
{
	if (dwMask == (u8)0x00)
		*(volatile u8 *)dwAddr = dwValue;
	else
	{
		u8 temp = *(volatile u8* )dwAddr;
		*(volatile u8* )dwAddr = (temp & dwMask) | (dwValue & (~dwMask));
	}
	return 0;
}

static int RegWrite16(void * blkinfo, u32 dwAddr, u16 dwMask, u16 dwValue)
{
	if (dwMask == (u16)0x0000)
		*(volatile u16*)dwAddr = dwValue;
	else
	{
		u16 temp = *(volatile u16* )dwAddr;
		*(volatile u16* )dwAddr = (temp & dwMask) | (dwValue & (~dwMask));
	}
	return 0;
}

static int RegWrite24(void * blkinfo, u32 dwAddr, u32 dwMask, u32 dwValue)
{
	if (dwMask == (u32)0x0)
	{
		*(volatile u8*)dwAddr = dwValue&0xff;
		*(volatile u8*)(dwAddr+1) = (dwValue>>8)&0xff;
		*(volatile u8*)(dwAddr+2)   = (dwValue>>16)&0xff;
	}
	else
	{
		u32 temp;
		u32 dwMaskedValue;
		int ret = RegRead24(blkinfo, dwAddr,&temp);
		if (0 !=ret)
		{
			return ret;
		}
		dwMaskedValue=(temp & dwMask) | (dwValue & (~dwMask));

		*(volatile u8*)dwAddr     = dwMaskedValue &0xff;
		*(volatile u8*)(dwAddr+1) = (dwMaskedValue>>8)&0xff;
		*(volatile u8*)(dwAddr+2) = (dwMaskedValue>>16)&0xff;

	}
	return 0;
}

static int RegWrite32(void * blkinfo, u32 dwAddr, u32 dwMask, u32 dwValue)
{
	if (dwMask == (u32)0x00000000)
		*(volatile u32 *)dwAddr = dwValue;
	else
	{
		u32 temp = *(volatile u32* )dwAddr;
		*(volatile u32* )dwAddr = (temp & dwMask) | (dwValue & (~dwMask));
	}
	return 0;
}

static void BlockInfoInit(struct BlockInfo *blkinfo, u32 BaseAddress)
{
    blkinfo->RegRead8  = RegRead8;
    blkinfo->RegRead16 = RegRead16;
    blkinfo->RegRead24 = RegRead32;
    blkinfo->RegRead32 = RegRead32;
    blkinfo->RegWrite8  = RegWrite8;
    blkinfo->RegWrite16 = RegWrite16;
    blkinfo->RegWrite24 = RegWrite24;
    blkinfo->RegWrite32 = RegWrite32;
    blkinfo->BaseAddress = BaseAddress;
}


// FIXME: these should be removed
#define I2C_M_NO_STOP   0x100   /* Force no stop */
#define I2C_M_FORCE_ACK 0x200   /* Force to send ACK when READ */
#define I2C_M_NO_RD_ACK 0x400   /* Force to send ACK when READ */
#define I2C_M_NOSTART   0x800
#define I2C_M_REV_DIR_ADDR 0x1000


//Flags to indicate interrupt status
#define INT_STATUS_NOERR    0x00    //Interrupt initial status
#define INT_STATUS_ERR      0x01    //Interrupt status of error
#define INT_STATUS_STOP     0x02    //Interrupt status of normal stop
#define INT_STATUS_NACK     0x04    //Interrupt status of NACK
#define INT_STATUS_READ_FIN 0x08    //Interrupt status of byte read done

//State of processing
#define STATE_SEND      0x00        //State to send commands
#define STATE_CHECK     0x01        //State to check results


enum {
	CMD_START,
	CMD_STOP,
	CMD_WRITE,
	CMD_READ,
	CMD_SNDACK = 7,
	CMD_INVALID
};

struct pw2wire_command {
	unsigned char cmd;
	unsigned char data;
};


struct pw2wire {
	struct device_d *dev;
	void *regs;
	struct BlockInfo blkinfo;
	struct i2c_adapter adapter;

	int clock_mode;				// Clock speed 1:high 0:low
	unsigned char rthreshold;		// Read FIFO threshold
	unsigned char wthreshold;		// Write FIFO threshold
	unsigned char c2wire_autohold;

	unsigned long interrupt_status;		// Interrupt status
        unsigned long err_flag;                 // Driver error information
        unsigned long status;                   // HW bus state


	struct pw2wire_command *cmdbuf;		// Command to be send to MWFIFO
	int cmdIndex;				// Index to current command
	int cmdIndexLast;			// Last index of command
	int cmdIndexMax;			// Max index of command
	unsigned char *recvbuf;			// Receive buffer
	int recvcount;				// Receive counter
	int recvmax;				// Receive max bytes
	int readcmdcnt;				// Count of read commands in WFIFO
};

#define to_pw2wire(adap) container_of(adap, struct pw2wire, adapter)

static void pw2wire_handler(struct pw2wire *pw2wire);


static int pw2wire_xtal_type = 1;        //XTAL TYPE (0-27M, 1-24M)
static int pw2wire_timeout_en = 0;       //TIMEOUT ENABLE(0-Ignore timeout, 1-Timeout enabled)


struct pw2wire_speedconfig {
	u32   speed_hz;
	u16   value;
};

static const struct pw2wire_speedconfig _speedconfig[] =
{
	{ 100000, 0x00 },
	{  25000, 0x01 },
	{  50000, 0x02 },
	{  75000, 0x03 },
	{ 150000, 0x04 },
	{ 200000, 0x05 },
	{ 250000, 0x06 },
	{ 275000, 0x07 },
	{ 300000, 0x08 },
	{ 325000, 0x09 },
	{ 350000, 0x0A },
	{ 375000, 0x0B },
	{ 400000, 0x0C },
	{ 600000, 0x0D },
	{ 800000, 0x0E },
	{1000000, 0x0F },
};
#define NUM_OF_SPEEDITEMS ARRAY_SIZE(_speedconfig)

static int get_clock_mode(u32 speed_hz)
{
	int i;
	int ret = 0; // default is 100kHz

	for (i = 0; i < NUM_OF_SPEEDITEMS; i++) {
		if (_speedconfig[i].speed_hz == speed_hz) {
			ret = _speedconfig[i].value;
			break;
		}
	}
	return ret;
}

static void pw2wire_master_enable(struct pw2wire *pw2wire)
{
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MASTER_EN(0x1));
}

static void pw2wire_master_disable(struct pw2wire *pw2wire)
{
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MASTER_EN(0x0));
}

static void pw2wire_disable_interrupt(struct pw2wire *pw2wire)
{
	TW_V1_INTEN_SET(&pw2wire->blkinfo,
			MNACK(0x0),
			MBYRCVD(0x0),
			MARBLOST(0x0),
			MBYCDONE(0x0),
			MRFLIM(0x0),
			MWFLIM(0x0),
			MILLCMD(0x0),
			MHOLD(0x0),
			SNACK(0x0),
			SBYRCVD(0x0),
			SWRREQ(0x0),
			SRDREQ(0x0),
			SRFLIM(0x0),
			SWFLIM(0x0),
			SWFLOCKED(0x0),
			SGENCALL(0x0),
			SHOLD(0x0),
			START(0x0),
			STOP(0x0));
}

static void pw2wire_enable_interrupt(struct pw2wire *pw2wire, int rwflg)
{
	if (rwflg == 0) {	//Receive
		TW_V1_INTEN_SET(&pw2wire->blkinfo,
				MNACK(0x1),
				MBYRCVD(0x1),
				MARBLOST(0x1),
				MBYCDONE(0x1),
				MILLCMD(0x1),
				STOP(0x1));
	} else {
		TW_V1_INTEN_SET(&pw2wire->blkinfo,
				MNACK(0x1),
				MARBLOST(0x1),
				MBYCDONE(0x1),
				MILLCMD(0x1),
				STOP(0x1));
	}
}

static void pw2wire_master_reset(struct pw2wire *pw2wire)
{
	//Reset control logic
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MASTER_RST(0x1));
	//Reset write FIFO
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MWFRST(0x1));
	//Reset read FIFO
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MRFRST(0x1));
	//Clear halt status
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,CLRHLT(0x1));

	//Clear all interrupt status
	TW_V1_INTCLR_SET(&pw2wire->blkinfo,
			 MNACK(0x1),
			 MBYRCVD(0x1),
			 MARBLOST(0x1),
			 MBYCDONE(0x1),
			 MRFLIM(0x1),
			 MWFLIM(0x1),
			 MILLCMD(0x1),
			 MHOLD(0x1),
			 SNACK(0x1),
			 SBYRCVD(0x1),
			 SWRREQ(0x1),
			 SRDREQ(0x1),
			 SRFLIM(0x1),
			 SWFLIM(0x1),
			 SWFLOCKED(0x1),
			 SGENCALL(0x1),
			 SHOLD(0x1),
			 START(0x1),
			 STOP(0x1));
}

static void pw2wire_master_init(struct pw2wire *pw2wire)
{
	pw2wire_master_disable(pw2wire);
	pw2wire_master_reset(pw2wire);

#if defined(AUTO_HOLD)
	// Enable master with auto-hold and local start/stop
	TW_V1_MCTL_SET(&pw2wire->blkinfo,
		       LCLSTST(0x1),
		       AUTOHOLD(0x1));
#else
	// Enable master with local start/stop
	TW_V1_MCTL_SET(&pw2wire->blkinfo,
		       LCLSTST(0x1),
		       AUTOHOLD(0x0));
#endif

	//Write FIFO threshold. Remaining entries 9->8 will cause INT
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MWFTHR(0x3));
	//Read FIFO threshold. 16 entries full will cause INT
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,MRFTHR(0x0));

	/* Set speed */
	TW_V1_MCTL_SET1(&pw2wire->blkinfo,SCLSPEED(pw2wire->clock_mode));

	/* Clear all interrupt status */
	TW_V1_INTCLR_SET(&pw2wire->blkinfo,
			 MNACK(0x1),
			 MBYRCVD(0x1),
			 MARBLOST(0x1),
			 MBYCDONE(0x1),
			 MRFLIM(0x1),
			 MWFLIM(0x1),
			 MILLCMD(0x1),
			 MHOLD(0x1),
			 SNACK(0x1),
			 SBYRCVD(0x1),
			 SWRREQ(0x1),
			 SRDREQ(0x1),
			 SRFLIM(0x1),
			 SWFLIM(0x1),
			 SWFLOCKED(0x1),
			 SGENCALL(0x1),
			 SHOLD(0x1),
			 START(0x1),
			 STOP(0x1));

	pw2wire_master_enable(pw2wire);
}

static int pw2wire_process_fifo(struct pw2wire *pw2wire,
				unsigned short flag, unsigned long timeout_us)
{
	struct pw2wire_command *current_cmd = NULL;
	unsigned short fifodepth;
	unsigned short fifopending;
	unsigned short rdfifodepth;
	unsigned short rdfifopending;
	int ret   = 0;
	int loop  = 1;
	int state = STATE_SEND;
	int i;
	uint64_t startwait_ns = get_time_ns();

	if (flag & I2C_M_RD)
		pw2wire_enable_interrupt(pw2wire, 0);
	else
		pw2wire_enable_interrupt(pw2wire, 1);

	while (loop) {
		pw2wire_handler(pw2wire);

		switch (state)
		{
		case STATE_SEND:     //Sending command
			//Get FIFO info
			TW_V1_MFIFOSTAT_GET(&pw2wire->blkinfo, WFNE(fifopending), WFSZ(fifodepth),
								RFNE(rdfifopending), RFSZ(rdfifodepth));

			//Check if fifo is full
			if (fifopending == fifodepth || rdfifopending == rdfifodepth)
				; //Yield
			else {
				TW_V1_MFIFOSTAT_GET(&pw2wire->blkinfo, WFNE(fifopending), WFSZ(fifodepth),
									RFNE(rdfifopending), RFSZ(rdfifodepth));
				for (i = fifopending; i < fifodepth; i++) {
					current_cmd = (struct pw2wire_command *)&pw2wire->cmdbuf[pw2wire->cmdIndex];
					// Check current command
					if (CMD_READ == current_cmd->cmd) {
						// For read command, control the level of WFIFO
						if( i - fifopending >= rdfifodepth - rdfifopending || pw2wire->readcmdcnt == rdfifodepth) {
							// Try again to wait for empty room
							dev_dbg(pw2wire->dev, "i=%d, w(%d,%d),r(%d,%d)\n",
									i, fifopending, fifodepth, rdfifopending, rdfifodepth);
							do {
								// Wait until WFIFO becomes empty
								pw2wire->interrupt_status &= ~INT_STATUS_READ_FIN;
								pw2wire_handler(pw2wire);
								wait_on_timeout(30 * MSECOND, INT_STATUS_READ_FIN & pw2wire->interrupt_status);
								TW_V1_MFIFOSTAT_GET1(&pw2wire->blkinfo, WFNE(fifopending));
							} while(fifopending > i/2);
							break;
						}
						pw2wire->readcmdcnt++;
					}
					//Advance the command index
					pw2wire->cmdIndex++;

					//Send one command
					dev_dbg(pw2wire->dev, "TX (%d, %d)\n",
						current_cmd->cmd, current_cmd->data);
					TW_V1_MWFIFO_SET(&pw2wire->blkinfo, WDATA(current_cmd->data), CMD(current_cmd->cmd));
					if (pw2wire->cmdIndex == pw2wire->cmdIndexLast) { //Last command
						//Finish writing
						state = STATE_CHECK;
						startwait_ns = get_time_ns();
						break;
					}
					//If received NACK, then stop the communication
					if (pw2wire->interrupt_status & INT_STATUS_NACK) {
						state = STATE_CHECK;
						break;
					}
				}
			}
			//Check if error happened
			if (pw2wire->interrupt_status & INT_STATUS_ERR)
				state = STATE_CHECK; //Error status is set in handler
			//If received NACK, then stop the communication
			else if (pw2wire->interrupt_status & INT_STATUS_NACK)
				state = STATE_CHECK;
			break;
		case STATE_CHECK:     //Waiting for result
			//Check if error happened
			if (pw2wire->interrupt_status & INT_STATUS_ERR) {
				dev_dbg(pw2wire->dev, "Error happened.\n");
				ret = -EIO;
				loop = 0;   //Error status is set in handler
				break;
			}
			// If received NACK, then force to send STOP.
			if (pw2wire->interrupt_status & INT_STATUS_NACK) {
				dev_dbg(pw2wire->dev, "Got NACK\n");
				current_cmd->cmd = CMD_STOP;
				current_cmd->data = 0;
			       ret = -EIO;
#if defined(AUTO_HOLD)
			       TW_V1_MCTL_SET1(&pw2wire->blkinfo,MWFRST(0x1));
			       TW_V1_MCTL_SET1(&pw2wire->blkinfo,MRFRST(0x1));
				//Send stop command
				dev_dbg(pw2wire->dev, "TX (%d, %d)\n",
					current_cmd->cmd, current_cmd->data);
				TW_V1_MWFIFO_SET(&pw2wire->blkinfo,
						WDATA(current_cmd->data),
						CMD(current_cmd->cmd));
#endif
			}
			//Check if last command is STOP
			if (current_cmd->cmd == CMD_STOP) {

				if (pw2wire->interrupt_status & INT_STATUS_STOP) {
					dev_dbg(pw2wire->dev, "Got STOP\n");
					loop = 0;
			       } else if (pw2wire->interrupt_status & INT_STATUS_NACK) {
				       dev_dbg(pw2wire->dev, "After NACK, not stop\n");
				       loop = 0;
				} else {
					while (!is_timeout(startwait_ns, timeout_us * 1000ULL)) {
						pw2wire_handler(pw2wire);
						if (INT_STATUS_READ_FIN & pw2wire->interrupt_status)
							break;
					}

					//Wait until STOP appears

					if (is_timeout(startwait_ns, timeout_us * 1000ULL) &&
					    (pw2wire->interrupt_status & INT_STATUS_STOP) == 0)
					{
						loop = 0;
						ret = -ETIMEDOUT;
						dev_dbg(pw2wire->dev, "Time out Can't get STOP\n");
					}
				}
			} else {
				//If last command is not a STOP, continue to process next message packet
				loop = 0;
			}
			break;
		}
	}
	pw2wire_disable_interrupt(pw2wire);
	return ret;
}

static void set_cmdbuf(struct pw2wire *pw2wire,
		       unsigned char cmd, unsigned char data)
{
	struct pw2wire_command *cmdbuf;
	int i;

	i = pw2wire->cmdIndexLast;
	cmdbuf = pw2wire->cmdbuf;

	cmdbuf[i].cmd  = cmd;
	cmdbuf[i].data = data;
	pw2wire->cmdIndexLast++;
}

static void pw2wire_parse_message(struct pw2wire *pw2wire,
				  struct i2c_msg *pmsg, int lastflg)
{
	unsigned char addr;
	unsigned short i;
	unsigned char *buf;

	//Check START/RESTART condition
	if ((pmsg->flags & I2C_M_NOSTART) != I2C_M_NOSTART) {
		//Set START
		set_cmdbuf(pw2wire, CMD_START, 0);

		//Set ADDR
		if (pmsg->flags & I2C_M_TEN) {
			//10 bit address
			addr = 0xf0 | ((pmsg->addr >> 7) & 0x03);
			set_cmdbuf(pw2wire, CMD_WRITE, addr);

			set_cmdbuf(pw2wire, CMD_WRITE, pmsg->addr & 0x7f);

			if (pmsg->flags & I2C_M_RD) {
				addr |= 0x01;

				//Set START
				set_cmdbuf(pw2wire, CMD_START, 0);

				//Set address
				set_cmdbuf(pw2wire, CMD_WRITE, addr);
			}
		} else {
			//7 bit address
			addr = pmsg->addr << 1;
			if (pmsg->flags & I2C_M_RD)
				addr |= 0x01;

			if (pmsg->flags & I2C_M_REV_DIR_ADDR)
				addr ^= 1;
			//Set address
			set_cmdbuf(pw2wire, CMD_WRITE, addr);
		}
	}

	//Set read or write data
	if(pmsg->flags & I2C_M_RD) {
		//Set read command
		for (i = 0; i < pmsg->len; i++)
			set_cmdbuf(pw2wire, CMD_READ, 0);
		//ACK and STOP
		if (lastflg) {
			if (!(pmsg->flags & I2C_M_NO_RD_ACK)) {
				if (pmsg->flags & I2C_M_FORCE_ACK)
					set_cmdbuf(pw2wire, CMD_SNDACK, 0); //Force to send ACK
				else
					set_cmdbuf(pw2wire, CMD_SNDACK, 1);  //NACK
			}

			if (!( pmsg->flags & I2C_M_NO_STOP))
				set_cmdbuf(pw2wire, CMD_STOP, 0);
		} else
			set_cmdbuf(pw2wire, CMD_SNDACK, 0);  //ACK
		//Set recv buffer for interrupt handler
		pw2wire->recvbuf   = pmsg->buf;
		pw2wire->recvcount = 0;
		pw2wire->recvmax   = pmsg->len;
		pw2wire->readcmdcnt = 0;
	} else {
		buf = pmsg->buf;

		//Set write command
		for (i = 0; i < pmsg->len; i++) {
			set_cmdbuf(pw2wire, CMD_WRITE, *buf);
			buf++;
		}
		//STOP
		if (lastflg) {
			if (!( pmsg->flags & I2C_M_NO_STOP))
				set_cmdbuf(pw2wire, CMD_STOP, 0);
		}
	}
}


static int pw2wire_xfer_msg(struct pw2wire *pw2wire,
			    struct i2c_msg *msg, int last)
{
	int ret;

	pw2wire->interrupt_status = INT_STATUS_NOERR;

	// Allocate command buffer for message
	pw2wire->cmdIndexMax = msg->len + 8;
	pw2wire->cmdIndex = 0;
	pw2wire->cmdIndexLast = 0;
	pw2wire->cmdbuf = kzalloc(pw2wire->cmdIndexMax *
				  sizeof(struct pw2wire_command), GFP_KERNEL);
	if (!pw2wire->cmdbuf)
		return -ENOMEM;

	// Parse input message into FIFO command
	pw2wire_parse_message(pw2wire, msg, last);

	// Process FIFO command
	ret = pw2wire_process_fifo(pw2wire, msg->flags, 10000);

	// Release cmd buffer
	kfree(pw2wire->cmdbuf);

	if (ret)
		// Failed
		pw2wire_master_reset(pw2wire);

	return ret;
}


static int pw2wire_xfer(struct i2c_adapter *adapter,
			   struct i2c_msg *msgs, int num)
{
	struct pw2wire *pw2wire = to_pw2wire(adapter);
	int i, ret;

	if (!num)
		return 0;

	// Check first msg only. Don't initialize H/W if special flags are set.
	if (!(msgs[0].flags & (I2C_M_NO_STOP | I2C_M_FORCE_ACK | I2C_M_NO_RD_ACK)))
		pw2wire_master_init(pw2wire);

	for (i = 0; i < num; i++) {
		ret = pw2wire_xfer_msg(pw2wire, &msgs[i], (i + 1) == num);
		if (ret) {
			dev_dbg(pw2wire->dev, "returning error %d\n", ret);
			return ret;
		}
	}
	dev_dbg(pw2wire->dev, "returning OK %d\n", num);
	return num; // Number of message sent
}


static void pw2wire_handler(struct pw2wire *pw2wire)
{
	unsigned long interrupt_status;
	unsigned char read_fifo_len;
	unsigned char read_cnt;

	TW_V1_INTSTAT_GET(&pw2wire->blkinfo, ALL(interrupt_status));
	dev_dbg(pw2wire->dev, "interrupt_status 0x%08lx\n", interrupt_status);
	//Check error
	if( interrupt_status & TW_V1_INTSTAT_m_STOP() )
	{
		pw2wire->interrupt_status |= INT_STATUS_STOP;
		TW_V1_INTCLR_SET(&pw2wire->blkinfo, STOP(0x1));
	}
	if (interrupt_status & TW_V1_INTSTAT_m_MARBLOST() ||
	    interrupt_status & TW_V1_INTSTAT_m_MILLCMD()) {
		pw2wire->interrupt_status |= INT_STATUS_ERR;
		if (interrupt_status & TW_V1_INTSTAT_m_MARBLOST())
			TW_V1_INTCLR_SET(&pw2wire->blkinfo, MARBLOST(0x1));

		if (interrupt_status & TW_V1_INTSTAT_m_MILLCMD())
			TW_V1_INTCLR_SET(&pw2wire->blkinfo, MILLCMD(0x1));
	}
	if (interrupt_status & TW_V1_INTSTAT_m_MNACK()) {
	       // reset read and write FIFO
	       TW_V1_MCTL_SET1(&pw2wire->blkinfo,MWFRST(0x1));
	       TW_V1_MCTL_SET1(&pw2wire->blkinfo,MRFRST(0x1));
	       pw2wire->interrupt_status |= INT_STATUS_NACK;
	       if (interrupt_status & TW_V1_INTSTAT_m_MNACK())
		       TW_V1_INTCLR_SET(&pw2wire->blkinfo, MNACK(0x1));
	}
	if (interrupt_status & TW_V1_INTSTAT_m_MBYCDONE())
		TW_V1_INTCLR_SET(&pw2wire->blkinfo, MBYCDONE(0x1));
	if (interrupt_status & TW_V1_INTSTAT_m_MBYRCVD()) {
		pw2wire->interrupt_status |= INT_STATUS_READ_FIN;
		TW_V1_INTCLR_SET(&pw2wire->blkinfo, MBYRCVD(0x1));
		TW_V1_MFIFOSTAT_GET1(&pw2wire->blkinfo,RFNE(read_fifo_len));
		for (read_cnt = 0; read_cnt < read_fifo_len; read_cnt++) {
			//Save the byte
			if (pw2wire->recvcount < pw2wire->recvmax) {
				TW_V1_MRFIFO_GET(&pw2wire->blkinfo, ALL(pw2wire->recvbuf[pw2wire->recvcount]));
				pw2wire->recvcount++;
				pw2wire->readcmdcnt--;
			}
		}
	}

	// Clear all interrupt status
	TW_V1_INTCLR_SET(&pw2wire->blkinfo,
			 MRFLIM(0x1),
			 MWFLIM(0x1),
			 MHOLD(0x1),
			 SNACK(0x1),
			 SBYRCVD(0x1),
			 SWRREQ(0x1),
			 SRDREQ(0x1),
			 SRFLIM(0x1),
			 SWFLIM(0x1),
			 SWFLOCKED(0x1),
			 SGENCALL(0x1),
			 SHOLD(0x1));
}


static int pw2wire_probe(struct device_d *dev)
{
	struct pw2wire *pw2wire;
	struct i2c_adapter *adapter = NULL;
	struct resource *res;
	int result = 0;
	u32 speed_hz = (u32)dev->platform_data;

	printk(KERN_INFO "pw2wire: %s%d, clock %uHz\n",
	       dev->name, dev->id, speed_hz);

	pw2wire = xzalloc(sizeof(struct pw2wire));
	pw2wire->dev = dev;
	res = dev_request_mem_resource(dev, 0);
	if (IS_ERR(res)) {
		dev_err(dev, "pw2wire_probe - no memory resource.\n");
		return PTR_ERR(res);
	}
	pw2wire->regs = IOMEM(res->start);
	BlockInfoInit(&pw2wire->blkinfo, (u32)pw2wire->regs);

	pw2wire->clock_mode = get_clock_mode(speed_hz);

	pw2wire_master_reset(pw2wire);
	pw2wire_master_enable(pw2wire);
	TW_V1_CRYSTAL_FREQ_SET1(&pw2wire->blkinfo,XTAL_TYPE(pw2wire_xtal_type));
	TW_V1_CRYSTAL_FREQ_SET1(&pw2wire->blkinfo,TIMEOUT_EN(pw2wire_timeout_en));

	adapter = &pw2wire->adapter;
	adapter->master_xfer = pw2wire_xfer;
	adapter->dev.parent = dev;
	adapter->dev.device_node = dev->device_node;
	adapter->nr = dev->id;

	result = i2c_add_numbered_adapter(adapter);
	if (result) {
		dev_err(dev, "pw2wire_probe - failed to add adapter.\n");
		return result;
	}

	dev_info(dev, "hw 2wire probe OK.\n");
	return 0;
}

static struct driver_d pw2wire_driver = {
	.name = "pw2wire",
	.probe = pw2wire_probe,
};
device_platform_driver(pw2wire_driver);


MODULE_AUTHOR("Pixelworks, Inc.");
MODULE_DESCRIPTION("Pixelworks two_wire master driver");
MODULE_LICENSE("GPL");
