/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <init.h>
#include <io.h>
#include <restart.h>
#include <asm/mmu.h>
#include <linux/sizes.h>
#include <linux/amba/bus.h>
#include <mach/sysmap.h>
#include <mach/ipc_wdog.h>
#include <platform_data/eth-designware.h>


void (*topazeh_reset_hook)(void);

static void topazeh_reset(struct restart_handler *rst)
{
	if (topazeh_reset_hook)
		topazeh_reset_hook();

	writel(1, WDOG_BASE + IPC_WDOG_LOAD);
	writel(IPC_WDOG_UNLOCK, WDOG_BASE + IPC_WDOG_LOCK);
	writel(IPC_WDOG_CONTROL_ENABLE | IPC_WDOG_CONTROL_RESEN,
	       WDOG_BASE + IPC_WDOG_CONTROL);
	mdelay(100);
	hang();
}

static int register_reset_handler(void)
{
	restart_handler_register_fn(topazeh_reset);
	return 0;
}
coredevice_initcall(register_reset_handler);

static struct dwc_ether_platform_data topazeh_eth_data = {
	.phy_addr = -1,         /* scan for PHY */
	.interface = PHY_INTERFACE_MODE_RGMII,
};

static void usb_setup(void)
{
	unsigned int p;
	unsigned int regs;

	// release usb host 0/1 phy reset
	p = SYS_V20_BASE + 0x430;
	regs = readl(p) & 0xfffffffc;
	writel(regs, p);

	// usb3 phy ref clk on
	p = SYS_V20_BASE + 0x10;
	regs = readl(p) & 0xfffffffe;
	writel(regs, p);

	// USBPWR0/1 pad configure output
	p = SYS_V20_BASE + 0x570;
	regs = readl(p) & 0xfffffffc;
	writel(regs, p);

	// USBPWR0/1 connect to pad
	p = USB_CONTROL_V1_BASE + 0x10;
	regs = readl(p) | 0x00000c00;
#if !defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)	
	writel(regs, p);
#endif

	// assert usb3 phy reset
	p = SYS_V20_BASE + 0x104;
	regs = readl(p) | 0x00004000;
	writel(regs, p);

	// choose internal ref clk
	p = USB_CONTROL_V1_BASE + 0x20;
	regs = readl(p) & 0xfffffeff;
	writel(regs, p);

	// wait 100ms
	mdelay(100);

	// release usb3 phy reset
	p = SYS_V20_BASE + 0x104;
	regs = readl(p) & 0xffffbfff;
	writel(regs, p);

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)	
	// change the USB2 PHY0/1 disconnect threshold
	p = USB_CONTROL_V1_BASE + 0x08;
	regs = readl(p) | 0x00003800;
	writel(regs, p);
	p = USB_CONTROL_V1_BASE + 0x0c;
	regs = readl(p) | 0x00003800;
	writel(regs, p);
	
	//fix USB device disconnected status issue
	p = USB_CONTROL_V1_BASE + 0x14;
	regs = readl(p) & 0xbfffff7f;
	writel(regs, p);
#endif
}

static int topazeh_devices_init(void)
{
	add_generic_device("designware_eth", 0, NULL, ETH_MAC_V4_BASE,
			   SZ_8K, IORESOURCE_MEM, &topazeh_eth_data);
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)
	add_generic_device("dwc3", 0, NULL, USB3_DEVICE_V2_BASE + 0xc100,
			   SZ_64K, IORESOURCE_MEM, NULL);
#endif
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
	add_generic_device("dwc3", 0, NULL, USB3_DEVICE_V2_BASE + 0xc100,
			   SZ_64K, IORESOURCE_MEM, NULL);
#endif
	usb_setup();
	return 0;
}
device_initcall(topazeh_devices_init);

static int topazeh_console_init(void)
{
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
	barebox_set_model("Pixelworks TopazEH");
	barebox_set_hostname("topazeh");
	amba_apb_device_add(NULL, "uart-pl011", 0, UART0_BASE, 4096, NULL, 0);
#endif

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)
	barebox_set_model("Pixelworks Topaz Prime");
	barebox_set_hostname("topaz prime");
	amba_apb_device_add(NULL, "uart-pl011", 0, UART0_BASE, 4096, NULL, 0);
#endif	
	return 0;
}
console_initcall(topazeh_console_init);

static int topazeh_l2_cache_init(void)
{
	writel(0x00000111, 0xffc00108); // TAG RAM latencies, 2 cycles
	writel(0x00000111, 0xffc0010c); // DATA RAM latencies, 2 cycles
	l2x0_init((void __iomem *)0xffc00000, 0x00000000, 0xffffffff);
	return 0;
}
postmmu_initcall(topazeh_l2_cache_init);
