/*
 * simple SPI flash access to copy the bootloader into RAM
 *
 * Copyright (C) 2009-2017 Pixelworks
 */

#include <linux/compiler.h>
#include <io.h>

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH) || defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)
#include <mach/sysmap.h>
#define SPI_V_SRAM_SIZE 0x800		// 2 KB
#define SPI_V_BASE SPI0_V3_BASE
#else
#error unknow platform
#endif

#define SPI_V_CR_OFF	0x00
#define SPI_V_TC_OFF	0x04
#define SPI_V_RC_OFF	0x08
#define SPI_V_IEN_OFF	0x0c
#define SPI_V_ST_OFF	0x10
#define SPI_V_ICLR_OFF	0x14
#define SPI_V_SS_OFF	0x18
#define SPI_V_CT_OFF	0x1c
#define SPI_V_TCD_OFF	0x20
#define SPI_V_RCD_OFF	0x24
#define SPI_V_SSD_OFF	0x28
#define SPI_V_TDC_OFF	0x30
#define SPI_V_RDC_OFF	0x34
#define SPI_V_TDF_OFF	0x38
#define SPI_V_RDF_OFF	0x3c

/* symbols defined by linker script */
extern char _stext;
extern char _barebox_image_size;
extern char _pre_boot_size;
extern char _itcm_boot_size;

extern char _pre_boot_itcm_addr;
extern char _pre_boot_dtcm_addr;
extern char _itcm_boot_len;
extern char _itcm_data_len;

__section(".pre_boot.c")
static inline void spi_write(unsigned long base, int off, unsigned int val)
{
	writel(val, base + off);
}

__section(".pre_boot.c")
static inline unsigned int spi_read(unsigned long base, int off)
{
	return *(volatile unsigned int *)(base + off);
}

#define SPI_OFF(reg) SPI_V_ ## reg ## _OFF
#define SPI_WRITE(reg, val) spi_write(SPI_V_BASE, SPI_OFF(reg), val)
#define SPI_READ(reg) spi_read(SPI_V_BASE, SPI_OFF(reg))

__section(".pre_boot.c")
void spi_init_f(void)
{
	// use sw-SS, SCLK pull-up, push-pull outputs, phase 1, polarity 1, master mode
	SPI_WRITE(CR, (1 << 6) | (1 << 8) | (3 << 20) | (1 << 24));
	// resend 0, SPI word lenght 8, MSB first, irq threshold = 31, rx data threshold = 3
	SPI_WRITE(RC, (3 << 26) | (31 << 20) | (1 << 12) | 8);
	// resend 0, SPI word lenght 8, MSB first, irq threshold = 31
	SPI_WRITE(TC, (31 << 20) | (1 << 12) | 8);
	SPI_WRITE(CT, 4);
	SPI_WRITE(RCD, 0);
	SPI_WRITE(TCD, 0);
	SPI_WRITE(SSD, 0);
	SPI_WRITE(SS, 0);
	SPI_WRITE(IEN, 0);
	SPI_WRITE(ICLR, ~0);
}

__section(".pre_boot.c")
static void spi_read_block(unsigned long dest, unsigned long offset, unsigned long size)
{
	unsigned long i;
	unsigned int addr_bytes;
	union {
		unsigned char bytes[5];
		unsigned int  words[2];
	} spicmd;

	/* after hw boot, we can read the TDC register to
	 * figure out if to use 3byte or 4byte addres mode
	 * (found by experiment, it would be better to have this
	 * information is a SYS register)
	 */
	addr_bytes = SPI_READ(TDC);
	if (addr_bytes != 4 && addr_bytes != 5)
		addr_bytes = 4;  // safe fallback, just in case...

	// prepare SPI read command (0x03) for address "offset"
	if (addr_bytes == 4)
		offset <<= 8;
	spicmd.bytes[0] = 0x03;
	spicmd.bytes[1] = (offset >> 24) & 0xff;
	spicmd.bytes[2] = (offset >> 16) & 0xff;
	spicmd.bytes[3] = (offset >> 8) & 0xff;
	spicmd.bytes[4] = offset & 0xff;

	spi_init_f();

	SPI_WRITE(TDC, addr_bytes); // 5 8-bit SPI words
	SPI_WRITE(RDC, (addr_bytes << 24) | size);

	SPI_WRITE(SS, 1);
	SPI_WRITE(CR, SPI_READ(CR) | (1 << 31)); // set ENable

	// send read command
	SPI_WRITE(TDF, spicmd.words[0]);
	if (addr_bytes == 5)
		SPI_WRITE(TDF, spicmd.bytes[4]);
	// change rx data threshold = 2
	SPI_WRITE(RC, (2 << 26) | (31 << 20) | (1 << 12) | 8);

	for (i = 0; i < size; i += 4) {
		while (!(SPI_READ(ST) & (1 << 5))) // check Receive Data FIFO Full
			;
		writel(SPI_READ(RDF), dest + i);
	}
	SPI_WRITE(CR, SPI_READ(CR) & ~(1 << 31)); // clear ENable
	SPI_WRITE(SS, 0);
}

__section(".pre_boot.c")
void spi_cp_boot(void)
{
	// clear boot bit to remove SPI SRAM and move SPI regs to SPI_V_BASE
	spi_write(SPI_V_BASE + SPI_V_SRAM_SIZE, SPI_OFF(CR), 0);

	spi_read_block((unsigned long)&_stext,
		       (unsigned long)&_pre_boot_size + (unsigned long)&_itcm_boot_size,
		       (unsigned long)&_barebox_image_size);
}

__section(".pre_boot.c")
void spi_cp_itcm_boot(void)
{
	// clear boot bit to remove SPI SRAM and move SPI regs to SPI_V_BASE
	spi_write(SPI_V_BASE + SPI_V_SRAM_SIZE, SPI_OFF(CR), 0);

	spi_read_block((unsigned long)&_pre_boot_itcm_addr + (unsigned long)&_pre_boot_size,
		       (unsigned long)&_pre_boot_size,
		       (unsigned long)&_itcm_boot_len);

	spi_read_block((unsigned long)&_pre_boot_dtcm_addr,
		       (unsigned long)&_pre_boot_size + (unsigned long)&_itcm_boot_len,
		       (unsigned long)&_itcm_data_len);
}
