/* -------------------------------------------------------------------------- *\
   itcm_serial_pl011.c - UART console control functions.
   Copyright (c) 2007,2008 Pixelworks Inc.
   Pixelworks owns the sole copyright to this software. Under international
   copyright laws you (1) may not make a copy of this software except for
   the purposes of maintaining a single archive copy, (2) may not derive
   works herefrom, (3) may not distribute this work to others. These rights
   are provided for information clarification, other restrictions of rights
   may apply as well.
   --------------------------------------------------------------------------
   This file is using to control UART function.
   Simple U-Boot driver for the PrimeCell PL011 UARTs on the IntegratorCP
   Should be fairly simple to make it work with the PL010 as well
\* -------------------------------------------------------------------------- */


#include <linux/compiler.h>
#include <io.h>
#include "itcm_serial_pl011.h"

#define CONFIG_XCLK_HZ 24000000

#define CONSOLE_PORT 0
#define baudRate CONFIG_BAUDRATE
static unsigned char *const itcm_serial[2] = { (void *) 0xfd800000 }; /*UART0*/

#define NUM_PORTS (sizeof(itcm_serial)/sizeof(itcm_serial[0]))

static void itcm_pl011_putc(int portnum, char c);
static int itcm_pl011_getc(int portnum);


/* -------------------------------------------------------------------------- *\
Function Name : itcm_serial_init
Parameter : slowclk
Return : None
Description : This function is used to initial console UART port. The setting is
	      115200-N-8-1 & FIFO Disable, receive Interrupt enable
\* -------------------------------------------------------------------------- */
int itcm_serial_init(int slowclk)
{
	unsigned int temp;
	unsigned int divider;
	unsigned int fraction;

	/*
	 ** First, disable everything.
	 */
	writel(0x0, itcm_serial[CONSOLE_PORT] + UART_PL011_CR);

	/*
	 * Set baud rate
	 * divider = UART_CLK / (16 * BAUD_RATE)
	 * fraction =
	 *  ROUND((64 * MOD(UART_CLK,(16 * BAUD_RATE))) / (16 * BAUD_RATE))
	 */
	/*
	 * TODO: When BAUD_RATE = 9600, UART_CLK = 1500000,
	 * program will call __aeabi_uidiv then runs fail.
	 * So configure the parameters from calculated result directly.
	 */
	if (slowclk) {
		/*BAUD_RATE = 9600; UART_CLK = 1500000(1.5MHz)*/
		divider = 9;
		fraction = 49;
	} else {
		/*BAUD_RATE = 115200; UART_CLK = 24000000(24MHz)*/
		divider = 13;
		fraction = 1;
	}

	writel(divider, itcm_serial[CONSOLE_PORT] + UART_PL011_IBRD);
	writel(fraction, itcm_serial[CONSOLE_PORT] + UART_PL011_FBRD);

	temp = readl(itcm_serial[CONSOLE_PORT] + UART_PL011_IMSC);
	temp = temp | UART_PL011_IMSC_RXIM;
	writel(temp, itcm_serial[CONSOLE_PORT] + UART_PL011_IMSC);

	writel((0x04 << 3), itcm_serial[CONSOLE_PORT] + UART_PL01x_IFLS);
	/*
	 ** Set the UART to be 8 bits, 1 stop bit, no parity, fifo disabled.
	 */
	writel((UART_PL011_LCRH_WLEN_8 /*| UART_PL011_LCRH_FEN*/),
		itcm_serial[CONSOLE_PORT] + UART_PL011_LCRH);

	/*
	 ** Finally, enable the UART
	 */
	writel((UART_PL011_CR_UARTEN | UART_PL011_CR_TXE | UART_PL011_CR_RXE),
		itcm_serial[CONSOLE_PORT] + UART_PL011_CR);

	return 0;
}

/* -------------------------------------------------------------------------- *\
Function Name : itcm_serial_putc
Parameter : const char c -- the data that will be output by console UART.
Return : None
Description : This function is used to put a data to console UART port.
\* -------------------------------------------------------------------------- */
void itcm_serial_putc(char c)
{
	if (c == '\n')
		itcm_pl011_putc(CONSOLE_PORT, '\r');

	itcm_pl011_putc(CONSOLE_PORT, c);
}

/* print 32bit value in hex */
void itcm_serial_puthex(unsigned int v)
{
	static const char hex[] = "0123456789abcdef";
	int i;

	for (i = 7; i >= 0; i--)
		itcm_pl011_putc(CONSOLE_PORT, hex[(v >> (4*i)) & 0x0f]);
}

/* -------------------------------------------------------------------------- *\
Function Name : itcm_serial_puts
Parameter : const char *s -- char strings that will be output by console UART.
Return : None
Description : This function is used to put data string to console UART port.
\* -------------------------------------------------------------------------- */
void itcm_serial_puts(char *s)
{
	while (s && *s)
		itcm_serial_putc(*s++);
}

/* -------------------------------------------------------------------------- *\
Function Name : itcm_serial_getc
Parameter : None
Return : Received Console port's Data
Description : This function is used to get console port's data.
\* -------------------------------------------------------------------------- */
int itcm_serial_getc(void)
{
	return itcm_pl011_getc(CONSOLE_PORT);
}


/* -------------------------------------------------------------------------- *\
Function Name : itcm_pl011_putc
Parameter : int portnum, 0 -- UART0, 1 -- UART1
	    char c -- Will be put to UART port.
Return : None
Description : This function is used to put data to UART port.
	      This function is a local function.
\* -------------------------------------------------------------------------- */
static void itcm_pl011_putc(int portnum, char c)
{
	/* Wait until there is space in the FIFO */
	while (readl(itcm_serial[portnum] + UART_PL01x_FR)
		& UART_PL01x_FR_TXFF)
		;
	/* Send the character */
	writel(c, itcm_serial[portnum] + UART_PL01x_DR);
}

/* -------------------------------------------------------------------------- *\
Function Name : itcm_pl011_getc
Parameter : int portnum, 0 -- UART0, 1 -- UART1
Return : Received Data
Description : This function is used to get received data.
	      This function is a local function.
\* -------------------------------------------------------------------------- */
static int itcm_pl011_getc(int portnum)
{
	unsigned int data;

	/* Wait until there is data in the FIFO */
	while (readl(itcm_serial[portnum] + UART_PL01x_FR)
		& UART_PL01x_FR_RXFE)
		;

	data = readl(itcm_serial[portnum] + UART_PL01x_DR);

	/* Check for an error flag */
	if (data & 0xFFFFFF00) {
		/* Clear the error */
		writel(0xFFFFFFFF, itcm_serial[portnum] + UART_PL01x_ECR);
		return -1;
	}

	return (int) data;
}

/* -------------------------------------------------------------------------- *\
Function Name : itcm_pl011_serial_Reenable
Parameter : None
Return : 0 -- No Interrupt happen, 1-- Interrupt happen
Description : This function is used for re-enable UART's interrupt.
\* -------------------------------------------------------------------------- */
void itcm_pl011_serial_Reenable(int portnum)
{
	unsigned int temp;
	temp = readl(itcm_serial[portnum] + UART_PL011_IMSC);
	temp = temp | UART_PL011_IMSC_RXIM;

	writel(temp, itcm_serial[portnum] + UART_PL011_IMSC);
	writel(temp, itcm_serial[portnum] + UART_PL010_ICR);
}


