/* -------------------------------------------------------------------------- *\
   itcm_power.c - power control file.
   Copyright (c) 2007,2008 Pixelworks Inc.
   Pixelworks owns the sole copyright to this software. Under international
   copyright laws you (1) may not make a copy of this software except for
   the purposes of maintaining a single archive copy, (2) may not derive
   works herefrom, (3) may not distribute this work to others. These rights
   are provided for information clarification, other restrictions of rights
   may apply as well.
   --------------------------------------------------------------------------
   This file is using to control POWER ON/OFF function.
\* -------------------------------------------------------------------------- */
#include <common.h>
#include <io.h>
#include "itcm_interrupts.h"
#include "itcm_serial_pl011.h"
#include "itcm_ir.h"
#include "itcm_power.h"

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME) || defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH) 
/* GPIO pins for power supply control using PORTC (GPIO2) */
#define GPIO_POWER_SW_BASE GPIO2_V2_BASE
#define GPIO_P1V2_SW 3
#define GPIO_P1V5_SW 2
#define GPIO_P2V5_SW 1
#define GPIO_P3V3_SW 0
#define GPIO_POWER_SWITCHES ((1 << GPIO_P1V2_SW) | (1 << GPIO_P1V5_SW) \
			     | (1 << GPIO_P2V5_SW) | (1 << GPIO_P3V3_SW))

/* GPIO registers */
#define GPIO_DATA(pin) (4 << (pin))
#define GPIO_DIR  0x400

/* delay loop for ARM running at 1.5MHz xclk */
static void udelay_slow(unsigned int usec)
{
	while (usec--)
		barrier();
}
#endif

/*wait until the step is reached*/
static int wait_status(unsigned int pmu_next_state, unsigned int loop_cnt)
{
	int i = 0;
	unsigned int pmu_status = 0;
	unsigned int pmu_step_status;

	for (i = 0; i < loop_cnt; i++) {
		pmu_status = readl(PMUCTRL);
		pmu_step_status = (pmu_status >> 25) & 0xf;

		if (pmu_next_state == pmu_step_status) {
			itcm_serial_puts(" reached\n");
			return 0;
		}
	}
	itcm_serial_puts(" timeout! :");
	itcm_serial_puthex(pmu_status);
	itcm_serial_putc('\n');

	return -1;
}

/* perform one PMU request  */
static int itcm_pmu_step(unsigned int step, unsigned int request,
						unsigned char bit_assert)
{
	unsigned int pmu_status = 0;

	itcm_serial_puts("PMU step 0x");
	if (step >= 10)
		itcm_serial_putc('a' + (step-10));
	else
		itcm_serial_putc('0' + step);

	pmu_status = readl(PMUCTRL);
	if (bit_assert)
		writel(pmu_status | request, PMUCTRL);
	else
		writel(pmu_status & (~request), PMUCTRL);
	return wait_status(step, 200);
}

/* -------------------------------------------------------------------------- *\
Function Name : itcm_power_down_seq
Parameter : None
Return : None
Description : The function is used for control power off sequence.
\* -------------------------------------------------------------------------- */
void itcm_power_down_seq(void)
{
	unsigned int pmu_status;
	unsigned int pmu_step_status;
	unsigned int data;

	/*Select VD3DMIF clock to XTAL firstly*/
	data = readl(_SYS_V20_BASE + 0x18);
	data |= (1<<4); /*VD3DMIFCLK_XTALSEL*/
	writel(data, _SYS_V20_BASE + 0x18);

	/*confirm STEP == 0 (NORMAL) */
	pmu_status = readl(PMUCTRL);
	pmu_step_status = (pmu_status >> 25) & 0xf;
	if (pmu_step_status != 0) {
		itcm_serial_puts("error: PMU step status != 0: ");
		itcm_serial_puthex(pmu_status);
		itcm_serial_putc('\n');
		return;
	}

	/*STEP 1 4'b0001: MI_READY */
	itcm_pmu_step(1, 1 << 19, 1);

	/*STEP 2 4'b0011: CLK_GATED */
	itcm_pmu_step(3, 1 << 20, 1);

	/*STEP 3 4'b0010: RESETED OPRST */
	itcm_pmu_step(2, 1 << 21, 1);

	/*STEP 4 4'b0110: PRE_ISOEN */
	itcm_pmu_step(6, 1 << 17, 1);

	/*STEP 5 4'b0111: POWERGATED */
	itcm_pmu_step(7, 1 << 16, 1);

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME) || defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH) 
	/*Slow clocks for CPU and MI and others.
	 *Switch ARM to 3MHz xclk, l2_cache = 1.5MHz, vic = 1.5MHz, MI = 24MHz
	 *NOTE: if using JTAG you must reduce the JTAG clock
	 *otherwise JTAG won't work and the system may hang!
	 */

	/*Select source to XTAL*/
	data = readl(_SYS_V20_BASE + 0x14);
	/*ARMCLK_XTALSEL, PWM2CLK_SRCSEL, PWM1CLK_SRCSEL, PWM0CLK_SRCSEL,
	 *KEYCLK_XTALSEL, MCLK_XCLKSEL
	 */
	data |= (0xf1 | 1 << 29);
	writel(data, _SYS_V20_BASE + 0x14);

	/*Set divider number*/
	data = 0x00000070;  /*ARMCLKDIV_DNUM = 7*/
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
	writel(data, _SYS_V20_BASE + 0x24);
#endif
	/*PWM0CLKDIV_DNUM, PWM0CLKDIV_DNUM, PWM0CLKDIV_DNUM = 0xf*/
	data = 0x000f0f0f;
	writel(data, _SYS_V20_BASE + 0x28);
	/*STATICREGCLKDIV_DNUM, PIALWAYSONCLKDIV_DNUM = 0xf*/
	data = 0x000ff000;
	writel(data, _SYS_V20_BASE + 0x30);
	/*IRCLKDIV_DNUM, UARTCLKDIV_DNUM, KEYCLKDIV_DNUM = 0xf*/
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)	
#if ENABLE_UART_LOWPOWER
	data = 0x000f00ff; /*uart clk -- XTAL/16*/
#else
	data = 0x000f000f; /*uart clk -- XTAL*/
#endif
#endif
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)	
	/*IRCLKDIV_DNUM, UARTCLKDIV_DNUM, KEYCLKDIV_DNUM = 0xf*/
	data = readl(_SYS_V20_BASE + 0x34);
#if ENABLE_UART_LOWPOWER
	data |= 0x000000ff; /*uart clk -- XTAL/16*/
#else
	data |= 0x00000000; /*uart clk -- XTAL*/
#endif
#endif
	writel(data, _SYS_V20_BASE + 0x34);
	data = 0x0f000f00; /*IR4CLKDIV_DNUM, TWOWCLKDIV_DNUM = 0xf*/
	writel(data, _SYS_V20_BASE + 0x38);

	/*Update new divider value. 1)Set En=1; 2) Set En=0;*/
	/*IR4CLKDIV_EN, COUNTERCLKDIV_EN, UARTCLKDIV_EN, IRCLKDIV_EN,
	 *PIALWAYSONCLKDIV_EN, STATICREGCLKDIV_EN, PWM2CLKDIV_EN,
	 *PWM1CLKDIV_EN, PWM0CLKDIV_EN, ARMCLKDIV_EN
	 */
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)
	data = 0x226c01c2;
	writel(data, _SYS_V20_BASE + 0x1c);
	writel(0, _SYS_V20_BASE + 0x1c);
	data = 0x2; /*TWOWCLKDIV_EN*/
	writel(data, _SYS_V20_BASE + 0x20);
	writel(0, _SYS_V20_BASE + 0x20);
#endif	
	/*Program sys to stop clocks of unnecessary blocks in always-on domain*/
	data = readl(_SYS_V20_BASE + 0);
	/*SPI0CLK_OFF, ARMCLK_OFF, SDIO0CLK_OFF*/
	data |= (1 << 5 | 1 << 20 | 1 << 22);
	writel(data, _SYS_V20_BASE + 0);

	data = readl(_SYS_V20_BASE + 0x04);
	data |= (3 << 15 | 0xb << 27); /*SADCCLK_OFF, XCLKOFF, UART2,3,5CLKOFF*/
	writel(data, _SYS_V20_BASE + 0x04);

	data = readl(_SYS_V20_BASE + 0x08);
	data |= (0xfefa << 13 | 1 << 31); /*GPIO1,3~7,9~15CLKOFF, PIFCLKOFF*/
	writel(data, _SYS_V20_BASE + 0x08);

	data = readl(_SYS_V20_BASE + 0x0c);
	/*
	 * TIMERCLKOFF, SDIO0BUSCLK_OFF, COUNTERCLK_OFF,
	 * WDTCLKOFF, TWOW2CLKOFF, TWOW3CLKOFF
	 */
	data |= (1 | 1 << 10 | 1 << 14 | 1 << 26 | 1 << 29 | 1 << 30);
	writel(data, _SYS_V20_BASE + 0x0c);

	/*MPLL POWEROFF*/
	data = readl(_SYS_V20_BASE + 0x54);
	data |= (1 << 28); /*MPLL_RESET*/
	writel(data, _SYS_V20_BASE + 0x54);
	data = readl(_SYS_V20_BASE + 0x58);
	data |= (1 << 7); /*MPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x58);
	data = readl(_SYS_V20_BASE + 0x58);
	data &= ~(1 << 7); /*MPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x58);

	/*PPLL POWEROFF*/
	data = readl(_SYS_V20_BASE + 0x60);
	data |= (1 << 28); /*PPLL_RESET*/
	writel(data, _SYS_V20_BASE + 0x60);
	data = readl(_SYS_V20_BASE + 0x64);
	data |= (1 << 7); /*PPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x64);
	data = readl(_SYS_V20_BASE + 0x64);
	data &= ~(1 << 7); /*PPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x64);

	/*DPLLA POWEROFF*/
	data = readl(_SYS_V20_BASE + 0x6c);
	data |= (1 << 28); /*DPLLA_RESET*/
	writel(data, _SYS_V20_BASE + 0x6c);
	data = readl(_SYS_V20_BASE + 0x70);
	data |= (1 << 7); /*DPLLA_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x70);
	data = readl(_SYS_V20_BASE + 0x70);
	data &= ~(1 << 7); /*DPLLA_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x70);

	/*DPLLB POWEROFF*/
	data = readl(_SYS_V20_BASE + 0x78);
	data |= (1 << 28); /*DPLLB_RESET*/
	writel(data, _SYS_V20_BASE + 0x78);
	data = readl(_SYS_V20_BASE + 0x7c);
	data |= (1 << 7); /*DPLLB_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x7c);
	data = readl(_SYS_V20_BASE + 0x7c);
	data &= ~(1 << 7); /*DPLLB_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x7c);

	/*
	 * Note: As desinged by IC, the ARMPLL can't be powered off.
	 * From the simulation result,
	 * the waveform shows the expected waveform to debug.
	 * it is found a clock-gating signal depends on PLL software reset.
	 * When the pll software reset is set, the armClk will be gated.
	 * It is a design bug.
	 * Luckily from the pll spec, there is another way to power off the PLL.
	 * It is to set PLL BYPASS2 to 1 instead of RESET.
	 */
	/*ARMPLL bypass1/2 enable and program PLL parameters*/
	writel(0x00001317, _SYS_V20_BASE + 0x80);/*ARMPLL_BYPASS1/2 = 1*/
	writel(0x07ae100e, _SYS_V20_BASE + 0x84);

	data = readl(_SYS_V20_BASE + 0x88);
	data |= (1 << 7); /*ARMPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x88);
	data = readl(_SYS_V20_BASE + 0x88);
	data &= ~(1 << 7); /*ARMPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x88);

	/*RGMIIPLL POWEROFF*/
	data = readl(_SYS_V20_BASE + 0x9c);
	data |= (1 << 28); /*RGMIIPLL_RESET*/
	writel(data, _SYS_V20_BASE + 0x9c);
	data = readl(_SYS_V20_BASE + 0x100);
	data |= (1 << 7); /*RGMIIPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x100);
	data = readl(_SYS_V20_BASE + 0x100);
	data &= ~(1 << 7); /*RGMIIPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x100);

	/*Full Power-down Sequence for Powering Off Topazeh*/
	/*first ensure all GPIO are on*/
	writel(GPIO_POWER_SWITCHES, GPIO_POWER_SW_BASE + (GPIO_POWER_SWITCHES << 2));
	/*make them output mode*/
	data = readl(GPIO_POWER_SW_BASE + GPIO_DIR);
	data |= GPIO_POWER_SWITCHES;
	writel(data, GPIO_POWER_SW_BASE + GPIO_DIR);
	/*sequence:
	 *1.5V/0.75V -> 1.2VA/1.2VD (internal and external) -> 2.5V -> 3.3V
	 */
	writel(1 << GPIO_P1V5_SW, GPIO_POWER_SW_BASE + GPIO_DATA(GPIO_P1V5_SW));
	udelay_slow(1500); /*wait until it is off, 1.5ms fall time max*/
	writel(1 << GPIO_P1V2_SW, GPIO_POWER_SW_BASE + GPIO_DATA(GPIO_P1V2_SW));
	udelay_slow(1500); /*wait until it is off, 1.5ms fall time max*/
	writel(1 << GPIO_P2V5_SW, GPIO_POWER_SW_BASE + GPIO_DATA(GPIO_P2V5_SW));
	writel(1 << GPIO_P3V3_SW, GPIO_POWER_SW_BASE + GPIO_DATA(GPIO_P3V3_SW));
#endif

	data = readl(LVDSPADCTL1);
	data |= (1<<18); /*LVDSPAD_REF_IPD[18]*/
	writel(data, LVDSPADCTL1);
	/*configure pads direction to input*/
	writel(0x0000001f, SYS_PAD_DIRCTRL2);

#if ENABLE_UART_LOWPOWER
	/*Re-configure uart parameters*/
	itcm_serial_init(1);
#endif
	itcm_serial_puts("wait for wake up\n\n");

	/*configure pads direction to input*/
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)
	writel(0x7fffffff, SYS_PAD_DIRCTRL1);
#endif
}


/* -------------------------------------------------------------------------- *\
Function Name : itcm_power_on_seq
Parameter : None
Return : None
Description : The function is used for control power on sequence.
\* -------------------------------------------------------------------------- */
void itcm_power_on_seq()
{
	int i = 0;
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME) || defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH) 
	unsigned int data = 0;
#endif

	itcm_serial_puts("start to power on seq\n");

	/*Power on seq*/
	/*
	 * PWR_CTRL_DISABLE:0
	 * -- Check the status of the external regulator,
	 * which compares the voltage level.
	 * PWR_CTRL_DISABLE:1
	 * -- Don't check the status of the external regulator,
	 * which compares the voltage level.
	 * PWR_CTRL_POL_CHANGE:0
	 * -- Don't change the polarity of the status of
	 * the external regulator.
	 * PWR_CTRL_POL_CHANGE:1
	 * -- Change the polarity of the status of
	 * the external regulator.
	*/
	data = readl(PMUCTRL2);
	data |= (1<<12); /*sys->PMUCTRL2-->PWR_CTRL_DISABLE */
	writel(data, PMUCTRL2);
	/*
	 * PMU_PLLISACTIVE:0
	 * -- PMU state transition depends on PLL stable status.
	 * PMU_PLLISACTIVE:1
	 * -- PMU state transition has no dependency on
	 * PLL stable status.
	*/
	data = readl(PMUCTRL);
	data |= (1<<22); /*sys->PMUCTRL1-->PMU_PLLISACTIVE */
	writel(data, PMUCTRL);

	data = readl(LVDSPADCTL1);
	data &= ~(1<<18); /*LVDSPAD_REF_IPD[18]*/
	writel(data, LVDSPADCTL1);

	/*configure pads direction to output*/
	writel(0, SYS_PAD_DIRCTRL1);
	writel(0, SYS_PAD_DIRCTRL2);

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME) || defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH) 
	/*Powering Up from Deep Standby Operation*/
	/*sequence:
	 *3.3V -> 2.5V -> 1.2VA/1.2VD (internal and external) -> 1.5V/0.75V
	 */
	writel(0, GPIO_POWER_SW_BASE + GPIO_DATA(GPIO_P3V3_SW));
	udelay_slow(200); /*arbitrary delay*/
	writel(0, GPIO_POWER_SW_BASE + GPIO_DATA(GPIO_P2V5_SW));
	udelay_slow(400); /*arbitrary delay + min 0.2ms delay*/
	writel(0, GPIO_POWER_SW_BASE + GPIO_DATA(GPIO_P1V2_SW));
	udelay_slow(200); /*arbitrary delay*/
	writel(0, GPIO_POWER_SW_BASE + GPIO_DATA(GPIO_P1V5_SW));
	udelay_slow(1500); /*1.5ms rise time max*/
	/*revert GPIOs back to input mode*/
	data = readl(GPIO_POWER_SW_BASE + GPIO_DIR);
	data &= ~GPIO_POWER_SWITCHES;
	writel(data, GPIO_POWER_SW_BASE + GPIO_DIR);

	/*MPLL POWERON*/
	/*MPLL CTRL setting has been configed outside, no need to set here.*/
	data = readl(_SYS_V20_BASE + 0x54);
	data &= ~(1 << 28); /*MPLL_RESET*/
	writel(data, _SYS_V20_BASE + 0x54);
	data = readl(_SYS_V20_BASE + 0x58);
	data |= (1 << 7); /*MPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x58);
	data = readl(_SYS_V20_BASE + 0x58);
	data &= ~(1 << 7); /*MPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x58);

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH) 
	/*PPLL POWERON*/
	writel(0x00611f8c, _SYS_V20_BASE + 0x5c);
	writel(0x07ce1004, _SYS_V20_BASE + 0x60);
#endif

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)
	/*PPLL POWERON*/
	writel(0x00411f8c, _SYS_V20_BASE + 0x5c);
	writel(0x07ce1009, _SYS_V20_BASE + 0x60);
#endif

	data = readl(_SYS_V20_BASE + 0x60);
	data &= ~(1 << 28); /*PPLL_RESET*/
	writel(data, _SYS_V20_BASE + 0x60);
	data = readl(_SYS_V20_BASE + 0x64);
	data |= (1 << 7); /*PPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x64);
	data = readl(_SYS_V20_BASE + 0x64);
	data &= ~(1 << 7); /*PPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x64);

	/*DPLLA POWERON*/
	writel(0x00009d68, _SYS_V20_BASE + 0x68);
	writel(0x24c610c0, _SYS_V20_BASE + 0x6c);

	data = readl(_SYS_V20_BASE + 0x6c);
	data &= ~(1 << 28); /*DPLLA_RESET*/
	writel(data, _SYS_V20_BASE + 0x6c);
	data = readl(_SYS_V20_BASE + 0x70);
	data |= (1 << 7); /*DPLLA_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x70);
	data = readl(_SYS_V20_BASE + 0x70);
	data &= ~(1 << 7); /*DPLLA_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x70);

	writel((1 << 20), _SYS_V20_BASE + 0x44); /*DPLLA_PROG*/

	while (1) {
		data = readl(_SYS_V20_BASE + 0x44); /*PLL_STATES*/
		/*DPLLA_LKDT1,DPLLA_LKDT2,DPLLA_DONE*/
		if (((data>>4) & 0x3) == 0x3 && ((data>>14) & 0x1) == 0x1)
			break;
		udelay_slow(1000);
	}
	writel(0, _SYS_V20_BASE + 0x44);

	/*DPLLB POWERON*/
	writel(0x02e596c4, _SYS_V20_BASE + 0x74);
	writel(0x23b03002, _SYS_V20_BASE + 0x78);

	data = readl(_SYS_V20_BASE + 0x78);
	data &= ~(1 << 28); /*DPLLB_RESET*/
	writel(data, _SYS_V20_BASE + 0x78);
	data = readl(_SYS_V20_BASE + 0x7c);
	data |= (1 << 7); /*DPLLB_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x7c);
	data = readl(_SYS_V20_BASE + 0x7c);
	data &= ~(1 << 7); /*DPLLB_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x7c);

	writel((1 << 21), _SYS_V20_BASE + 0x44); /*DPLLB_PROG*/
	while (1) {
		data = readl(_SYS_V20_BASE + 0x44); /*PLL_STATES*/
		if (((data>>6) & 0x3) == 0x3) /*DPLLB_LKDT1,DPLLB_LKDT2*/
			break;
		udelay_slow(1000);
	}
	writel(0, _SYS_V20_BASE + 0x44);

	/*ARMPLL bypass1/2 disable*/
	data = readl(_SYS_V20_BASE + 0x80);
	data &= ~(0x3);/*ARMPLL_BYPASS1/2 = 0*/
	writel(data, _SYS_V20_BASE + 0x80);
	data = 1 << 22; /*ARMPLL_PROG*/
	writel(data, _SYS_V20_BASE + 0x44);
	while (1) {
		data = readl(_SYS_V20_BASE + 0x44); /*PLL_STATES*/
		/*ARMPLL_LKDT1,ARMPLL_LKDT2,ARMPLL_DONE*/
		if (((data >> 8) & 0x3) == 0x3 && ((data>>16) & 0x1) == 0x1)
			break;
		udelay_slow(1000);
	}
	writel(0, _SYS_V20_BASE + 0x44);

	/*RGMIIPLL POWERON*/
	data = readl(_SYS_V20_BASE + 0x53c); /*Bootstrap*/
	/*ethernet mac interface selection*/
	if ((data >> 6) & 0x1) {
		/*RGMIIEN = 1, use 500MHZ setting*/
		writel(0x00009d8c, _SYS_V20_BASE + 0x98);
		writel(0x00c61000, _SYS_V20_BASE + 0x9c);
	} else {
		/*RGMIIEN = 0, use 50MHZ setting*/
		writel(0x02ebab8c, _SYS_V20_BASE + 0x98);
		writel(0x00d61000, _SYS_V20_BASE + 0x9c);
	}
	data = readl(_SYS_V20_BASE + 0x9c);
	data &= ~(1 << 28); /*RGMIIPLL_RESET*/
	writel(data, _SYS_V20_BASE + 0x9c);
	data = readl(_SYS_V20_BASE + 0x100);
	data |= (1 << 7); /*RGMIIPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x100);
	data = readl(_SYS_V20_BASE + 0x100);
	data &= ~(1 << 7); /*RGMIIPLL_UPDATE*/
	writel(data, _SYS_V20_BASE + 0x100);

	/*Enable clocks of unnecessary blocks in always-on domain*/
	data = readl(_SYS_V20_BASE + 0);
	/*SPI0CLK_OFF, ARMCLK_OFF, SDIO0CLK_OFF*/
	data &= ~(1 << 5 | 1 << 20 | 1 << 22);
	writel(data, _SYS_V20_BASE + 0);

	data = readl(_SYS_V20_BASE + 0x04);
	/*SADCCLK_OFF, XCLKOFF, UART2,3,5CLKOFF*/
	data &= ~(3 << 15 | 0xb << 27);
	writel(data, _SYS_V20_BASE + 0x04);

	data = readl(_SYS_V20_BASE + 0x08);
	data &= ~(0xfefa << 13 | 1 << 31); /*GPIO1,3~7,9~15CLKOFF, PIFCLKOFF*/
	writel(data, _SYS_V20_BASE + 0x08);

	data = readl(_SYS_V20_BASE + 0x0c);
	/*
	 * TIMERCLKOFF, SDIO0BUSCLK_OFF, COUNTERCLK_OFF,
	 * WDTCLKOFF, TWOW2CLKOFF, TWOW3CLKOFF
	 */
	data &= ~(1 | 1 << 10 | 1 << 14 | 1 << 26 | 1 << 29 | 1 << 30);
	writel(data, _SYS_V20_BASE + 0x0c);
#endif

	/*POWERGATE_EN */
	itcm_pmu_step(5, 1 << 16, 0);
	/*TODO delay 2ms */
	for (i = 0; i < 50000; i++)
		;

	/*ISOEN */
	itcm_pmu_step(0xc, 1 << 17, 0);

	/*IPRST */
	itcm_pmu_step(0xa, 1 << 21, 0);

	/*CLKGATE */
	itcm_pmu_step(8, 1 << 20, 0);

	/*MIIDLEREQ */
	itcm_pmu_step(0, 1 << 19, 0);

#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME) || defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH) 
	/*Revert clock speed*/
	/*Clear source from XTAL*/
	data = readl(_SYS_V20_BASE + 0x14);
	/*ARMCLK_XTALSEL, PWM2CLK_SRCSEL, PWM1CLK_SRCSEL, PWM0CLK_SRCSEL,
	 *KEYCLK_XTALSEL, MCLK_XCLKSEL
	 */
	data &= ~(0xf1 | 1 << 29);
	writel(data, _SYS_V20_BASE + 0x14);

	data = readl(_SYS_V20_BASE + 0x18);
	data &= ~(1<<4); /*VD3DMIFCLK_XTALSEL*/
	writel(data, _SYS_V20_BASE + 0x18);

	/*Clear divider number*/
	writel(0, _SYS_V20_BASE + 0x24);
	writel(0x00010101, _SYS_V20_BASE + 0x28);
	writel(0, _SYS_V20_BASE + 0x30);
	writel(0, _SYS_V20_BASE + 0x34);
	writel(0, _SYS_V20_BASE + 0x38);

	/*Update new divider value. 1)Set En=1; 2) Set En=0;*/
	/*IR4CLKDIV_EN, COUNTERCLKDIV_EN, UARTCLKDIV_EN, IRCLKDIV_EN,
	 *PIALWAYSONCLKDIV_EN, STATICREGCLKDIV_EN, PWM2CLKDIV_EN,
	 *PWM1CLKDIV_EN, PWM0CLKDIV_EN, ARMCLKDIV_EN
	 */
	data = 0x226c01c2;
	writel(data, _SYS_V20_BASE + 0x1c);
	writel(0, _SYS_V20_BASE + 0x1c);
	data = 0x2; /*TWOWCLKDIV_EN*/
	writel(data, _SYS_V20_BASE + 0x20);
	writel(0, _SYS_V20_BASE + 0x20);
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)
	writel(0x03744003, _SYS_V20_BASE + 0x24);
	writel(0x0000002d, _SYS_V20_BASE + 0x30);
#endif
#endif

#if ENABLE_UART_LOWPOWER
	/*Re-configure uart parameters*/
	itcm_serial_init(0);
#endif
	itcm_serial_puts("power on seq done.\n");
}

