/* -------------------------------------------------------------------------- *\
   itcm_main.c - Main system flow control.
   Copyright (c) 2013 Pixelworks Inc.
   Pixelworks owns the sole copyright to this software. Under international
   copyright laws you (1) may not make a copy of this software except for
   the purposes of maintaining a single archive copy, (2) may not derive
   works herefrom, (3) may not distribute this work to others. These rights
   are provided for information clarification, other restrictions of rights
   may apply as well.
   --------------------------------------------------------------------------
   This is first C file when enter ITCM mode. The file will be ITCM entry file.
   The main function's major job is control TCM software flow. Normally, the
   function will implement device drivers initial. Execute power on/off control
   force CPU enter power down mode and wait for event to wake up.
\* -------------------------------------------------------------------------- */
#include <linux/compiler.h>
#include <io.h>
#include <mach/sysmap.h>
#include <mach/ipc_wdog.h>
#include "itcm_main.h"
#include "itcm_interrupts.h"
#include "itcm_serial_pl011.h"
#include "itcm_ir.h"
#include "itcm_power.h"
#include "itcm_gpio.h"
#include "itcm_eth.h"

void itcm_start(void)
{
	unsigned int r;

	/* disable irqs */
	__asm__ __volatile__("mrs %0, cpsr" : "=r"(r));

	r |= 0x80;
	__asm__ __volatile__("msr cpsr, %0" : : "r"(r));

	itcm_main();

	__asm__ __volatile__("mrs %0, cpsr" : "=r"(r));
	r &= ~0x1f;
	r |= 0xd3;
	__asm__ __volatile__("msr cpsr, %0" : : "r"(r));

}

void itcm_main(void)
{
	unsigned int irqs0, irqs1;
	unsigned char event = 0;
	unsigned int data;

	itcm_serial_init(0);
	itcm_interrupt_init();
	itcm_gpio_init(0);
	itcm_ruby_ir_init(0);

	itcm_serial_puts("\nitcm_start_armboot\n");

	/*Power off seq*/
	itcm_serial_puts("Low Power\n");

	/*
	 * RESERVED_REG bit 0,1 --> 1, this is a workaround for FPGA test.
	 * bit 0 -- FORCE_MIF_IDLE, bit 1 -- FORCE_MI_IDLE.
	 * In PMU design, MI and MIF idle is a precondition to do power
	 * down. MI status is 64-bit, and MIF status is 8-bit. On FPGA
	 * platform, there is no enough IO pads resource to connect status
	 * signal out. So use this way as a workaround to let PMU to
	 * ignore MI/MIF status.
	 * The real chip needs this workaround too.
	 */
	writel(0xffff0003, (_SYS_V20_BASE+0x00008000));

	/*
	 * Clear ARM sys reset enable bit, as this bit won't be cleard
	 * during standby reset.
	 */
	data = readl(PMUCTRL2);
	data &= ~(1<<10); /*ARMSYS_RST[10]: 0 */
	writel(data, PMUCTRL2);

	/*Read/write value in static register to validate result after reset.*/
	data = readl(PXLW_STATIC_REGS_V1_BASE+4);
	itcm_serial_puts("Static_regs_2: ");
	itcm_serial_puthex(data);
	itcm_serial_putc('\n');
	writel(0x11223344, PXLW_STATIC_REGS_V1_BASE+4);

	itcm_power_down_seq();
#if 0 // not working in low power mode
	itcm_serial_puts("eth init:\n");
	itcm_eth_init();
	itcm_serial_puts("eth init done\n");
#endif
	while (1) {
		/*Program ARM register to enter into WFI state*/
		itcm_wait_for_interrupt();

		itcm_interrupt_pending(&irqs0, &irqs1);

		/*config uart TXD and RTS pad direction to output*/
		writel(0x7fffff00, SYS_PAD_DIRCTRL1);
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZEH)
		if (irqs0 & VICINTSOURCE_UARTINTR_0) {
#endif
#if defined(CONFIG_ARCH_PIXELWORKS_TOPAZ_PRIME)
		if (irqs0 & VICINTSOURCE_UARTINTR_0) {
#endif
			itcm_serial_puts("got UART irq\n");
			event = itcm_serial_getc();

			if (event == '9') {
				itcm_serial_putc('\n');
				itcm_serial_putc('9');
				itcm_serial_putc('\n');
				itcm_power_on_seq();
				break;
			} else if (event == '7') {
				itcm_serial_putc('7');
				itcm_serial_putc('\n');
				itcm_serial_puts("watch dog reset\n");
				itcm_power_on_seq();
				writel(0x1acce551, 0xfd06001c); /*Wdog lock */
				writel(0xfff, 0xfd060000); /*Wdog load */
				writel(0x500, 0xfd060018); /*Wdog intval */
				writel(0xe0, 0xfd060008); /*Wdog control */
				break;
			} else if (event == '5') {
				itcm_serial_putc('5');
				itcm_serial_putc('\n');
				itcm_serial_puts("full chip reset\n");
				itcm_power_on_seq();
				data = readl(MISCCTRL);
				/*CHIP_RST_CTRL[5:4]: 0 */
				data &= ~(3<<4);
				writel(data, MISCCTRL);
				/*CHIP_RST_CTRL[5:4]: 1 */
				data |= (1<<4);
				writel(data, MISCCTRL);
				/*CHIP_RST_CTRL[5:4]: 3 */
				data |= (3<<4);
				writel(data, MISCCTRL);
				/*CHIP_RST_CTRL[5:4]: 2 */
				data &= ~(1<<5);
				writel(data, MISCCTRL);
				break;
			} else if (event == '3') {
				itcm_serial_putc('3');
				itcm_serial_putc('\n');
				itcm_serial_puts("standby reset\n");
				itcm_power_on_seq();
				/*
				 * Clear MI address remap control, as standby
				 * reset won't clear this bit automatically.
				 */
				data = readl(MISCCTRL);
				data |= 1; /*REMAP_2_MI */
				writel(data, MISCCTRL);

				data = readl(PMUCTRL2);
				data |= (1<<10); /*ARMSYS_RST[10]: 1 */
				writel(data, PMUCTRL2);

				break;
			}
			itcm_pl011_serial_Reenable(0);
		}

		if (irqs0 & VICINTSOURCE_IR0) {
			itcm_serial_puts("got IR irq\n");
			event = itcm_IRCode(0);
			itcm_serial_puts("IR key: ");
			itcm_serial_puthex(event);
			itcm_serial_putc('\n');

			if (IR_BUTTON_POWER == event) {
				itcm_IRIntReenable(0);
				itcm_power_on_seq();
				break;
			}
		}

		if (irqs0 & VICINTSOURCE_GPIO_0_1) {
			itcm_serial_puts("got GPIO irq\n");
			if (itcm_gpio_pinget(0, 1)) {
				itcm_serial_puts("\nPA_1\n");
				itcm_power_on_seq();
				break;
			}
		}
		if (irqs0 & VICINTSOURCE_ETH_WAKEUP) {
			itcm_serial_puts("got eth wakeup irq\n");
			itcm_power_on_seq();
			break;
		}
	}

	writel(0x52454254, PXLW_STATIC_REGS_V1_BASE);  //set static reg "REBT"
	itcm_IRIntDisable(0);
	itcm_interrupt_ack(irqs0, irqs1);
	itcm_serial_puts("Exit Low Power\n");

	/*TODO: eixt mode6 PMU steps */

}
