/* -------------------------------------------------------------------------- *\
   itcm_interrupt.c - interrupt setting control functions.
   Copyright (c) 2007,2008 Pixelworks Inc.
   Pixelworks owns the sole copyright to this software. Under international
   copyright laws you (1) may not make a copy of this software except for
   the purposes of maintaining a single archive copy, (2) may not derive
   works herefrom, (3) may not distribute this work to others. These rights
   are provided for information clarification, other restrictions of rights
   may apply as well.
   --------------------------------------------------------------------------
   This file is using to control VIC function.
\* -------------------------------------------------------------------------- */

#include <linux/compiler.h>
#include <io.h>

#include "itcm_interrupts.h"
#include "itcm_serial_pl011.h"

static unsigned char *const itcm_pl192_int[] = CONFIG_PL192_PORTS;

#define NUM_PORTS (sizeof(itcm_pl192_int)/sizeof(itcm_pl192_int[0]))

/* return pending irqs */
void itcm_interrupt_pending(unsigned int *irqs0, unsigned int *irqs1)
{
	*irqs0 = readl(itcm_pl192_int[0] + VIC_PL192_IRQSTATUS);
	*irqs1 = readl(itcm_pl192_int[1] + VIC_PL192_IRQSTATUS);
}

/* ack pending irqs */
void itcm_interrupt_ack(unsigned int irqs0, unsigned int irqs1)
{
	writel(irqs0, itcm_pl192_int[0] + VIC_PL192_INTENCLEAR);
	writel(irqs1, itcm_pl192_int[1] + VIC_PL192_INTENCLEAR);
}

/* execute power-saving wait-for-interrupt instruction */
void itcm_wait_for_interrupt(void)
{
	int temp = 0;
	__asm__ __volatile__("mcr p15,0,%0,c7,c10,4\n"  /* drain write buffer*/
			     "mcr p15,0,%0,c7,c0,4\n"   /* wait for irq*/
			     "nop\n"
			     "nop\n"
			     "nop\n"
			     "nop\n"
			     : "=r" (temp)
			     :
			     : "memory");
}

/* -------------------------------------------------------------------------- *\
Function Name : itcm_interrupt_init
Parameter : None
Return : None
Description :
Initialise each vectored interrupt handler address &
priority with the appropriate interrupt & enable bits,
e.g.

"<vic_base_address>.<VectAddrs>[X] = &interrupt;"
"<vic_base_address>.<VectCntl>[X]  = INT_VALUE;"

Where;
X          - Interrupt Priority (0-15)
&interrupt - Address of matching "interrupt" handler
INT_VALUE  - Interrupt & enable bits for "interrupt"

See vic.h for enable & interrupt value definitions.
\* -------------------------------------------------------------------------- */
void itcm_interrupt_init(void)
{

	unsigned int enable = 0;
	/* ------------------------------------- *\
	don't set up irq vectors, just enable the
	VIC to signal the irq to the ARM to wake-up from
	wait-for-irq; this relies on the ARM I bit being set
	in the CPSR to block the actual irq handling
	\* ------------------------------------- */
	writel(~0, itcm_pl192_int[0] + VIC_PL192_INTENCLEAR);
	writel(~0, itcm_pl192_int[0] + VIC_PL192_SOFTINTCLEAR);
	writel(~0, itcm_pl192_int[1] + VIC_PL192_INTENCLEAR);
	writel(~0, itcm_pl192_int[1] + VIC_PL192_SOFTINTCLEAR);

	enable |= VICINTSOURCE_UARTINTR_0;
	enable |= VICINTSOURCE_IR0;
	enable |= VICINTSOURCE_GPIO_0_1;
	writel(enable, itcm_pl192_int[0] + VIC_PL192_INTENABLE);

	enable = VIC1_INTSOURCE_UARTINTR_4;
	writel(enable, itcm_pl192_int[1] + VIC_PL192_INTENABLE);
}

