/* ------------------------------------------------------------------ *\
   itcm_gpio.h - GPIO control functions.
   Copyright (c) 2007,2008 Pixelworks Inc.
   Pixelworks owns the sole copyright to this software. Under international
   copyright laws you (1) may not make a copy of this software except for
   the purposes of maintaining a single archive copy, (2) may not derive
   works herefrom, (3) may not distribute this work to others. These rights
   are provided for information clarification, other restrictions of rights
   may apply as well.
   ------------------------------------------------------------------
   This file is using to control GPIO function.
\* ------------------------------------------------------------------ */

/**********************************************************************/
/*** Register  Offset  Width Type Function                          ***/
/*** ============================================================== ***/
/*** GPIO Normal Registers                                          ***/
/*** =====================                                          ***/
/***                                                                ***/
/*** GPIO_PL061_DATA      0x000  8  R/W  Data register                     ***/
/*** GPIO_PL061_DIR       0x400  8  R/W  Data Direction register           ***/
/*** GPIO_PL061_IS        0x404  8  R/W  Interrupt Sense register          ***/
/*** GPIO_PL061_IBE       0x408  8  R/W  Interrupt Both Edges register     ***/
/*** GPIO_PL061_IEV       0x40C  8  R/W  Interrupt EVent register          ***/
/*** GPIO_PL061_IE        0x410  8  R/W  Interrupt Enable register         ***/
/*** GPIO_PL061_RIS       0x414  8  R    Raw Interrupt register            ***/
/*** GPIO_PL061_MIS       0x418  8  R    Masked Interrupt register         ***/
/*** GPIO_PL061_IC        0x41C  8  W    Interrupt Clear register          ***/
/*** GPIO_PL061_AFSEL     0x420  8  R/W  Alternate Functionality reg.      ***/
/***                                                                ***/
/*** GPIO Integration Vector Registers                              ***/
/*** =================================                              ***/
/*** GPIO_PL061_ITCR      0x600  1  R/W  Integration Test Control reg.     ***/
/*** GPIO_PL061_ITIP1     0x604  8  R/W  Integr. Vect. I/P for GPAFOUT     ***/
/*** GPIO_PL061_ITIP2     0x608  8  R/W  Integr. Vect. O/P for GPAFIN      ***/
/*** GPIO_PL061_ITOP1     0x60C  8  W    Integration MIS Test O/P Set reg. ***/
/*** GPIO_PL061_ITOP2     0x610  1  R    Integr. Interrupt Test O/P Read.  ***/
/*** GPIO_PL061_ITOP3     0x614  8  R/W  Integr. Vect. I/P for nGPAFEN     ***/
/***                                                                ***/
/*** GPIO Identification Registers                                  ***/
/*** ===================                                            ***/
/***                                                                ***/
/*** GPIO_PL061_PeriphID0 0xFE0  8  R    Identification(0) register        ***/
/*** GPIO_PL061_PeriphID1 0xFE4  8  R    Identification(1) register        ***/
/*** GPIO_PL061_PeriphID2 0xFE8  8  R    Identification(2) register        ***/
/*** GPIO_PL061_PeriphID3 0xFEC  8  R    Identification(3) register        ***/
/*** GPIO_PL061_PCellID0  0xFF0  8  R    Octopus Identification(0) reg.    ***/
/*** GPIO_PL061_PCellID1  0xFF4  8  R    Octopus Identification(1) reg.    ***/
/*** GPIO_PL061_PCellID2  0xFF8  8  R    Octopus Identification(2) reg.    ***/
/*** GPIO_PL061_PCellID3  0xFFC  8  R    Octopus Identification(3) reg.    ***/
/***                                                                ***/
/*** GPIO Trickbox Registers                                        ***/
/*** =======================                                        ***/
/***                                                                ***/
/**********************************************************************/

/**********************************************************************/
/*** For more information on the Gpio, please refer to PL061 AMBA   ***/
/*** GPIO Block Specification                                       ***/
/**********************************************************************/

/**********************************************************************/
/******************* GPIO NORMAL-MODE REGISTERS  **********************/
/**********************************************************************/
#ifndef __ITCM_GPIO_H__
#define __ITCM_GPIO_H__

#include <mach/sysmap.h>

#define CONFIG_PL06x_PORTS	{ (void *) GPIO0_V2_BASE , \
					(void *) GPIO1_V2_BASE , \
					(void *) GPIO2_V2_BASE , \
					(void *) GPIO3_V2_BASE , \
					(void *) GPIO4_V2_BASE , \
					(void *) GPIO5_V2_BASE , \
					(void *) GPIO6_V2_BASE , \
					(void *) GPIO7_V2_BASE , \
					(void *) GPIO8_V2_BASE , \
					(void *) GPIO9_V2_BASE ,}
#define GPIO_PL061_DATA         0x000         /*0x000 to 0x3FC*/
#define GPIO_PL061_DIR          0x400
#define GPIO_PL061_IS           0x404
#define GPIO_PL061_IBE          0x408
#define GPIO_PL061_IEV          0x40C
#define GPIO_PL061_IE           0x410
#define GPIO_PL061_RIS          0x414
#define GPIO_PL061_MIS          0x418
#define GPIO_PL061_IC           0x41C
#define GPIO_PL061_AFSEL        0x420


/**********************************************************************/
/******************** GPIO INTEGRATION VECTORS ************************/
/**********************************************************************/
#define GPIO_PL061_ITCR         0x600
#define GPIO_PL061_ITIP1        0x604
#define GPIO_PL061_ITIP2        0x608
#define GPIO_PL061_ITOP1        0x60C
#define GPIO_PL061_ITOP2        0x610
#define GPIO_PL061_ITOP3        0x614

/**********************************************************************/
/***************** GPIO IDENTIFICATION REGISTERS **********************/
/**********************************************************************/
#define GPIO_PL061_PeriphID0    0xFE0
#define GPIO_PL061_PeriphID1    0xFE4
#define GPIO_PL061_PeriphID2    0xFE8
#define GPIO_PL061_PeriphID3    0xFEC

#define GPIO_PL061_PCellID0     0xFF0
#define GPIO_PL061_PCellID1     0xFF4
#define GPIO_PL061_PCellID2     0xFF8
#define GPIO_PL061_PCellID3     0xFFC

/**********************************************************************/
/****************** Bit Masking Constants *****************************/
/**********************************************************************/
#define NoMask           0x000000FF
#define MaskAll          0x00000000

#define GPIO_PORTA	(0x0)
#define GPIO_PORTB	(0x1)
#define GPIO_PORTC	(0x2)
#define GPIO_PORTD	(0x3)
#define GPIO_PORTE	(0x4)
#define GPIO_PORTF	(0x5)
#define GPIO_PORTG	(0x6)
#define GPIO_PORTH	(0x7)
#define GPIO_PORTI	(0x8)
#define GPIO_PORTJ	(0x9)

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_init
Parameter : unsigned char portnum
Return : 0 -- Always return 0;
Description : This function is setting GPIO to default state.
\* ------------------------------------------------------------------ */
int itcm_gpio_init(unsigned char portnum);

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_set_outpindir
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char pins -- pin direction value.
	    0 means input/ 1 means output
Return : 0 -- None
Description : This function is setting GPIO pin direction to output.
\* ------------------------------------------------------------------ */
void itcm_gpio_set_outpindir(int portnum, const char pins);

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_get_pindir
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
Return : 0 -- pin direction value of port.
Description : This function is getting GPIO pin direction.
\* ------------------------------------------------------------------ */
int itcm_gpio_get_pindir(int portnum);

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_set_inpindir
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char pins -- pin direction value.
	    0 means output/ 1 means input
Return : 0 -- None
Description : This function is setting GPIO pin direction to input.
\* ------------------------------------------------------------------ */
void itcm_gpio_set_inpindir(int portnum, const char pins);

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_putc
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const unsigned short c -- pin value.
	    0 means low/ 1 means high
Return : 0 -- None
Description : This function is setting GPIO pin to drive pin's high/low accord
	to variable's value.
\* ------------------------------------------------------------------ */
void itcm_gpio_putc(int portnum, const unsigned short c);

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_puts
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char *s -- pin value string.
Return : 0 -- None
Description : This function is setting GPIO pin to drive pin's high/low accord
	to variable's value.
\* ------------------------------------------------------------------ */
void itcm_gpio_puts(int portnum, const char *s);

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_getc
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
Return : 0 -- return port's pin state value.
Description : This function is getting GPIO pin's high/low status.
\* ------------------------------------------------------------------ */
int itcm_gpio_getc(int portnum);

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_pinput
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char pin -- 0 ~ 7, which pin of port
	    const unsigned char hi -- set pin to high or low
Return : 0 -- None
Description : This function is setting one pin of GPIO port to high/low.
\* ------------------------------------------------------------------ */
void itcm_gpio_pinput(int portnum, const char pin, const unsigned char hi);

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_pinget
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char pin -- 0 ~ 7, which pin of port
Return : 0 -- pin's status.
Description : This function is getting one pin of GPIO port's status.
\* ------------------------------------------------------------------ */
int itcm_gpio_pinget(int portnum, const char pin);

#endif
