/* ------------------------------------------------------------------ *\
   itcm_gpio.c - GPIO control functions.
   Copyright (c) 2007,2008 Pixelworks Inc.
   Pixelworks owns the sole copyright to this software. Under international
   copyright laws you (1) may not make a copy of this software except for
   the purposes of maintaining a single archive copy, (2) may not derive
   works herefrom, (3) may not distribute this work to others. These rights
   are provided for information clarification, other restrictions of rights
   may apply as well.
   ------------------------------------------------------------------
   This file is using to control GPIO function.
\* ------------------------------------------------------------------ */

#include <common.h>
#include <io.h>
#include "itcm_gpio.h"

static unsigned char *itcm_gpio[10] = {
	(unsigned char *)GPIO0_V2_BASE,
	(unsigned char *)GPIO1_V2_BASE,
	(unsigned char *)GPIO2_V2_BASE,
	(unsigned char *)GPIO3_V2_BASE,
	(unsigned char *)GPIO4_V2_BASE,
	(unsigned char *)GPIO5_V2_BASE,
	(unsigned char *)GPIO6_V2_BASE,
	(unsigned char *)GPIO7_V2_BASE,
	(unsigned char *)GPIO8_V2_BASE,
	(unsigned char *)GPIO9_V2_BASE
};
#define NUM_PORTS (sizeof(itcm_gpio)/sizeof(itcm_gpio[0]))

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_init
Parameter : unsigned char portnum
Return : 0 -- Always return 0;
Description : This function is setting GPIO to default state.
\* ------------------------------------------------------------------ */
int itcm_gpio_init(unsigned char portnum)
{
	writel(0x0, itcm_gpio[portnum] + GPIO_PL061_AFSEL);
	writel(0x2, itcm_gpio[portnum] + GPIO_PL061_IE);
	writel(0x0, itcm_gpio[portnum] + GPIO_PL061_IS);
	writel(0x0, itcm_gpio[portnum] + GPIO_PL061_IBE);
	writel(0x2, itcm_gpio[portnum] + GPIO_PL061_IEV);
	/*clear all interrupts*/
	writel(0xFF, itcm_gpio[portnum] + GPIO_PL061_IC);
	return 0;
}

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_set_outpindir
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char pins -- pin direction value.
	    0 means input/ 1 means output
Return : 0 -- None
Description : This function is setting GPIO pin direction to output.
\* ------------------------------------------------------------------ */
void itcm_gpio_set_outpindir(int portnum, const char pins)
{
	writel(pins, itcm_gpio[portnum] + GPIO_PL061_DIR);
}

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_get_pindir
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
Return : 0 -- pin direction value of port.
Description : This function is getting GPIO pin direction.
\* ------------------------------------------------------------------ */
int itcm_gpio_get_pindir(int portnum)
{
	unsigned int data;
	data = readl(itcm_gpio[portnum] + GPIO_PL061_DIR);
	return (int) data;
}

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_set_inpindir
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char pins -- pin direction value.
	    0 means output/ 1 means input
Return : 0 -- None
Description : This function is setting GPIO pin direction to input.
\* ------------------------------------------------------------------ */
void itcm_gpio_set_inpindir(int portnum, const char pins)
{
	writel((~pins), itcm_gpio[portnum] + GPIO_PL061_DIR);
}

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_putc
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const unsigned short c -- pin value. 0 means low/ 1 means high
Return : 0 -- None
Description : This function is setting GPIO pin
			to drive pin's high/low accord to variable's value.
\* ------------------------------------------------------------------ */
void itcm_gpio_putc(int portnum, const unsigned short c)
{
	writel(c, itcm_gpio[portnum] + GPIO_PL061_DATA + 0x3FC);
}

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_puts
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char *s -- pin value string.
Return : 0 -- None
Description : This function is setting GPIO pin
			to drive pin's high/low accord to variable's value.
\* ------------------------------------------------------------------ */
void itcm_gpio_puts(int portnum, const char *s)
{
	if (s == NULL)
		return;
	while (*s)
		itcm_gpio_putc(portnum, *s++);
}

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_getc
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
Return : 0 -- return port's pin state value.
Description : This function is getting GPIO pin's high/low status.
\* ------------------------------------------------------------------ */
int itcm_gpio_getc(int portnum)
{
	unsigned int data;
	data = readl(itcm_gpio[portnum] + GPIO_PL061_DATA + 0x3FC);

	return (int) data;
}

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_pinput
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char pin -- 0 ~ 7, which pin of port
	    const unsigned char hi -- set pin to high or low
Return : 0 -- None
Description : This function is setting one pin of GPIO port to high/low.
\* ------------------------------------------------------------------ */
void itcm_gpio_pinput(int portnum, const char pin, const unsigned char hi)
{
	unsigned int data = 0;
	data = itcm_gpio_getc(portnum);
	data = (data & ~(1 << pin));
	if (hi)
		data = data | (1<<pin);

	writel(data, itcm_gpio[portnum] + GPIO_PL061_DATA + 0x3FC);
}

/* ------------------------------------------------------------------ *\
Function Name : itcm_gpio_pinget
Parameter : int portnum -- 0 ~ 9, it means port0 ~ port9 (PORT A ~ PORTJ)
	    const char pin -- 0 ~ 7, which pin of port
Return : 0 -- pin's status.
Description : This function is getting one pin of GPIO port's status.
\* ------------------------------------------------------------------ */
int itcm_gpio_pinget(int portnum, const char pin)
{
	unsigned int data;
	data = readl(itcm_gpio[portnum] + GPIO_PL061_DATA + 0x3FC);

	return (int) ((data >> pin) & 0x1);
}
