/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/of.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/regulator/consumer.h>

#define RICOH_SIMVA

#ifdef RICOH_SIMVA
#include <linux/sched.h>
#include <linux/wait.h>
#endif

#include "asic/Xena_regheaders.h"
#include "upc.h"

// register writing defines
// note that you *must* have defined a local variable called xena_regbase
#define upcWrite(val, rname)    iowrite32(val, &(((XENA_REGS_t*)(xena_regbase))->rname))
#define upcRead(rname)          ioread32(&(((XENA_REGS_t*)(xena_regbase))->rname))

#define MAX_DRAGONITE_ITCM_SIZE   0x08000  

/*  RICOH SIMVA  */
#define FGATE_MASK        (0x000003c0)
#define FGATE_BLACK_BIT   (0x00000200)
#define FGATE_CYAN_BIT    (0x00000100)
#define FGATE_MAGENTA_BIT (0x00000080)
#define FGATE_YELLOW_BIT  (0x00000040)

#define FGATE_YELLOW_ASSERT  (1)
#define FGATE_MAGENTA_ASSERT (2)
#define FGATE_CYAN_ASSERT    (3)
#define FGATE_BLACK_ASSERT   (4)
#define FGATE_YELLOW_NEGATE  (5)
#define FGATE_MAGENTA_NEGATE (6)
#define FGATE_CYAN_NEGATE    (7)
#define FGATE_BLACK_NEGATE   (8)

#define FGATE_BUF_SIZE    (64)

static unsigned long     fgate_value=0x000003c0;
static unsigned char  fgatebuf[2][FGATE_BUF_SIZE];
static unsigned char *wptr=&fgatebuf[0][0];
char   g_upc_ver[] = "UPC driver based on LSP_20150828, support fgates queueing.";
/*  RICOH SIMVA  */

// force pegmatite regulator to be probed first, always returns 1
extern int pegmatite_reg_check(void);

// structure to store state of upcs during suspend/resume
// cpu_soft_reset controls the running of the dragonite cpu's if they are OFF then the close of the 
// device handle will disable the power island via the regulator.  Normal regulator reference counting is 
// disabled to support the free running behaviour of the dragonites without an open file handle on the linux
// side of things.
struct upc_suspend_context_struct
{
	uint8_t code[MAX_NUM_DRAGONITES][MAX_DRAGONITE_ITCM_SIZE];    // upc code
	uint32_t cpu_soft_reset;                    // current value of CPUSoftReset
	uint32_t num_open;  // 0 1 2
	struct regulator *power;
};

struct upc_data
{
	void __iomem *base; // kernel virtual
	int irq;
	int dragonite_num;
	uint32_t register_addr; // address of the registers
	uint32_t itcm_addr;  // address of the itcm 
	uint32_t itcm_size;  // size of the itcm (in bytes)
	uint32_t dtcm_addr;  // address of the DTCM
	uint32_t dtcm_size;  // size of the dtcm (in bytes)
	
	int upc_ifileptr;    // char interface can break up writes into multiple calls - track current file location
	int upc_dfileptr;    // char interface can break up writes into multiple calls - track current file location
	uint8_t cdev_open;
	struct cdev upc_cdev;
	struct device *dev;
	struct mutex upc_var_lock; 
};

///////// start of "framework" stuff

static bool tcm_memory_select = ITCM_MEMORY;

struct upc_overall_struct
{
	int upc_major;           // the major number - for making the device
	spinlock_t upc_reg_lock; // one set of registers for all dragonites
	struct class *upc_class; // for class creation
	struct upc_data *upc_data_array[MAX_NUM_DRAGONITES];
	struct upc_suspend_context_struct upc_suspend_context;
#ifdef RICOH_SIMVA
#define  NUM_UPC_MAILBOX  (6)
	uint32_t  mailbox_data[NUM_UPC_MAILBOX];
	int       rcv_irq;
	wait_queue_head_t  read_queue;
	int       rcv_fgate;
	wait_queue_head_t  fgate_queue;
#endif
};
static struct upc_overall_struct upc_object;

// only 1 set of registers shared by all dragonites, get that address
// (it's based on the dragonite 0 base address)
static uint32_t upc_get_regbase_addr(void)
{
	return (upc_object.upc_data_array[0]->register_addr);
}

//////////////  end of framework "stuff"

//////////////////// start of public interface functions
// macro for upc_register_setup
#define upcRegConfigCopy(regname)					\
	if (upcconfig->operation.regname != OP_IGNORE) {		\
		if (upcconfig->operation.regname == OP_WRITE) {		\
			tempvar = upcconfig->value.regname;		\
		} else {						\
			tempvar = upcRead(regname);			\
			if (upcconfig->operation.regname == OP_OR)	\
				tempvar |= upcconfig->value.regname;	\
			else						\
				tempvar &= upcconfig->value.regname;	\
		}							\
		upcWrite(tempvar, regname);				\
	}


// take the upc_reg_config parameter and set up every single register
// in the upc register set (one shared register set for all dragonites)
void upc_register_setup(struct upc_reg_config *upcconfig)
{
	uint32_t *xena_regbase;
	uint32_t tempvar;
	int i;
    
	xena_regbase = (uint32_t *) upc_get_regbase_addr();
        
	upcRegConfigCopy(EncPos);
	upcRegConfigCopy(Phase0);
	upcRegConfigCopy(Phase1);
	upcRegConfigCopy(Phase2);
	upcRegConfigCopy(Phase3);
	upcRegConfigCopy(Mailbox0);
	upcRegConfigCopy(Mailbox1);
	upcRegConfigCopy(GPStatus);
#ifdef RICOH_SIMVA
	upcRegConfigCopy(GPStatusClear);
#endif
	upcRegConfigCopy(CPUSoftReset);
	upcRegConfigCopy(PhaseRef);
	upcRegConfigCopy(XRR);
	upcRegConfigCopy(XPS);
	upcRegConfigCopy(XCR);
	upcRegConfigCopy(XMC);
	upcRegConfigCopy(XMA);
	upcRegConfigCopy(XMJ);
	upcRegConfigCopy(XMA2);
	upcRegConfigCopy(XML);
	upcRegConfigCopy(XDA);
	upcRegConfigCopy(XDJ);
	upcRegConfigCopy(XDA2);
	upcRegConfigCopy(XDL);
	upcRegConfigCopy(MaskTileBurst);
	upcRegConfigCopy(MaskTileWord);
	upcRegConfigCopy(XCA);
	upcRegConfigCopy(XCJ);
	upcRegConfigCopy(XNW);
	upcRegConfigCopy(XST);
	upcRegConfigCopy(EncControl);
	upcRegConfigCopy(EncFilt);
	upcRegConfigCopy(Coeff);
	upcRegConfigCopy(FWPredPeriod);
	upcRegConfigCopy(LeftMargin0);
	upcRegConfigCopy(RightMargin0);
	upcRegConfigCopy(Res0);
	upcRegConfigCopy(TOF0);
	upcRegConfigCopy(LeftMargin1);
	upcRegConfigCopy(RightMargin1);
	upcRegConfigCopy(Res1);
	upcRegConfigCopy(TOF1);
	upcRegConfigCopy(LeftMargin2);
	upcRegConfigCopy(RightMargin2);
	upcRegConfigCopy(Res2);
	upcRegConfigCopy(TOF2);
	upcRegConfigCopy(LeftMargin3);
	upcRegConfigCopy(RightMargin3);
	upcRegConfigCopy(Res3);
	upcRegConfigCopy(TOF3);
	upcRegConfigCopy(LeftMargin4);
	upcRegConfigCopy(RightMargin4);
	upcRegConfigCopy(Res4);
	upcRegConfigCopy(TOF4);
	upcRegConfigCopy(LeftMargin5);
	upcRegConfigCopy(RightMargin5);
	upcRegConfigCopy(Res5);
	upcRegConfigCopy(TOF5);
	upcRegConfigCopy(LeftMargin6);
	upcRegConfigCopy(RightMargin6);
	upcRegConfigCopy(Res6);
	upcRegConfigCopy(TOF6);
	upcRegConfigCopy(LeftMargin7);
	upcRegConfigCopy(RightMargin7);
	upcRegConfigCopy(Res7);
	upcRegConfigCopy(TOF7);
	upcRegConfigCopy(Status);
	upcRegConfigCopy(PseudoEncoder);
	upcRegConfigCopy(Mailbox2);
	upcRegConfigCopy(Mailbox3);            
	upcRegConfigCopy(Mailbox4);
	upcRegConfigCopy(Mailbox5);    
	upcRegConfigCopy(Timer0);
	upcRegConfigCopy(Timer1);
	upcRegConfigCopy(Timer2);
	upcRegConfigCopy(Timer3);
	upcRegConfigCopy(XIEnSel0);
	upcRegConfigCopy(XIEnSel1);    
	upcRegConfigCopy(XCT);            
	upcRegConfigCopy(XIE0);
	upcRegConfigCopy(XIFire);
	upcRegConfigCopy(XISync);
	for (i=0;i<8;i++)
		upcRegConfigCopy(XIC[i]);            
	upcRegConfigCopy(XICST0);
	upcRegConfigCopy(XICST);
	upcRegConfigCopy(XIFC);
	upcRegConfigCopy(XIFI);
	for (i=0;i<8;i++) {
		upcRegConfigCopy(XIFD[i]);
		upcRegConfigCopy(XIFSt[i]);
		upcRegConfigCopy(XIFBF[i]);
		upcRegConfigCopy(XIFBC[i]);
	}
	upcRegConfigCopy(XISI);
	upcRegConfigCopy(XISC);
	for (i=0;i<32;i++) {
		upcRegConfigCopy(XIIC[i]);
		upcRegConfigCopy(XIP0[i]);
		upcRegConfigCopy(XIP1[i]);
		upcRegConfigCopy(XIP2[i]);
	}
	upcRegConfigCopy(XIRW);
	upcRegConfigCopy(XIRD);
	upcRegConfigCopy(XIRD1);
	upcRegConfigCopy(XIRS);            
	upcRegConfigCopy(XIII);
	for (i=0;i<8;i++)
		upcRegConfigCopy(XISyncConfig[i]);
	upcRegConfigCopy(SafetyData0);
	upcRegConfigCopy(SafetyOE0);            
	upcRegConfigCopy(SafetyWatchdog);
	for (i=0;i<32;i++)
		upcRegConfigCopy(LFSR[i]);
	for (i=0;i<3;i++) {
		upcRegConfigCopy(IRQEnableFM[i]);
		upcRegConfigCopy(FIQEnableFM[i]);
		upcRegConfigCopy(IntPostFM[i]);
		upcRegConfigCopy(IntClearFM[i]);
		upcRegConfigCopy(IRQEnableIO[i]);
		upcRegConfigCopy(FIQEnableIO[i]);
		upcRegConfigCopy(IntPostIO[i]);
		upcRegConfigCopy(IntClearIO[i]);
	}
	for (i=0;i<1024;i++) 
		upcRegConfigCopy(APB[i]);                    
#if 0
	// Dump registers 
	do {
		int i;
		int regbase = (int) xena_regbase;
		for(i=0x1160; i<=0x1694; i=i+4) {
			printk("Addr 0x%08x = 0x%08x\n", (uint32_t)(regbase+i), *(uint32_t *)(regbase+i));
		}
	} while (0);
#endif    
    
}

EXPORT_SYMBOL(upc_register_setup);

#ifdef RICOH_SIMVA
void upc_set_gp_status_clear(int value)
{
	uint32_t  *xena_regbase;
	xena_regbase = (uint32_t *) upc_get_regbase_addr();

	upcWrite(value, GPStatusClear);
}
EXPORT_SYMBOL(upc_set_gp_status_clear);
#endif
void upc_set_gp_status(int value)
{
	uint32_t *xena_regbase;
	xena_regbase = (uint32_t *) upc_get_regbase_addr();
    
	upcWrite(value,GPStatus);
}

EXPORT_SYMBOL(upc_set_gp_status);

uint32_t upc_get_gp_status(void)
{
	uint32_t *xena_regbase;
	xena_regbase = (uint32_t *) upc_get_regbase_addr();   
    
	return(upcRead(GPStatus));
}

EXPORT_SYMBOL(upc_get_gp_status);

int upc_write_mailbox(uint32_t mailboxnum, int value)
{
	uint32_t *xena_regbase;

	xena_regbase = (uint32_t *) upc_get_regbase_addr();
    
	if (mailboxnum > 5)
		return -1;
	switch(mailboxnum) {
	case 0:
		upcWrite(value, Mailbox0);
		break;
	case 1:
		upcWrite(value, Mailbox1);
		break;
	case 2:
		upcWrite(value, Mailbox2);
		break;
	case 3:
		upcWrite(value, Mailbox3);
		break;
	case 4:
		upcWrite(value, Mailbox4);
		break;
	case 5:
		upcWrite(value, Mailbox5);
		break;
	}
	return 0;
}

EXPORT_SYMBOL(upc_write_mailbox);

int upc_read_mailbox(uint32_t mailboxnum, uint32_t *value)
{
	uint32_t *xena_regbase;

	xena_regbase = (uint32_t *) upc_get_regbase_addr();
    
	if (mailboxnum > 5)
		return -1;
    switch(mailboxnum) {
    case 0:
	    *value = upcRead(Mailbox0);
	    break;
    case 1:
	    *value = upcRead(Mailbox1);
	    break;
    case 2:
	    *value = upcRead(Mailbox2);
	    break;
    case 3:
	    *value = upcRead(Mailbox3);
	    break;
    case 4:
	    *value = upcRead(Mailbox4);
	    break;
    case 5:
	    *value = upcRead(Mailbox5);
	    break;
    }
    return 0;
}
EXPORT_SYMBOL(upc_read_mailbox);

void upc_soft_reset(int value)
{
	uint32_t *xena_regbase;
    
	xena_regbase = (uint32_t *) upc_get_regbase_addr();
	upcWrite(value, CPUSoftReset);
}

EXPORT_SYMBOL(upc_soft_reset);

/*
 * For UPC RAM, writing a whole block (e.g., 4k) of data with iowrite8 and then
 * reading them back often shows inconsistency. Trying iowrite8 followed by a
 * ioread8 doesn't show any problem. So it appears that a iowrite8 may cause
 * neighboring bytes to change. This could be a hardware issue.
 * However no issue is observed with iowrite32. This function is a hack to
 * get around the issue.
 */
static int upc_write_align(uint8_t *data, int length, int offset, uint8_t *addr)
{
	int i, length4;
	uint8_t *addr_begin = addr + offset;
	int total_length = length;
	uint32_t *data32 = (uint32_t*)data;
	int need_align_buffer = (uint32_t)data & 3; /* data is not 32-bit aligned */
	int pad_begin = offset & 3;
	int pad_end = (pad_begin + length) & 3;

	if (pad_end) pad_end = 4 - pad_end;

	if (pad_begin || pad_end) {
		need_align_buffer = 1;
		total_length += pad_begin + pad_end;
	}

	length4 = total_length/4;
	if (need_align_buffer) {
		addr_begin -= pad_begin;
		data32 = (uint32_t *)kmalloc(total_length, GFP_KERNEL);

		if (pad_begin) {
			*data32 = ioread32(addr_begin);
		}
		if (pad_end) {
			data32[length4 - 1] = ioread32(addr_begin + total_length - 4);
		}
		memcpy(((uint8_t *)data32) + pad_begin, data, length);
	}

	for (i=0; i<length4; i++, addr_begin += 4) {
		iowrite32(data32[i], addr_begin);
	}
	
	if (need_align_buffer) kfree(data32);

	return length;
}

// write the instruction memory (itcm) for a given dragonite cpu *byte at a time* 
// The upc_write() function may need to pass in a byte offset - use it
int upc_write_itcm(int dragonite_num, uint8_t *data, int length, int write_byte_offset)
{
	uint8_t *dragonite_itcmbase;

	dragonite_itcmbase = (uint8_t *) upc_object.upc_data_array[dragonite_num]->itcm_addr;
	return upc_write_align(data, length, write_byte_offset, dragonite_itcmbase);
}

EXPORT_SYMBOL(upc_write_itcm);

// read the instruction memory (itcm) for a given dragonite cpu *byte at a time*
// The upc_read() function may need to pass in a byte offset - use it
int upc_read_itcm(int dragonite_num, uint8_t *buff, int length, int read_byte_offset)
{
	uint8_t *dragonite_itcmbase;
	int i;
	
	dragonite_itcmbase = (uint8_t *) upc_object.upc_data_array[dragonite_num]->itcm_addr + read_byte_offset;

	for (i=0; i<length; i++) {
		buff[i] = ioread8(dragonite_itcmbase + i);
	}
	return i;
}

EXPORT_SYMBOL(upc_read_itcm);
    
// write the data memory (dtcm) for the dragonites (they share the dtcm) 8 bits at a time
int upc_write_dtcm(int dragonite_num, uint8_t *data, int length, int write_byte_offset)
{
	int dragonite_dtcmbase;

	// ignore the dragonite_num since both share the same dtcm, so the addr of dtcm is based on dragonite 1
	dragonite_dtcmbase = upc_object.upc_data_array[dragonite_num]->dtcm_addr;

	return upc_write_align(data, length, write_byte_offset, (uint8_t *)dragonite_dtcmbase);
}

EXPORT_SYMBOL(upc_write_dtcm);

// read the data memory (dtcm) for the dragonites (they share the dtcm) 8 bits at a time
// The upc_read() function may need to pass in a byte offset - use it
int upc_read_dtcm(int dragonite_num, uint8_t *data, int length, int read_byte_offset)
{
	uint32_t dragonite_dtcmbase;
	uint8_t *dtcm_ptr;
	int i;

	// ignore the dragonite_num since both share the same dtcm, so the addr of dtcm is based on dragonite 0
	dragonite_dtcmbase = upc_object.upc_data_array[dragonite_num]->dtcm_addr + (uint32_t)read_byte_offset;

	dtcm_ptr = (uint8_t *)dragonite_dtcmbase;
    for (i=0;i<length;i++) {
	    data[i] = ioread8(dtcm_ptr + i);
    }
    return i;
}

EXPORT_SYMBOL(upc_read_dtcm);

//////////////////// end of public interface functions

static int upc_open(struct inode *inode, struct file *filep)
{
	struct upc_data *dev;
	struct regulator *r = upc_object.upc_suspend_context.power;
	int retval = -EBUSY;
    
	dev = container_of(inode->i_cdev, struct upc_data, upc_cdev);

	mutex_lock(&dev->upc_var_lock);

	if (!dev->cdev_open) {
		dev->cdev_open = 1;
		filep->private_data = dev;
		dev->upc_ifileptr = 0; // initialize our internal fileptr for writes and reads
		dev->upc_dfileptr = 0; // initialize our internal fileptr for writes and reads

		if ( r && !upc_object.upc_suspend_context.num_open 
		     &&   !upc_object.upc_suspend_context.cpu_soft_reset ) {
			if (regulator_enable( r )) 
				printk(KERN_ERR "can't enable upc regulator \n");
		} else {
			printk(KERN_WARNING "already running regulator \n");
		}
		upc_object.upc_suspend_context.num_open++ ;
	}
#ifdef RICOH_SIMVA
	else
	{
		dev->cdev_open++;
		filep->private_data = dev;
		if ( r && !upc_object.upc_suspend_context.cpu_soft_reset )
		{
			if (regulator_enable( r )) 
				printk(KERN_ERR "can't enable upc regulator \n");
		} else {
			printk(KERN_WARNING "already running regulator \n");
		}
		upc_object.upc_suspend_context.num_open++ ;
	}
#endif
	retval = 0;
	mutex_unlock(&dev->upc_var_lock);
    
	return retval;
}

static int upc_read(struct file *filep, char __user *buf, size_t len, loff_t *ptr)
{
	struct upc_data *dev = filep->private_data;
	int readlength, retval;
	uint8_t *readbuf;
	int fp;
	int max_size;
    
	retval = -EINVAL;
    
	if (len == 0)
		return retval;

	mutex_lock(&dev->upc_var_lock);

	if (ITCM_MEMORY == tcm_memory_select) {
		fp = dev->upc_ifileptr;
		max_size = dev->itcm_size;
	} else {
		fp = dev->upc_dfileptr;
		max_size = dev->dtcm_size;
	}

	if (len+fp > max_size )
		readlength = max_size-fp; // don't read more than dragonite's memory
	else
		readlength = (int) len;

	/* eof */
	if (readlength == 0) {
		mutex_unlock(&dev->upc_var_lock);            
		return 0;
	}

	readbuf = (uint8_t *) kmalloc(readlength, GFP_KERNEL);
    
	if (ITCM_MEMORY == tcm_memory_select)
		upc_read_itcm(dev->dragonite_num, readbuf, readlength, fp);
	else
		upc_read_dtcm(dev->dragonite_num, readbuf, readlength, fp);

	if (copy_to_user(buf,readbuf,readlength)) {
		retval = -EFAULT;
		goto cleanexit;
	}
	if (ITCM_MEMORY == tcm_memory_select)
		dev->upc_ifileptr += readlength;
	else
		dev->upc_dfileptr += readlength;
	
	retval = readlength;

cleanexit:
	kfree(readbuf);
	mutex_unlock(&dev->upc_var_lock);            
	return retval;
}

static int upc_write(struct file *filep, const char __user *buf, size_t len, loff_t *ptr)
{
	struct upc_data *dev = filep->private_data;
	int  writelength, retval;
	uint8_t *writebuf; 
	int fp;
	int max_size;
    
	retval = -EINVAL;
	if (len == 0)
		return retval;
    
	mutex_lock(&dev->upc_var_lock);

	if (ITCM_MEMORY == tcm_memory_select) {
		fp = dev->upc_ifileptr;
		max_size = dev->itcm_size;
	} else {
		fp = dev->upc_dfileptr;
		max_size = dev->dtcm_size;
	}

	if (len+fp > max_size )
		writelength = max_size-fp; // don't write more than dragonite's memory
	else
		writelength = (int) len;

	/* eof */
	if (writelength == 0) {
		mutex_unlock(&dev->upc_var_lock);
		return -0;
	}
	
	writebuf = (uint8_t *) kmalloc(writelength, GFP_KERNEL);
    
	if (copy_from_user(writebuf,buf,writelength)) {
		retval = -EFAULT;
		goto cleanexit;
	}
    
	if (ITCM_MEMORY == tcm_memory_select) {
		upc_write_itcm(dev->dragonite_num, writebuf, writelength, fp);
		dev->upc_ifileptr += writelength;
	} else {
		upc_write_dtcm(dev->dragonite_num, writebuf, writelength, fp);
		dev->upc_dfileptr += writelength;
	}
    
	if (writelength > 0)
		retval = writelength;
	else
		retval = -EAGAIN;
	upc_object.upc_suspend_context.cpu_soft_reset = 1;
cleanexit:
	kfree(writebuf);
	mutex_unlock(&dev->upc_var_lock);
	return retval;
}

static int upc_close(struct inode *inode, struct file *filep)
{
	struct upc_data *dev = filep->private_data;
	struct regulator *r = upc_object.upc_suspend_context.power;

	mutex_lock(&dev->upc_var_lock);
#ifdef RICOH_SIMVA
	dev->cdev_open--;
#else
	dev->cdev_open = 0;
#endif
	upc_object.upc_suspend_context.num_open-- ;
	if ( !upc_object.upc_suspend_context.num_open && 
	     !upc_object.upc_suspend_context.cpu_soft_reset ) {
		// power off island
		if ( r && regulator_is_enabled( r ) ) 
		{
			if (regulator_disable( r ))
				regulator_force_disable(r);
			printk("!!!!!!!! upc regulator disabled !!!!!!!!\n");
		}
	}
	dev->upc_ifileptr = 0;
	dev->upc_dfileptr = 0;
	filep->private_data = NULL;
	mutex_unlock(&dev->upc_var_lock);
    
	return 0;
}

static long upc_ioctl(struct file *filep, unsigned int cmd, unsigned long arg)
{
	// struct upc_data *dev = filep->private_data; // uncomment if private data needed
	uint32_t *xena_regbase;
	unsigned long flags;
	int error = 0;
    	struct upc_mailbox_arg a;
#ifdef RICOH_SIMVA
	struct upc_data *dev = filep->private_data;
	struct regulator *r = upc_object.upc_suspend_context.power;
	unsigned char *copybuf;
#endif
	xena_regbase = (uint32_t *) upc_get_regbase_addr();

	spin_lock_irqsave(&(upc_object.upc_reg_lock), flags); // one reg set for all upcs
	switch (cmd) {
	case UPC_START_PROCESSORS:
		// tell the processors to start executing code loaded by upc_write
		upc_soft_reset(1);
		upc_object.upc_suspend_context.cpu_soft_reset = 1;
		printk(KERN_ERR "upc start  \n");
		break;
		// tel the processors to stop executing code
	case UPC_STOP_PROCESSORS:
		upc_soft_reset(0);
		upc_object.upc_suspend_context.cpu_soft_reset = 0;
#ifdef RICOH_SIMVA
		dev->upc_ifileptr = 0;
		dev->upc_dfileptr = 0;
		if ( r && regulator_is_enabled( r ) ) 
		{
			if (regulator_disable( r ))
				regulator_force_disable(r);
			printk("!!!!!!!! upc regulator disabled !!!!!!!!\n");
		}
		else
		{
			printk("!!!!!!!! upc regulator *doesn't* disabled !!!!!!!!\n");
		}
#endif
		printk(KERN_ERR "upc stop  \n");
// = upcRead(CPUSoftReset);
		break;
	case UPC_SELECT_ITCM:
		// Set flag indicating code read/write by upc_read/upc_write goes to ITCM memory.
		tcm_memory_select = ITCM_MEMORY;
		break;
	case UPC_SELECT_DTCM:
		// Set flag indicating code read/write by upc_read/upc_write goes to DTCM memory.
		tcm_memory_select = DTCM_MEMORY;
		break;
	case UPC_WRITE_MAILBOX: 
		if (copy_from_user(&a, (void __user *)arg, sizeof(a))) {
			error = -EFAULT;
			break;
		}
		if (upc_write_mailbox(a.boxnum, a.value)) {
			error = -EINVAL;
			break;
		}

		break;
	case UPC_READ_MAILBOX: 
		if (copy_from_user(&a, (void __user *)arg, sizeof(a))) {
			error = -EFAULT;
			break;
		}
#ifdef RICOH_SIMVA
		if(a.boxnum >= NUM_UPC_MAILBOX) {
			error = -EINVAL;
			break;
		}
		while (upc_object.rcv_irq == 0) {
			spin_unlock_irqrestore(&(upc_object.upc_reg_lock), flags);
			error = wait_event_interruptible(upc_object.read_queue,
							(upc_object.rcv_irq != 0));
			spin_lock_irqsave(&(upc_object.upc_reg_lock), flags);
			if (error)
				break;
		}
		printk("[%s] READ_MAILBOX\n", __FUNCTION__);
		if (error) {
			error = -EINTR;
			break;
		}
		a.value = upc_object.mailbox_data[a.boxnum]; /* omo 8/25 */
		upc_object.rcv_irq = 0;
		/*  RICOH : add  */
#else
		if (upc_read_mailbox(a.boxnum, &a.value)) {
			error = -EINVAL;
			break;
		}
#endif
		if (copy_to_user((void __user *)arg, &a, sizeof(a))) {
			error = -EFAULT;
			break;
		}

		break;
#ifdef RICOH_SIMVA
	case 5:
	case UPC_GET_FGATE:
		printk("[%s] GET_FGATE start\n", __FUNCTION__);
		while (upc_object.rcv_fgate == 0) {
			spin_unlock_irqrestore(&(upc_object.upc_reg_lock), flags);
			error = wait_event_interruptible(upc_object.fgate_queue,
							(upc_object.rcv_fgate!= 0));
			spin_lock_irqsave(&(upc_object.upc_reg_lock), flags);
			if (error)
				break;
		}
		if (error) {
			error = -EINTR;
			break;
		}
		upc_object.rcv_fgate = 0;
		spin_unlock_irqrestore(&(upc_object.upc_reg_lock), flags);
		if((&fgatebuf[0][0] <= wptr) && (wptr < &fgatebuf[0][FGATE_BUF_SIZE])) {
			copybuf = &fgatebuf[0][0];
			wptr = &fgatebuf[1][0];
		} else {
			copybuf = &fgatebuf[1][0];
			wptr = &fgatebuf[0][0];
		}
		printk("[%s] copybuf: %02x %02x %02x %02x %02x %02x %02x %02x\n",
			__FUNCTION__, 
			copybuf[0], copybuf[1],	copybuf[2], copybuf[3], copybuf[4], copybuf[5], copybuf[6], copybuf[7]);
		if(copy_to_user((void __user *)arg, copybuf, FGATE_BUF_SIZE)) {
			error = -EFAULT;
		}
		memset(copybuf, 0x00, FGATE_BUF_SIZE);

		printk("[%s] GET_FGATE end\n", __FUNCTION__);
		return(error);
		break;
#endif
	default:
		printk(KERN_CRIT"[upc driver] cmd error(cmd=%d)\n", cmd);
		error = -ENOTTY;
		break;
	}
	spin_unlock_irqrestore(&(upc_object.upc_reg_lock), flags);
	return error;
}

static loff_t upc_seek(struct file *file, loff_t offset, int origin)
{
	struct upc_data *dev = file->private_data;
	loff_t retval;
	int fp;
	int max_size;

	if (ITCM_MEMORY == tcm_memory_select) {
		fp = dev->upc_ifileptr;
		max_size = dev->itcm_size;
	} else {
		fp = dev->upc_dfileptr;
		max_size = dev->dtcm_size;
	}

	mutex_lock(&dev->upc_var_lock);
	/* switch implements generic seek, but bounds checking is below */
	switch (origin) {
        case SEEK_END:
		offset += max_size;
		break;
        case SEEK_CUR:
		if (offset == 0) {
			retval = fp;
			goto out;
		}
		offset += fp;
		break;
        case SEEK_DATA:
		/*
		 * In the generic case the entire file is data, so as
		 * long as offset isn't at the end of the file then the
		 * offset is data.
		 */
		if (offset >= max_size) {
			retval = -ENXIO;
			goto out;
		}
		break;
        case SEEK_HOLE:
		/*
		 * There is a virtual hole at the end of the file, so
		 * as long as offset isn't i_size or larger, return
		 * i_size.
		 */
		if (offset >= max_size) {
			retval = -ENXIO;
			goto out;
		}
		offset = max_size;
		break;
	}
	retval = -EINVAL;
	/* check bounds */
	if ( (offset >= 0) && (offset < max_size) ) {
		if (offset != fp) {
			if (ITCM_MEMORY == tcm_memory_select) {
				dev->upc_ifileptr = offset;
			} else {
				dev->upc_dfileptr = offset;
			}
		}
		retval = offset;
	}
out:
	mutex_unlock(&dev->upc_var_lock);
	return retval;
}

struct file_operations upc_fops =
{
	.owner = THIS_MODULE,
	.read = upc_read,
	.write = upc_write,
	.unlocked_ioctl = upc_ioctl,
	.open = upc_open,
	.release = upc_close,
	.llseek = upc_seek,
	
};

// called once for each device on install (called by probe)
static int __init upc_char_init(struct upc_data *dev_data, int deviceID)
{
	if (deviceID >= MAX_NUM_DRAGONITES) {
		printk(KERN_ERR "too many devices found!\n");
		return -EIO;
	}

	dev_data->dev = device_create(upc_object.upc_class, NULL, 
				      MKDEV(upc_object.upc_major, dev_data->dragonite_num),
				      NULL, "upc%u", dev_data->dragonite_num);
	if (IS_ERR(dev_data->dev)) {
		printk(KERN_ERR "Failed to create device: upc%d\n",dev_data->dragonite_num);
		return -1;
	}
    
	// register a character device
	cdev_init(&dev_data->upc_cdev, &upc_fops);
	dev_data->upc_cdev.owner = THIS_MODULE;

	if (cdev_add(&dev_data->upc_cdev, MKDEV(upc_object.upc_major, dev_data->dragonite_num), 1) < 0) {
		printk(KERN_ERR "Can't add device driver\n");
		return -1;
	}

	upc_object.upc_suspend_context.num_open = 0;
	upc_object.upc_suspend_context.cpu_soft_reset = 0;

	

	dev_data->cdev_open = 0; // init local boolean to unopened
	mutex_init(&dev_data->upc_var_lock);

	return 0;
}

// called once for each device on platform remove
// (cleaning up upc_char_init)
static void upc_char_cleanup(struct upc_data *dev_data)
{   
	mutex_destroy(&dev_data->upc_var_lock);
	cdev_del(&dev_data->upc_cdev);
	device_destroy(upc_object.upc_class, MKDEV(upc_object.upc_major, dev_data->dragonite_num));
}

// called at module install time
static int __init upc_onetime_init(void)
{
	dev_t upc_dev;
	int retval;
	struct device *dev;
    
	upc_object.upc_class = class_create(THIS_MODULE, "upc");
	if (IS_ERR(upc_object.upc_class)) {
		retval = PTR_ERR(upc_object.upc_class);
		printk(KERN_ERR "upc: unable to create class %d\n", retval);
		return retval;
	}

	retval = alloc_chrdev_region(&upc_dev, 0, MAX_NUM_DRAGONITES, "upc_processor");
	if (retval) {
		printk(KERN_ERR "can't register character driver error %d!!!\n", retval);
		class_destroy(upc_object.upc_class);
		return retval;
	}
	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (!dev)
		return -1;
	device_initialize(dev);
	dev->class = upc_object.upc_class;


	tcm_memory_select = ITCM_MEMORY;
	upc_object.upc_major = MAJOR(upc_dev);
	spin_lock_init(&(upc_object.upc_reg_lock)); // 1 reg set for all dragonites

	return retval;
}

// called at module removal time (cleanup upc_onetime_init)
static void upc_onetime_cleanup(void)
{
	unregister_chrdev_region(MKDEV(upc_object.upc_major,0), MAX_NUM_DRAGONITES);
	class_destroy(upc_object.upc_class);
}

// platform 

/**
 * upc_platform_irq: interrupt handler
 * @irq:  irq number
 * @pdev: interrupt source
 *
 * This function returns IRQ_HANDLED if the IRQ has been handled
 * This is an ISR don't trace, use attribute interface instead
 */
static irqreturn_t upc_platform_irq(int irq, void *pdev)
{
#ifdef RICOH_SIMVA
	int i;
	int j;
	unsigned long  mboxdata;
	unsigned long  changed;

#if defined(_MSUGI_DEBUG)
	printk(KERN_ALERT "upc get interrupt!!\n");
#endif
	for (i = 0; i < NUM_UPC_MAILBOX; i++) {
		upc_read_mailbox(i, &(upc_object.mailbox_data[i]));
	}
	upc_object.rcv_irq = 1;

	//  XXX : refine 
	//check fgate
	mboxdata = upc_object.mailbox_data[1];
	if((mboxdata&FGATE_MASK)!=fgate_value)
	{
		printk("[upc driver] detect fgate bit change.\n");
		changed = mboxdata^fgate_value;
		if(changed&FGATE_BLACK_BIT)
		{
			*wptr = (mboxdata&FGATE_BLACK_BIT)?FGATE_BLACK_NEGATE:FGATE_BLACK_ASSERT;
			wptr++;
			printk("get black fgate(%08x):fgate_value(%08x)\n", (mboxdata&FGATE_BLACK_BIT), fgate_value);
		}
		if(changed&FGATE_CYAN_BIT)
		{
			*wptr = (mboxdata&FGATE_CYAN_BIT)?FGATE_CYAN_NEGATE:FGATE_CYAN_ASSERT;
			wptr++;
			printk("get cyan fgate(%08x):fgate_value(%08x)\n", (mboxdata&FGATE_CYAN_BIT), fgate_value);
		}
		if(changed&FGATE_MAGENTA_BIT)
		{
			*wptr = (mboxdata&FGATE_MAGENTA_BIT)?FGATE_MAGENTA_NEGATE:FGATE_MAGENTA_ASSERT;
			wptr++;
			printk("get magenta fgate(%08x):fgate_value(%08x)\n", (mboxdata&FGATE_MAGENTA_BIT), fgate_value);
		}
		if(changed&FGATE_YELLOW_BIT)
		{
			*wptr = (mboxdata&FGATE_YELLOW_BIT)?FGATE_YELLOW_NEGATE:FGATE_YELLOW_ASSERT;
			wptr++;
			printk("get yellow fgate(%08x):fgate_value(%08x)\n", (mboxdata&FGATE_YELLOW_BIT), fgate_value);
		}
		fgate_value = (mboxdata&FGATE_MASK);
		upc_object.rcv_fgate = 1;
		wake_up_interruptible(&(upc_object.fgate_queue));
	}

	wake_up_interruptible(&(upc_object.read_queue));
	upc_set_gp_status_clear(0x1);
	return IRQ_HANDLED;
#else
    return IRQ_NONE;
#endif
}

static int upc_platform_probe(struct platform_device *pdev)
{
	struct resource *res;
	int retval;
	int irq;
	struct upc_data *dev_data;
	struct device_node *of_node = pdev->dev.of_node;
#ifdef RICOH_SIMVA
	init_waitqueue_head(&(upc_object.read_queue));
	init_waitqueue_head(&(upc_object.fgate_queue));
#endif
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "platform_get_resource failed\n");
		return -ENXIO;
	}
	
	irq = platform_get_irq(pdev, 0);
	if (irq < 0) {
		dev_err(&pdev->dev, "platform_get_irq failed\n");
		return -ENXIO;
	}
    
	dev_data = (struct upc_data *)kzalloc(sizeof(struct upc_data), GFP_KERNEL);
	if (!dev_data) {
		dev_err(&pdev->dev, "kzalloc failed\n");
		return -ENOMEM;
	}
	dev_data->irq = irq;
    
	if (!request_mem_region(res->start, resource_size(res), UPC_DRIVER_NAME)) {
		dev_err(&pdev->dev, "request_mem_region failed\n");
		retval = -EBUSY;
		goto req_mem_reg_failed;
	}
    
	dev_data->base = ioremap(res->start, resource_size(res));
	if (!dev_data->base) {
		dev_err(&pdev->dev, "ioremap failed\n");
		retval = -ENOMEM;
		goto ioremap_failed;
	}
    
	retval = request_irq(dev_data->irq, upc_platform_irq, 0, UPC_DRIVER_NAME, pdev);
	if (retval) {
		dev_err(&pdev->dev, "request_irq failed\n");
		goto request_irq_failed;
	}
    
#ifndef RICOH_SIMVA
	disable_irq(dev_data->irq); // since we really don't have any interrupts
	                            // disable the irq we just requested
#endif
	
	of_property_read_u32(of_node, "id", &dev_data->dragonite_num);
	of_property_read_u32(of_node, "register_address", &dev_data->register_addr);
	of_property_read_u32(of_node, "itcm_address", &dev_data->itcm_addr);
	of_property_read_u32(of_node, "itcm_size", &dev_data->itcm_size);
	if (dev_data->itcm_size > MAX_DRAGONITE_ITCM_SIZE) {
		dev_err(&pdev->dev, "Bad ITCM Size - fix dtsi or MAX_* macro in this driver");
		goto upc_probe_failed;
	}
	of_property_read_u32(of_node, "dtcm_address", &dev_data->dtcm_addr);
	of_property_read_u32(of_node, "dtcm_size", &dev_data->dtcm_size);

	retval = upc_char_init(dev_data, dev_data->dragonite_num);
	if (retval) {
		dev_err(&pdev->dev, "upc_probe failed\n");
		goto upc_probe_failed;
	}
	
	if ( dev_data->dragonite_num == 0 && pegmatite_reg_check()) 
		upc_object.upc_suspend_context.power = regulator_get(dev_data->dev, "pegmatite_upc");

	upc_object.upc_data_array[dev_data->dragonite_num] = dev_data; // register with our private upc "framework"
	platform_set_drvdata(pdev, dev_data);                  // register as a platform driver
	// printk(KERN_ERR "UPC%d: device registration complete\n", dev_data->dragonite_num);
        
	return 0;
    
upc_probe_failed:
	free_irq(dev_data->irq, pdev);
request_irq_failed:
	iounmap(dev_data->base);
ioremap_failed:
	release_mem_region(res->start, resource_size(res));
req_mem_reg_failed:
	kfree(dev_data);
	return retval;
}


static int upc_platform_remove(struct platform_device *pdev)
{
	struct resource *res;
	struct upc_data *dev_data = platform_get_drvdata(pdev);
	
	upc_char_cleanup(dev_data);
    
	free_irq(dev_data->irq, pdev); 
	iounmap(dev_data->base);
    
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	release_mem_region(res->start, resource_size(res));
	if ( dev_data->dragonite_num == 0 )
		regulator_put( upc_object.upc_suspend_context.power );
	// printk(KERN_ERR "UPC%d: device remove complete\n", dev_data->dragonite_num);

	kfree(dev_data);
	platform_set_drvdata(pdev, NULL);

	return 0;
}

static int upc_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
//	struct upc_data *dev_data = platform_get_drvdata(pdev);
	struct regulator *r = upc_object.upc_suspend_context.power;
	printk(KERN_WARNING "%s event = %d\n",__func__, state.event);

	if ( !upc_object.upc_suspend_context.num_open && 
	     !upc_object.upc_suspend_context.cpu_soft_reset ) {
		if ( r && !regulator_is_enabled( r ) ) {
			regulator_put( r );
			return 0;
		}
	}
	return -1;
}

static int upc_platform_resume(struct platform_device *pdev)
{
	printk(KERN_WARNING "%s\n",__func__);
	return(0);
}

static struct of_device_id upc_of_match[] = {
	{ .compatible = UPC_DRIVER_NAME, },
	{}
};

MODULE_DEVICE_TABLE(of, upc_of_match);

static struct platform_driver upc_platform_driver =
{
	.probe = upc_platform_probe,
	.remove  = upc_platform_remove,
	.suspend = upc_platform_suspend,
	.resume = upc_platform_resume,   
	.driver  = {
        .name  = UPC_DRIVER_NAME,
		.owner = THIS_MODULE,
		.of_match_table = upc_of_match,
	},
};

/**
 * upc_platform_init: module init
 *
 * Driver load
 */
static int __init upc_platform_init(void)
{
    int retval;
    
    retval = upc_onetime_init();
    if (retval)
    {
        printk(KERN_ERR "%s: error initializing driver\n", __func__);
    }
    else
    {
        retval = platform_driver_register(&upc_platform_driver);
        if (retval)
        {
            upc_onetime_cleanup();
            printk(KERN_ERR "%s: error registering platform driver\n", __func__);
        }
        else
        {
            printk(KERN_ERR "UPC:  platform registration complete\n");
        }
    }
        return retval;
}
module_init(upc_platform_init);


/**
 * upc_platform_exit: module exit
 *
 * Driver unload
 */
static void __exit upc_platform_exit(void)
{
    platform_driver_unregister(&upc_platform_driver);
    upc_onetime_cleanup();
    printk(KERN_ERR "UPC:  platform remove complete\n");
}
module_exit(upc_platform_exit);

MODULE_AUTHOR("Copyright (c) 2011 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Marvell Universal Print Controller (UPC) Driver");

MODULE_LICENSE("Dual MPL/GPL");
MODULE_VERSION("2012_Aug_2");

