#include <stdint.h>
#include "sha256.h"
#include "fcert_v3.h"
#include "polarssl/bignum.h"

#define KEY_LEN 256

static const uint8_t pub_modulus[] = {
    0xdd, 0xad, 0x4c, 0x60, 0xe9, 0xbe, 0x89, 0x20, 0xb9, 0x94, 0xc9, 0x83, 0x99, 0x11, 0x4b, 0x63,
    0x3d, 0x98, 0xeb, 0x41, 0x03, 0x53, 0x69, 0x41, 0xd5, 0x75, 0x36, 0x5b, 0xb5, 0x10, 0xc0, 0x8b,
    0x14, 0x22, 0x44, 0xfc, 0x63, 0x93, 0x24, 0x31, 0x9a, 0x57, 0x65, 0xc7, 0x97, 0x01, 0xa8, 0x2c,
    0x0a, 0xfc, 0x22, 0x68, 0x60, 0x2d, 0x9a, 0x26, 0xa6, 0x4b, 0xf3, 0x52, 0x90, 0x53, 0x43, 0x7d,
    0x86, 0x21, 0x9e, 0x26, 0x9b, 0x66, 0x6a, 0x1a, 0x49, 0x52, 0x2d, 0x16, 0x5a, 0xe9, 0x5c, 0x55,
    0x86, 0x73, 0x44, 0x83, 0x18, 0xa5, 0x24, 0xa3, 0x0e, 0xb8, 0x1c, 0x82, 0x90, 0xb4, 0xd1, 0x8c,
    0x9b, 0x61, 0x4d, 0xea, 0x53, 0x7a, 0xab, 0xd7, 0x2a, 0x6b, 0x61, 0x37, 0x88, 0x79, 0x1c, 0x2a,
    0xc7, 0xc7, 0x7f, 0x0a, 0xf5, 0xd1, 0x6a, 0x13, 0x3c, 0xba, 0x33, 0x7f, 0xff, 0xff, 0x79, 0x8b,
    0xd4, 0x33, 0xe1, 0x69, 0x9c, 0x64, 0xda, 0x0c, 0xee, 0xe6, 0x0a, 0x0d, 0xbb, 0x63, 0x7d, 0xb3,
    0xf7, 0xbe, 0x4e, 0xee, 0x07, 0x32, 0xc0, 0x3f, 0x66, 0xda, 0x93, 0xbd, 0xfc, 0x58, 0x5d, 0x82,
    0xff, 0xad, 0x3c, 0xc2, 0x3a, 0x88, 0x8e, 0xfc, 0x3b, 0xe2, 0x90, 0x30, 0x1e, 0xbf, 0x5c, 0x3d,
    0xc2, 0xa8, 0x32, 0x99, 0x2a, 0x66, 0xca, 0x2e, 0xd3, 0xb7, 0x7a, 0x29, 0xd3, 0xae, 0x7c, 0x49,
    0xc6, 0xdc, 0xde, 0x48, 0xb4, 0x46, 0x9f, 0xcc, 0xfd, 0x6c, 0xe1, 0x89, 0xe8, 0x7e, 0x33, 0x92,
    0xb9, 0x72, 0xd9, 0x4f, 0x4f, 0xad, 0x8a, 0xc7, 0x25, 0x4a, 0x5d, 0x9c, 0xd1, 0x8b, 0x4c, 0xa9,
    0x77, 0x5a, 0x4e, 0x38, 0x80, 0x87, 0x27, 0x98, 0x2d, 0x57, 0x22, 0x52, 0xd8, 0x53, 0xd1, 0xe6,
    0x90, 0xbd, 0x01, 0x52, 0x14, 0xc2, 0x03, 0x25, 0x3e, 0x1e, 0xf6, 0xbe, 0x92, 0xdb, 0xb5, 0x85
};

static const uint8_t pub_exponent[] = { 0x00, 0x01, 0x00, 0x01 };

static const uint8_t sha256_encoding_string[] = {
    0x00, 0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60,
    0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01,
    0x05, 0x00, 0x04, 0x20
};

static int decrypt_signature(const uint8_t *signature, const uint8_t *exponent, const uint8_t *modulus, uint8_t *output) {
    int ret = 0;
    mpi T, E, N, RN;

    mpi_init(&T);
    mpi_init(&E);
    mpi_init(&N);
    mpi_init(&RN);

    MPI_CHK(mpi_read_binary(&T, signature, KEY_LEN));
    MPI_CHK(mpi_read_binary(&E, exponent, FCERT_V3_RSA_KEY_EXPONENT_LEN));
    MPI_CHK(mpi_read_binary(&N, modulus, KEY_LEN));

    if (mpi_cmp_mpi(&T, &N) >= 0) {
        ret = -1;
        goto cleanup;
    }

    MPI_CHK(mpi_exp_mod(&T, &T, &E, &N, &RN));
    MPI_CHK(mpi_write_binary(&T, output, KEY_LEN));

cleanup:
    mpi_free(&T);
    mpi_free(&E);
    mpi_free(&N);
    mpi_free(&RN);

    return ret;
}

static int rsa_verify(const uint8_t *signature, const uint8_t *exponent, const uint8_t *modulus, const uint8_t *hash, const size_t hash_len) {
    int rc;
    int i;
    uint8_t output[KEY_LEN];

    /* Decrypt the signature */
    rc = decrypt_signature(signature, exponent, modulus, output);
    if (rc < 0)
        return -1;

    /* Compare padding */
    if ((output[0] != 0) || (output[1] != 1))
        return -1;

    for (i = 2; i < (int)(KEY_LEN - hash_len - sizeof(sha256_encoding_string)); ++i)
        if (output[i] != 0xff)
            return -1;

    if (memcmp(&output[KEY_LEN - hash_len - sizeof(sha256_encoding_string)], sha256_encoding_string, sizeof(sha256_encoding_string)))
        return -1;

    /* Compare decrypted hash to calculated hash */
    return memcmp(hash, &output[KEY_LEN - hash_len], hash_len);
}

int verify_code(const uint8_t *code_hash, const struct fcert_v3 *cert) {
    uint8_t cert_hash[SHA256_DIGEST_LEN];

    sha256_csum_wd((unsigned char *)cert, sizeof(*cert) - FCERT_V3_SIGNATURE_LEN, cert_hash, 0);

    /*
     * Verify the hash of the cert minus the signature is the same as the hash encrypted
     * in the signature
     */
    if (rsa_verify(cert->sig, pub_exponent, pub_modulus, cert_hash, sizeof(cert_hash)))
        return -1;

    /* Verify the hash computed against the hash in the cert */
    return (memcmp(code_hash, cert->code.sha256, SHA256_DIGEST_LEN)); 
}
