#include <asm/arch/gpio.h>
#include "mv61fb.h"
#include <lcd.h>
#include <mv_mmp_panel.h>
#include "mv_mmp_eeprom_info.h"
#include <asm/arch/board_priv.h>

#define LCD_BASE (0xF9070000)
#define REF_RATE_HZ (60)
#define LCD_MAX_COLS (800)
#define LCD_MAX_ROWS (480)
#define LCD_COLOR_DEPTH (LCD_COLOR16)

#define SHOWREGW(base, reg)  serial_printf("offset 0x%04x = 0x%08x\n", \
						(unsigned int)reg, \
						((u32)readl(base + reg)))

void lcd_enable (void)
{
}

vidinfo_t panel_info = 
{
       vl_col  : LCD_MAX_COLS,
       vl_row  : LCD_MAX_ROWS,
       vl_bpix : LCD_COLOR_DEPTH,
       cmap    : NULL,
       priv    : NULL,
       refresh : NULL,
};

/**
 * mv_mmp_vram_size - get video buffer size
 * 
 * This will return the buffer size allocated by lcd_setmem(), which
 * happens before the real size is known.
 */
unsigned int mv_mmp_vram_size(void)
{
	unsigned int size;
	int line_length = (LCD_MAX_COLS * NBITS(LCD_COLOR_DEPTH)/ 8);

	size = line_length * LCD_MAX_ROWS;

	/* Round up to nearest full page */
	size = (size + (PAGE_SIZE - 1)) & ~(PAGE_SIZE - 1);
	
	return size;
}

/**
 * mv_mmp_eeprom_version_ge - test for minimum eeprom version
 * @major - minimum version major
 * @minor - minimum version minor
 * @vpi - pointer to eeprom data image
 */
int mv_mmp_eeprom_version_ge(int major, int minor, vga_panel_info *vpi) {
	return ((vpi->version_major > major) ||
				((vpi->version_major == major) &&
				(vpi->version_minor >= minor)));
}

/**
 * mv_mmp_eeprom_version_eq - test for exact eeprom version
 * @major - required version major
 * @minor - required version minor
 * @vpi - pointer to eeprom data image
 */
int mv_mmp_eeprom_version_eq(int major, int minor, vga_panel_info *vpi) {
	return ((vpi->version_major == major) ||
				(vpi->version_minor == minor));
}

/**
 * mv_mmp_eeprom_version_blank - test for blank or corrupted version
 * @vpi - pointer to eeprom data image
 *
 * Blank or brownout-corrupted state is usually 0xff, so make it illegal in the
 * version fields.
 */
int mv_mmp_eeprom_version_blank(vga_panel_info *vpi) {
    /* v1.255 is valid */
    return ((vpi->version_major == 0xff)
            || (vpi->version_minor == 0xff && vpi->version_major != 1));
}

/**
 * mv_mmp_eeprom_checksum - verify eeprom contents if supported
 * @vpi - pointer to eeprom data image
 */
int mv_mmp_eeprom_checksum(vga_panel_info *vpi) {
  if(!vpi) {
     printf("%s:%d: vpi null\n", __func__, __LINE__);
     return -1;
  }
  
  if(mv_mmp_eeprom_version_ge(2, 3, vpi)) {
     uint16_t eechecksum = 0;
     int check_len = offsetof(vga_panel_info, check_msb);
     int i;
     uint8_t *data = (uint8_t *)vpi;

     for(i=0;i<check_len;i++) {
        /* 
	 * Assume data could be placed after checksum bytes
	 * in later versions.
	 */
        if((i == offsetof(panel_info_t, check_msb)) ||
          		(i == offsetof(panel_info_t, check_lsb))) {
           continue;
        }
        /* panel_info.check_*sb included in sum, this time probably non-zero */
        eechecksum += data[i] & 0x0ff; 
     }
     eechecksum += (data[offsetof(panel_info_t, check_msb)] & 0x0ff) << 8;
     eechecksum += data[offsetof(panel_info_t, check_lsb)] & 0x0ff;
     if(eechecksum != (uint16_t)MV_MMP_EEPROM_CHECKSUM_MAGIC) {
        printf("%s:%d: Uicc eeprom corrupted\n",__func__, __LINE__);
        return -1;
     }
     
     return 0;
  }
  else {
     printf("%s:%d: Checksum not supported\n", __func__, __LINE__);
     return 0;
  }
}
  
void mv_mmp_mv61fb_dump_regs(void *reg_base)
{
#ifdef __MV_MMP_VGA_MV61FB_DEBUG__
	printf("******** LCD2 registers ********\n");

	SHOWREGW(reg_base, LCD_SOFT_RESET);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_Y0);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_U0);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_V0);
	SHOWREGW(reg_base, LCD_CFG_DMA_START_ADDR_0);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_Y1);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_U1);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_V1);
	SHOWREGW(reg_base, LCD_CFG_DMA_START_ADDR_1);
	SHOWREGW(reg_base, LCD_SPU_DMA_PITCH_YC);
	SHOWREGW(reg_base, LCD_SPU_DMA_PITCH_UV);
	SHOWREGW(reg_base, LCD_SPUT_DMA_OVSA_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPU_DMA_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPU_DZM_HPXL_VLN);
	SHOWREGW(reg_base, LCD_CFG_GRA_START_ADDR0);
	SHOWREGW(reg_base, LCD_CFG_GRA_START_ADDR1);
	SHOWREGW(reg_base, LCD_CFG_GRA_PITCH);
	SHOWREGW(reg_base, LCD_SPU_GRA_OVSA_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPU_GRA_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPU_GZM_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPUT_V_H_TOTAL);
	SHOWREGW(reg_base, LCD_SPU_V_H_ACTIVE);
	SHOWREGW(reg_base, LCD_SPU_H_PORCH);
	SHOWREGW(reg_base, LCD_SPU_V_PORCH);
	SHOWREGW(reg_base, LCD_SPU_BLANKCOLOR);
	SHOWREGW(reg_base, LCD_SPU_COLORKEY_Y);
	SHOWREGW(reg_base, LCD_SPU_COLORKEY_U);
	SHOWREGW(reg_base, LCD_SPU_COLORKEY_V);
	SHOWREGW(reg_base, LCD_PN_SEPXLCNT);
	//SHOWREGW(reg_base, LCD_SPU_ISA_RSDATA);
	SHOWREGW(reg_base, LCD_READ_IOPAD);
	SHOWREGW(reg_base, LCD_DMAVLD_YC);
	SHOWREGW(reg_base, LCD_DMAVLD_UV);
	SHOWREGW(reg_base, LCD_TVGGRAVLD_HLEN);
	SHOWREGW(reg_base, LCD_SPU_GAMMA_RDDAT);
	SHOWREGW(reg_base, LCD_SPU_PALETTE_RDDAT);
	SHOWREGW(reg_base, LCD_SLV_DBG);
	SHOWREGW(reg_base, LCD_FRAME_CNT);
	SHOWREGW(reg_base, LCD_SPU_SMPN_CTRL);
	SHOWREGW(reg_base, LCD_SLV_PORT);
	SHOWREGW(reg_base, LCD_SPU_DMA_CTRL0);
	SHOWREGW(reg_base, LCD_SPU_DMA_CTRL1);
	SHOWREGW(reg_base, LCD_SPU_SRAM_CTRL);
	SHOWREGW(reg_base, LCD_SPU_SRAM_WRDAT);
	SHOWREGW(reg_base, LCD_SPU_SRAM_PARA1);
	SHOWREGW(reg_base, LCD_CFG_SCLK_DIV);
	SHOWREGW(reg_base, LCD_SPU_CONTRAST);
	SHOWREGW(reg_base, LCD_SPU_SATURATION);
	SHOWREGW(reg_base, LCD_SPU_CBSH_HUE);
	SHOWREGW(reg_base, LCD_SPU_DUMB_CTRL);
	SHOWREGW(reg_base, SPU_IOPAD_CONTROL);
	SHOWREGW(reg_base, SPU_IRQ_ENA);
	SHOWREGW(reg_base, SPU_IRQ_ISR);
	SHOWREGW(reg_base, SPU_IRQ_RSR);
	SHOWREGW(reg_base, LCD_GRA_CUTHPXL);
	SHOWREGW(reg_base, LCD_GRA_CUTVLN);
	SHOWREGW(reg_base, LCD_TOP_CTRL);
	SHOWREGW(reg_base, LCD_AFA_ALL2ONE);
	SHOWREGW(reg_base, LCD_MISR_CONTROL);
	SHOWREGW(reg_base, LCD_MISR_STATUS);

	printf("********************************\n");
#endif
}

unsigned int mv_mmp_mv61fb_get_clk_div(unsigned target_hz) {
	unsigned int rem_hz, div_val;

	/* Calculate the divider value -- round to the closest match */
	div_val = (board_get_core_freq_hz() / target_hz);
	rem_hz = board_get_core_freq_hz() % target_hz;
	if (rem_hz >= target_hz/2)
		div_val++;
	
	/* clamp to legal values */
	if(div_val > 256)
		div_val = 256;
	if(div_val < 8)
		div_val = 8;
		
	/* actual hw divider is reg value + 1 */
	div_val -= 1;

	return div_val;
}

void mv_mmp_mv61fb_default_gamma(void *reg_base)
{
	/* allow writing gamma sram even if gamma is disabled */
	writel((CFG_CSB_256x24(0x1)|CFG_CSB_256x8(0x1)), 
					(reg_base + LCD_SPU_SRAM_PARA1));
	/* init gamma sram to direct-mapped */
	writel(CFG_SRAM_INIT_WR_RD(3), (reg_base + LCD_SPU_SRAM_CTRL));
	udelay(25);
	writel(CFG_SRAM_INIT_WR_RD(0), (reg_base + LCD_SPU_SRAM_CTRL));
	/* disable writing gamma sram unless gamma is enabled */
	writel(0, (reg_base + LCD_SPU_SRAM_PARA1));

	dmb();
	
}

void mv_mmp_mv61fb_paint_buffer(void *buffer, unsigned int bufsize)
{ 
#ifdef __MV_MMP_VGA_MV61FB_DEBUG__
	/*
	 * Paint buffer with some dummy data for debug.
	 */
	int i;
	short tmp;
	short *ip = (short *)buffer;
		
	for (i = 0; i < (bufsize >> 1); i++) {
		tmp = (i & 0x0ff);
		tmp |= tmp << 8;
		if(i & 32) {
			ip[i] = tmp;
		}
		else {
			if(i & 8)
				ip[i] = 0x7fef;
			else
				ip[i] = 0xfbff;
		}
	}
#else
	memset(buffer, 0, bufsize);
#endif
}

/*
 */
void lcd_ctrl_init (void *buffer)
{
	void *reg_base = (void *)LCD_BASE;
	MV_MMP_PANEL_TYPE display_type;

	display_type = mv_mmp_panel_get_type();
	MV_MMPPANELDBG("%s:%d: panel display type = %d\n", __func__, __LINE__, display_type);

	if(display_type == MV_MMP_PANEL_TYPE_INVALID) {
	        printf("%s:%d: no vga panel found\n", __func__, __LINE__);
		return;
	}
		
	switch(display_type) {
#ifdef CONFIG_MV_MMP_PANEL_SMPN
	case MV_MMP_PANEL_TYPE_320X240:
		mv_mmp_mv61fb_smart_panel_init(reg_base, buffer);
		break;
#endif
	case MV_MMP_PANEL_TYPE_480X272:
	case MV_MMP_PANEL_TYPE_480X272_NON_SEQ:
	case MV_MMP_PANEL_TYPE_800X480:
	case MV_MMP_PANEL_TYPE_800X480_SCALED:
	case MV_MMP_PANEL_TYPE_480X272_FIXW:
	case MV_MMP_PANEL_TYPE_800X480_FIXW:
		mv_mmp_mv61fb_dumb_panel_init(reg_base, buffer);
		break;
	default:
		return;
	}
}																																																	
