#include <common.h>
#include <i2c.h>
#include <asm/io.h>
#include <lcd.h>
#include <mv_mmp_panel.h>
#include <stdio_dev.h>
#include <lcd.h>
#include <mv_mmp_panel.h>
#include <mv_mmp_panel.h>
#include <asm/arch/board_priv.h>
#include <video_font.h>
#include "mv_mmp_eeprom_info.h"
#include "mv_mmp_panel_common.h"

#define I2CADDR_PANEL_JAM (0x48 >> 1) /* MFP */
#define I2CADDR_PANEL_BAH (0x4C >> 1) /* SFP */

static int ext_clock_required = MV_MMP_PANEL_EXT_CLK_USED;

static lb_vga_panel_data vpd = {
	.subtype = NULL,
	.use_smpnclk = 0,
	.use_te = 0,
	.backlight_enable = NULL,
	.gamma = NULL,
};

#ifndef CONFIG_PANEL_I2C_BUS
#error CONFIG_PANEL_I2C_BUS must be defined in board config
#endif
static mv_mmp_panel_i2c_info_t i2c_info = {
	.bus_num = -1,
	.read_retries = 0,
	.write_retries = 0,
	.delay = 1000,
	.protocol = PROTOCOL_GEN1,
};

static uint8_t hw_ver = 0;

/*-----------------------------------------------------------------------------
 *
 * Name:        mv_mmp_panel_vga_extclkreq
 *
 * Description: Report whether the display requires external free-running clock.
 *
 * Input:       None
 *
 * Output:
 *
 * Notes:       Clock may still be required internally by lcd block even if not
 *              used externally.
 *---------------------------------------------------------------------------*/
mv_mmp_panel_ext_clk_requirement_t mv_mmp_panel_lb_vga_extclkreq(void)
{
	return ext_clock_required;
}

int panel_lb_keyscan_init(void)
{
   int rc = 0;
   int rows, cols;

   /* Key scan */
   if(i2c_info.dev_addr == I2CADDR_PANEL_BAH)
   {
      /* SFP -- Bahamas */
      rows = 5;
      cols = 4;
   }
   else if(i2c_info.dev_addr == I2CADDR_PANEL_JAM)
   {
      /* MFP -- Jamaica */
      rows = 6;
      cols = 5;
   }
   else
   {
      rc = -1;
   }
   
   if(!rc)
      rc =  mv_mmp_panel_keyscan_init(&i2c_info, rows, cols);

   return rc;
}

int panel_lb_led_init(void)
{
   int cmd, rc;
   unsigned char msg[2];
   unsigned char buff;

   cmd = MCU_GPIOD_OUT_REG;
   rc = mv_mmp_panel_i2c_read(&i2c_info, cmd,1,&buff,1);
   if(rc)
      return rc;
   msg[0] = cmd;
   buff |= 0x80; /* red @ PD7 - off */
   buff &= ~0x40; /* green @ PD6 - on */
   msg[1] = buff;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

   return rc;
}

int panel_lb_init(void)
{
   int cmd, rc;
   unsigned char msg[2];
   unsigned char buff;

   cmd = MCU_IRQ_STATUS_REG;
   rc = mv_mmp_panel_i2c_read(&i2c_info, cmd,1,&buff,1);
   if(rc)
      return rc;

   msg[0] = MCU_ALT_FUNC_CONFIG_REG;
   msg[1] = 0x03;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
   if(rc)
      return rc;

   msg[0] = MCU_GPIOB_OER_REG;
   /* keep default setting of two pins most likely to be resets */
   rc = mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], sizeof(msg[1]));
   if(rc)
      return rc;
   msg[1] &= 0x80;
   msg[1] |= 0x3D;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
   if(rc)
      return rc;

   msg[0] = MCU_GPIOB_OUT_REG;
   msg[1] = 0x80;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
   if(rc)
      return rc;

   msg[1] = 0x0;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
   if(rc)
      return rc;

   msg[1] = 0x80;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
   if(rc)
      return rc;

   msg[0] = MCU_GPIOD_OER_REG;
   msg[1] = 0x3F;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

   return rc;
}

/*-----------------------------------------------------------------------------
 *
 * Name:        backlightEnable
 *
 * Description: turn backlight off or on using duty cycle
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:
 *---------------------------------------------------------------------------*/
static int backlightEnable(int enable) {
  unsigned char
    data[2];
  int rc;

  data[0] = PWM_OCR1AL;

  if(!enable) {
    /* turn off backlight by setting pwm duty to min */
    data[1] = 0;
  }
  else {
    vga_panel_info *vpi = (vga_panel_info *)&vpd.vpi;

    /* turn on backlight by setting pwm duty to default */
    if((!mv_mmp_eeprom_version_blank(vpi) && mv_mmp_eeprom_version_ge(2, 3, vpi))) {
       data[1] = vpi->backlight_pwm_value;
    }
    else {
       data[1] = 0xff;
    }
  }

  rc = mv_mmp_panel_i2c_write(&i2c_info, data, 2);

  return rc;
}

int panel_lb_bl_init(void)
{
   unsigned char WGM10, WGM11, WGM12, WGM13;
   int cmd, rc;
   unsigned char data[2];
   unsigned char cntr_clock,
                 mode,
                 cmpr_level;

   cntr_clock = 0x01; /* div 1 = no prescaling, use system clock of 8 MHz */
   mode = 0x05; /* fast PWM, top = 0xff */
   cmpr_level = 0;            // Sets the level of the PWM output (goes low during compare match OCR1AL)

   WGM10 = mode & 1;
   WGM11 = (mode & 2) >> 1;
   WGM12 = (mode & 4) >> 2;
   WGM13 = (mode & 8) >> 3;

   cmd = MCU_ALT_FUNC_CONFIG_REG;
   rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, data, 1);
   if(rc)
      return rc;

   data[0] |= 0x02; /* PWM enable */
   data[1] = data[0];
   data[0] = MCU_ALT_FUNC_CONFIG_REG;
   rc = mv_mmp_panel_i2c_write(&i2c_info, data, 2);
   if(rc)
      return rc;

   backlightEnable(0);

   data[0] = PWM_CONFIG;
   data[1] = ((cmpr_level << 7) | (WGM11 << 6) | (WGM10 << 5) | (WGM13 << 4) | (WGM12 << 3) | cntr_clock); //0xD9;
   rc = mv_mmp_panel_i2c_write(&i2c_info, data, 2);

   return rc;
}

int mv_mmp_panel_lb_vga_probe(void)
{
  char porKeys[11];
  int cmd;
  unsigned char *dest, data[2];
  int rc = -1;
  vga_panel_info *vpi = (vga_panel_info *)&vpd.vpi;

  MV_MMPPANELDBG("%s:%d\n", __func__, __LINE__);
  
  i2c_info.dev_addr = I2CADDR_PANEL_BAH;

  rc = mv_mmp_panel_i2c_probe(&i2c_info);
  if(rc) {
     MV_MMPPANELDBG("%s:%d: communication error or no response, try alternate address\n",
     								__func__, __LINE__);
     i2c_info.dev_addr = I2CADDR_PANEL_JAM;

     rc = mv_mmp_panel_i2c_probe(&i2c_info);
     
     if(rc) {
        MV_MMPPANELDBG("%s:%d: communication error or no response\n", __func__, __LINE__);
	return rc;
     }
  }

  rc = mv_mmp_panel_reset_mcu(&i2c_info);

  if(rc >= 0)
     rc = panel_lb_init();

  if(rc) {
     MV_MMPPANELDBG("%s:%d: communication error or no response\n", __func__, __LINE__);
     return rc;
  }

  /* read eeprom, might eventually need this before panel_lb_init */
  data[0] = MCU_EEPROM_ADDR_REG;
  data[1] = 0x00;
  mv_mmp_panel_i2c_write(&i2c_info, data, sizeof(data));

  cmd = MCU_EEPROM_DATA_REG;
  dest = (unsigned char *) vpi;
  rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, dest, sizeof(vga_panel_info));
  if (rc) {
    MV_MMPPANELDBG("%s:%d: couldn't read eeprom, bailing out...\n", __func__, __LINE__);
    return rc;
  }

  /* get hardware version, byte 63 */
  data[0] = MCU_EEPROM_ADDR_REG;
  data[1] = 0x3f;
  mv_mmp_panel_i2c_write(&i2c_info, data, sizeof(data));

  cmd = MCU_EEPROM_DATA_REG;
  rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, &hw_ver, sizeof(hw_ver));
  if (rc) {
    MV_MMPPANELDBG("%s:%d: couldn't read eeprom, bailing out...\n", __func__, __LINE__);
    return rc;
  }

  rc = panel_lb_keyscan_init();

  if(rc) {
     MV_MMPPANELDBG("%s:%d : keyscan init failed\n", __func__, __LINE__);
     return rc;
  }
  rc = panel_lb_led_init();

  if(rc) {
     MV_MMPPANELDBG("%s:%d : led init failed\n", __func__, __LINE__);
     return rc;
  }

  rc = mv_mmp_eeprom_version_blank(vpi);
  if (rc == 1 || hw_ver == 0x0ff) {
    /* blank EEPROM -- use defaults */
    vpd.use_smpnclk = 0;
    vpd.use_te = 0;
    vpd.smpn_type = MV_MMP_PANEL_SMPN_TYPE_TM024HDH49;
    vpd.smpn_info = NULL; /* let mv_mmp_mv61fb_smart_panel_init() populate this */
    vpd.xres = 320;
    vpd.yres = 240;
    vpi->flags1 = 0;
    vpi->flags2 = 0x0f;
    vpi->flags3 = 0;
    vpi->lvds_type = 0xff; /* unknown */
    printf("%s:%d: Uicc eeprom corrupted\n",__func__, __LINE__);
    printf("Using defaults\n");
  } else {
    /* programmed EEPROM -- do some checks */
    rc = mv_mmp_eeprom_checksum(vpi);
    if(rc < 0) {
      printf("%s:%d: checksum failed\n", __func__, __LINE__);
      return 1;
    }

    vpd.smpn_info = NULL; /* let mv_mmp_mv61fb_smart_panel_init() populate this */

    rc = mv_mmp_eeprom_version_ge(2, 2, vpi);
    if (rc == 1) {
      switch (vpi->lvds_type) {
        case 2:
        case 3:
        case 4:
          vpd.use_smpnclk = 1;
          break;
        default:
          vpd.use_smpnclk = 0;
	  if(mv_mmp_eeprom_version_ge(2, 3, vpi))
	     ext_clock_required = MV_MMP_PANEL_EXT_CLK_UNUSED;
          break;
      } /* switch */

      if ((hw_ver >= 2) && board_supports_lcd_vsync_in()) {
        vpd.use_te = 1;
      }

      vpd.xres = ((vpi->xres_msb & 0xff) << 8) | (vpi->xres_lsb & 0xff);
      vpd.yres = ((vpi->yres_msb & 0xff) << 8) | (vpi->yres_lsb & 0xff);
      vpd.smpn_type = vpi->display_type;
    } else {
      /* use defaults */
      vpd.use_smpnclk = 0;
      vpd.use_te = 0;
      vpd.smpn_type = MV_MMP_PANEL_SMPN_TYPE_TM024HDH49;
      vpd.smpn_info = NULL; /* let mv_mmp_mv61fb_smart_panel_init() populate this */
      vpd.xres = 320;
      vpd.yres = 240;
      vpi->flags1 = 0;
      vpi->flags2 = 0x0f;
      vpi->flags3 = 0;
      vpi->lvds_type = 0xff; /* unknown */
      printf("%s:%d: Uicc eeprom version not supported\n",__func__, __LINE__);
      printf("Using defaults\n");
    } /* version check */
  } /* blank check */

  /* reset assertion sequence */
  rc = mv_mmp_panel_disp_ctrl(&i2c_info, vpi, SECONDARY_RESET_OUT, 0);
  if(!rc)
     rc = mv_mmp_panel_disp_ctrl(&i2c_info, vpi, PRIMARY_RESET_OUT, 0);
  if(rc) {
        MV_MMPPANELDBG("%s:%d: communication error or no response asserting reset\n",
							__func__, __LINE__);
	return rc;
  }

  sprintf(porKeys, "0x%08x", (unsigned int)mv_mmp_panel_lb_vga_buttons());
  setenv("PORKEYS", porKeys);

  /* panel_info defaults to the maximum legal values, which predetermine the buffer size */
  if((vpd.yres < 0) || (vpd.yres > panel_info.vl_row) ||
		(vpd.xres < 0) || (vpd.xres > panel_info.vl_col)) {
     printf("%s:%d: unsupported display size rows=%d cols=%d\n", __func__, __LINE__,
     								vpd.yres, vpd.xres);
     printf("Uicc eeprom corrupted\n");
     return -1;
  }

  if(PANEL_FLAGS1_PIXEL_DEPTH(vpi->flags1) != PIXEL_DEPTH_16) {
     printf("%s:%d: unsupported color depth, flags1=0x%02x\n", __func__, __LINE__,
     								vpi->flags1);
     printf("Uicc eeprom corrupted\n");
     return -1;
  }

  /* update the defaults to real values for common/lcd.c*/
  panel_info.vl_row = vpd.yres;
  panel_info.vl_col = vpd.xres;

  /* hardware info for mv_mmp_vga_mv61fb */
  panel_info.priv = &vpd;

  lcd_line_length = (panel_info.vl_col * NBITS (panel_info.vl_bpix)) / 8;

  printf("Little Brother Detected\n");
  printf("panel_type 0x%x, hw_ver %d, eeprom v%d.%d\n", vpi->panel_type, hw_ver, vpi->version_major, vpi->version_minor);
  if(vpi->lvds_type != 0x0ff)
  	printf("lvds_type %d\n", vpi->lvds_type);
  else
  	printf("lvds_type unknown\n");

  /* reset deassertion sequence */
  udelay(50*1000);
  rc = mv_mmp_panel_disp_ctrl(&i2c_info, vpi, PRIMARY_RESET_OUT, 1);
  if(!rc) {
     udelay(50*1000);
     rc = mv_mmp_panel_disp_ctrl(&i2c_info, vpi, SECONDARY_RESET_OUT, 1);
  }
  if(rc) {
        MV_MMPPANELDBG("%s:%d: communication error or no response deasserting reset\n",
								__func__, __LINE__);
	return rc;
  }

  rc = panel_lb_bl_init();

  if(rc) {
     MV_MMPPANELDBG("%s:%d : bl init failed\n", __func__, __LINE__);
     return rc;
  }

  vpd.backlight_enable = backlightEnable;

  return(0);
}

/*-----------------------------------------------------------------------------
 * Panel status message function removed.
 * Using mv_mmp_panel_vga_status() from mv_mmp_vga.c
 *---------------------------------------------------------------------------*/

unsigned long long mv_mmp_panel_lb_vga_buttons(void)
{
   unsigned int buttons;
   const key_map_t *map = mv_mmp_common_key_map;
   
   buttons = mv_mmp_panel_read_buttons(&i2c_info, map, 0, 0);

   return (long long) mv_mmp_panel_buttons_alias(buttons);
}

int mv_mmp_panel_lb_vga_type(void)
{
  return (int)MV_MMP_PANEL_TYPE_320X240;
}

char *mv_mmp_panel_lb_vga_subtype(void)
{
	/*
	 * As a placeholder, just keep it simple.
	 * TODO: build the subtype string in the probe function switch.
	 */
	vga_panel_info *vpi = (vga_panel_info *)&vpd.vpi;

	if(!hw_ver || (hw_ver == 0x0ff) || !mv_mmp_eeprom_version_ge(2, 2, vpi))
		return "TM024HDH49";

	if(vpd.use_te) {
		if(vpd.use_smpnclk)
			return "TM024HDH49_te_sclk";
		else
			return "TM024HDH49_te";
	}
	else {
		if(vpd.use_smpnclk)
			return "TM024HDH49_sclk";
		else
			return "TM024HDH49";
	}
}
